use strict;
use warnings;

package RT::Extension::SMSWebhook::Twilio;

our $VERSION = '0.01';

=head1 NAME

RT-Extension-SMSWebhook-Twilio - Add a webhook to receive SMS from Twilio

=head1 DESCRIPTION

A Twilio SMS number can be configured to call a webhook anytime it receives a
text.

This RT extension adds a webhook to your RT install to receive the webhook call
and either create a new ticket or add a comment to an existing ticket.

=head1 RT VERSION

Works with RT 5.

=head1 INSTALLATION

=over

=item C<perl Makefile.PL>

=item C<make>

=item C<make install>

May need root permissions

=item Edit your F</opt/rt5/etc/RT_SiteConfig.pm>

Add this line:

    Plugin('RT::Extension::SMSWebhook::Twilio');

To allow Twilio to send data to RT without a referrer:

    Set( %ReferrerComponents,
        '/SMSWebhook/Twilio.html' => 1,
    );

To define the interactions between RT and Twilio:

    Set(
        $SMSWebhookTwilio,
        {
            url   => 'https://YourRTServer.com/SMSWebhook/Twilio.html',
            token => 'twilio auth token',
            queue => 'General',
        }
    );

The C<url> value is the full url for the webhook, without the username and
password portion used for web server basic authorization as described next.

The C<token> is the Twilio auth token. See L<https://support.twilio.com/hc/en-us/articles/223136027-Auth-Tokens-and-How-to-Change-Them>
for details on finding your auth token.

The C<url> and C<token> values are used to validate the signature of the call
made to the webhook to ensure it is coming from Twilio. If the signature is
failing to validate make sure both of these values are correct.

The C<queue> value is an optional value to specify which queue new tickets
should be created in when receiving an SMS. It defaults to General if not
specified.

=item Configure your web server to require Basic Authorization for the
location $WebPath/SMSWebhook/Twilio.html.  This is I<very important> because
this plugin does not do any authentication and must be used with HTTP Basic
Authorization.

For example:

    <Location /SMSWebhook/Twilio.html>
        AuthType Basic
        AuthName "Twilio WebHook"
        AuthUserFile "/etc/apache2/twiliopass"
        Require valid-user
    </Location>

F</etc/apache2/twiliopass> could be generated by command C<htpasswd>:

    htpasswd -c /etc/apache2/twiliopass twilio

=item Clear your mason cache

    rm -rf /opt/rt5/var/mason_data/obj

=item Restart your webserver

Z<>

=item Set up a webhook in Twilio

See the video at L<https://www.twilio.com/docs/usage/webhooks/getting-started-twilio-webhooks>
to learn how to set a Messaging webhook for your Twilio phone number.

When setting the webhook url include the username and password you are using in
the web server basic authorization as described previously:

    https://username:password@YourRTServer.com/SMSWebhook/Twilio.html

=back

=head1 WEBHOOK LOGIC

When the Twilio webhook receives a call from Twilio it searches for a user with
a Pager number that matches the from number of the SMS.

If it does not find a user it creates a new one with the name 'SMS-###-###-####'
using the from number in place of the number signs. It then creates a new ticket
with the new user as the requestor.

If it does find a user it then looks for active tickets where that user has any
role on a ticket. If it finds one active ticket it adds a comment to that ticket
with the content of the SMS. If it finds zero or more than one active ticket it
creates a new ticket with the user as the requestor.

=head1 AUTHOR

Best Practical Solutions, LLC E<lt>modules@bestpractical.comE<gt>

=head1 BUGS

=for html <p>All bugs should be reported via email to <a
href="mailto:bug-RT-Extension-SMSWebhook-Twilio@rt.cpan.org">bug-RT-Extension-SMSWebhook-Twilio@rt.cpan.org</a>
or via the web at <a
href="http://rt.cpan.org/Public/Dist/Display.html?Name=RT-Extension-SMSWebhook-Twilio">rt.cpan.org</a>.</p>

=for text
    All bugs should be reported via email to
        bug-RT-Extension-SMSWebhook-Twilio@rt.cpan.org
    or via the web at
        http://rt.cpan.org/Public/Dist/Display.html?Name=RT-Extension-SMSWebhook-Twilio

=head1 LICENSE AND COPYRIGHT

This software is Copyright (c) 2022 by BPS

This is free software, licensed under:

  The GNU General Public License, Version 2, June 1991

=cut

1;
