/*
 * Copyright (C) 1997, 1998, 1999, 2000, 2002, 2003, 2004, 2005, 2006 Free
 * Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "config.h"

#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>

#include <gtk/gtk.h>
#include <gdk/gdkprivate.h>
#include <gdk/gdkx.h>

#include <libguile.h>
#include <guile/gh.h>
#include <libguile/dynl.h>
#include <libguile/tags.h>

#include "guile-gtk.h"
#include "compat.h"


/* Define this to enable some output during GC and other interesting
   actions. */
#undef DEBUG_PRINT

static void *
xmalloc (size_t sz)
{
  void *ptr = malloc (sz);
  if (ptr == NULL && sz != 0)
    scm_memory_error ("xmalloc");
  return ptr;
}

static void *
xrealloc (void *old, size_t sz)
{
  void *new = realloc (old, sz);
  if (new == NULL && sz != 0)
    scm_memory_error ("xrealloc");
  return new;
}


/* Miscellaneous helpers */

#if SCM_MAJOR_VERSION == 1 && SCM_MINOR_VERSION == 6
/* return a newly-malloced copy of STR, in space obtained from scm_malloc */
static char *
sgtk_strdup (const char *str)
{
  size_t size = strlen (str) + 1;
  return memcpy (scm_malloc (size), str, size);
}
#endif /* Guile 1.6 */

/* Return a newly-malloced C string which is the name of the keyword OBJ.
   An exception is thrown if OBJ is not a keyword.  */
static char *
sgtk_keyword_to_locale_string (SCM obj)
{
#if SCM_MAJOR_VERSION == 1 && SCM_MINOR_VERSION == 6
  /* A replacement scm_keyword_to_symbol for guile 1.6 would run fairly
     slowly, since in that version keywords use a "dash symbol", so
     scm_keyword_to_symbol would end up creating a new symbol every time.
     Hence the following direct implementation to get the name part.  */
  return sgtk_strdup (SCM_SYMBOL_CHARS (scm_keyword_dash_symbol (obj)) + 1);

#else /* Guile 1.8 */
  return scm_to_locale_string
    (scm_symbol_to_string (scm_keyword_to_symbol (obj)));
#endif
}

/* Return a newly-malloced C string which is the name of the symbol OBJ.
   An exception is thrown if OBJ is not a symbol.  */
static char *
sgtk_symbol_to_locale_string (SCM obj)
{
  return scm_to_locale_string (scm_symbol_to_string (obj));
}



/* C strings held in SCM objects.

   The idea here is that the contents of a string are picked out with
   scm_to_locale_string and then that malloced block is held in a cstr smob
   object.  This is done through the "convert" attribute in
   build-guile-gtk-1.2 (for string and cstring) and in sgtk_type_info (for
   GTK_TYPE_STRING).

   A smob object is a pretty easy way to get stuff freed when no longer in
   use.  That includes in C code like gdk_window_new_interp.

   The alternative would be frames with scm_dynwind_begin.  That might even
   be the best idea for the straightforward bits of code generated by
   build-guile-gtk-1.2.  But for a few special cases like gtk-signal-emit
   which do a callback to scheme code some care is needed to ensure that
   continuations can still be captured like they have been in the past.
   (There's always been restrictions on that stuff, but don't want to
   restrict it any further.)  In any case that's something to think about
   for the futures.

   ("dnywind" can be implemented well enough, mostly, as compatiblity code
   for Guile 1.6.  Under an error throw the cleanups might not be done
   immediately, but that doesn't much for memory frees.  Supporting nested
   dynwind frames is a problem though, that might have to be avoided.)  */

static const char cstr_name[] = "sgtk_cstr";
static scm_t_bits cstr_smob_type;
#define CSTR_PTR(cstr)  ((char *) SCM_SMOB_DATA(cstr))

int
sgtk_valid_cstr (SCM obj)
{
  return SCM_SMOB_PREDICATE (cstr_smob_type, obj);
}

static size_t
cstr_free (SCM obj)
{
  char *s = CSTR_PTR(obj);
  scm_gc_unregister_collectable_memory (s, strlen (s) + 1, cstr_name);
  free (s);
  return 0;
}

SCM
sgtk_to_cstr (SCM obj)
{
  char *s;

  /* convert comes first, on any type object, type check is later in
     sgtk_cstr2ptr or sgtk_valid_cstr */
  if (scm_is_string (obj))
    {
      s = scm_to_locale_string (obj);
      scm_gc_register_collectable_memory (s, strlen (s) + 1, cstr_name);
      SCM_RETURN_NEWSMOB (cstr_smob_type, (scm_t_bits) s);
    }

  return obj;
}

char *
sgtk_cstr2ptr (SCM obj, unsigned long pos, const char *func_name)
{
  SCM_ASSERT_TYPE (SCM_SMOB_PREDICATE (cstr_smob_type, obj), obj, pos,
                   func_name, "string");
  return CSTR_PTR (obj);
}



/* C memory blocks held in SCM objects. */

static const char cblk_name[] = "sgtk_cblk";
static scm_t_bits cblk_smob_type;
#define CBLK_PTR(cblk)  ((char *) SCM_SMOB_DATA(cblk))
#define CBLK_LEN(cblk)  ((size_t) SCM_SMOB_DATA_2(cblk))

static size_t
cblk_free (SCM obj)
{
  void *p = CBLK_PTR(obj);
  scm_gc_unregister_collectable_memory (p, CBLK_LEN(obj), cblk_name);
  free (p);
  return 0;
}

SCM
sgtk_make_cblk (void *p, size_t len)
{
  scm_gc_register_collectable_memory (p, len, cblk_name);
  SCM_RETURN_NEWSMOB2 (cblk_smob_type, (scm_t_bits) p, (scm_t_bits) len);
}



/* Associating SCM values with Gtk pointers.

   We keep a hash table that can store a SCM value for an arbitray
   gpointer.  This is used for the proxies of GtkObjects and the boxed
   types.  */

static GHashTable *proxy_tab;

static guint
gpointer_hash (gpointer a)
{
  return (guint)a;
}

static gint
gpointer_compare (gpointer a, gpointer b)
{
  return a == b;
}

static void
enter_proxy (gpointer obj, SCM proxy)
{
  if (proxy_tab == NULL)
    proxy_tab = g_hash_table_new ((GHashFunc)gpointer_hash,
				  (GCompareFunc)gpointer_compare);
  g_hash_table_insert (proxy_tab, obj, (gpointer)proxy);
}

static SCM
get_proxy (gpointer obj)
{
  if (proxy_tab)
    {
      gpointer val = g_hash_table_lookup (proxy_tab, obj);
      return val? (SCM) val : SCM_BOOL_F;
    }
  return SCM_BOOL_F;
}

static void
forget_proxy (gpointer obj)
{
  g_hash_table_remove (proxy_tab, obj);
}



/* Storing additional info about a GtkType.

   type_info_tab is an array of pointers to sgtk_type_info records, or
   records like sgtk_boxed_info which have sgtk_type_info as an initial
   header.  The array is grown dynamically when necessary.

   Back in Gtk 1.2 each GtkType had a unique sequence number which was used
   as an index into the array.  This is no longer so, we now just search
   looking at type_info_tab[i]->type for a desired GtkType.  */

#define TYPE_INFO_INCR_MASK 0xFF

static sgtk_type_info **type_info_tab;
static guint n_type_info_tab = 0;

static int
boxed_info_to_seqno (const sgtk_boxed_info *info)
{
  guint i;
  for (i = 0; i < n_type_info_tab; i++)
    if (type_info_tab[i] == &info->header)
      return i;
  printf ("info not found %p\n", info);
  abort();
}

static void
enter_type_info (sgtk_type_info *info)
{
  type_info_tab = (sgtk_type_info **)
    xrealloc ((char *)type_info_tab,
              sizeof(sgtk_type_info*) * (n_type_info_tab + 1));
  type_info_tab[n_type_info_tab] = info;
  n_type_info_tab++;
}

sgtk_type_info*
sgtk_get_type_info (guint seqno)
{
  if (seqno >= n_type_info_tab)
    return NULL;
  return type_info_tab[seqno];
}

static sgtk_type_info*
must_get_type_info (guint seqno)
{
  sgtk_type_info *info = sgtk_get_type_info (seqno);
  if (info == NULL)
    abort ();
  return info;
}

typedef struct _type_infos {
  struct _type_infos *next;
  sgtk_type_info **infos;
} type_infos;

static type_infos *all_type_infos;

/* Find types that are mentioned in our *.defs files but are not
   provided by the Gtk run-time system.  This is only used
   occasionally to update the table in sgtk_try_missing_type.  */
#ifdef NEED_UNUSED_CODE
static void
sgtk_find_missing_types (type_infos *infos)
{
  sgtk_type_info **ip;
  for (ip = infos->infos; *ip; ip++)
    {
      if (gtk_type_from_name ((*ip)->name) == GTK_TYPE_INVALID
	  && (*ip)->type != GTK_TYPE_OBJECT)
	printf ("missing: %s, %s\n",
		(*ip)->name, gtk_type_name ((*ip)->type));
    }
}
#endif

void
sgtk_register_type_infos (sgtk_type_info **infos)
{
  type_infos *t;

  sgtk_init ();

  t = (type_infos *) xmalloc (sizeof(type_infos));
  t->infos = infos;
  t->next = all_type_infos;
  all_type_infos = t;

#if 0
  sgtk_find_missing_types (t);
#endif
}

static gpointer
dummy_boxed_copy (gpointer boxed)
{
  fprintf (stderr, "guile-gtk: oops, boxed_copy not supported\n");
  abort ();
}
static void
dummy_boxed_free (gpointer boxed)
{
  fprintf (stderr, "guile-gtk: oops, boxed_free not supported\n");
  abort ();
}

/* This was "gtk_type_unique (GTK_TYPE_BOXED, *t)" in Gtk 1.2, but in Gtk 2
   gtk_type-unique will only create subtypes of GtkObject */
void
sgtk_register_type_infos_gtk (GtkTypeInfo **infos)
{
  GtkTypeInfo **t;

  for (t = infos; t && *t; t++)
    g_boxed_type_register_static ((*t)->type_name,
                                  dummy_boxed_copy, dummy_boxed_free);
}

/* When INFO refers to one of the known `missing' types, we initialize
   that type ourselves.  This is used to fix certain discrepancies
   between old Gtk versions and our *.defs files.  It is not OK to do
   this in general because we should not assume that we can safely
   initialize types from other modules.

   In Gtk 2 this is now unused, and might well be able to be deleted
   entirely.  */

static GtkType
sgtk_try_missing_type (char *name)
{
  static sgtk_type_info missing[] = {
    {NULL, GTK_TYPE_NONE, NULL}
  };

  sgtk_type_info *m;
  for (m = missing; m->name; m++)
    if (!strcmp (m->name, name))
      {
	GtkTypeInfo info = { NULL };
	info.type_name = name;
	return gtk_type_unique (m->type, &info);
      }

  return GTK_TYPE_INVALID;
}

static int
sgtk_fillin_type_info (sgtk_type_info *info)
{
  if (info->type != GTK_TYPE_OBJECT
      && info->type == GTK_FUNDAMENTAL_TYPE (info->type)
      && info->type != GTK_TYPE_INVALID)
    {
      GtkType parent_type = info->type;
      GtkType this_type = sgtk_type_from_name (info->name);
      if (this_type == GTK_TYPE_INVALID)
	this_type = sgtk_try_missing_type (info->name);
      if (this_type == GTK_TYPE_INVALID)
	{
	  fprintf (stderr, "unknown type `%s'.\n", info->name);
	  return 0;
	}
      info->type = this_type;
      if (GTK_FUNDAMENTAL_TYPE (info->type) != parent_type)
	{
	  fprintf (stderr, "mismatch for type `%s'.\n", info->name);
	  info->type = GTK_TYPE_INVALID;
	  return 0;
	}
      enter_type_info (info);
    }

  return 1;
}      
     
sgtk_type_info*
sgtk_maybe_find_type_info (GtkType type)
{
  sgtk_type_info *info;
  type_infos *infos;
  char *name;
  int i;

  for (i = 0; i < n_type_info_tab; i++)
    if (type_info_tab[i]->type == type)
      return type_info_tab[i];

  /* XXX - merge this with the GObject code.  I don't have the brain
     right now to do it. */

  name = gtk_type_name (type);
  for (infos = all_type_infos; infos; infos = infos->next)
    {
      sgtk_type_info **ip;
      for (ip = infos->infos; *ip; ip++)
	if (!strcmp ((*ip)->name, name))
	  {
	    if (GTK_FUNDAMENTAL_TYPE (type) != (*ip)->type)
	      {
		fprintf (stderr, "mismatch for type `%s' -- %ld %ld.\n",
                         name,
                         GTK_FUNDAMENTAL_TYPE (type),
                         (*ip)->type);
		info->type = GTK_TYPE_INVALID;
		abort ();
	      }
	    (*ip)->type = type;
	    enter_type_info (*ip);
	    return *ip;
	  }
    }

  /* XXX - should use the Gtk+ type introspection here instead of
     giving up. */

  return NULL;
}

sgtk_type_info *
sgtk_find_type_info (GtkType type)
{
  sgtk_type_info *info = sgtk_maybe_find_type_info (type);

  if (info)
    return info;

  fprintf (stderr, "unknown type `%s'.\n", gtk_type_name (type));
  abort ();
}

static SCM
sgtk_type_name (GtkType type)
{
  /* type name can be NULL */
  const char *name = gtk_type_name (type);
  if (name == NULL)
    return SCM_BOOL_F;
  else
    return scm_from_locale_string (name);
}

/* GObjects.

   GObjects are wrapped with a smob.  The smob of a GObject is
   called its proxy.  The proxy and its GObject are strongly
   connected; that is, the GObject will stay around as long as the
   proxy is referenced from Scheme, and the proxy will not be
   collected as long as the GObject is used from outside of Scheme.

   The lifetime of GObjects is controlled by a reference count,
   while Scheme objects are managed by a tracing garbage collector
   (mark/sweep).  These two techniques are made to cooperate like
   this: the pointer from the proxy to the GObject is reflected in
   the reference count of the GObject.  All proxies are kept in a
   list and those that point to GObjects with a reference count
   greater than the number of `internal' references are marked during
   the marking phase of the tracing collector.  An internal reference
   is one that goes from a GObject with a proxy to another GObject
   with a proxy.  We can only find a subset of the true internal
   references (because Gtk does not yet cooperate), but this should be
   good enough.

   By using this combination of tracing and reference counting it is
   possible to break the cycle that is formed by the proxy pointing to
   the GObject and the GObject pointing back.  It is
   straightforward to extend this to other kind of cycles that might
   occur.  For example, when connecting a Scheme procedure as a signal
   handler, the procedure is very likely to have the GObject that it
   is connected to in its environment.  This cycle can be broken by
   including the procedure in the set of Scheme objects that get
   marked when we are tracing GObjects with a reference count
   greater than the number of internal references.

   Therefore, each proxy contains a list of `protects' that are marked
   when the proxy itself is marked.  In addition to this, there is
   also a global list of `protects' that is used for Scheme objects
   that are somewhere in Gtk land but not clearly associated with a
   particular GObject (like timeout callbacks).

  */

struct sgtk_protshell {
  SCM object;
  struct sgtk_protshell *next;
  struct sgtk_protshell **prevp;
};

static GMemChunk *sgtk_protshell_chunk;

/* Analogous to the PROTECTS list of a proxy but for SCM values that
   are not associated with a particular GObject. */

static struct sgtk_protshell *global_protects;

void
sgtk_unprotect (sgtk_protshell *prot)
{
  if ((*prot->prevp = prot->next))
    prot->next->prevp = prot->prevp;
  g_chunk_free (prot, sgtk_protshell_chunk);
}

static void
sgtk_mark_protects (sgtk_protshell *prots)
{
  while (prots)
    {
      scm_gc_mark (prots->object);
      prots = prots->next;
    }
}

/* The CDR of a GObject smob points to one of these.  PROTECTS is a
   Scheme list of all SCM values that need to be protected from the GC
   because they are in use by OBJ.  PROTECTS includes the smob cell
   itself.  NEXT and PREVP are used to chain all proxies together for
   the marking mentioned above.  NEXT simply points to the next proxy
   struct and PREVP points to the pointer that points to us.  */

typedef struct _sgtk_object_proxy {
  /*
    FIXME: Maybe what we want is to have a GObject instead of GObject
  */

  GObject *obj;
  struct sgtk_protshell *protects;
  int traced_refs;
  struct _sgtk_object_proxy *next;
  struct _sgtk_object_proxy **prevp;
} sgtk_object_proxy;

/* The list of all existing proxies. */

static sgtk_object_proxy *all_proxies = NULL;

/* Insert the list of protshells starting at PROTS into the global
   protects list.  This is used when a proxy is freed so that we don't
   forget about its protects. */

static void
sgtk_move_prots_to_global (sgtk_protshell *prots)
{
  if (prots)
    {
      sgtk_protshell *g = global_protects;
      global_protects = prots;
      global_protects->prevp = &global_protects;
      if (g)
	{
	  sgtk_protshell *p;
	  for (p = prots; p->next; p = p->next)
	    ;
	  p->next = g;
	  g->prevp = &p->next;
	}
    }
}

#if 0
static int
sgtk_check_protshell (sgtk_protshell *prot)
{
  sgtk_object_proxy *proxy;
  sgtk_protshell *walk;

  for (proxy = all_proxies; proxy; proxy = proxy->next)
    for (walk = proxy->protects; walk; walk = walk->next)
      if (walk == prot)
	return 1;
  for (walk = global_protects; walk; walk = walk->next)
    if (walk == prot)
      return 1;

  fprintf (stderr, "unknown protshell %p\n", prot);
  return 0;
}
#endif

/* The smob for GObjects.  */

static long tc16_gtkobj;

#define GTKOBJP(x)       (SCM_SMOB_PREDICATE(tc16_gtkobj, x))
#define GTKOBJ_PROXY(x)  ((sgtk_object_proxy *)SCM_SMOB_DATA(x))

sgtk_protshell *
sgtk_protect (SCM protector, SCM obj)
{
  sgtk_protshell *prot = g_chunk_new (sgtk_protshell, sgtk_protshell_chunk);
  sgtk_protshell **prevp;

  prot->object = obj;

  if (GTKOBJP (protector))
    prevp = &(GTKOBJ_PROXY(protector)->protects);
  else
    prevp = &global_protects;
  
  if ((prot->next = *prevp))
	prot->next->prevp = &prot->next;
  *prevp = prot;
  prot->prevp = prevp;

  return prot;
}

static void
mark_traced_ref (GtkWidget *obj, void *data)
{
  SCM p = (SCM)get_proxy (obj);
  if (! scm_is_false (p))
    {
      sgtk_object_proxy *proxy = GTKOBJ_PROXY (p);
#ifdef DEBUG_PRINT
      fprintf (stderr, "marking trace %p %s\n",
	       proxy->obj, gtk_type_name (GTK_OBJECT_TYPE (proxy->obj)));
#endif
      sgtk_mark_protects (proxy->protects);
    }
}

static SCM
gtkobj_mark (SCM obj)
{
  sgtk_object_proxy *proxy = GTKOBJ_PROXY(obj);

#ifdef DEBUG_PRINT
  fprintf (stderr, "marking %p %s\n",
	   proxy->obj, gtk_type_name (GTK_OBJECT_TYPE (proxy->obj)));
#endif

  if (GTK_IS_CONTAINER (proxy->obj))
    gtk_container_foreach (GTK_CONTAINER(proxy->obj), mark_traced_ref, NULL);
  sgtk_mark_protects (proxy->protects);
  return SCM_EOL;
}

static int
gtkobj_print (SCM obj, SCM port, scm_print_state *pstate)
{
  sgtk_object_proxy *proxy = GTKOBJ_PROXY (obj);
  GtkType tid = GTK_OBJECT_TYPE (proxy->obj);

  scm_puts ("#<", port);
  scm_puts (gtk_type_name (tid), port);
  scm_puts (" ", port);
  scm_intprint ((long)proxy->obj, 16, port);
  scm_puts (">", port);
  return 1;
}

static size_t
gtkobj_free (SCM obj)
{
  sgtk_object_proxy *proxy = GTKOBJ_PROXY (obj);

  // fprintf (stderr, "freeing proxy %p\n", proxy);

#ifdef DEBUG_PRINT
  fprintf (stderr, "freeing %p %s\n",
	   proxy->obj, gtk_type_name (GTK_OBJECT_TYPE (proxy->obj)));
#endif

  forget_proxy (proxy->obj);
  g_object_unref (proxy->obj);
  if ((*proxy->prevp = proxy->next)) proxy->next->prevp = proxy->prevp;

  assert (proxy->protects && scm_is_eq (proxy->protects->object, obj));
  sgtk_move_prots_to_global (proxy->protects->next);

  scm_gc_free (proxy, sizeof(sgtk_object_proxy), "GtkObject proxy");
  return 0;
}

/* Treating GObject proxies right during GC.  We need to run custom
   code during the mark phase of the Scheme GC.  We do this by
   creating a new smob type and allocating one actual smob of it.
   This smob is made permanent and thus its marking function is
   invoked for every GC.  We hijack this function to do the tracing of
   all existing proxies as well. */

static long tc16_gtkobj_marker_hook;

static void
count_traced_ref (GtkWidget *obj, void *data)
{
  SCM p = (SCM)get_proxy (obj);
  if (p != SCM_BOOL_F)
    {
      sgtk_object_proxy *proxy = GTKOBJ_PROXY (p);
#ifdef DEBUG_PRINT
      fprintf (stderr, "counting %p %s\n",
	       proxy->obj, gtk_type_name (GTK_OBJECT_TYPE (proxy->obj)));
#endif
      proxy->traced_refs++;
    }
}

static SCM
gtkobj_marker_hook (SCM obj)
{
  sgtk_object_proxy *proxy;

  /* We do two passes here.  The first pass counts how many references
     an object has from other objects that have a proxy.  The second
     pass marks all objects that have more than this number of
     references.  For the first pass to work, we need to enumerate all
     references that an object has to other objects.  We can't do that
     precisely without help from Gtk+ itself.  But luckily, *not*
     knowing about an `internal' reference is the conservative thing.
     Missing a reference will make it appear to us that an object has
     more `external' references to it than it really has, thus making
     us keep the proxy alive.  Only when these `external' references
     form a cycle over some Scheme values, we loose.  As a first
     approximation to the true set of references of a GObject, we
     just traverse its children with gtk_container_foreach.  */

  /* First pass. */
  for (proxy = all_proxies; proxy; proxy = proxy->next)
    {
      GObject *obj = proxy->obj;
#ifdef DEBUG_PRINT
      fprintf (stderr, "on %p %p\n", proxy, obj);
#endif
      if (GTK_IS_CONTAINER (obj))
	gtk_container_foreach (GTK_CONTAINER(obj), count_traced_ref, NULL);
    }
#ifdef DEBUG_PRINT
  fprintf (stderr, "done with pass 1.\n");
#endif

  /* Second pass. */
  for (proxy = all_proxies; proxy; proxy = proxy->next)
    {
      /* 
	 FIXME: proxy struct may need to be changed!
      */
/*       if (proxy->obj->parent_instance.ref_count > proxy->traced_refs + 1) */
      if (proxy->obj->ref_count > proxy->traced_refs + 1)
	{
#ifdef DEBUG_PRINT
	  fprintf (stderr, "hooking %p %s\n",
		   proxy->obj, gtk_type_name (GTK_OBJECT_TYPE (proxy->obj)));
#endif
	  sgtk_mark_protects (proxy->protects);
	}
      proxy->traced_refs = 0;
    }
  sgtk_mark_protects (global_protects);
  return SCM_EOL;
}

static int
gtkobj_marker_hook_print (SCM obj, SCM port, scm_print_state *pstate)
{
  scm_puts ("#<the invisible GObject marker hook>", port);
  return 1;
}

static void
install_marker_hook ()
{
  scm_permanent_object (scm_cell (tc16_gtkobj_marker_hook, 0));
}

/* Create a proxy for OBJ.  There are three cases for reference counting,

   1. Ordinary pointer return.  Gtk doesn't change the ref count.
      Eg. gtk_bin_get_child.  We must g_object_ref for our reference in the
      proxy.

   2. Newly created object in a subclass of GInitiallyUnowned, being
      everything under GtkObject.  Gtk starts with a so-called "floating"
      reference count of 1.  We want to convert that to a proper ref for our
      reference in the proxy.  It's assumed any floating ref is a newly
      created object we're acquiring (ie. not case 1 above).

   3. Newly created object in some other class.  Gtk starts with a reference
      count of 1.  That becomes our reference in the proxy, ie. no further
      change to the count.

   The use of g_object_ref_sink covers cases 1 and 2.  Case 3 must be
   distinguished by "(copy #f)" on relevant creator functions in the .defs
   files, eg. gdk_gc_new.  */

static SCM
make_gobject (GObject *obj, int add_ref)
{
  sgtk_object_proxy *proxy;
  SCM z;

  proxy = (sgtk_object_proxy *) scm_gc_malloc (sizeof(sgtk_object_proxy),
                                               "GObject proxy");
  if (add_ref)
    g_object_ref_sink (obj);
#ifdef DEBUG_PRINT
  fprintf (stderr, "New proxy %p for %p %s\n", proxy, obj,
	   gtk_type_name (GTK_OBJECT_TYPE (obj)));
#endif
  proxy->obj = obj;
  proxy->protects = NULL;
  proxy->traced_refs = 0;
  proxy->next = all_proxies;
  all_proxies = proxy;
  proxy->prevp = &all_proxies;
  if (proxy->next)
    proxy->next->prevp = &proxy->next;

  z = scm_cell (tc16_gtkobj, (scm_t_bits)proxy);
  enter_proxy (obj, z);

  sgtk_protect (z, z); /* this one is never removed. */

  return z;
}

/* Return the proxy for OBJ if it already has one, else create a new
   one.  When OBJ is NULL, return `#f'. */

SCM
sgtk_wrap_gtkobj (GObject *obj)
{
  SCM handle;

  if (obj == NULL)
    return SCM_BOOL_F;

  handle = get_proxy (obj);
  if (scm_is_false (handle))
    handle = make_gobject (obj, 1);
  return handle;
}

SCM
sgtk_wrap_gtkobj_nocopy (GObject *obj)
{
  SCM handle;

  if (obj == NULL)
    return SCM_BOOL_F;

  handle = get_proxy (obj);
  if (scm_is_false (handle))
    handle = make_gobject (obj, 0);
  return handle;
}

int
sgtk_is_a_gtkobj (GType type, SCM obj)
{

  if (!(SCM_NIMP (obj) && GTKOBJP (obj)))
    return 0;
  return gtk_type_is_a (GTK_OBJECT_TYPE(GTKOBJ_PROXY(obj)->obj), type);
}

GObject*
sgtk_get_gtkobj (SCM obj)
{
  if (scm_is_false (obj))
    return NULL;
  else
    return GTKOBJ_PROXY(obj)->obj;
}

/* Enums.

   Enumerations are described by a `sgtk_enum_info' structure.  That
   structure contains a list of all literals and their respective
   values.  In Scheme, an enum element is represented by a symbol
   whose name is the literal. */

SCM sgtk_flags_symbol_protector = SCM_BOOL_F;

static int
sgtk_flags_comp (const void *first, const void *second)
{
  if (SCM_UNPACK (((sgtk_enum_literal *) first)->symbol)
      > SCM_UNPACK (((sgtk_enum_literal *) second)->symbol))
    return 1;
  else if (SCM_UNPACK (((sgtk_enum_literal *) first)->symbol)
           < SCM_UNPACK (((sgtk_enum_literal *) second)->symbol))
    return -1;
  else
    return 0;
}

void 
sgtk_enum_flags_init (sgtk_enum_info *info)
{
  int	i;
  SCM	s;

  if (scm_is_false (sgtk_flags_symbol_protector))
    {
      sgtk_flags_symbol_protector = scm_cons (SCM_BOOL_F, SCM_EOL);
      scm_gc_protect_object (sgtk_flags_symbol_protector);
    }

  for (i = 0; i < info->n_literals; i++)
    {
      info->literals[i].symbol = scm_from_locale_symbol (info->literals[i].name);

      s = scm_cons (info->literals[i].symbol, 
		    SCM_CDR (sgtk_flags_symbol_protector));
      SCM_SETCDR (sgtk_flags_symbol_protector, s);
    }

  qsort (info->literals, info->n_literals, sizeof (sgtk_enum_literal), sgtk_flags_comp);
}

int
sgtk_enum_flags_bin_search (SCM key, sgtk_enum_info *info, int *rval)
{
  int			upper, lower, half;
  sgtk_enum_literal	*ls;
  
  ls = info->literals;

  upper = info->n_literals - 1;
  lower = 0;

  while (upper >= lower)
    {
      half = (upper + lower) >> 1;
      if (key > ls[half].symbol)
	lower = half + 1;
      else
	if (key == ls[half].symbol)
	  { 
	    *rval = ls[half].value; 
	    return TRUE; 
	  }
	else
	  upper = half - 1;
    } 

  *rval = -1;
  return FALSE;
}

SCM
sgtk_enum2scm (gint val, sgtk_enum_info *info)
{
  int i;
  for (i = 0; i < info->n_literals; i++)
    if (info->literals[i].value == val)
      return info->literals[i].symbol;
  SCM_ASSERT (0, scm_from_int (val), SCM_ARG1, "enum->symbol");
  return SCM_BOOL_F;
}

gint
sgtk_scm2enum (SCM obj, sgtk_enum_info *info, int pos, char *sname)
{
  int rval;

  if (scm_is_symbol (obj) &&
      (sgtk_enum_flags_bin_search (obj, info, &rval) == TRUE))
    return rval;

  /* if obj is not integer (or it is not correct symbol)
   * scm_num2long throws an exception for us 
   */
  return scm_num2int (obj, (long) pos, sname);
}

gint
sgtk_valid_enum (SCM obj, sgtk_enum_info *info)
{
  int tmp;

  if (scm_is_symbol (obj))
    return sgtk_enum_flags_bin_search (obj, info, &tmp);

  return scm_is_signed_integer (obj, INT_MIN, INT_MAX);
}

/* Flags.

   Like enums, flags are described by a `sgtk_enum_info' structure.
   In Scheme, flags are represented by a list of symbols, one for each
   bit that is set in the flags value. */

/* The test in sgtk_flags2scm is for all the bits of info->literals[i].value
   appearing in the given "val" so that GDK_ALL_EVENTS_MASK, which has lots
   of bits, is not returned for some lesser set of bits.

   FIXME: There's still a problem here; if "val" is in fact
   GDK_ALL_EVENTS_MASK then sometimes the return will have each flag bit
   individually, ie. "(exposure-mask pointer-motion-mask ...)", or sometimes
   you get "(all-events-mask)".  It depends whether all-events if first in
   the info->literals array or not.  That array is sorted by SCM pointer
   value, so it's basically a lottery as to where the addresses fall in a
   given guile run.  Probably need something where all-events-mask is
   recognised for input (ie. sgtk_scm2flags), but not used for output
   (ie. sgtk_flags2scm).  */

SCM
sgtk_flags2scm (gint val, sgtk_enum_info *info)
{
  SCM ans = SCM_EOL;
  int i;
  for (i = 0; i < info->n_literals; i++)
    if ((val & info->literals[i].value) == info->literals [i].value)
      {
	ans = scm_cons (info->literals[i].symbol, ans);
	val &= ~info->literals[i].value;
      }
  return ans;
}

gint
sgtk_scm2flags (SCM obj, sgtk_enum_info *info, int pos, char *sname)
{
  if (scm_is_pair (obj) || scm_is_null (obj))
    {
      int ans = 0, m;
      while (scm_is_pair (obj))
        {
          SCM sym = SCM_CAR (obj);
          if (scm_is_symbol (sym))
            {
              if (sgtk_enum_flags_bin_search (sym, info, &m) == FALSE)
                break;
            }
          else
            m = scm_num2int (sym, (long) pos, sname);

          ans |= m;
          obj = SCM_CDR (obj);
        }
      if (! scm_is_null (obj))
        SCM_ASSERT (0, obj, pos, sname);
      return ans;
    }
  else
    {
      return scm_num2int (obj, (long) pos, sname);
    }
}

gint
sgtk_valid_flags (SCM obj, sgtk_enum_info *info)
{
  int tmp;

  /* FIXME: should be "fits an `int'" here, actually, but this function is
     presently unused. */

  /* an integer */
  if (scm_is_integer (obj))
    return TRUE;

  /* or a list of integers and known symbols */
  for ( ; scm_is_pair (obj); obj = SCM_CDR (obj))
    {
      SCM sym = SCM_CAR (obj);

      if (scm_is_symbol (sym))
	{
	  if (sgtk_enum_flags_bin_search (sym, info, &tmp) == FALSE)
	    return FALSE;
	}
      else
	if (! scm_is_integer (sym))
	  return FALSE;
    }
  if (! scm_is_null (obj))
    return FALSE;

  return TRUE;
}

/* String enums.

   A string enum is like an enum, but the values are strings.  The
   range of values can be extended, so anywhere a "string enum" value
   is accepted, we also accept a string (but not a symbol).  */

int
sgtk_valid_senum (SCM obj, sgtk_senum_info *info)
{
  int i;
  char *name;

  if (scm_is_string (obj))
    return 1;
  if (! scm_is_symbol (obj))
    return 0;

  name = sgtk_symbol_to_locale_string (obj);
  for (i = 0; i < info->n_literals; i++)
    if (strcmp (info->literals[i].name, name) == 0)
      {
        free (name);
        return 1;
      }

  free (name);
  return 0;
}

SCM
sgtk_senum2scm (char *val, sgtk_senum_info *info)
{
  int i;
  for (i = 0; i < info->n_literals; i++)
    if (! strcmp (info->literals[i].value, val))
      return scm_from_locale_symbol (info->literals[i].name);
  return scm_makfrom0str (val);
}

char *
sgtk_scm2senum (SCM obj, sgtk_senum_info *info)
{
  int i;
  char *name;

  if (scm_is_string (obj))
    {
      SCM_STRING_COERCE_0TERMINATION_X (obj);
      return SCM_STRING_CHARS (obj);
    }

  name = sgtk_symbol_to_locale_string (obj);
  for (i = 0; i < info->n_literals; i++)
    if (strcmp (info->literals[i].name, name) == 0)
      {
        free (name);
        return info->literals[i].value;
      }

  free (name);
  return NULL;
}

/* Boxed Values.

   A boxed value is a pointer, held in a scheme cell.  Cell word 1 is the
   pointer and cell word 0 is the type and also a BOXED_SEQNO.

   The seqno is an index into the type_info_tab[] array of sgtk_boxed_info
   pointers.  For example for a GdkFont it's a pointer to the
   sgtk_gdk_font_info record (created by build-guile-gtk-2.0).  That record
   has stuff like conversion and free functions, and a size.
 */

static long tc16_boxed;

#define BOXED_P(x)     (SCM_NIMP(x) && (SCM_TYP16(x) == tc16_boxed))
#define BOXED_SEQNO(x) (((guint)SCM_CAR(x))>>16)
#define BOXED_PTR(x)   ((gpointer)SCM_CDR(x))
#define BOXED_INFO(x)  ((sgtk_boxed_info*)must_get_type_info(BOXED_SEQNO(x)))
#define BOXED_SET_PTR(x,d) SCM_SETCDR(x,d)

static size_t
boxed_free (SCM obj)
{
  sgtk_boxed_info *info = BOXED_INFO (obj);
  scm_gc_unregister_collectable_memory (BOXED_PTR (obj), info->size,
					"GtkBoxed");
  info->destroy (BOXED_PTR (obj));
  return 0;
}

static int
boxed_print (SCM exp, SCM port, scm_print_state *pstate)
{
  sgtk_boxed_info *info = BOXED_INFO (exp);
  scm_puts ("#<", port);
  scm_puts (info->header.name, port);
  scm_puts (" ", port);
  if (BOXED_PTR (exp) == NULL)
    scm_puts ("Invalidated", port);
  else
    scm_intprint ((long)BOXED_PTR (exp), 16, port);
  scm_puts (">", port);
  return 1;
}

SCM
sgtk_boxed2scm (gpointer ptr, sgtk_boxed_info *info, int copyp)
{
  SCM z;

  if (ptr == NULL)
    return SCM_BOOL_F;

  if (!sgtk_fillin_type_info (&info->header))
    return SCM_BOOL_F;

  if (copyp && info->copy)
    ptr = info->copy (ptr);

  scm_gc_register_collectable_memory (ptr, info->size, "GtkBoxed");
  z = scm_cell (tc16_boxed | boxed_info_to_seqno (info) << 16,
                (scm_t_bits) ptr);
  return z;
}

void *
sgtk_scm2boxed (SCM obj)
{
  if (scm_is_false (obj))
    return NULL;
  return BOXED_PTR (obj);
}

int
sgtk_valid_boxed (SCM obj, sgtk_boxed_info *info)
{
  return (BOXED_P (obj) && BOXED_PTR (obj) != NULL && 
	  BOXED_INFO (obj) == info);
}

void
sgtk_boxed_invalidate (SCM obj)
{ BOXED_SET_PTR (obj, NULL); }

int
sgtk_valid_point (SCM obj)
{
  return scm_is_pair (obj)
    && scm_is_signed_integer (SCM_CAR (obj), INT_MIN, INT_MAX)
    && scm_is_signed_integer (SCM_CDR (obj), INT_MIN, INT_MAX);
}

GdkPoint
sgtk_scm2point (SCM obj)
{
  GdkPoint res;
  res.x = scm_num2int (SCM_CAR (obj), 1, "scheme->point");
  res.y = scm_num2int (SCM_CDR (obj), 1, "scheme->point");
  return res;
}

SCM
sgtk_point2scm (GdkPoint p)
{
  return scm_cons (scm_from_int (p.x),
		   scm_from_int (p.y));
}

int
sgtk_valid_rect (SCM obj)
{
  return scm_is_pair (obj)
    && sgtk_valid_point (SCM_CAR (obj))
    && sgtk_valid_point (SCM_CDR (obj));
}

GdkRectangle
sgtk_scm2rect (SCM obj)
{
  GdkRectangle res;
  res.x = scm_num2int (SCM_CAAR (obj), 1, "scheme->rectangle");
  res.y = scm_num2int (SCM_CDAR (obj), 1, "scheme->rectangle");
  res.width = scm_num2int (SCM_CADR (obj), 1, "scheme->rectangle");
  res.height = scm_num2int (SCM_CDDR (obj), 1, "scheme->rectangle");
  return res;
}

struct sgtk_rectangle
sgtk_scm2rect_null_ok (SCM obj)
{
  struct sgtk_rectangle res;
  res.null = scm_is_false (obj);
  if (! res.null)
    {
      res.r.x = scm_num2int (SCM_CAAR (obj), 1, "scheme->rectangle");
      res.r.y = scm_num2int (SCM_CDAR (obj), 1, "scheme->rectangle");
      res.r.width = scm_num2int (SCM_CADR (obj), 1, "scheme->rectangle");
      res.r.height = scm_num2int (SCM_CDDR (obj), 1, "scheme->rectangle");
    }
  return res;
}

SCM
sgtk_rect2scm (GdkRectangle r)
{
  return scm_cons (scm_cons (scm_from_int (r.x),
			     scm_from_int (r.y)),
		   scm_cons (scm_from_int (r.width),
			     scm_from_int (r.height)));
}

int
sgtk_valid_segment (SCM obj)
{
  return scm_is_pair (obj)
    && sgtk_valid_point (SCM_CAR (obj))
    && sgtk_valid_point (SCM_CDR (obj));
}

GdkSegment
sgtk_scm2segment (SCM obj)
{
  GdkSegment seg;
  seg.x1 = scm_num2int (SCM_CAAR (obj), 1, "scheme->segment");
  seg.y1 = scm_num2int (SCM_CDAR (obj), 1, "scheme->segment");
  seg.x2 = scm_num2int (SCM_CADR (obj), 1, "scheme->segment");
  seg.y2 = scm_num2int (SCM_CDDR (obj), 1, "scheme->segment");
  return seg;
}

SCM
sgtk_segment2scm (GdkSegment seg)
{
  return scm_cons (scm_cons (scm_from_int (seg.x1),
			     scm_from_int (seg.y1)),
		   scm_cons (scm_from_int (seg.x2),
			     scm_from_int (seg.y2)));
}

GdkAtom
sgtk_scm2atom (SCM symbol)
{
  char *name = sgtk_symbol_to_locale_string (symbol);
  GdkAtom ret = gdk_atom_intern (name, FALSE);
  free (name);
  return ret;
}

SCM
sgtk_atom2scm (GdkAtom atom)
{
  char *name = gdk_atom_name (atom);
  if (name == NULL)
    return SCM_BOOL_F;
  else
    return scm_take_locale_symbol (name);
}

SCM_SYMBOL (sym_gnome_file, "gnome-file");

int
sgtk_port2fileno (SCM port)
{
  return SCM_FSTREAM(port)->fdes;
}

SCM
sgtk_fileno2port (int fd)
{
  SCM res;

  res = scm_fdes_to_port (fd, "r+0", sym_gnome_file);
  if (SCM_OPFPORTP (res))
    scm_setvbuf (res, scm_from_int (_IONBF), scm_from_int (0));
  return res;
}

static long tc16_gtktype;

#define GTKTYPEP(x)     (SCM_SMOB_PREDICATE(tc16_gtktype, x))
#define GTKTYPE(x)      ((GtkType)SCM_SMOB_DATA(x))

static int
gtktype_print (SCM obj, SCM port, scm_print_state *pstate)
{
  GtkType type = GTKTYPE (obj);
  scm_puts ("#<GtkType ", port);
  scm_puts (gtk_type_name (type), port);
  scm_puts (">", port);
  return 1;
}

static SCM
gtktype_equalp (SCM obj1, SCM obj2)
{
  return GTKTYPE (obj1) == GTKTYPE (obj2)? SCM_BOOL_T : SCM_BOOL_F;
}

/* Look for a record in all_type_infos with the given "name".
   Return a pointer to it if found, or NULL if not.  */
sgtk_type_info *
all_type_infos_find (const char *name)
{
  type_infos *infos;
  for (infos = all_type_infos; infos; infos = infos->next)
    {
      sgtk_type_info **ip;
      for (ip = infos->infos; *ip; ip++)
        if (!strcmp ((*ip)->name, name))
          return *ip;
    }
  return NULL;
}


/* This is like g_type_from_name, but when "name" is not known to Glib yet
   we search among the type names we have in all_type_infos and initialize
   the type with the init_func in our sgtk_type_info.  */
GtkType
sgtk_type_from_name (char *name)
{
  GType type = g_type_from_name (name);
  if (type == GTK_TYPE_INVALID)
    {
      sgtk_type_info *info = all_type_infos_find (name);
      if (info && info->init_func)
        {
          type = (*info->init_func) ();
          /* check we put the right func in our info */
          assert (strcmp (name, g_type_name (type)) == 0);
        }
    }
  return type;
}

int
sgtk_valid_type (SCM obj)
{
  char  *name;
  int   ret;

  if (scm_is_false (obj))
    return 1;

  if (GTKTYPEP (obj))
    return 1;

  if (! scm_is_symbol (obj))
    return 0;
  name = sgtk_symbol_to_locale_string (obj);
  ret = (sgtk_type_from_name (name) != 0);
  free (name);
  return ret;
}

GtkType
sgtk_scm2type (SCM obj)
{
  char *name;
  GtkType type;

  if (scm_is_false (obj))
    return GTK_TYPE_INVALID;

  if (GTKTYPEP (obj))
    return GTKTYPE (obj);

  name = sgtk_symbol_to_locale_string (obj);
  type = sgtk_type_from_name (name);
  free (name);
  return type;
}

SCM
sgtk_type2scm (GtkType t)
{
  if (t == GTK_TYPE_INVALID)
    return SCM_BOOL_F;

  return scm_cell (tc16_gtktype, (scm_t_bits) t);
}

/* Illegal objects.  Guile-gtk constructs one of these when it sees a
   object with illegal type.  The use can't do anything with them, but
   the failure is clearly labelled and doesn't pop up until such an
   object is really used. */

static long tc16_illobj;

#define ILLOBJP(x)     (SCM_NIMP(x) && SCM_CAR(x) == tc16_illobj)
#define ILLOBJ_TYPE(x) ((GtkType)SCM_CDR(x))

static int
illobj_print (SCM obj, SCM port, scm_print_state *pstate)
{
  GtkType type = ILLOBJ_TYPE (obj);
  scm_puts ("#<object of illegal type ", port);
  scm_puts (gtk_type_name (type), port);
  scm_puts (">", port);
  return 1;
}
/*
#ifdef OLG_GUILE
struct scm_smobfuns illobj_smob = {
  scm_mark0,
  scm_free0,
  illobj_print,
  NULL
};
#endif 
*/
static SCM
sgtk_make_illegal_type_object (GtkType type)
{
  return scm_cell (tc16_illobj, (scm_t_bits) type);
}

/* Composites. */

int
sgtk_valid_composite (SCM obj, int (*predicate)(SCM))
{
  return sgtk_valid_complen (obj, predicate, -1);
}

int
sgtk_valid_complen (SCM obj, int (*predicate)(SCM), int len)
{
  int actual_len;

  if ((actual_len = scm_ilength (obj)) >= 0)
    {
      if (len >= 0 && len != actual_len)
	return 0;

      if (predicate)
	{
	  while (scm_is_pair (obj))
	    {
	      if (!predicate (SCM_CAR(obj)))
		return 0;
	      obj = SCM_CDR(obj);
	    }
	}
      return 1;
    }
  else if (scm_is_vector (obj))
    {
      int i;

      actual_len = scm_c_vector_length (obj);
      if (len >= 0 && len != actual_len)
	return 0;

      if (predicate)
	{
	  for (i = 0; i < actual_len; i++)
	    if (!predicate(scm_c_vector_ref(obj,i)))
	      return 0;
	}
      return 1;
    }
  else
    return 0;
}

SCM
sgtk_composite_inconversion (SCM obj, SCM (*conversion)(SCM))
{
  if (conversion == NULL)
    return obj;

  if (scm_is_null (obj) || scm_is_pair (obj))
    {
      int pos = 0;
      SCM list = obj;
      SCM newlist = list;
      while (scm_is_pair (obj))
	{
	  SCM newelt = conversion (SCM_CAR(obj));
	  if (! scm_is_eq (newelt, SCM_CAR(obj)))
	    {
	      if (scm_is_eq (newlist, list))
		{
		  newlist = scm_list_copy (list);
		  obj = newlist;
		  while (pos > 0)
		    obj = SCM_CDR(obj);
		}
	      SCM_SETCAR(obj, newelt);
	    }
	  obj = SCM_CDR(obj);
	  pos++;
	}
      return newlist;
    }
  else if (scm_is_vector (obj))
    {
      SCM vec = obj;
      SCM newvec = vec;
      size_t len = scm_c_vector_length(newvec), i;
      for (i = 0; i < len; i++)
	{
	  SCM newelt = conversion (scm_c_vector_ref (newvec, i));
	  if (! scm_is_eq (newelt, scm_c_vector_ref (newvec, i)))
	    {
	      if (scm_is_eq (newvec, vec))
		{
		  size_t j;
		  newvec = scm_c_make_vector (len, SCM_UNDEFINED);
		  for (j = 0; j < len; j++)
		    SCM_SIMPLE_VECTOR_SET(newvec, j, scm_c_vector_ref (vec, j));
		}
	      scm_c_vector_set_x(newvec, i, newelt);
	    }
	}
      return newvec;
    }
  else
    return obj;
}

SCM
sgtk_composite_outconversion (SCM obj, SCM (*conversion)(SCM))
{
  if (conversion == NULL)
    return obj;

  if (scm_is_null (obj) || scm_is_pair (obj))
    {
      SCM list = obj;
      while (scm_is_pair (obj))
	{
	  SCM_SETCAR(obj, conversion (SCM_CAR(obj)));
	  obj = SCM_CDR(obj);
	}
      return list;
    }
  else if (scm_is_vector (obj))
    {
      int len = scm_c_vector_length(obj), i;
      for (i = 0; i < len; i++)
	scm_c_vector_set_x(obj,i, conversion (scm_c_vector_ref (obj, i)));
      return obj;
    }
  else
    return obj;
}
  
SCM
sgtk_slist2scm (GSList *list, SCM (*toscm)(void*))
{
  SCM res = SCM_EOL, *tail = &res;
  while (list)
    {
      *tail = scm_cons (toscm (&list->data), *tail);
      tail = SCM_CDRLOC (*tail);
      list = list->next;
    }
  *tail = SCM_EOL;
  return res;
}

GSList*
sgtk_scm2slist (SCM obj, void (*fromscm)(SCM, void*))
{
  GSList *res, **tail = &res;

  if (scm_is_false (obj))
    return NULL;
  else if (scm_is_null (obj) || scm_is_pair (obj))
    {
      while (scm_is_pair (obj))
	{
	  *tail = g_slist_alloc ();
	  if (fromscm)
	    fromscm (SCM_CAR (obj), &(*tail)->data);
	  else
	    (*tail)->data = NULL;
	  obj = SCM_CDR(obj);
	  tail = &(*tail)->next;
	}
    }
  else if (scm_is_vector (obj))
    {
      int len = scm_c_vector_length (obj), i;
      for (i = 0; i < len; i++)
	{
	  *tail = g_slist_alloc ();
	  if (fromscm)
	    fromscm (scm_c_vector_ref (obj, i), &(*tail)->data);
	  else
	    (*tail)->data = NULL;
	  tail = &(*tail)->next;
	}
    }
  *tail = NULL;
  return res;
}

void
sgtk_slist_finish (GSList *list, SCM obj, SCM (*toscm)(void*))
{
  if (list == NULL)
    return;

  if (toscm)
    {
      if (scm_is_null (obj) || scm_is_pair (obj))
	{
	  while (scm_is_pair (obj) && list)
	    {
	      SCM_SETCAR (obj, toscm (list->data));
	      obj = SCM_CDR(obj);
	      list = list->next;
	    }
	}
      else if (scm_is_vector (obj))
	{
	  int len = scm_c_vector_length (obj), i;
	  for (i = 0; i < len && list; i++)
	    {
	      scm_c_vector_set_x (obj, i, toscm (list->data));
	      list = list->next;
	    }
	}
    }

  g_slist_free (list);
}

SCM
sgtk_list2scm (GList *list, SCM (*toscm)(void*))
{
  SCM res = SCM_EOL, *tail = &res;
  while (list)
    {
      *tail = scm_cons (toscm (&list->data), *tail);
      tail = SCM_CDRLOC (*tail);
      list = list->next;
    }
  *tail = SCM_EOL;
  return res;
}

GList*
sgtk_scm2list (SCM obj, void (*fromscm)(SCM, void*))
{
  GList *res = NULL, *tail;

  if (scm_is_false (obj))
    return NULL;
  else if (scm_is_null (obj) || scm_is_pair (obj))
    {
      while (scm_is_pair (obj))
      {
        GList *n = g_list_alloc ();
	if (res == NULL)
	  res = tail = n;
	else 
	  {
	    g_list_concat (tail, n);
	    tail = n;
	  }
	if (fromscm)
	  fromscm (SCM_CAR (obj), &(n->data));
	else
	  n->data = NULL;
	obj = SCM_CDR(obj);
      }
    }
  else if (scm_is_vector (obj))
    {
      int len = scm_c_vector_length (obj), i;
      for (i = 0; i < len; i++)
	{
	  GList *n = g_list_alloc ();
	  if (res == NULL)
	    res = tail = n;
	  else 
	    {
	      g_list_concat (tail, n);
	      tail = n;
	    }
	  if (fromscm)
	    fromscm (scm_c_vector_ref (obj, i), &(n->data));
	  else
	    n->data = NULL;
	}
    }

  return res;
}

void
sgtk_list_finish (GList *list, SCM obj, SCM (*toscm)(void*))
{
  if (list == NULL)
    return;

  if (toscm)
    {
      if (scm_is_null (obj) || scm_is_pair (obj))
	{
	  while (scm_is_pair (obj) && list)
	    {
	      SCM_SETCAR (obj, toscm (list->data));
	      obj = SCM_CDR(obj);
	      list = list->next;
	    }
	}
      else if (scm_is_vector (obj))
	{
	  int len = scm_c_vector_length (obj), i;
	  for (i = 0; i < len && list; i++)
	    {
	      scm_c_vector_set_x (obj, i, toscm (list->data));
	      list = list->next;
	    }
	}
    }
  
  g_list_free (list);
}

sgtk_cvec
sgtk_scm2cvec (SCM obj, void (*fromscm)(SCM, void*), size_t sz)
{
  sgtk_cvec res;
  int i;
  char *ptr;

  if (scm_is_false (obj) || scm_is_null (obj))
    {
      res.vec = xmalloc (sz);	/* for NULL-termination */
      res.count = 0;
    }
  else if ((res.count = scm_ilength (obj)) >= 0)
    {
      res.vec = xmalloc ((res.count + 1) * sz);
      if (fromscm)
	{
	  for (i = 0, ptr = res.vec; i < res.count; i++, ptr += sz)
	    {
	      fromscm (SCM_CAR (obj), ptr);
	      obj = SCM_CDR(obj);
	    }
	  memset ((char *) res.vec + res.count * sz, 0, sz);
	}
      else
	memset (res.vec, 0, (res.count + 1) * sz);
    }
  else if (scm_is_vector (obj))
    {
      res.count = scm_c_vector_length (obj);
      res.vec = (void *)xmalloc ((res.count + 1) * sz);
      if (fromscm)
	{
	  for (i = 0, ptr = res.vec; i < res.count; i++, ptr += sz)
	    fromscm (scm_c_vector_ref (obj, i), ptr);
	  memset ((char *) res.vec + res.count * sz, 0, sz);
	}
      else
	memset (res.vec, 0, (res.count + 1) * sz);
    }

  return res;
}

void
sgtk_cvec_finish (sgtk_cvec *cvec, SCM obj, SCM (*toscm)(void *), size_t sz)
{
  if (cvec->vec == NULL)
    return;

  if (toscm)
    {
      if (scm_is_null (obj) || scm_is_pair (obj))
	{
	  int i, len = cvec->count;
	  char *ptr;

	  for (i = 0, ptr = cvec->vec;
	       i < len && scm_is_pair (obj);
	       i++, ptr += sz, obj = SCM_CDR (obj))
	    {
	      SCM_SETCAR (obj, toscm (ptr));
	    }
	}
      else if (scm_is_vector (obj))
	{
	  int len1 = scm_c_vector_length (obj), len2 = cvec->count, i;
	  char *ptr;

	  for (i = 0, ptr = cvec->vec; i < len1 && i < len2; i++, ptr += sz)
	    scm_c_vector_set_x (obj, i, toscm (ptr));
	}
    }

  free (cvec->vec);
}

SCM
sgtk_cvec2scm (sgtk_cvec *cvec, SCM (*toscm)(void *), size_t sz)
{
    int len, i;
    SCM obj = scm_c_make_vector (len = cvec->count, SCM_UNSPECIFIED);
    char *ptr;

    for (i = 0, ptr = cvec->vec; i < len; i++, ptr += sz)
      SCM_SIMPLE_VECTOR_SET (obj, i, toscm (ptr));

    g_free (cvec->vec);
    return obj;
}

sgtk_raw
sgtk_scm2raw (SCM obj, int pos, char* func)
#define FUNC_NAME func
{
  SCM val;
  long i,v;
  sgtk_raw ret;

  if (scm_is_false (obj) || scm_is_null (obj))
    {
      ret.count = 0;
      ret.raw = NULL;
      ret.keep = SCM_BOOL_F;
    }
  else if (scm_is_string (obj))   /* string bytes */
    {
      size_t len;
      ret.raw = (guchar *) scm_to_locale_stringn (obj, &len);
      ret.count = len;
      ret.keep = sgtk_make_cblk (ret.raw, len);
    }
  else if (scm_is_vector (obj)    /* vector (or weak vector) of byte values */
           || scm_u8vector_p (obj)
           || scm_s8vector_p (obj))
    {
      /* ENHANCE-ME: Use the array handle stuff for greater speed.  Some
         Guile 1.6 compats for that wouldn't be too hard (only vectors and
         byvects are supposed to work there).  */
      ret.count = scm_c_generalized_vector_length (obj);
      ret.raw = scm_malloc (ret.count);
      ret.keep = sgtk_make_cblk (ret.raw, ret.count);

      for (i = 0; i < ret.count; ++i)
        {
          val = scm_c_generalized_vector_ref (obj, i);
          v = scm_to_int (val);
          if (v < -128 || v > 255)
            SCM_OUT_OF_RANGE (pos, val);
          ret.raw [i] = (guchar) v;
        }
    }
  else
    {
      SCM_WRONG_TYPE_ARG (pos, obj);
    }

  return ret;
}
#undef FUNC_NAME

/* converting between SCM and GtkArg */

SCM
sgtk_arg2scm (GtkArg *a, int free_mem)
{
  switch (G_TYPE_FUNDAMENTAL (a->type))
    {
    case GTK_TYPE_NONE:
      return SCM_UNSPECIFIED;
    case GTK_TYPE_CHAR:
      return SCM_MAKE_CHAR (GTK_VALUE_CHAR(*a));
    case GTK_TYPE_BOOL:
      return GTK_VALUE_BOOL(*a)? SCM_BOOL_T : SCM_BOOL_F;
    case GTK_TYPE_INT:
      return scm_from_int (GTK_VALUE_INT(*a));
    case GTK_TYPE_UINT:
      return scm_from_uint (GTK_VALUE_UINT(*a));
    case GTK_TYPE_LONG:
      return scm_from_long (GTK_VALUE_LONG(*a));
    case GTK_TYPE_ULONG:
      return scm_from_ulong (GTK_VALUE_ULONG(*a));
    case GTK_TYPE_FLOAT:
      return scm_from_double ((double) GTK_VALUE_FLOAT(*a));
    case GTK_TYPE_DOUBLE:
      return scm_from_double (GTK_VALUE_DOUBLE(*a));
    case GTK_TYPE_STRING:
      if (free_mem)
        return scm_take_locale_string (GTK_VALUE_STRING(*a));
      else
        return scm_from_locale_string (GTK_VALUE_STRING(*a));
    case GTK_TYPE_ENUM:
      return sgtk_enum2scm (GTK_VALUE_FLAGS(*a),
			     (sgtk_enum_info *)sgtk_find_type_info (a->type));
    case GTK_TYPE_FLAGS:
      return sgtk_flags2scm (GTK_VALUE_FLAGS(*a),
			     (sgtk_enum_info *)sgtk_find_type_info (a->type));
    case GTK_TYPE_BOXED:
      return sgtk_boxed2scm (GTK_VALUE_BOXED(*a),
			     (sgtk_boxed_info *)sgtk_find_type_info (a->type),
			     TRUE);
    case G_TYPE_OBJECT:
      return sgtk_wrap_gtkobj ((GObject *) GTK_VALUE_OBJECT(*a));
    default:
      return sgtk_make_illegal_type_object (a->type);
    }
}

SCM
sgtk_gvalue2scm (const GValue *value, int free_mem)
{
  GType type = G_VALUE_TYPE (value);

#ifdef G_VALUE_HOLDS_GTYPE  /* new in glib 2.12 */
  if (g_type_is_a (type, G_TYPE_GTYPE))
    return sgtk_type2scm (g_value_get_gtype (value));
#endif

  switch (G_TYPE_FUNDAMENTAL (type))
    {
    case G_TYPE_NONE:
      return SCM_UNSPECIFIED;
    case G_TYPE_CHAR:
      return SCM_MAKE_CHAR ((int) (guchar) g_value_get_char (value));
    case G_TYPE_BOOLEAN:
      return scm_from_bool (g_value_get_boolean (value));
    case G_TYPE_INT:
      return scm_from_long (g_value_get_int (value));
    case G_TYPE_UINT:
      return scm_from_uint (g_value_get_uint (value));
    case G_TYPE_LONG:
      return scm_from_long (g_value_get_long (value));
    case G_TYPE_ULONG:
      return scm_from_ulong (g_value_get_ulong (value));
    case G_TYPE_FLOAT:
      return scm_from_double ((double) g_value_get_float (value));
    case G_TYPE_DOUBLE:
      return scm_from_double (g_value_get_double (value));
    case G_TYPE_STRING:
      return scm_from_locale_string (g_value_get_string (value));
    case G_TYPE_ENUM:
      return sgtk_enum2scm (g_value_get_enum (value),
                            (sgtk_enum_info *) sgtk_find_type_info (type));
    case G_TYPE_FLAGS:
      return sgtk_enum2scm (g_value_get_flags (value),
                            (sgtk_enum_info *) sgtk_find_type_info (type));
    case G_TYPE_BOXED:
      return sgtk_boxed2scm (g_value_get_boxed (value),
			     (sgtk_boxed_info *) sgtk_find_type_info (type),
			     TRUE);
    case G_TYPE_OBJECT:
      return sgtk_wrap_gtkobj (g_value_get_object (value));
    default:
      return sgtk_make_illegal_type_object (type);
    }
}

int
sgtk_valid_for_type (GType type, SCM obj)
{
#ifdef G_VALUE_HOLDS_GTYPE  /* new in glib 2.12 */
  if (g_type_is_a (type, G_TYPE_GTYPE))
    return sgtk_valid_type (obj);
#endif
  
  switch (G_TYPE_FUNDAMENTAL (type))
    {
    case GTK_TYPE_NONE:
      return TRUE;
    case GTK_TYPE_CHAR:
      return SCM_CHARP (obj);
    case GTK_TYPE_BOOL:
      return TRUE;
    case GTK_TYPE_INT:
      return scm_is_signed_integer (obj, INT_MIN, INT_MAX);
    case GTK_TYPE_UINT:
      return scm_is_unsigned_integer (obj, 0, UINT_MAX);
    case GTK_TYPE_LONG:
      return scm_is_signed_integer (obj, LONG_MIN, LONG_MAX);
    case GTK_TYPE_ULONG:
      return scm_is_unsigned_integer (obj, 0, ULONG_MAX);
    case GTK_TYPE_FLOAT:
    case GTK_TYPE_DOUBLE:
      return scm_is_real (obj);
    case GTK_TYPE_STRING:
      return sgtk_valid_cstr (obj);
    case GTK_TYPE_ENUM:
      return sgtk_valid_enum (obj, ((sgtk_enum_info *)
				    sgtk_find_type_info (type)));
    case GTK_TYPE_FLAGS:
      return sgtk_valid_flags (obj, ((sgtk_enum_info *)
				     sgtk_find_type_info (type)));
    case GTK_TYPE_BOXED:
      return sgtk_valid_boxed (obj, ((sgtk_boxed_info *)
				     sgtk_find_type_info (type)));
      break;
#if 0
      /* Gone in Gtk 2 */
    case GTK_TYPE_CALLBACK:
      return gh_procedure_p (obj);
#endif
    case G_TYPE_OBJECT:
      return sgtk_is_a_gtkobj (type, obj);
    default:
      fprintf (stderr, "unhandled arg type %s\n", gtk_type_name (type));
      return FALSE;
    }
}

void
sgtk_scm2ret (GtkArg *a, SCM obj)
{
  switch (GTK_FUNDAMENTAL_TYPE (a->type))
    {
    case GTK_TYPE_NONE:
      return;
    case GTK_TYPE_CHAR:
      *GTK_RETLOC_CHAR(*a) = SCM_CHAR (obj);
      break;
    case GTK_TYPE_BOOL:
      *GTK_RETLOC_BOOL(*a) = scm_is_true (obj);
      break;
    case GTK_TYPE_INT:
      *GTK_RETLOC_INT(*a) = scm_num2int (obj, (long) SCM_ARG1, "scm->gtk");
      break;
    case GTK_TYPE_UINT:
      *GTK_RETLOC_UINT(*a) = scm_num2uint (obj, (long) SCM_ARG1, "scm->gtk");
      break;
    case GTK_TYPE_LONG:
      *GTK_RETLOC_LONG(*a) = scm_num2long (obj, (long) SCM_ARG1, "scm->gtk");
      break;
    case GTK_TYPE_ULONG:
      *GTK_RETLOC_ULONG(*a) = scm_num2ulong (obj, (long) SCM_ARG1, "scm->gtk");
      break;
    case GTK_TYPE_FLOAT:
      *GTK_RETLOC_FLOAT(*a) = (float) scm_to_double (obj);
      break;
    case GTK_TYPE_DOUBLE:
      *GTK_RETLOC_DOUBLE(*a) = scm_to_double (obj);
      break;
    case GTK_TYPE_STRING:
      /* FIXME: This case never seems to get used, not sure whether it
         should be a freshly malloced string here.  */
      GTK_VALUE_STRING(*a) = scm_to_locale_string (obj);
      break;
    case GTK_TYPE_ENUM:
      *GTK_RETLOC_ENUM(*a) =
	sgtk_scm2enum (obj, (sgtk_enum_info *)sgtk_find_type_info (a->type),
		       SCM_ARG1, "scm->gtk");
      break;
    case GTK_TYPE_FLAGS:
      *GTK_RETLOC_ENUM(*a) =
	sgtk_scm2flags (obj, (sgtk_enum_info *)sgtk_find_type_info (a->type),
		       SCM_ARG1, "scm->gtk");
      break;
    case GTK_TYPE_BOXED:
      *GTK_RETLOC_BOXED(*a) = sgtk_scm2boxed (obj);
      break;
    case G_TYPE_OBJECT:
      SCM_ASSERT (sgtk_is_a_gtkobj (a->type, obj), obj, SCM_ARG1, "scm->gtk");
      *GTK_RETLOC_OBJECT(*a) = sgtk_get_gtkobj (obj);
      break;
    default:
      fprintf (stderr, "unhandled return type %s\n", gtk_type_name (a->type));
      break;
    }
}

void
sgtk_scm2gvalue (GValue *v, SCM obj)
{
  GType type = G_VALUE_TYPE (v);

#ifdef G_VALUE_HOLDS_GTYPE  /* new in glib 2.12 */
  if (g_type_is_a (type, G_TYPE_GTYPE))
    {
      g_value_set_gtype (v, sgtk_scm2type (obj));
      return;
    }
#endif

  switch (G_TYPE_FUNDAMENTAL (type))
    {
    case 0: /* uninitialized (ie. all zeros) GValue */
    case G_TYPE_NONE:
      return;
    case G_TYPE_CHAR:
      g_value_set_char (v, SCM_CHAR (obj));
      break;
    case G_TYPE_UCHAR:
      g_value_set_uchar (v, SCM_CHAR (obj));
      break;
    case G_TYPE_BOOLEAN:
      g_value_set_boolean (v, scm_is_true (obj));
      break;
    case G_TYPE_INT:
      g_value_set_int (v, scm_num2int (obj, SCM_ARGn, "scm->gtk"));
      break;
    case G_TYPE_UINT:
      g_value_set_uint (v, scm_num2uint (obj, SCM_ARGn, "scm->gtk"));
      break;
    case G_TYPE_LONG:
      g_value_set_long (v, scm_num2long (obj, SCM_ARGn, "scm->gtk"));
      break;
    case G_TYPE_ULONG:
      g_value_set_ulong (v, scm_num2ulong (obj, SCM_ARGn, "scm->gtk"));
      break;
    case G_TYPE_FLOAT:
      g_value_set_float (v, (float) scm_to_double (obj));
      break;
    case G_TYPE_DOUBLE:
      g_value_set_double (v, scm_to_double (obj));
      break;
    case G_TYPE_STRING:
      if (sgtk_valid_cstr (obj))
        g_value_set_string (v, sgtk_cstr2ptr (obj, SCM_ARGn, "scm->gtk"));
      else
        g_value_take_string (v, scm_to_locale_string (obj));
      break;
    case G_TYPE_ENUM:
      g_value_set_enum (v, sgtk_scm2enum (obj, (sgtk_enum_info *) sgtk_find_type_info (type), SCM_ARGn, "scm->gtk"));
      break;
    case G_TYPE_FLAGS:
      g_value_set_enum (v, sgtk_scm2flags (obj, (sgtk_enum_info *)sgtk_find_type_info (type), SCM_ARGn, "scm->gtk"));
      break;
    case G_TYPE_BOXED:
      g_value_set_boxed (v, sgtk_scm2boxed (obj));
      break;
    case G_TYPE_OBJECT:
      SCM_ASSERT (sgtk_is_a_gtkobj (type, obj), obj, SCM_ARGn, "scm->gtk");
      g_value_set_object (v, sgtk_get_gtkobj (obj));
      break;
    default:
      fprintf (stderr, "unhandled GValue type %lu %s\n",
               type, gtk_type_name (type));
      break;
    }
}

void
sgtk_arg_cleanup (GtkArg *a, SCM obj)
{
  switch (GTK_FUNDAMENTAL_TYPE (a->type))
    {
    case GTK_TYPE_BOXED:
      if (BOXED_P (obj) && BOXED_INFO (obj)->cleanup)
       BOXED_INFO (obj)->cleanup (obj);
      break;

    default: 
      break;
    }
}

/* Callbacks.

   Callbacks are executed within a new dynamic root.  That means that
   the flow of control can't leave them without Gtk noticing.  Throws
   are catched and briefly reported.  Calls to continuations that have
   been made outside the dynamic root can not be activated.

   Callbacks are invoked with whatever arguments that are specified by
   the Gtk documentation.  They do not, however, receive the GtkObject
   that has initiated the callback.

   When callback_trampoline is non-#f, we treat it as a procedure and
   call it as

      (trampoline proc args)

   PROC is the real callback procedure and ARGS is the list of
   arguments that should be passed to it.  */

static SCM callback_trampoline;

/* The SCM_PROC for gtk-callback-trampoline is in gtk-support.c to
   have it be snarfed for sgtk_init_support */

SCM
sgtk_callback_trampoline (SCM new)
{
  SCM old = SCM_CAR (callback_trampoline);
  if (! SCM_UNBNDP (new))
    SCM_SETCAR (callback_trampoline, new);
  return old;
}

struct callback_info {
  SCM proc;
  gint n_args;
  GtkArg *args;
};

static SCM
inner_callback_marshal (void *data)
{
  struct callback_info *info = (struct callback_info *)data;
  int i;
  SCM args = SCM_EOL, ans;

  for (i = info->n_args-1; i >= 0; i--)
    args = scm_cons (sgtk_arg2scm (info->args+i, 0), args);
  if (SCM_FALSEP (SCM_CAR(callback_trampoline)))
    ans = scm_apply (info->proc, args, SCM_EOL);
  else
    ans = scm_apply (SCM_CAR(callback_trampoline),
		     scm_cons2 (info->proc, args, SCM_EOL), SCM_EOL);
  if (info->args[info->n_args].type != GTK_TYPE_NONE)
    sgtk_scm2ret (info->args+info->n_args, ans);

  for (i = 0; i < info->n_args; ++i, args = SCM_CDR (args))
    sgtk_arg_cleanup (info->args+i, SCM_CAR (args));

  return SCM_UNSPECIFIED;
}

/* Be carefull when this macro is true.
   scm_gc_heap_lock is set during gc.  */
#define SCM_GC_P (scm_gc_running_p)

void
sgtk_callback_marshal (GtkObject *obj,
		       gpointer data,
		       guint n_args,
		       GtkArg *args)
{
  SCM_STACKITEM stack_item;
  struct callback_info info;

  if (SCM_GC_P)
    {
      /* This should only happen for the "destroy" signal and is then
         harmless. */
      fprintf (stderr, "callback ignored during GC!\n");
      return;
    }
  
  info.proc = ((sgtk_protshell *)data)->object;
  info.n_args = n_args;
  info.args = args;

  scm_internal_cwdr ((scm_t_catch_body)inner_callback_marshal, &info,
		     scm_handle_by_message_noexit, "gtk",
		     &stack_item);
}

struct closure_info {
  SCM proc;
  GValue *return_value;
  guint n_param_values;
  const GValue *param_values;
};

static SCM
inner_closure_marshal (void *data)
{
  struct closure_info *info = (struct closure_info *)data;
  int i;
  SCM args = SCM_EOL, ans;

  /* param[0] is the originating GObject, which we ignore */
  for (i = info->n_param_values-1; i > 0; i--)
    args = scm_cons (sgtk_gvalue2scm (&info->param_values[i], 1), args);

  if (SCM_FALSEP (SCM_CAR (callback_trampoline)))
    ans = scm_apply (info->proc, args, SCM_EOL);
  else
    ans = scm_apply (SCM_CAR(callback_trampoline),
		     scm_cons2 (info->proc, args, SCM_EOL), SCM_EOL);

  /* info->return_value can be uninitialized (zero in the `type' field), for
     example from gtk_binding_entry_activate().  Believe this means there's
     no return wanted or expected.  */
  if (info->return_value != NULL && G_VALUE_TYPE (info->return_value) != 0)
    sgtk_scm2gvalue (info->return_value, ans);

  return SCM_UNSPECIFIED;
}

void
sgtk_closure_marshal (GClosure *closure,
                      GValue *return_value,
                      guint n_param_values,
                      const GValue *param_values,
                      gpointer invocation_hint,
                      gpointer marshal_data)
{
  SCM_STACKITEM stack_item;
  struct closure_info info;
  sgtk_protshell *protshell = (sgtk_protshell *) closure->data;

  if (SCM_GC_P)
    {
      /* This should only happen for the "destroy" signal and is then
         harmless. */
      fprintf (stderr, "callback ignored during GC!\n");
      return;
    }

  info.proc = protshell->object;
  info.return_value = return_value;
  info.n_param_values = n_param_values;
  info.param_values = param_values;

  scm_internal_cwdr (inner_closure_marshal, &info,
		     scm_handle_by_message_noexit, "gtk",
		     &stack_item);
}

void
sgtk_callback_destroy (gpointer data)
{
  sgtk_unprotect ((sgtk_protshell *)data);
}



/* Type conversions */

SCM
sgtk_color_conversion (SCM color)
{
  if (scm_is_string (color))
    {
      GdkColor colstruct;
      GdkColormap *colmap;
      char *c_str;
      int ret;

      c_str = scm_to_locale_string (color);
      ret = gdk_color_parse (c_str, &colstruct);
      free (c_str);
      if (! ret)
	{
	  scm_misc_error ("string->color", "no such color: ~S",
			  scm_cons (color, SCM_EOL));
	}
      colmap = gtk_widget_get_default_colormap ();
      if (!gdk_color_alloc (colmap, &colstruct))
	{
	  scm_misc_error ("string->color", "can't allocate color: ~S",
			  scm_cons (color, SCM_EOL));
	}
      return sgtk_boxed2scm (&colstruct, &sgtk_gdk_color_info, 1);
    }
  return color;
}

SCM
sgtk_font_conversion (SCM font)
{
  SCM orig_font = font;

  if (scm_is_string (font))
    {
      font = sgtk_gdk_font_load (font);
      if (scm_is_false (font))
	scm_misc_error ("string->font", "no such font: ~S",
			scm_cons (orig_font, SCM_EOL));
    }
  return font;
}

/* If there's a conversion function for parameters of `type' then run it on
   `obj' and return the result.  If the conversion makes a change then cons
   the old value onto `*keep_list'.

   This conversion includes strings becoming cstr for G_TYPE_STRING.  */

SCM
sgtk_apply_conversion (GType type, SCM obj, SCM *keep_list)
{
  const sgtk_type_info *type_info = sgtk_maybe_find_type_info (type);
  if (type_info && type_info->conversion)
    {
      SCM new_obj = type_info->conversion (obj);
      if (! scm_is_eq (new_obj, obj))
        {
          *keep_list = scm_cons (new_obj, *keep_list);
          obj = new_obj;
        }
    }
  return obj;
}


GtkTargetEntry
sgtk_scm2gtk_target_entry (SCM entry, int pos, char *subr)
{
  GtkTargetEntry ret;

  SCM_ASSERT (scm_ilength (entry) == 3, entry, pos, subr);
  SCM_ASSERT (scm_is_string (SCM_CAR(entry)), entry, pos, subr);
  SCM_ASSERT (SCM_INUMP (SCM_CADR(entry)), entry, pos, subr);
  SCM_ASSERT (SCM_INUMP (SCM_CADDR(entry)), entry, pos, subr);

  ret.target = g_strdup (SCM_STRING_CHARS (SCM_CAR (entry)));
  ret.flags = SCM_INUM (SCM_CADR (entry));
  ret.info = SCM_INUM (SCM_CADDR (entry));
  
  return ret;
}

void
sgtk_gtk_target_entry_free (GtkTargetEntry* entry)
{ g_free (entry->target); }




/* Support for gtk_object_new, gtk_object_set, ... */

/* The SCM_PROC for the exported functions is in gtk-support.c to have
   it be snarfed for sgtk_init_gtk_support. */

sgtk_object_info *sgtk_find_object_info (char *name);

sgtk_object_info *
sgtk_find_object_info_from_type (GtkType type)
{
  if (type == GTK_TYPE_INVALID)
    return NULL;

  /* must lookup by name, since the GtkType values are not immediately
     filled into the object info tables */
  return sgtk_find_object_info (gtk_type_name (type));
}

sgtk_object_info *
sgtk_find_object_info (char *name)
{
  GtkType type, parent;
  sgtk_object_info *info;
  type_infos *infos;
  int i;

  type = gtk_type_from_name (name);

  for (infos = all_type_infos; infos; infos = infos->next)
    {
      sgtk_type_info **ip;
      for (ip = infos->infos; *ip; ip++)
	if (!strcmp ((*ip)->name, name))
	  {
	    if (GTK_FUNDAMENTAL_TYPE((*ip)->type) != GTK_TYPE_OBJECT)
	      return NULL;

	    info = (sgtk_object_info *)*ip;
	    info->header.type = info->header.init_func ();
	    enter_type_info ((sgtk_type_info*)info);
	    goto query_args;
	  }
    }

  /* Not found among our precompiled types.  Construct a fresh
     sgtk_object_info, if it's known to Gtk+. */

  if (type != GTK_TYPE_INVALID)
    {
      /* fprintf (stderr, "Fresh info for %s, %d\n", name, type); */

      info = (sgtk_object_info *)xmalloc (sizeof(sgtk_object_info));
      info->header.type = type;
      info->header.name = name;
      info->header.init_func = NULL;
      enter_type_info ((sgtk_type_info*)info);
    }
  else
    return NULL;

 query_args:
  gtk_type_class (info->header.type);
#if 0
  /* FIXME: This is gone from Gtk 2, should be unused by us now though. */
  info->args = gtk_object_query_args (info->header.type,
				      &info->args_flags,
				      &info->n_args);
#endif
  info->args_flags = 0;
  info->n_args = 0;

  info->args_short_names =
    (char **)xmalloc (info->n_args*(sizeof(char*)));
  for (i = 0; i < info->n_args; i++)
    {
      char *l = info->args[i].name;
      char *d = strchr (l, ':');
      if (d == NULL || d[1] != ':')
	{
	  fprintf (stderr, "`%s' has no class part.\n", l);
	  info->args_short_names[i] = l;
	}
      else
	info->args_short_names[i] = d+2;
    }
  
  parent = gtk_type_parent (info->header.type);
  if (parent != GTK_TYPE_INVALID)
    info->parent = sgtk_find_object_info_from_type (parent);
  else
    info->parent = NULL;
  
  return info;
}

static void
sgtk_find_arg_info (GtkArg *arg, sgtk_object_info *info, char *name)
{
  /* XXX - handle signal handlers.  Do not use '::', use '.' instead. */

  char *d = strchr (name, ':');
  if (d && d[1] == ':')
    {
      /* A long name.  Find the object_info for the class part. */
      int len = d-name;

      while (info)
	{
	  if (info->header.name[len] == '\0'
	      && !strncmp (info->header.name, name, len))
	    break;
	  info = info->parent;
	}
      name = d+2;
    }
  
#ifdef DEBUG_PRINT
  fprintf (stderr, "searching short `%s'\n", name);
#endif
  while (info)
    {
      int i;
      for (i = 0; i < info->n_args; i++)
	{
#ifdef DEBUG_PRINT
	  fprintf (stderr, " on %s\n", info->args[i].name);
#endif
	  if (!strcmp (info->args_short_names[i], name))
	    {
	      *arg = info->args[i];
	      return;
	    }
	}
      info = info->parent;
    }
  
  arg->type = GTK_TYPE_INVALID;
  return;
}
      
static SCM
sgtk_build_params (GType type, GParameter **paramsp, int *n_paramsp,
                   SCM scm_params, const char *subr)
{
  int i, n_params = *n_paramsp;
  GObjectClass *class = g_type_class_ref (type);
  GParamSpec *ps;
  GParameter *params;
  SCM kw, val;
  SCM keep_list = scm_params;

  params = g_new0 (GParameter, n_params);

  for (i = 0; i < n_params; i++)
    {
      kw = SCM_CAR (scm_params);
      val = SCM_CADR (scm_params);
      scm_params = SCM_CDDR (scm_params);

      if (scm_is_symbol (kw))
        kw = scm_symbol_to_string (kw);
      else if (scm_is_keyword (kw))
        kw = scm_symbol_to_string (scm_keyword_to_symbol (kw));
      else
        scm_misc_error (subr, "bad keyword: ~S", scm_list_1 (kw));
      kw = sgtk_to_cstr (kw);
      keep_list = scm_cons (kw, keep_list);
      params[i].name = sgtk_cstr2ptr (kw, 2*i+1, subr);

      ps = g_object_class_find_property (class, params[i].name);
      if (ps == NULL)
        {
      	  fprintf (stderr, "no such arg for type `%s': %s\n",
      		   g_type_name (type), params[i].name);
          n_params--;
          i--;
          continue;
        }

      /* run conversion like from a string for GdkFont, including string
         becomes cstr for GTK_TYPE_STRING */
      val = sgtk_apply_conversion (ps->value_type, val, &keep_list);

      g_value_init (&params[i].value, ps->value_type);
      sgtk_scm2gvalue (&params[i].value, val);
    }

  g_type_class_unref (class);
  *paramsp = params;
  *n_paramsp = n_params;
  return keep_list;
}

SCM
sgtk_gtk_object_new (SCM type_obj, SCM scm_params)
{
  const char func_name[] = "gtk-object-new";

  int n_params, i;
  GParameter *params;
  GObject *obj;
  SCM keep_list, scm_obj = SCM_EOL;
  GType type;

  SCM_ASSERT (type_obj != SCM_BOOL_F
              && sgtk_valid_type (type_obj), type_obj, SCM_ARG1,
              func_name);

  type = sgtk_scm2type (type_obj);

  if (G_TYPE_IS_ABSTRACT (type))
    scm_misc_error (func_name,
                    "type is abstract, cannot create instance: ~S",
                    scm_list_1 (type_obj));

  n_params = scm_ilength (scm_params);
  SCM_ASSERT (n_params >= 0 && (n_params%2) == 0, scm_params,
	      SCM_ARG2, func_name);
  n_params = n_params/2;

  keep_list = sgtk_build_params (type, &params, &n_params,
                                 scm_params, func_name);
  obj = g_object_newv (type, n_params, params);

  for (i = 0; i < n_params; i++)
    g_value_unset (&params[i].value);
  g_free (params);
  scm_remember_upto_here_1 (keep_list);

  scm_obj = sgtk_wrap_gtkobj (obj);
  return scm_obj;
}

SCM
sgtk_gtk_object_set (SCM scm_obj, SCM scm_params)
{
  static char func_name[] = "gtk-object-new";

  int n_params, i;
  GParameter *params;
  GObject *obj;
  SCM keep_list;
  GType type;

  SCM_ASSERT (GTKOBJP(scm_obj), scm_obj, SCM_ARG1, func_name);
  obj = GTKOBJ_PROXY(scm_obj)->obj;

  n_params = scm_ilength (scm_params);
  SCM_ASSERT (n_params >= 0 && (n_params%2) == 0, scm_params,
	      SCM_ARG2, func_name);
  n_params = n_params/2;

  type = G_OBJECT_TYPE (obj);
  keep_list = sgtk_build_params (type, &params, &n_params,
                                 scm_params, func_name);
  for (i = 0; i < n_params; i++)
    {
      g_object_set_property (obj, params[i].name, &params[i].value);
      g_value_unset (&params[i].value);
    }

  g_free (params);
  scm_remember_upto_here_1 (keep_list);
  return SCM_UNSPECIFIED;
}

SCM
sgtk_gtk_object_get (SCM scm_obj, SCM argsym)
{
  static const char func_name[] = "gtk-object-get";
  GObject *obj;
  GObjectClass *class;
  char *name;
  GParamSpec *ps;
  GValue value;
  SCM ret;

  SCM_ASSERT (GTKOBJP(scm_obj), scm_obj, SCM_ARG1, func_name);
  SCM_ASSERT (scm_is_keyword (argsym) || scm_is_symbol (argsym), argsym,
	      SCM_ARG2, func_name);

  obj = GTKOBJ_PROXY(scm_obj)->obj;

  if (scm_is_symbol (argsym))
    name = sgtk_symbol_to_locale_string (argsym);
  else
    name = sgtk_keyword_to_locale_string (argsym);

  class = G_OBJECT_GET_CLASS (G_OBJECT (obj));
  ps = g_object_class_find_property (class, name);
  if (ps == NULL)
    {
      free (name);
      return SCM_BOOL_F;
    }

  memset (&value, '\0', sizeof(value));
  g_value_init (&value, ps->value_type);
  g_object_get_property (obj, name, &value);
  ret = sgtk_gvalue2scm (&value, 0);

  g_value_unset (&value);
  free (name);
  return ret;
}



/* Creating new object classes */

GtkType
gtk_class_new (GtkType parent_type, gchar *name)
{
  GTypeInfo *info;
  GTypeQuery parent_query;

  g_type_query (parent_type, &parent_query);
  if (parent_query.type == 0)
    return GTK_TYPE_INVALID;

  info = g_new0 (GTypeInfo, 1);
  info->class_size = parent_query.class_size;
  info->base_init = NULL;
  info->base_finalize = NULL;
  info->class_init = NULL;
  info->class_finalize = NULL;
  info->class_data = NULL;
  info->instance_size = parent_query.instance_size;
  info->n_preallocs = 0;
  info->instance_init = NULL;
  info->value_table = NULL;

  return g_type_register_static (parent_type, name, info, 0);
}


void
sgtk_closure_destroy (gpointer data, GClosure *closure)
{
  sgtk_protshell *protshell = (sgtk_protshell *) data;
  sgtk_unprotect (protshell);
}

static const char s_gtk_signal_connect[] = "gtk-signal-connect";

gulong
gtk_signal_connect_interp (SCM p_object, const gchar *name, SCM p_func, gboolean object_signal, gboolean after)
{
  GObject *c_object;
  GClosure *closure;
  sgtk_protshell *protshell;

  SCM_ASSERT (GTKOBJP(p_object), p_object, SCM_ARG1, s_gtk_signal_connect);
  c_object = GTKOBJ_PROXY(p_object)->obj;

  SCM_ASSERT (scm_is_true (scm_procedure_p (p_func)), p_func, SCM_ARG3,
              s_gtk_signal_connect);

  protshell = sgtk_protect (p_object, p_func);

  closure = g_closure_new_simple (sizeof (GClosure), protshell);
  g_closure_set_marshal (closure, sgtk_closure_marshal);
  g_closure_add_finalize_notifier (closure, protshell, sgtk_closure_destroy);

  return g_signal_connect_closure (c_object, name, closure, after);
}


/* Note that gtk_marshal_NONE__NONE given to gtk_signal_newv clearly won't
   match the return_type and params.  This means in general the created
   signal cannot be used by plain C code gtk_signal_connect.  The Scheme
   level gtk-signal-connect is fine though, it always does a
   gtk_signal_connect_full asking for sgtk_callback_marshal.  */

guint
gtk_signal_new_generic (const gchar     *name,
			GtkSignalRunType signal_flags,
			GtkType          type,
			GtkType          return_type,
			guint            nparams,
			GtkType         *params)
{
  guint signal_id;

  if (! G_TYPE_IS_OBJECT (type))
    scm_misc_error ("gtk-signal-new-generic",
                    "type not subclass of GObject: ~S",
                    scm_list_1 (sgtk_type_name (type)));

  signal_id = gtk_signal_newv (name, signal_flags, type,
			       0, gtk_marshal_NONE__NONE,
			       return_type, nparams, params);

  /* Believe this is unnecessary in Gtk 2.0. */
  /*   if (signal_id > 0) */
  /*     gtk_object_class_add_signals (gtk_type_class (type), */
  /* 				  &signal_id, 1); */

  return signal_id;
}

SCM
sgtk_signal_emit (GObject *obj, char *name, SCM scm_params)
{
  static char func_name[] = "gtk-signal-emit";
  
  GSignalQuery info;
  guint signal_id, i;
  guint n_params;
  GValue *params;
  SCM params_cblk;
  SCM keep_list = scm_params;
  SCM ret;

  signal_id = g_signal_lookup (name, G_OBJECT_TYPE (obj));
  if (signal_id == 0)
    scm_misc_error (func_name, "no such signal: ~S",
                    scm_cons (scm_makfrom0str (name), SCM_EOL));

  g_signal_query (signal_id, &info);
  n_params = info.n_params;

  if (scm_ilength (scm_params) != n_params)
    scm_misc_error (func_name, "wrong number of signal arguments",
                    SCM_EOL);

  params = g_new0 (GValue, n_params+1);
  params_cblk = sgtk_make_cblk (params, sizeof (params[0]) * (n_params+1));

  g_value_init (&params[0], G_TYPE_OBJECT);
  g_value_set_object (&params[0], obj);

  for (i = 0; i < n_params; i++)
    {
      GType type = info.param_types[i];
      GValue *par = &params[i+1];
      SCM orig_val = scm_car (scm_params);
      SCM val = orig_val;
      scm_params = scm_cdr (scm_params);

      /* run conversion like from a string for GdkFont, including string
         becomes cstr for GTK_TYPE_STRING */
      val = sgtk_apply_conversion (type, val, &keep_list);

      if (!sgtk_valid_for_type (type, val))
        scm_misc_error (func_name, "wrong type argument for type ~A: ~S",
                        scm_list_2 (sgtk_type_name (type), orig_val));

      g_value_init (par, type);
      sgtk_scm2gvalue (par, val);
    }

  if (G_TYPE_FUNDAMENTAL (info.return_type) == G_TYPE_NONE)
    {
      g_signal_emitv (params, signal_id, 0, NULL);
      ret = SCM_UNSPECIFIED;
    }
  else
    {
      GValue retval;
      memset (&retval, '\0', sizeof(retval));
      g_value_init (&retval, info.return_type);
      g_signal_emitv (params, signal_id, 0, &retval);
      ret = sgtk_gvalue2scm (&retval, 0);
      g_value_unset (&retval);
    }

  for (i = 0; i < n_params+1; i++)
    g_value_unset (&params[i]);

  scm_remember_upto_here_1 (params_cblk);
  scm_remember_upto_here_1 (keep_list);

  return ret;
}

SCM_SYMBOL (sgtk_symbol_g_error, "g-error");

/* `msg' "~a" is used in case gerr->message contains any "~" characters.  */
void
sgtk_throw_gerror (const char *func_name, GError *gerr)
{
  SCM key = sgtk_symbol_g_error;
  SCM subr = scm_from_locale_string (func_name);
  SCM msg = scm_from_locale_string ("~a");
  SCM args = scm_list_1 (scm_from_locale_string (gerr->message));
  SCM data = scm_list_2 (scm_from_locale_symbol (g_quark_to_string
                                                 (gerr->domain)),
                         scm_from_int (gerr->code));
  g_error_free (gerr);
  scm_error_scm (key, subr, msg, args, data);
}



/* Initialization */

static int standalone_p = 1;

void
sgtk_set_standalone (int flag)
{
  standalone_p = flag;
}

int
sgtk_is_standalone ()
{
  return standalone_p;
}

SCM
sgtk_standalone_p ()
{
  return standalone_p? SCM_BOOL_T : SCM_BOOL_F;
}

void
sgtk_register_glue (char *name, void (*func)(void))
{
  scm_c_register_extension (NULL, name, (void (*)(void *))func, NULL);
}

#ifdef HAVE_LIBGUILE_DEPRECATION_H
#include <libguile/deprecation.h>
#endif

void
sgtk_issue_deprecation_warning (const char* msg)
{
#ifdef HAVE_SCM_C_ISSUE_DEPRECATION_WARNING
  scm_c_issue_deprecation_warning (msg);
#endif
}


SCM_SYMBOL (sym_top_repl, "top-repl");
SCM_SYMBOL (sym_quit, "quit");
SCM_SYMBOL (sym_use_modules, "use-modules");
SCM_SYMBOL (sym_gtk, "gtk");
SCM_SYMBOL (sym_repl, "repl");
SCM_SYMBOL (sym_gtk_repl, "gtk-repl");
SCM_SYMBOL (sym_gtk_version, "gtk-2.0");

static void
sgtk_init_substrate (void)
{	
  cstr_smob_type = scm_make_smob_type (cstr_name, 0);
  scm_set_smob_free (cstr_smob_type, cstr_free);
  
  cblk_smob_type = scm_make_smob_type (cblk_name, 0);
  scm_set_smob_free (cblk_smob_type, cblk_free);
  
  tc16_gtkobj_marker_hook = scm_make_smob_type ("gtkobj_marker_hook", sizeof(sgtk_object_proxy));
  scm_set_smob_mark (tc16_gtkobj_marker_hook, gtkobj_marker_hook);
  scm_set_smob_print (tc16_gtkobj_marker_hook, gtkobj_marker_hook_print);
    
  tc16_gtkobj = scm_make_smob_type ("gtkobj", sizeof(sgtk_object_proxy));
  scm_set_smob_mark (tc16_gtkobj, gtkobj_mark);
  scm_set_smob_free (tc16_gtkobj, gtkobj_free);
  scm_set_smob_print (tc16_gtkobj, gtkobj_print);
  
  tc16_boxed = scm_make_smob_type ("gtkboxed", sizeof(sgtk_boxed_info));
  scm_set_smob_free (tc16_boxed, boxed_free);
  scm_set_smob_print (tc16_boxed, boxed_print);
   
  tc16_gtktype = scm_make_smob_type ("gtktype", sizeof(sgtk_type_info));
  scm_set_smob_mark (tc16_gtktype, scm_mark0);
  scm_set_smob_free (tc16_gtktype, scm_free0);
  scm_set_smob_print (tc16_gtktype, gtktype_print);
  scm_set_smob_equalp (tc16_gtktype, gtktype_equalp);
  
  tc16_illobj = scm_make_smob_type ("gtkillobj", sizeof(GtkType));
  scm_set_smob_mark (tc16_illobj, scm_mark0);
  scm_set_smob_free (tc16_illobj, scm_free0);
  scm_set_smob_print (tc16_illobj, illobj_print);
  
  global_protects = NULL;
  sgtk_protshell_chunk = g_mem_chunk_create (sgtk_protshell, 128,
					     G_ALLOC_AND_FREE);
  install_marker_hook ();

  callback_trampoline = scm_permanent_object (scm_cons (SCM_BOOL_F, SCM_EOL));

  /* Get our conversion function sgtk_to_cstr() into the sgtk_type_info
     record.  This conversion is as per build-guile-gtk-1.2 and is used from
     here by sgtk_signal_emit, sgtk_gtk_object_new, sgtk_gtk_object_set.  */
  {
    static sgtk_type_info string_info = {
      "string", GTK_TYPE_STRING, sgtk_to_cstr
    };
    enter_type_info (&string_info);
  }

#ifndef SCM_MAGIC_SNARFER
#ifndef MKDEP
#include "guile-gtk.x"
#endif /* MKDEP */
#endif /* SCM_MAGIC_SNARFER */
}

static int sgtk_inited = 0;

void
sgtk_init_with_args (int *argcp, char ***argvp)
{
  if (sgtk_inited)
    return;

  /* XXX - Initialize Gtk only once.  We assume that Gtk has already
     been initialized when Gdk has.  That is not completely correct,
     but the best I can do. */

  /* must have glib threads setup before gdk_init runs, since gdk_init
     will use that to initialize gdk_threads_mutex */
  sgtk_init_threads ();

  if (gdk_display == NULL)
    {
      gtk_set_locale ();
      gtk_init (argcp, argvp);
    }
  sgtk_init_substrate ();
  sgtk_inited = 1;
}

/* Initialize guile-gtk, passing the Guile level `program-arguments' to
   gdk_init and gtk_init, and putting back in `program-arguments' whatever
   those two functions leave (they strip arguments they understand, like
   "--display").

   The strings passed in the argv[] are new malloced copies, and they're not
   freed.  Probably they could be.  It's been this way (not freeing) for
   quite a while.

   There's no need for any trouble over memory leaks on error throws here.
   This code is executed just once, and if it fails then nothing at all can
   be used.  */

void
sgtk_init (void)
{
  SCM lst = scm_program_arguments ();
  int argc = scm_to_int (scm_length (lst));
  char **argv;
  int i;

  argv = (char **) scm_malloc ((argc+1) * sizeof(char*));
  for (i = 0; i < argc; i++, lst = SCM_CDR (lst))
      argv[i] = scm_to_locale_string (SCM_CAR (lst));
  argv[argc] = NULL;

  sgtk_init_with_args (&argc, &argv);
  scm_set_program_arguments (argc, argv, NULL);
  free (argv);
}

static SCM
hack_compiled_switches (SCM script)
{
  SCM last_action;

  script = scm_reverse_x (script, SCM_UNDEFINED);
  last_action = SCM_CAR (script);
  SCM_SETCAR (script, scm_list_2 (sym_use_modules,
				  scm_list_2 (sym_gtk_version, sym_gtk)));
  script = scm_cons (scm_list_2 (sym_use_modules,
				 scm_list_2 (sym_gtk, sym_repl)),
		     script);
  
  if (scm_is_eq (SCM_CAR (last_action), sym_top_repl))
    {
      script = scm_cons (scm_list_1 (sym_gtk_repl), script);
      sgtk_set_standalone (0);
    }
  else if (! scm_is_eq (SCM_CAR (last_action), sym_quit))
    {
      fprintf (stderr, "guile-gtk: unknown action in startup script\n");
      scm_display (last_action, SCM_UNDEFINED);
      scm_newline (SCM_UNDEFINED);
      exit (1);
    }

  return scm_reverse_x (script, SCM_UNDEFINED);
}

void
sgtk_shell (int argc, char **argv)
{
  SCM script;

  sgtk_init_with_args (&argc, &argv);

  /* If present, add SCSH-style meta-arguments from the top of the
     script file to the argument vector.  See the SCSH manual: "The
     meta argument" for more details.  */
  {
    char **new_argv = scm_get_meta_args (argc, argv);

    if (new_argv)
      {
	argv = new_argv;
	argc = scm_count_argv (new_argv);
      }
  }

  script = hack_compiled_switches (scm_compile_shell_switches (argc, argv));
  scm_eval_x (script, scm_current_module ());
  exit (0);
}
