/* $NetBSD$ */

/*
 * Copyright (c) 2003 Dennis I. Chernoivanov
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. The name of the author may not be used to endorse or promote products
 *    derived from this software without specific prior written permission
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
 * IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
 * INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 * NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
 * THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ctype.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "paneld.h"

#define INET6_BUFSZ	45
#define INET6_FORMAT \
"%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x:%02x%02x"

static int	inet6_open(struct menu *m);
static int	inet6_close(struct menu *m);
static int	inet6_next(struct menu *m);
static int	inet6_prev(struct menu *m);
static int	inet6_inc(struct menu *m);
static int	inet6_dec(struct menu *m);
static int	inet6_select(struct menu *m);
static char*	inet6_allocbuf(int *sz);

static int
inet6_open(struct menu *m)
{
	const u_char *p;
	struct sockaddr_in6 addr;

	if (exec_read(m) != E_OK)
		return E_IO;
	if (inet_pton(AF_INET6, m->io.buf, &addr.sin6_addr) != 1)
		return E_IO;

	p = (const u_char*)&addr.sin6_addr;
	sprintf(m->io.buf, INET6_FORMAT,
			p[0],  p[1],  p[2],  p[3],
			p[4],  p[5],  p[6],  p[7],
			p[8],  p[9],  p[10], p[11],
			p[12], p[13], p[14], p[15]);
	return E_OK;
}

static int
inet6_close(struct menu *m)
{
	struct sockaddr_in6 addr;

	util_ipstrip(AF_INET6, m->io.buf);
	if (inet_pton(AF_INET6, m->io.buf, &addr.sin6_addr) != 1)
		return E_BADV;

	return exec_write(m);
}

static int
inet6_next(struct menu *m)
{
	if (window->edit()->right() == E_OK) {
		int c;
		if (window->edit()->getchr(&c) == E_OK) {
			if (c == ':')
				return window->edit()->right();
			return E_OK;
		}
	}
	return E_IO;
}

static int
inet6_prev(struct menu *m)
{
	if (window->edit()->left() == E_OK) {
		int c;
		if (window->edit()->getchr(&c) == E_OK) {
			if (c == ':')
				return window->edit()->left();
			return E_OK;
		}
	}
	return E_IO;
}

static int
inet6_inc(struct menu *m)
{
	int c;
	if (window->edit()->getchr(&c) == E_OK) {
		if (c == '9')
			window->edit()->putchr('a');
		else if (c == 'f')
			window->edit()->putchr('0');
		else
			window->edit()->putchr(++c);
		return E_OK;
	}
	return E_IO;
}

static int
inet6_dec(struct menu *m)
{
	int c;
	if (window->edit()->getchr(&c) == E_OK) {
		if (c == '0')
			window->edit()->putchr('f');
		else if (c == 'a')
			window->edit()->putchr('9');
		else
			window->edit()->putchr(--c);
		return E_OK;
	}
	return E_IO;
}

static int
inet6_select(struct menu *m)
{
	if (inet6_open(m) == E_OK) {
		window->set_title(m->nm);
		window->edit()->setbuf(m->io.buf, ALIGN_ANY);

		if (enter_menu(m) == E_SELECT)
			return inet6_close(m);
	}

	return E_IO;
}

static char*
inet6_allocbuf(int *sz)
{
	*sz = INET6_BUFSZ + 1;
	return (char*)cf_malloc(INET6_BUFSZ + 1);
}

struct menu_ops*
get_inet6_ops(void)
{
	static struct menu_ops inet6_ops = {
		inet6_next,
		inet6_prev,
		inet6_inc,
		inet6_dec,
		inet6_select,
		inet6_allocbuf
	};
	return &inet6_ops;
}
