
/*
 * hfs.c
 *
 *   July 10, 1994 -- Lawrence Kesteloot
 */

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <fcntl.h>
#include <ctype.h>
#include <string.h>

#include "util.h"
#include "hfs.h"
#include "btree.h"

int	debug = 0;

/*
 * hfsGetVolumeInfo()
 *
 *   Returns 1 and fills vi if the filename corresponds to a device
 *   (partition) on which there is an HFS file system.  Returns 0
 *   for any other error.
 */

int hfsGetVolumeInfo (char *fn, struct hfsVolumeInfo *vi)
{
	int	f, n;

	f = open (fn, O_RDONLY);
	if (f == -1) {
		return 0;
	}

	if (lseek (f, BLKSIZ * 2, SEEK_SET) == -1) {
		close (f);
		return 0;
	}

	n = read (f, vi, sizeof (struct hfsVolumeInfo));

	close (f);

	if (n != sizeof (struct hfsVolumeInfo)) {
		return 0;
	}

	if (vi->drSigWord != 0x4244) {
		return 0;
	}

	return 1;
}

/*
 * hfsGetRootDir()
 *
 *   Fills ei with the information for the root (top-level) directory
 *   of the partition.  The top-level directory is named after the
 *   partition name.  Returns 1 on success or 0 on failure.
 */

int hfsGetRootDir (int f, int ctblk, char *partname, struct hfsEntryInfo *ei)
{
	int	namelen, keylen;
	uchar	buf[64];

	dprintf ("hfsGetRootDir: enter\n");
	namelen = strlen (partname);
	keylen = namelen + 6;  /* ?(byte) + strlen(byte) + parentID(long) */

	memset (buf, '\0', keylen);
	*(ulong *)&buf[1] = 1; /* Root dir parent ID */
	buf[5] = namelen;
	memcpy (&buf[6], partname, namelen);
	ei->blk = btreeGetLeaf (f, ctblk, 0, buf, keylen, ISCATALOG);
	dprintf ("Result = 0x%x\n", ei->blk);
	ei->rec = btreeGetRecordNum (f, ctblk, ei->blk, buf, keylen, ISCATALOG);
	if (ei->rec == -1) {
		printf ("hfsGetRootDir: Not found.\n");
		return 0;
	}
	strcpy (ei->name, partname);
	ei->left = 0;  /* Filled by hfsGetFirst */
	{ int i;
	i = hfsGetRecord (f, ctblk, ei);
	dprintf ("hfsGetRootDir: exit\n");
	return i;
	}
}

/*
 * hfsGetRecord()
 *
 *   Given an entry (with block and record), fills the information for
 *   it and gets its name.  Returns 1 on success or 0 on failure.
 */

int hfsGetRecord (int f, int ctblk, struct hfsEntryInfo *ei)
{
	uchar	key[64];
	int	res;

	res = btreeGetRecord (f, ctblk, &ei->blk, &ei->rec, &ei->info, key);
	memcpy (ei->name, &key[7], key[6]);
	ei->name[key[6]] = '\0';

	return res;
}

/*
 * hfsGetFirst()
 *
 *   Given the information for a directory entry, this routine fills
 *   ei with the information for the first entry in that directory.
 *   The rest of the files in the directory can be retrieved with
 *   the hfsGetNext() call.  Returns 1 on success or 0 on failure.
 */

int hfsGetFirst (int f, int ctblk, struct hfsEntryInfo *ei)
{
	uchar	buf[32];

	dprintf ("hfsGetFirst: enter\n");

	ei->left = ei->info.dir.dirVal; /* # of files in directory */

	memset (buf, '\0', 7);
	*(long *)(buf + 1) = ei->info.dir.dirDirID;
	ei->blk = btreeGetLeaf (f, ctblk, 0, buf, 7, ISCATALOG);
	if (ei->blk == -1) {
		printf ("hfsGetFirst: Didn't find thread for directory.\n");
		return 0;
	}

	ei->rec = btreeGetRecordNum (f, ctblk, ei->blk, buf, 7, ISCATALOG);

	return hfsGetNext (f, ctblk, ei);
}

/*
 * hfsGetNext()
 *
 *   Given an entry in a directory, fills ei with the next entry
 *   in that directory.  Returns 1 on success or 0 on failure, such
 *   as the end of the directory having been reached.
 */

int hfsGetNext (int f, int ctblk, struct hfsEntryInfo *ei)
{
	ei->rec++;

	if (ei->left == 0 || hfsGetRecord (f, ctblk, ei) == 0) {
		return 0;
	}

	ei->left--;

	return 1;
}

/*
 * hfsDump()
 *
 *   This not-particularly-useful routine dumps a block to the screen.
 */

void hfsDump (int f, int blk)
{
	char	buf[512];

	getbytes (f, blk * BLKSIZ, buf, 512);
	printbytes (buf, 512);
}

/*
 * hfsReadBlock()
 *
 *   Given a filesystem, entry information, and a block number, this
 *   routine fills "buf" with that block.  The size of the block is
 *   vi->drAlBlkSiz.
 *
 *   This information is not in the HFS documentation: the file
 *   information (hfsFileInfo) has a filExtRec array which keeps
 *   three startblock/length pairs.  These are the first three
 *   extents of the file.  Other extents are kept in the extents
 *   tree and referenced by the file number (filFlNum).  Each record
 *   in the btree also has three pairs.
 *
 *   This routine is very slow because it may possibly go through the
 *   extents tree for every access.  It could be sped up drastically
 *   by taking advantage of the locality between calls.  Maybe the
 *   ei structure could be used to keep hints.
 *
 *   This routine should probably not be called directly by the
 *   user -- hfsRead() should be used instead because it checks
 *   the parameters against the length of the file.
 */

int hfsReadBlock (int f, struct hfsVolumeInfo *vi, int xtblk,
	struct hfsEntryInfo *ei, void *buf, int blknum)
{
	int	ext, blkcnt;
	ulong	blk, rec;
	uchar	key[16];
	ushort	info[6];

	if (ei->info.filetype == 1) {
		printf ("\"%s\" is a directory.\n", ei->name);
		return 0;
	}
	if (ei->info.filetype != 2) {
		printf ("hfsReadBlock: can't handle files of type %d\n",
			(int)ei->info.filetype);
		return 0;
	}

	blkcnt = blknum;
	for (ext = 0; ext < 3; ext++) {
		if (ei->info.fil.filExtRec[ext * 2 + 1] <= blkcnt) {
			blkcnt -= ei->info.fil.filExtRec[ext * 2 + 1];
		} else {
			break;
		}
	}
	dprintf ("Looking for block %d, found blk %d, ext %d\n", blknum,
		blkcnt, ext);

	if (ext < 3) {
		blk = ei->info.fil.filExtRec[ext * 2] + blkcnt;
		getbytes (f, vi->drAlBlSt * BLKSIZ + blk * vi->drAlBlkSiz,
			buf, vi->drAlBlkSiz);
		return 1;
	}

	/* ext >= 3, so we must use the extents tree to find the data */

	key[0] = 0x00; /* Data fork */
	*(ulong *)&key[1] = ei->info.fil.filFlNum; /* File number */
	*(ushort *)&key[5] = blknum; /* Allocation block number */

	blk = btreeGetLeaf (f, xtblk, 0, key, 7, ISEXTENT);
	if (blk == -1) {
		printf ("hfsReadBlock: btreeGetLeaf failed\n");
		return 0;
	}

	rec = btreeGetRecordNum (f, xtblk, blk, key, 7, ISEXTENT);
	if (rec == -1) {
		printf ("hfsReadBlock: btreeGetRecordNum failed.\n");
		return 0;
	}

	if (btreeGetRecord (f, xtblk, &blk, &rec, info, key) == 0) {
		printf ("hfsReadBlock: btreeGetRecord failed.\n");
		return 0;
	}

	if (*(ulong *)&key[2] != ei->info.fil.filFlNum) {
		printf ("hfsReadBlock: file not found in data extents tree.\n");
		return 0;
	}

	blkcnt = blknum - *(ushort *)&key[6];
	dprintf ("Looking for block %d, found rec %d, cnt = %d\n",
		blknum, (int)*(ushort *)&key[6], blkcnt);
	for (ext = 0; ext < 3; ext++) {
		dprintf ("len = %d\n", (int)info[ext*2+1]);
		if (info[ext * 2 + 1] <= blkcnt) {
			blkcnt -= info[ext * 2 + 1];
		} else {
			break;
		}
	}

	if (ext < 3) {
		blk = info[ext * 2] + blkcnt;
		getbytes (f, vi->drAlBlSt * BLKSIZ + blk * vi->drAlBlkSiz,
			buf, vi->drAlBlkSiz);
		return 1;
	}

	printf ("hfsReadBlock: Block %d is past the end of the file.\n",
		blknum);

	return 0;
}

/*
 * hfsRead()
 *
 *   Reads an arbitrary chuck of data from a file and returns the number
 *   of bytes read.  Returns -1 on failure.
 */

int hfsRead (int f, struct hfsVolumeInfo *vi, int xtblk,
	struct hfsEntryInfo *ei, void *buf, int loc, int len)
{
	int	startblk, endblk, filelen, pos, i, left;
	uchar	localbuf[32768]; /* XXX Is this ever too small? */

	if (ei->info.filetype == 1) {
		printf ("\"%s\" is a directory.\n", ei->name);
		return -1;
	}
	if (ei->info.filetype != 2) {
		printf ("hfsRead: can't handle files of type %d\n",
			(int)ei->info.filetype);
		return -1;
	}

	if (sizeof (localbuf) < vi->drAlBlkSiz) {
		printf ("hfsRead: local block too small (should be %d)\n",
			vi->drAlBlkSiz);
		return -1;
	}

	filelen = ei->info.fil.filLgLen;
	if (loc + len > filelen) {
		len = filelen - loc;
	}
	if (len <= 0) {
		return 0;
	}

	left = len;

	startblk = loc / vi->drAlBlkSiz;
	endblk = (loc + len - 1) / vi->drAlBlkSiz;

	if (hfsReadBlock (f, vi, xtblk, ei, localbuf, startblk) == 0) {
		printf ("hfsRead: hfsReadBlock (1st) failed.\n");
		return -1;
	}
	if (startblk == endblk) {
		memcpy (buf, localbuf + loc % vi->drAlBlkSiz, len);
		return len;
	}
	pos = vi->drAlBlkSiz - loc % vi->drAlBlkSiz;
	memcpy (buf, localbuf + loc % vi->drAlBlkSiz, pos);
	left -= pos;

	for (i = startblk + 1; i < endblk; i++) {
		if (hfsReadBlock (f, vi, xtblk, ei, localbuf, i) == 0) {
			printf ("hfsRead: hfsReadBlock (2nd) failed.\n");
			return -1;
		}
		memcpy ((uchar *)buf + pos, localbuf, vi->drAlBlkSiz);
		pos += vi->drAlBlkSiz;
		left -= vi->drAlBlkSiz;
	}

	if (hfsReadBlock (f, vi, xtblk, ei, localbuf, endblk) == 0) {
		printf ("hfsRead: hfsReadBlock (3rd) failed.\n");
		return -1;
	}
	memcpy ((uchar *)buf + pos, localbuf, left);

	return len;
}
