/* elf2aout.c

   This program converts an elf executable to a NetBSD a.out executable.
   The minimal symbol table is copied, but the debugging symbols and
   other informational sections are not. */

#include <sys/types.h>
#include <fcntl.h>
#include <unistd.h>
#include <machine/elf.h>

main (int argc, char **argv, char **envp)
{
  int count;
  struct ehdr ex;
  struct phdr ph;
  struct shdr sh [32];
  char *contents [32];
  struct sym *symtab;
  int nsyms;
  char *strtab;
  int i;
  off_t off;
  unsigned char buf [4];

  count = read (0, &ex, sizeof ex);
  if (count != sizeof ex)
    {
      if (count < 0)
        perror ("ex: read");
      else
	printf ("ex: short read: %d bytes instead of %d.\n",
		count, sizeof ex);
      exit (1);
    }
  printf ("magic %d %3.3s %x %x %x\n",
	  ex.elf_magic [0], &ex.elf_magic [1],
	  ex.magic [0], ex.magic [1], ex.magic [2]);
  printf ("type %x  machine %x  version %x\n",
	  ex.type, ex.machine, ex.version);
  printf ("entry point %x  phoff %x  shoff %x  flags %x\n",
	  ex.entry, ex.phoff, ex.shoff, ex.flags);
  printf ("ehsize %x  phsize %x  phcount %x\n",
	  ex.ehsize, ex.phsize, ex.phcount);
  printf ("shsize %x  shcount %x  shstrndx %x\n",
	  ex.shsize, ex.shcount, ex.shstrndx);
  for (i = 0; i < ex.phcount; i++)
    {
      if ((off = lseek (0, ex.phoff + i * sizeof ph, SEEK_SET)) < 0)
	{
	  perror ("ph: lseek");
	  exit (1);
	}
      if ((count = read (0, &ph, sizeof ph)) < 0)
	{
	  perror ("ph: read");
	  exit (1);
	}
      else if (count != sizeof ph)
	{
	  printf ("ph: short read: %d bytes instead of %d.\n",
		  count, sizeof ph);
	  exit (1);
	}
      if (ph.type == PT_LOAD)
	{
	  if ((off = lseek (0, ph.offset, SEEK_SET)) < 0)
            {
              perror ("ph contents: lseek");
              exit (1);
            }
	  if ((count = read (0, buf, sizeof buf)) < 0)
	    {
	      perror ("ph contents: read");
	      exit (1);
	    }
	  else if (count != sizeof buf)
	    {
	      printf ("ph contents: short read: %d bytes instead of %d.\n",
		      count, sizeof buf);
	    }
	  printf ("Section %d at %x, len %x  addr %x  size %x\n",
		  i, ph.offset, ph.filesz, ph.vaddr, ph.memsz);
	}
    }
  for (i = 0; i < ex.shcount; i++)
    {
      if ((off = lseek (0, ex.shoff + i * sizeof (struct shdr), SEEK_SET)) < 0)
	{
	  perror ("sh: lseek");
	  exit (1);
	}
      if ((count = read (0, &sh [i], sizeof (struct shdr))) < 0)
	{
	  perror ("sh: read");
	  exit (1);
	}
      else if (count != sizeof (struct shdr))
	{
	  printf ("sh: short read: %d bytes instead of %d.\n",
		  count, sizeof (struct shdr));
	  exit (1);
	}
      contents [i] = (char *)malloc (sh [i].size);
      if (!contents [i])
	{
	  printf ("Can't allocate section %d contents\n", i);
	  exit (1);
	}
      if ((off = lseek (0, sh [i].offset, SEEK_SET)) < 0)
	{
	  perror ("contents: lseek");
	  exit (1);
	}
      if ((count = read (0, contents [i], sh [i].size)) < 0)
	{
	  perror ("contents: read");
	  exit (1);
	}
      else if (count != sh [i].size)
	{
	  printf ("contents: short read: %d bytes instead of %d.\n",
		  count, sh [i].size);
	  exit (1);
	}
    }
  for (i = 0; i < ex.shcount; i++)
    {
      char *name = contents [ex.shstrndx] + sh [i].name;
      printf ("name %s  type %x  flags %x  addr %x\n",
	      name, sh [i].type, sh [i].flags, sh [i].addr);
      printf ("offset %x  size %x  link %x  info %x\n",
	      sh [i].offset, sh [i].size, sh [i].link, sh [i].info);
      printf ("align %x  esize %x\n", sh [i].align, sh [i].esize);
      if (!strcmp (name, ".symtab"))
	{
	  symtab = (struct sym *)contents [i];
	  nsyms = sh [i].size / sizeof (struct sym);
        }
      else if (!strcmp (name, ".strtab"))
	{
	  strtab = contents [i];
        }
    }
  for (i = 0; i < nsyms; i++)
    {
      printf ("name = %s; value = %x; size = %x; ",
	      strtab + symtab [i].name, symtab [i].value, symtab [i].size);
      printf ("type = %x; binding = %x; other = %x; sect = %s\n",
	      symtab [i].type, symtab [i].binding, symtab [i].other,
	      symtab [i].shndx < ex.shcount
	      ? contents [ex.shstrndx] + sh [symtab [i].shndx].name
	      : "--junk--");
    }
}
