/* This file is part of the KDE libraries
    Copyright (c) 1998 Emmeran Seehuber (the_emmy@hotmail.com)
 
    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#ifndef KLDEVICE_H
#define KLDEVICE_H

#include "klchild.h"


/**
* This is the baseclass for all display devices.
*
* A display device may be e.g. 
* <UL>
* <LI> a dialog </LI>
* <LI> a KTMainWindow <LI>
* <LI> a toolbar </LI>
* <LI> or whatever you want </LI>
* </UL>
*
* This is an abstract baseclass.
*/

class KLDevice {
friend class KLDesignerPrivate;
public:
  KLDevice();
  virtual ~KLDevice();

  /**
  * Showes the device and all its containting childs
  *
  * The rootchild must have been set !
  *
  * You normaly dont need to call this directly. Use the
  * show() method of a dervided class
  *
  * @see #hideDevice
  */
  bool showDevice();

  /**
  * Hides the device.
  *
  * You normaly dont need to call this directly.
  *
  * @see #showDevice
  */
  void hideDevice();

  virtual void setDeviceSize(unsigned long xSize, unsigned long ySize);

  void relayout();
  void stopRelayout();
  void startRelayout();

  virtual bool refreshDisplay();
  void stopRefresh();
  void startRefresh();

  /**
  * Sets the root child of this device. You should
  * only do this, if the device is not showed !!!
  *
  * The old root child will NOT be deleted !!
  *
  * But this child will be deleted in the destructor.
  *
  * @param rootChild root of the layout
  * @see #rootChild
  */
  void setRootChild( KLChild *rootChild );

  /**
  * @return the actual root child
  * @see #setRootChild
  */
  KLChild *rootChild();

  /**
  * Set configuration environmet to get the UI settings
  * from. UI settings are user specific font/space/...
  * settings.
  */
  void setConfig( KConfigBase *newconfig );

#ifdef VIELLEICHT_WOLLEN_WIR_UNS_DIE_BASIS_PORTABEL_HALTEN
  virtual void *getDeviceHandle() = 0; 
#else
  virtual class QWidget *getDeviceHandle() = 0;
#endif

  /**
  * Grap the input from this device.
  *
  * NOTE: All mouseevent-coords have to be converted to screencoords
  */
  virtual void grapDeviceInput(QObject *graper);

  /**
  * UnGrapes the input
  */
  virtual void releaseDeviceInput();

  /**
  * @return the actual event filter, or 
  *  0 if non set.
  */
  QObject *getGrapper() { return a_grapper; };
  
protected:

  class KLDeviceShowinfo {
  public:
    ulong x;
    ulong y;
    ulong xSize;
    ulong ySize;
    ulong minXSize;
    ulong minYSize;
    ulong maxXSize;
    ulong maxYSize;
    ulong xOff;     // Offset in the REAL window (not in the logical device)
    ulong yOff;
    void init();
    KLDeviceShowinfo() { init(); };
  } dsi;

  // Create the window object
  virtual bool createWindow() = 0;
  // Calculate the window position and size (perhaps overwrite this 
  // in a derivided class)
  virtual bool calcWindowPos( KLDeviceShowinfo *deviceShowInfo );
  // Show the window (at the given pos)
  virtual bool showWindow( KLDeviceShowinfo *deviceShowInfo );
  // Set the window min/max limits
  virtual void setLimits( KLDeviceShowinfo *deviceShowInfo ) = 0;
  // Hide the window
  virtual void hideWindow();
  // destroy the window 
  virtual void deleteWindow();

  // Calcluation funcs: Get the border of the window
  virtual ulong getLeftBorder() const { return 0; };
  virtual ulong getRightBorder() const { return 0; };
  virtual ulong getTopBorder() const { return 0; };
  virtual ulong getBottomBorder() const { return 0; };

  // Calc real window sizes of device size
  void calcRWSize(ulong &xSize, ulong &ySize) const;
  // Calc device sizes of real window sizes
  void calcDSize(ulong &xSize, ulong &ySize) const;
  
  // Get root childs sizes in device sizes
  void getRCSize(ulong &xSize, ulong &ySize) const;

  void deviceGrap(QEvent *e);

protected:
  bool             a_oldAcceptDrops;
  QObject          *a_grapper;
  KLChild          *a_rootChild; // Rootobject (normal a KLGroup)
  KConfigBase      *config;    // Configuration
  bool             showed;      
  bool             setupdone;
  unsigned long    layoutStop;
  bool             doRelayout;
  bool             doingRelayout; // TRUE: A relayout is current in progress
  unsigned long    refreshStop; 
  bool             doRefresh;
};

#endif 
