/***************************************************************************
    Description          : KPuzzle - A KDE Jigsaw Puzzle Game
    Version              : 0.2
    Copyright            : (C) 2000-2001 by Michael Wand
    EMail                : mwand@gmx.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef KPUZZLEAPP_H
#define KPUZZLEAPP_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <qobject.h>

#include <qpoint.h>
#include <qsize.h>
#include <qstring.h>
#include <qpixmap.h>
#include <qlist.h>
#include <qpaintdevice.h>
#include <qtimer.h>

#include <kaction.h>

class KPuzzleView;
class CPicview;

#include "piece.h"

// Game Types - when do you lose a game
// Only the max time counts
#define STD_GAME 1
// You have a certain amount of time for every piece
#define PIECE_TIME_GAME 2
// You have a certain amount of allowed faults for every piece
#define PIECE_FAULTS_GAME 3
// You have a certain amount of allowed faults for the entire game
#define FAULTS_SUM_GAME 4
// You have got a limited time for every piece, then the next piece is shown
// (but there are various rounds)
#define UNBEARABLE_GAME 5

// Ways to turn a piece
#define TURN_LR 1
#define TURN_UD 2

//GameStatus
#define GS_NOT_STARTED 1
#define GS_RUNNING 2
#define GS_SHOW_LARGE 3
#define GS_PAUSED 4

#define GS_WINNING 10
#define GS_SHOW_LARGE_WINNING 11

#define GS_LOSING 20
#define GS_SHOW_LARGE_LOSING 21

#define GS_SHOW_HIGHSCORES 30

// Reasons to terminate
#define TM_WON 1
#define TM_LOST 2
#define TM_QUIT 3
#define TM_FORCE 4

class KPuzzleApp;

class ScoreData // Scoring system
{
public:
	int _maxTime;
	int _timeScore1; // You get ->this score if you stay below
	int _score1Time; // ->this time
	int _timeScore2;
	int _score2Time;
	int _timeScore3;
	int _score3Time;

	int _rightPieceScore;
	int _wrongPieceScore;

	int _bonus; // winning bonus

	int _maxFaults; // PIECE_FAULTS_GAME or FAULTS_SUM_GAME
	int _maxPieceTime; // PIECE_TIME_GAME or UNBEARABLE_GAME (time for every round)
	int _maxRounds; // UNBEARABLE_GAME

        bool save(QDataStream* f) const;
        bool load(QDataStream* f);

        KPuzzleApp* _owner;
};

class GameData // Data which does not change during a game
{
public:
	QString _filename;
        bool _savePixmap;
        QSize _originalPixmapSize;
        QString _annotation;

        char _gameType;
	char _dialog_pieceSize;
	char _difficulty;
	bool _scale;
	bool _useMask;
	int _displace;

	QSize _pieceSize;
	QSize _piecesCount;

        bool save(QDataStream* f) const;
        bool load(QDataStream* f); // ctrlPixmapSize is for assuring that the right pixmap is loaded

        KPuzzleApp* _owner;
};

class CurrentData // Data which changes during the game
{
public:
	QPixmap* _currentPiecePixmap;
	//	int _currentPieceNr;
	CPiece* _currentPieceData;
	QList<CPiece>* _pieceList; // Holds information about the status of every piece
        QListIterator<CPiece>* _pieceListIterator;

	int _score;
	//        char* _gameStatus;
        int _faults;
        int _rounds;
	bool _running; // false when the game has been terminated, e.g. by signal "won"
	int _elapsed; // Elapsed timer ticks
	bool _dirty; // game cannot be won

        bool save(QDataStream* f) const;
        bool load(QDataStream* f);

        KPuzzleApp* _owner;
};

// Main game class. There is only one instance of it, which makes the code
// a bit ugly. Subject to change in future versions. One reason why this is not yet
// a beta version.

class KPuzzleApp : public QObject
{
	Q_OBJECT
 public:
	KPuzzleApp(KPuzzleView* view);
// common init method
	bool initialize();
// init method for new game
	bool initializeNew();
// init method for loaded game
	bool initializeLoad();
// called when the game terminates
	void done();
	virtual ~KPuzzleApp();

// Initializes all game data which must not be saved
	void initGameData();
	void calcPiecesCount();
// Scale the main pixmap, according to the size of the pieces. Returns if the size of the pixmap has changed.
        bool scalePixmap();

// Data
 public:
	// GUI data
	KPuzzleView* mainWidget() const { return _mainWidget; }
	void setMainWidget(KPuzzleView* w) { _mainWidget = w; }
	QPixmap* mainPixmap() const { return _mainPixmap; }
	QPixmap* gamePixmap() const { return _gamePixmap; }
	QPixmap* largePxm() const { return _largePxm; }

        QString gameFilename() const { return _gameFilename; }

        KAction*& actSave() { return _actSave; }
	KAction*& actSaveAs()  { return _actSaveAs; }
	KAction*& actTerminate() { return _actTerminate; }

	KAction*& actShowLarge() { return _actShowLarge; }
        KAction*& actShowMainPixmap() { return _actShowMainPixmap; }
	KAction*& actPause() { return _actPause; }

	// Data which does not change during the game
	ScoreData* scoreData() const { return _scoreData; }
	GameData* gameData() const { return _gameData; }
	int gameType() const { return gameData()->_gameType; }
	QSize pixmapSize() const { return mainPixmap()->size(); }
	QSize pieceSize() const { return gameData()->_pieceSize; }
	QSize pieceSizeDisp() const { return pieceSize() + QSize(2 * displace(),2 * displace()); }
	bool maskedPieces() const { return gameData()->_useMask; }
	int displace() const { return gameData()->_displace; }

	// Data which *does* change
	CurrentData* currentData() const { return _currentData; }
	QPixmap* currentPiecePixmap() const { return currentData()->_currentPiecePixmap; }
	CPiece* currentPieceData() const { return currentData()->_currentPieceData; }
	QPixmap* currentPieceTurned() const;
	QSize piecesCount() const { return gameData()->_piecesCount; }
	int piecesTotal() const { return piecesCount().width() * piecesCount().height(); }
	QList<CPiece>* pieceList() const { return currentData()->_pieceList; }
        QListIterator<CPiece>* pieceListIterator() const { return currentData()->_pieceListIterator; }
	//	int availablePiecesCount() { return currentData()->_pieceList->count(); }

	int status() const { return _status; }
        bool paused() const { return status() == GS_PAUSED; }
	bool running() const { return currentData()->_running; }
	int score() const { return currentData()->_score; }

 protected:
	// Protected access functions
	int elapsed() const { return currentData()->_elapsed; }

	// Protected writing functions
	void setScore(int s) { currentData()->_score = s; emit sigScore(s); }
        QString& gameFilename() { return _gameFilename; }

 protected:
	// GUI data
	KPuzzleView* _mainWidget;
	CPicview* _picview;
	CPicview* _fullview;
	QTimer _timer;
	QPixmap* _mainPixmap;
	QPixmap* _gamePixmap;
	QPixmap* _largePxm;

        QString _gameFilename; // name under which this game was saved

	int _status;

	// This does not change during the game
	// Data from the dialog or the datafile
/* 	QString _filename; */
/* 	char _gameType; */
/* 	char _dialog_pieceSize; */
/* 	char _difficulty; */
/* 	bool _scale; */
/* 	bool _useMask; */
/* 	int _displace; */


	ScoreData* _scoreData;
	GameData* _gameData;
	CurrentData* _currentData;

	// This data forms the game status
/* 	QPixmap* _currentPiece; */
/* 	int _currentPieceNr; */
/* 	QList<PieceState>* _pieceList; // Holds information about the status of every piece */
/* 	QSize _pieceSize; */
/* 	QSize _piecesCount; */

/* 	int _score;  */
/*         char* _gameStatus; */
/*         int _faults; */
/*         int _rounds; */
/* 	bool _running; // false when the game has been terminated, e.g. by signal "won" */
/* 	int _elapsed; // Elapsed timer ticks */
/* 	int _status; */
/* 	bool _dirty; // game cannot be won */


	// Actions for disabling/toggling
	KAction* _actSave;
	KAction* _actSaveAs;
	KAction* _actTerminate;

	KAction* _actShowLarge;
        KAction* _actShowMainPixmap;
	KAction* _actPause;

 private:

// Implementation
 public:
// switch to next/previous piece
	bool setNextPiece();
	bool setPrevPiece();

// Tries to place a piece at the current position. If sucessfull, returns true
	bool setPiece(QPoint pos);
// called by CPiece on timeout
	void hidePiece();

	void changeLRTurn() { currentPieceData()->turn() ^= TURN_LR; }
	void changeUDTurn() { currentPieceData()->turn() ^= TURN_UD; }
	void turnCW() { if (++(currentPieceData()->turn()) > 3)
		currentPieceData()->turn() = 0; }
	void turnCCW() { if (--(currentPieceData()->turn()) < 0)
		currentPieceData()->turn() = 3; }

	QPixmap getLargePixmap(QSize size) const;
        QPixmap getLargeMainPixmap(QSize size) const;

	void setStatus(int s);
	void pause(bool p);

// Main game timer
	void startTimer() { _timer.start(1000); }
	bool timerActive() const { return _timer.isActive(); }
	void stopTimer() { _timer.stop(); }

	void updateAll();

        void stopGame(int why) { emit sigTerm(why); }

// Changes menu items, according to the status and to configuration settings
	void enableMenuItems(int enable);

 protected:
// Writes pixmap of piece pc or current pixmap to pxm of _currentData->_currentPiecePixmap
	void getPiecePixmap(CPiece* pc = NULL,QPixmap* pxm = NULL) const;

// Mingles pieces, creates the objects
	void minglePieces();

	void changeScore(char reason);
	void showHighscoreDialog() const;

// main saving/loading functions
        bool save(QDataStream* f) const;
        bool load(QDataStream* f);

 public slots:
	void slotNewGame();
	void slotOpenGame();
	void slotSaveGame();
	void slotSaveGameAs();
        void slotStopGame() { stopGame(TM_QUIT); }
        void slotHighscores();
	void slotPause();
	void slotPauseToggle(bool on);
        void slotShowLarge();
        void slotShowMainPixmap();

	void slotTimer();
        void slotTerm(int reason);

	//	void slotHidePiece(bool dirty);

 signals:
	void sigPiecePlaced();
	void sigWinning();
	void sigTerm(int);
	void sigScore(int);
//	void sigTime(const char*);
	void sigUpdate();
};

#endif // KPUZZLEAPP_H
