/***************************************************************************
                          ktransferwgetimpl.cpp  -  description
                             -------------------
    begin                : Thu Oct 19 2000
    copyright            : (C) 2000 by Sergio Moretti
    email                : sermore@libero.it
    revision             : $Revision: 1.9 $
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <cstdlib>
#include <kdebug.h>
#include <qregexp.h>
//#include <kapp.h>
#include "qdir.h"
#include "utils.h"
#include "ktmanagerimpl.h"
#include "ktransferwgetimpl.h"


const char KTransferWgetImpl::RE_STR[RE_NUM][200] = {
  // RE_INT
  "^--([[:digit:]]{2}:[[:digit:]]{2}:[[:digit:]]{2})--[[:space:]]+([^\n]+)\n[[:space:]]+(\\(try:[[:space:]]*([[:digit:]]+)[[:space:]]*\\)[[:space:]]+)?=>[[:space:]]+`([^\n]+)'\n",
  // RE_CNN
  "^Connecting to ([[:alnum:]._]+):([[:digit:]]+)[[:space:]]*\\.{3}",
  // RE_FTP_LOGIN
  "^Logging in as ([[:alnum:]_]+)[[:space:]]*\\.{3}",
  // RE_FTP_TYPE
  "^[[:space:]]*==>[[:space:]]+TYPE[[:space:]]+(I|A)[[:space:]]*\\.{3}",
  // RE_FTP_CWD
  "^[[:space:]]*==>[[:space:]]+CWD[[:space:]]+((not required\\.\n|not needed\\.\n)|([^\n]+ \\.{3}))",
  // RE_FTP_PORT
  "^[[:space:]]*==>[[:space:]]+PORT[[:space:]]+\\.{3}",
  // RE_FTP_PASV
  "^[[:space:]]*==>[[:space:]]+PASV[[:space:]]+\\.{3}",
  // RE_FTP_LIST
  "^[[:space:]]*==>[[:space:]]+LIST[[:space:]]+\\.{3}",
  // RE_FTP_REST
  "^[[:space:]]*==>[[:space:]]+REST[[:space:]]+([[:digit:]]+)[[:space:]]*\\.{3}",
  // RE_FTP_REST_ACK
  "^[[:space:]]*\nREST failed, starting from scratch.\n",
  // RE_FTP_RETR
  "^[[:space:]]*==>[[:space:]]+RETR[[:space:]]+([^\n]+) \\.{3}",
  // RE_FTP_LENGTH
  "^Length:[[:space:]]+([[:digit:],-]+)[[:space:]]+(\\[([[:digit:],-]+)[[:space:]]+to go\\][[:space:]]+)?(\\(unauthoritative\\))?\n",
  // RE_DWN
  "^\n*( *\\[ skipping +([[:digit:]]+)K \\]\n)? *([[:digit:]]+)K ->( ,+( ,+)*)?",
  // RE_DW1
  "^\n* *([[:digit:]]+)K -> ",
  // RE_DW2
  "^ ?\\.+( \\.+)*",
  // RE_DW3
  "^ *(\\[ *([[:digit:]-]+)%\\])?\n",
  // RE_FTP_DW4
  "^[^\n]+: [^\n]+, (closing [^\n]+\\.)\n+|"
  "^[^\n]+ \\([^\n]+\\) - ("
  "(Data c.+: [^\n]+; )\n*|"
  "(Control [^\n]+\\.|Data [^\n]+\\.)\n((Giving up|Retrying)\\.\n)?\n*|"
  "`[^\n]+' saved \\[([^\n]+)\\]\n+)",
  /* RE_FTP_ACK
   * 2: success message: 'done.', 'connected!', 'Logged in!'
   * 3: error message
   * 5: resuming from error: 'Retrying', 'Giving up'
   */
  "^[[:space:]]*(done\\.|connected!|Logged in!)[[:space:]]*\n?",
  /* RE_ACKERR
   * 1: error message
   * 3: retrying/giving up
   */
  "^[^\n]*\n([^\n]+)\n((Retrying|Giving up)\\.\n\n)?\n*",
  // RE_HTTP_REQUEST
  "^(HTTP|Proxy)[[:space:]]+request sent, awaiting response *\\.{3}",
  // RE_HTTP_AUTHFAIL
  "^(Authorization failed|Unknown authentication scheme)\\.\n",
  // _RE_HTTP_LOCATION
  "^Location: *(unspecified|[^ \n]+)( +\\[following\\])? *\n",
  // RE_HTTP_LENGTH
  "^Length: +((([[:digit:],-]+) (\\(([[:digit:],-]+) to go\\))?)|(unspecified|ignored)) *(\\[[^\n]+\\])? *\n",
  // RE_HTTP_DW4 parse end string
  "^[^\n]+ \\([^\n]+\\) - `[^\n]+' saved \\[([[:digit:]]+)(/([[:digit:]]+))?\\] *\n+|"
  "^([^\n]+)\n(Giving up|Retrying)\\.\n+",
  // RE_HTTP_ACK
  // 1: success message: 'connected!', '200 OK.', '206 Partial Content'
  "^ *(connected!|[2-3][[:digit:]]+ [^\n]+)\n"
};

regex_t KTransferWgetImpl::_preg[RE_NUM];
bool KTransferWgetImpl::_reInit = false;
	

/** create empty transfer */
KTransferWgetImpl::KTransferWgetImpl(int type) : KTransferImpl(type) {
  _parsebuf = "";
  initRE();
  connect(&process, SIGNAL(receivedStdout(KProcess*, char*, int)), 
	  this, SLOT(slotProcessOutput(KProcess*, char*, int)));
  connect(&process, SIGNAL(receivedStderr(KProcess*, char*, int)), 
	  this, SLOT(slotProcessOutput(KProcess*, char*, int)));
  connect(&process, SIGNAL(processExited(KProcess*)), 
	  this, SLOT(slotProcessExited(KProcess*)));
}

/** clone operation */
KObjectImpl * KTransferWgetImpl::clone() const {
  kdFatal(id() != 0) << name() << " CLONE NOT PROTOTYPE" << endl;
  return new KTransferWgetImpl(type());
}

void KTransferWgetImpl::loadData() {
  KTransferImpl::loadData();
  _tmpFile = QString::null;
  _tmpFile.setPath(dom().attribute("TmpFile", QString::null));
  _cfg_cmdPath = dom().attribute("CommandPath", "/usr/bin/wget");
  _cfg_passiveFtp = dom().attribute("PassiveFtp", "0").toInt();
  _cfg_HTTPCache = dom().attribute("HTTPCache", "1").toInt();
  _cfg_proxy = dom().attribute("Proxy", "0").toInt();
  _cfg_readTimeout = dom().attribute("ReadTimeout", "900").toInt();
  _cfg_proxyFTP = dom().attribute("FTPProxy", "");
  _cfg_proxyPortFTP = dom().attribute("FTPProxyPort", "8080").toInt();
  _cfg_proxyHTTP = dom().attribute("HTTPProxy", "");
  _cfg_proxyPortHTTP = dom().attribute("HTTPProxyPort", "8080").toInt();
  _cfg_noProxy = QStringList::split(",", dom().attribute("NoProxy", ""));
  _cfg_speed = dom().attribute("Speed", "0").toInt();

  _protocol = 0;
  _tryResuming = false;
  _block = 0;
  _localStatus = 0;
  _localError = 0;
}

void KTransferWgetImpl::updateConf() {
  KTransferImpl::updateConf();
  dom().setAttribute("TmpFile", _tmpFile.path());
  dom().setAttribute("CommandPath", _cfg_cmdPath);
  dom().setAttribute("PassiveFtp", _cfg_passiveFtp);
  dom().setAttribute("HTTPCache", _cfg_HTTPCache);
  dom().setAttribute("Proxy", _cfg_proxy);
  dom().setAttribute("ReadTimeout", _cfg_readTimeout);
  dom().setAttribute("FTPProxy", _cfg_proxyFTP);
  dom().setAttribute("FTPProxyPort", _cfg_proxyPortFTP);
  dom().setAttribute("HTTPProxy", _cfg_proxyHTTP);
  dom().setAttribute("HTTPProxyPort", _cfg_proxyPortHTTP);
  dom().setAttribute("NoProxy", _cfg_noProxy.join(","));
  dom().setAttribute("Speed", _cfg_speed);
}

const KTransferWgetImpl * KTransferWgetImpl::global() const {
  return dynamic_cast<const KTransferWgetImpl*>(KTransferImpl::global());
}

bool KTransferWgetImpl::isProtocolSupported(const QString &proto) const {
  return (proto == "ftp" || proto == "http");
}

bool KTransferWgetImpl::isRunning() const {
  return KTransferImpl::isRunning() || process.isRunning();
}

KURL KTransferWgetImpl::getTempFile(const KURL &rmt, const KURL &) const {
  KURL tmp;
  tmp.setPath(QFileInfo(manager()->getWorkingDir(), 
			rmt.fileName()).absFilePath());
  return tmp;
}

KURL KTransferWgetImpl::getTmpFile() const {
  if (_tmpFile.path().isEmpty())
    return getTempFile(remote(), local());
  else
    return _tmpFile;
}

/** use passive transfer mode */
bool KTransferWgetImpl::getPassiveFtp() const {
  return isGlobal() ? global()->getPassiveFtp() : _cfg_passiveFtp;
}

/** get wget command path */
QString KTransferWgetImpl::getCmdPath() const {
  return isGlobal() ? global()->getCmdPath() : _cfg_cmdPath;
}

bool KTransferWgetImpl::getHTTPCache() const {
  return isGlobal() ? global()->getHTTPCache() : _cfg_HTTPCache;
}

int KTransferWgetImpl::getReadTimeout() const {
  return isGlobal() ? global()->getReadTimeout() : _cfg_readTimeout;
}

bool KTransferWgetImpl::getProxy() const {
  return isGlobal() ? global()->getProxy() : _cfg_proxy;
}

QString KTransferWgetImpl::getProxyFTP() const {
  return isGlobal() ? global()->getProxyFTP() : _cfg_proxyFTP;
}

int KTransferWgetImpl::getProxyPortFTP() const {
  return isGlobal() ? global()->getProxyPortFTP() : _cfg_proxyPortFTP;
}

QString KTransferWgetImpl::getProxyHTTP() const {
  return isGlobal() ? global()->getProxyHTTP() : _cfg_proxyHTTP;
}

int KTransferWgetImpl::getProxyPortHTTP() const {
  return isGlobal() ? global()->getProxyPortHTTP() : _cfg_proxyPortHTTP;
}

const QStringList & KTransferWgetImpl::getNoProxyList() const {
  return isGlobal() ? global()->getNoProxyList() : _cfg_noProxy;
}

int KTransferWgetImpl::getSpeed() const {
  return isGlobal() ? global()->getSpeed() : _cfg_speed;
}

void KTransferWgetImpl::initRE() {
  if (_reInit)
    return;
  int chk;
  char buf[100];
  for (int i = 0; i < RE_NUM; i++) {
    chk = regcomp(&_preg[i], RE_STR[i], REG_EXTENDED | REG_ICASE);
    if (chk) {
      regerror(chk, &_preg[i], buf, 100);
      kdFatal() << "RE " << i << ":" << RE_STR[i] << " ERROR:" 
		<< buf << endl;
    }
  }
  _reInit = true;
}

bool KTransferWgetImpl::processStart() {
  _directory = false;
  _processExit = false;
  _parsebuf = "";
  _tryResuming = false;
  _localStatus = 0;
  _localError = 0;
  QString tmpfile = getTmpFile().path();
  _parseCmd = (remote().protocol() == QString("ftp") ? 
	       &KTransferWgetImpl::parseFTP : &KTransferWgetImpl::parseHTTP);
  process.clearArguments();
  process << getCmdPath();
  if (QDir::current().exists(tmpfile)) {
    if (rsmStatus() == RSM_NO)
      QDir::current().remove(tmpfile);
    else
      process << "-c";
  }
  setPartial(QFileInfo(tmpfile).size());
  if (getPassiveFtp())
    process << "--passive-ftp";
  process << (getHTTPCache() ? "--cache=on" : "--cache=off");
  QStringList lst = getNoProxyList();
  bool proxy = getProxy();
  for (QStringList::Iterator item = lst.begin(); item != lst.end() && proxy; 
       ++item) {
    QRegExp re(*item, false, true);
    proxy = re.match(remote().host(), 0, 0) == -1;
  }
  switch (getSpeed()) {
  case SPD_DEFAULT:
    process << "--dot-style=default";
    _block = 1024;
    break;
  case SPD_BINARY:
    process << "--dot-style=binary";
    _block = 8192;
    break;
  case SPD_MEGA:
    process << "--dot-style=mega";
    _block = 65536;
    break;
  case SPD_MICRO:
    process << "--dot-style=micro";
    _block = 128;
    break;
  default:
    kdFatal() << "TRANSFER SPEED UNKNOWN" << endl;
  }
  process << (proxy ? "--proxy=on" : "--proxy=off");
  if (proxy) {
    // set environment variable
    char str[80];
    if (!getProxyHTTP().isEmpty()) {
      sprintf(str, "%s:%d", (const char*)getProxyHTTP(), getProxyPortHTTP());
      int check = setenv("http_proxy", str, 1);
      kdFatal(check != 0, D_RUN) << "SETENV ERROR" << endl;
    } else
      unsetenv("http_proxy");
    if (!getProxyFTP().isEmpty()) {
      sprintf(str, "%s:%d", (const char*)getProxyFTP(), getProxyPortFTP());
      int check = setenv("ftp_proxy", str, 1);
      kdFatal(check != 0, D_RUN) << "SETENV ERROR" << endl;
    } else
      unsetenv("ftp_proxy");
  }
  // set locale to C
  int check = setenv("LC_ALL", "C", 1);
  kdFatal(check != 0, D_RUN) << "SETENV ERROR" << endl;
  
  process << "-t" << QString().setNum(getMaxRetry())
	  << "-w" << QString().setNum(getWaitRetry())
	  << "-P" << manager()->getWorkingDir().absPath()
	  << "-v"
	  << "-T" << QString().setNum(getReadTimeout())
	  << "-nr" // if you are downloading a directory from ftp
	  << "--htmlify=off" // don't htmlify directories
	  << remote().url();
  //arguments.append("2>&1");
  //FIXME!! linee x debug
  QString cmdline;
  for (char *arg = process.args()->first(); arg != 0; 
       arg = process.args()->next())
    cmdline += arg + QString(" ");
  kdDebug(D_RUN) << name() << ": cmdline " << cmdline << endl;

  if (process.start(KProcess::NotifyOnExit, KProcess::AllOutput)) {
    opProcessStart();
    return true;
  }
  return false;
}

bool KTransferWgetImpl::processKill() {
  return process.kill();
}

void KTransferWgetImpl::parseFTP() {
  Match res = MTC_OK;
  while (!_parsebuf.isEmpty() 
	 && (res == MTC_OK || (res != MTC_FAIL && _processExit))) {
    //kdDebug(D_PRS) << name() << "#" << _parsebuf << "#" << endl;
    switch (_localStatus) {
    case FTP_DISCONNECT:
      res = parseProlog(FTP_CONNECT_REQ);
      // call opRunStart
      break;
    case FTP_CONNECT_REQ:
      //if fail goto CWD
      res = parseToken(RE_CNN, FTP_CONNECT_ACK, FTP_CWD_REQ);
      break;
    case FTP_CONNECT_ACK:
      res = parseAck(RE_FTP_ACK, FTP_LOGIN_REQ);
      if (res == MTC_OK)
	opConnect();
      break;
    case FTP_LOGIN_REQ:
      res = parseToken(RE_FTP_LOGIN, FTP_LOGIN_ACK);
      break;
    case FTP_LOGIN_ACK:
      res = parseAck(RE_FTP_ACK, FTP_TYPE_REQ);
      break;
    case FTP_TYPE_REQ:
      res = parseToken(RE_FTP_TYPE, FTP_TYPE_ACK);
      break;
    case FTP_TYPE_ACK:
      res = parseAck(RE_FTP_ACK, FTP_CWD_REQ);
      break;
    case FTP_CWD_REQ:
      res = parseCwd(); // if cwd not executed, goto FTP_(PORT/PASV)_REQ
      break;
    case FTP_CWD_ACK:
      // warn: fatal error with one more \n
      if (getPassiveFtp())
	res = parseAck(RE_FTP_ACK, FTP_PASV_REQ);
      else 
	res = parseAck(RE_FTP_ACK, FTP_PORT_REQ);
      break;
    case FTP_PORT_REQ:
      res = parseToken(RE_FTP_PORT, FTP_PORT_ACK);
      break;
    case FTP_PORT_ACK:
      res = parseAck(RE_FTP_ACK, FTP_LIST_REQ); // next try LIST
      break;
    case FTP_PASV_REQ:
      res = parseToken(RE_FTP_PASV, FTP_PASV_ACK);
      break;
    case FTP_PASV_ACK:
      res = parseAck(RE_FTP_ACK, FTP_LIST_REQ); // next try LIST
      break;
    case FTP_LIST_REQ:
      // if fail goto REST
      res = parseToken(RE_FTP_LIST, FTP_LIST_ACK, FTP_REST_REQ);
      break;
    case FTP_LIST_ACK:
      // warn: fatal error with one more \n
      res = parseAck(RE_FTP_ACK, FTP_LENGTH); // if match goto LENGTH
      _directory = (res == MTC_OK);
      if (res == MTC_OK) {
	setRsmStatus(RSM_NO);
	setModified(MOD_STATE, PRP_ALL);
      }
      break;
    case FTP_REST_REQ:
      res = parseRest(); //if fail goto RETR, else setPartial && tryResuming
      break;
    case FTP_REST_ACK:
      res = parseAck(RE_FTP_ACK, FTP_RETR_REQ);
      if (res != MTC_OK)
	res = parseRestAck();
      break;
    case FTP_RETR_REQ:
      res = parseToken(RE_FTP_RETR, FTP_RETR_ACK);
      break;
    case FTP_RETR_ACK:
      // warn: fatal error witn one more \n
      res = parseAck(RE_FTP_ACK, FTP_LENGTH);
      break;
    case FTP_LENGTH:
      res = parseFtpLength(); // setLen, goto DOWNLOAD_START
      if (res == MTC_FAIL) {
	// check for a fatal error
	res = parseAck(RE_FTP_ACK, 0);
	if (res == MTC_FAIL) {
	  // if the file transfer is complete, try to continue
	  setLen(partial());
	  res = MTC_CONT;
	  _localStatus = FTP_DOWNLOAD_START;
	}
      }
      break;
    case FTP_DOWNLOAD_START:
      res = parseDownloadStart(FTP_DOWNLOAD_RUN);
      // call opDownload
      break;
      case FTP_DOWNLOAD_RUN:
      // call opDataRead()
      res = parseDownloadRun();
      if (res == MTC_FAIL) // check for end of transfer
	// call opRunEnd if succesful
	res = parseFtpDownloadEnd();
      break;
    case FTP_DOWNLOAD_END:
      //kdFatal(D_PRS) << name() << ": error reached FTP_END" << endl;
      res = MTC_FAIL;
      break;
    default:
      res = MTC_FAIL;
      //kdFatal(D_PRS) << name() << ":" << _localStatus << " unknown FTP state" << endl;
    }
    //kdFatal(res == MTC_FAIL) << name() << " FTP state " << _localStatus << " unable to parse #" << _parsebuf << "#" << endl;
    if (res == MTC_FAIL) {
      opFatalError(QString("FTP state %1 : unable to parse #%2#").arg(_localStatus).arg(_parsebuf));
      _localError = ERR_ABORT;
    }
    if (_localError == ERR_RETRY) { 
      // restart from beginning
      kdDebug(D_PRS) << name() << ": retry" << endl;
      _localStatus = FTP_DISCONNECT;
      _localError = ERR_OK;
      opRunEnd();
    }
    // remove parsed string from buffer
    if (res == MTC_OK) {
      //kdDebug() << "!" << matchStr(0) << "!" << endl;
      _parsebuf.remove(0, _pmatch[0].rm_eo - _pmatch[0].rm_so);
    }
    kdDebug(D_PRS) << name() << " match " << res << " FTP state " << _localStatus << endl;
  } // end while
  parseExit();
  kdDebug(D_PRS) << name() << ":" << statusStr() << ":" << cmdStatus() << " exit FTP parse" << endl;
}

QString KTransferWgetImpl::matchStr(int i) const {
  return _parsebuf.mid(_pmatch[i].rm_so, _pmatch[i].rm_eo - _pmatch[i].rm_so);
}

KTransferWgetImpl::Match 
KTransferWgetImpl::parseToken(REId token, int nextState, int failState) {
  int check = regexec(&_preg[token], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    // goto nextState and mark string as parsed
    _localStatus = nextState;
    return MTC_OK;
  }
  if (failState != 0 && _parsebuf.length() > 100 || _processExit) { 
    // goto to failState, but not advance in string parsing
    _localStatus = failState;
    return MTC_CONT;
  }
  // fallback condition
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseProlog(int nextState) {
  int check = regexec(&_preg[RE_INT], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) { // ok
    QString str = matchStr(5);
    kdWarning(QFileInfo(str).absFilePath() != getTmpFile().path(), D_PRS) << name() << ": file " << str << " != " << getTmpFile().path() << endl;
    _tmpFile = QFileInfo(str).absFilePath();
    _localStatus = nextState;
    opRunStart();
    return MTC_OK;
  }
  return (_parsebuf.length() > 200) ? MTC_FAIL : MTC_CONT;
}
  
KTransferWgetImpl::Match KTransferWgetImpl::parseAck(REId token, int nextState) {
  // check for ack
  int check = regexec(&_preg[token], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) { // ack ok
    _localStatus = nextState;
    return MTC_OK;
  }
  // be sure that check for ack is failed
  if (_parsebuf.length() < 40 && !_processExit)
    return MTC_CONT;
  // check for error ack
  check = regexec(&_preg[RE_ACKERR], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) { // error ack 
    QString str = matchStr(3);
    // retry, give up or abort
    if (str.isEmpty() && _processExit)
      // fatal error
      _localError = ERR_ABORT;
    else if (str == "Giving up")
      _localError = ERR_GIVEUP;
    else if (str == "Retrying")
      _localError = ERR_RETRY;
    else
      goto fail;
    // error message
    str = matchStr(1);
    kdDebug(D_PRS) << name() << ":" << _localStatus << " error :" << str << endl;
    opError(str);
    kdDebug(D_PRS) << name() << ": error action " << matchStr(3) << endl;
    return MTC_OK;
  }
 fail:
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}
  
KTransferWgetImpl::Match KTransferWgetImpl::parseCwd() {
  int check = regexec(&_preg[RE_FTP_CWD], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    if (_pmatch[2].rm_so != -1) // cwd not required/not needed, goto PORT/PASV
      _localStatus = getPassiveFtp() ? FTP_PASV_REQ : FTP_PORT_REQ;
    else
      _localStatus = FTP_CWD_ACK;
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseRest() {
  int check = regexec(&_preg[RE_FTP_REST], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    QString str = matchStr(1);
    setPartial(str.toInt());
    _tryResuming = true;
    _localStatus = FTP_REST_ACK;
    return MTC_OK;
  }
  if (_parsebuf.length() > 40 || _processExit) { // REST not present
    _localStatus = FTP_RETR_REQ;
    kdError(partial() > 0, D_PRS) << name() << " REST not present and partial != 0" << endl;
    setPartial(0);
    _tryResuming = false;
    return MTC_CONT;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseRestAck() {
  int check = regexec(&_preg[RE_FTP_REST_ACK], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    setPartial(0);
    _localStatus = FTP_RETR_REQ;
    if (_tryResuming) {
      setRsmStatus(RSM_NO);
      setModified(MOD_STATE, PRP_ALL);
    }
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseFtpLength() {
  int check = regexec(&_preg[RE_FTP_LENGTH], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    QString str = matchStr(1);
    setLen(parseNumber(str) + partial());
    _localStatus = FTP_DOWNLOAD_START;
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseDownloadStart(int nextState) {
  int check = regexec(&_preg[RE_DWN], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    if (_tryResuming) {
      setRsmStatus(RSM_YES);
      setModified(MOD_STATE, PRP_ALL);
    }
    opDownload();
    _localStatus = nextState;
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}
  
KTransferWgetImpl::Match KTransferWgetImpl::parseDownloadRun() {
  int check = regexec(&_preg[RE_DW1], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    // parse: 120K ->
    return MTC_OK;
  }
  check = regexec(&_preg[RE_DW2], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    // parse:  .... ....
    QString str = matchStr(0);
    int datalen = str.contains('.') * _block;
    opDataRead(datalen);
    return MTC_OK;
  }
  check = regexec(&_preg[RE_DW3], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    // parse: [ 33%]\n"
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseFtpDownloadEnd() {
  int check = regexec(&_preg[RE_FTP_DW4], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    QString str;
    if (_pmatch[7].rm_so != -1) {
      // transfer succesfully completed
      str = matchStr(7);
      int l = str.toInt();
      kdWarning(l != len(), D_PRS) << name() << " length " << l << "!=" << len() << endl;
      if (l != len())
	setLen(l);
      if (partial() < len())
	opDataRead(len() - partial());
      _localStatus = FTP_DOWNLOAD_END;
      opRunEnd();
      return MTC_OK;
    }
    // errors
    if (_pmatch[1].rm_so != -1) {
      // match 1: fatal error
      str = matchStr(1);
      _localError = ERR_ABORT;
    } else if (_pmatch[3].rm_so != -1) {
      // match 3: ???, continue
      str = matchStr(3);
    } else if (_pmatch[4].rm_so != -1) {
      // match 4: error: retry, giveup or ok
      str = matchStr(6);
      if (str.isEmpty())
	; //ok
      else if (str == "Giving up")
	// give up 
	_localError = ERR_GIVEUP;
      else
	// retry
	_localError = ERR_RETRY;
      // error message
      str = matchStr(4);
    }
    // output error message
    opError(str);
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

void KTransferWgetImpl::parseExit() {
  // set resume if we are checking for it
  if (rsmStatus() == RSM_CHECKING && _processExit) {
    if (_localError != ERR_OK) {
      // if we are checking for resuming and it give a recoverable error after RETR
      setRsmStatus((_localStatus == FTP_RETR_ACK 
		    && _localError == ERR_GIVEUP) ? RSM_NO : RSM_CHECKFAIL);
      setModified(MOD_STATE, PRP_ALL);
    } else
      setRsmStatus(RSM_CHECKFAIL);
  }
  switch (_localError) {
  case ERR_OK:
    if (isComplete())
      setCmdStatus(CMD_EXITED);
    break;
  case ERR_RETRY:
    kdFatal(D_PRS) << name() << " exiting with ERR_RETRY" << endl;
    break;
  case ERR_GIVEUP:
    setCmdStatus(CMD_EXITED);
    break;
  case ERR_ABORT:
    setCmdStatus(CMD_FATALERROR);
    //setModified(MOD_STATE, PRP_ALL); 
  }
}

void KTransferWgetImpl::parseHTTP() {
  Match res = MTC_OK;
  while (!_parsebuf.isEmpty() 
	 && (res == MTC_OK || (res != MTC_FAIL && _processExit))) {
    //kdDebug(D_PRS) << name() << "#" << _parsebuf << "#" << endl;
    switch (_localStatus) {
    case HTTP_DISCONNECT:
      res = parseProlog(HTTP_CONNECT_REQ);
      // call opRunStart
      break;
    case HTTP_CONNECT_REQ:
      res = parseToken(RE_CNN, HTTP_CONNECT_ACK);
      break;
    case HTTP_CONNECT_ACK:
      res = parseAck(RE_HTTP_ACK, HTTP_REQUEST_REQ);
      if (res == MTC_OK)
	opConnect();
      break;
    case HTTP_REQUEST_REQ:
      res = parseToken(RE_HTTP_REQUEST, HTTP_REQUEST_ACK);
      break;
    case HTTP_REQUEST_ACK:
      res = parseAck(RE_HTTP_ACK, HTTP_LENGTH);
      break;
    case HTTP_LENGTH:
      res = parseHttpLength();
      if (res == MTC_FAIL) {
	// if parse then raise a FATALERROR
	res = parseToken(RE_HTTP_AUTHFAIL, 0);
	if (res == MTC_OK)
	  _localError = ERR_ABORT;
	else
	  // if LOCATION is found, then restart else goto LENGTH
	  res = parseToken(RE_HTTP_LOCATION, HTTP_DISCONNECT);
      }
      break;
    case HTTP_DOWNLOAD_START:
      res = parseDownloadStart(HTTP_DOWNLOAD_RUN);
      // call opDownloadStart();
      break;
    case HTTP_DOWNLOAD_RUN:
      // call opDataRead()
      res = parseDownloadRun();
      if (res == MTC_FAIL)
	res = parseHttpDownloadEnd();
      break;
    case HTTP_DOWNLOAD_END:
      //kdFatal(D_PRS) << name() << ": error reached HTTP_END" << endl;
      res = MTC_FAIL;
      break;
    default:
      res = MTC_FAIL;
      //kdFatal(D_PRS) << name() << ":" << _localStatus << " unknown HTTP state" << endl;
    }
    //kdFatal(res == MTC_FAIL) << name() << " HTTP state " << _localStatus << " unable to parse #" << _parsebuf << "#" << endl;
    if (res == MTC_FAIL) {
      opFatalError(QString("HTTP state %1 : unable to parse #%2#").arg(name()).arg(_localStatus).arg(_parsebuf));
      _localError = ERR_ABORT;
    }
    if (_localError == ERR_RETRY) { 
      // restart from beginning
      kdDebug(D_PRS) << name() << ": retry" << endl;
      _localStatus = HTTP_DISCONNECT;
      _localError = ERR_OK;
      opRunEnd();
    }
    // remove parsed string from buffer
    if (res == MTC_OK) {
      _parsebuf.remove(0, _pmatch[0].rm_eo - _pmatch[0].rm_so);
    }
    kdDebug(D_PRS) << name() << " match " << res << " HTTP state " << _localStatus << endl;
  }
  parseExit();
  kdDebug(D_PRS) << name() << ":" << statusStr() << ":" << cmdStatus() << " exit HTTP parse" << endl;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseHttpLength() {
  int check = regexec(&_preg[RE_HTTP_LENGTH], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    if (_pmatch[3].rm_so != -1) {
      setLen(parseNumber(matchStr(3)));
      QString str = matchStr(5);
      if (str.isEmpty())
	setPartial(0);
      else
	setPartial(len() - parseNumber(str));
      _tryResuming = partial() > 0;
    } else {
      setLen(0);
      setPartial(0);
      _tryResuming = false;
      if (rsmStatus() == RSM_CHECKING) {
	setRsmStatus(RSM_NO);
	setModified(MOD_STATE, PRP_ALL);
      }
    }
    _localStatus = HTTP_DOWNLOAD_START;
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

KTransferWgetImpl::Match KTransferWgetImpl::parseHttpDownloadEnd() {
  int check = regexec(&_preg[RE_HTTP_DW4], _parsebuf, 10, &_pmatch[0], 0);
  if (!check) {
    QString str;
    if (_pmatch[1].rm_so != -1) {
      // match 1: transfer successfully completed
      int l = matchStr(1).toInt();
      kdWarning(l != len(), D_PRS) << name() << " length " << l << "!=" << len() << endl;
      if (l != len())
	setLen(l);
      if (partial() < len())
	opDataRead(len() - partial());
      _localStatus = HTTP_DOWNLOAD_END;
      opRunEnd();
    } else if (_pmatch[4].rm_so != -1) {
      // match 5: error: retry or giveup
      opError(matchStr(4));
      str = matchStr(5);
      if (str == "Giving up")
	// give up 
	_localError = ERR_GIVEUP;
      else 
	// retry
	_localError = ERR_RETRY;
    }
    return MTC_OK;
  }
  return (_parsebuf.length() > 100 || _processExit) ? MTC_FAIL : MTC_CONT;
}

/////////////// SLOTS

void KTransferWgetImpl::slotProcessOutput(KProcess *, char *buffer, int buflen) {
  QCString cdata(buffer, buflen+1);
  _parsebuf += cdata;
  opProcessOutput(cdata);
  (this->*_parseCmd)();
}

void KTransferWgetImpl::slotProcessExited(KProcess *) {
  _processExit = true;
  (this->*_parseCmd)();
  opProcessEnd(process.exitStatus());
} 

#include "ktransferwgetimpl.moc"
