/**
 * manager.cpp
 *
 * Copyright (C)  2003  Zack Rusin <zack@kde.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 */
#include "manager.h"

#include "mainwidget.h"
#include "editorwidget.h"
#include "items.h"

#include <kpopupmenu.h>
#include <kinputdialog.h>
#include <kmessagebox.h>
#include <klistview.h>
#include <ktextedit.h>
#include <klineedit.h>
#include <kcombobox.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kdebug.h>

#include <qcheckbox.h>

namespace KCfgCreator {

Manager::Manager( MainWidget* parent )
    : QObject( parent ),
      m_entryView( parent->listView() ),
      m_editWidget( parent->editor() )
{
    initConnections();
}

void Manager::initConnections()
{
    connect( m_entryView,
             SIGNAL(contextMenuRequested(QListViewItem*, const QPoint&, int)),
             SLOT(slotContextMenuRequested(QListViewItem*, const QPoint&, int)) );
    connect( m_entryView, SIGNAL(selectionChanged(QListViewItem*)),
             m_editWidget, SLOT(treeSelectionChanged(QListViewItem*)) );
}

void Manager::slotContextMenuRequested( QListViewItem* lvi, const QPoint& p, int )
{
    if ( !lvi && m_entryView->childCount() )
        return;

    KPopupMenu *treeMenu = new KPopupMenu();
    if ( m_entryView->childCount() == 0 ) {
        treeMenu->insertTitle( i18n( "CFG Menu" ) );
        treeMenu->insertItem( SmallIcon( "folder_new" ),
                              i18n( "&New File" ),
                              static_cast<MainWidget*>( parent() ),
                              SLOT(newConfig()) );
    } else if ( lvi ) {
        m_entryView->setCurrentItem( lvi );
        m_entryView->setSelected( lvi, true );

        ConfItem *item = dynamic_cast<ConfItem*>( lvi );
        if ( item ) {
            switch( item->type() ) {
            case ConfItem::Entry:
                treeMenu->insertItem( QPixmap(),
                                      i18n( "&Remove Entry" ),
                                      this, SLOT(removeEntry()) );
                break;
            case ConfItem::Group:
                treeMenu->insertItem( QPixmap(),
                                      i18n( "&Add Entry..." ),
                                      this, SLOT(addEntry()) );
                treeMenu->insertItem( QPixmap(),
                                      i18n( "&Remove Group" ),
                                      this, SLOT(removeGroup()) );
                break;
            case ConfItem::App:
                treeMenu->insertItem( QPixmap(),
                                      i18n( "&Add Group..." ),
                                      this, SLOT(addGroup()) );
                treeMenu->insertItem( QPixmap(),
                                      i18n( "&Preferences" ),
                                      static_cast<MainWidget*>( parent() ),
                                      SLOT(preferences()) );
                break;
            default:
                kdWarning()<<"Switch case is missing an entry"<<endl;
                break;
            }
        }
    }

    treeMenu->exec( p, 0 );
    delete treeMenu;
}

void Manager::slotSelectionChanged( QListViewItem* )
{
}

void Manager::slotNameChanged( const QString& )
{
}

void Manager::slotTypeChanged( const QString& )
{
}

void Manager::slotHiddenToggled( bool )
{
}

void Manager::slotLabelChanged( const QString& )
{
}

void Manager::slotWhatsThisChanged()
{
}

void Manager::renameConfig()
{
    QString name = KInputDialog::getText(
        i18n( "Rename Config" ),
        i18n( "Please enter the new name of the config file: " ) );

    if ( !name.isEmpty() ) {
        AppItem *item = dynamic_cast<AppItem*>( m_entryView->firstChild() );
        if ( item ) item->setName( name );
    }
}

void Manager::addGroup()
{
    AppItem *app = dynamic_cast<AppItem*>( m_entryView->firstChild() );
    if ( app ) {
        QString name = KInputDialog::getText(
        i18n( "Add Group" ),
        i18n( "Please enter the name of the group you wish to add: " ) );

        if (  findGroup( name ) ) {
            KMessageBox::information(
                m_entryView,
                i18n( "Group with the specified name already exists!" ),
                i18n( "Group Exists" ) );
            return;
        }

        if ( !name.isEmpty() ) {
            GroupItem *gi = new GroupItem( app, name );
            m_entryView->setOpen( app, true );
            m_entryView->setSelected( gi, true );
        }
    }
}

void Manager::addEntry()
{
    GroupItem *item = dynamic_cast<GroupItem*>( m_entryView->selectedItem() );
    if ( item ) {
        QString name = KInputDialog::getText(
        i18n( "Add Entry" ),
        i18n( "Please enter the name of the entry you wish to add: " ) );

        if (  findEntry( item, name ) ) {
            KMessageBox::information(
                m_entryView,
                i18n( "Entry with the specified name already exists!" ),
                i18n( "Entry Exists" ) );
            return;
        }
        if ( !name.isEmpty() ) {
           EntryItem *ei = new EntryItem( item, name );
           m_entryView->setOpen( item, true );
           m_entryView->setSelected( ei, true );
        }
    }
}

void Manager::removeGroup()
{
    GroupItem *group = dynamic_cast<GroupItem*>( m_entryView->selectedItem() );
    if ( group && group->childCount() ) {
        int ans = KMessageBox::questionYesNo(
            m_entryView,
            i18n( "Are you sure you want to remove group %1 containing "
                  "%2 entries?" )
            .arg( group->name() )
            .arg( group->childCount() ),
            i18n( "Remove Group" ) );
        if ( ans != KMessageBox::Yes )
            return;
    }
    delete group;
}

void Manager::removeEntry()
{
    delete m_entryView->selectedItem();
}

GroupItem* Manager::findGroup( const QString& name )
{
    AppItem *conf = dynamic_cast<AppItem*>( m_entryView->firstChild() );

    if ( conf ) {
        GroupItem *group = static_cast<GroupItem*>( conf->firstChild() );

        while ( group ) {
            if ( group->name() == name )
                return group;
            group = static_cast<GroupItem*>( group->nextSibling() );
        }
    }
    return 0;
}

EntryItem* Manager::findEntry( GroupItem* parent, const QString& name )
{
    if ( parent ) {
        EntryItem *entry = static_cast<EntryItem*>( parent->firstChild() );

        while ( entry ) {
            if ( entry->name() == name )
                return entry;
            entry = static_cast<EntryItem*>( entry->nextSibling() );
        }
    }
    return 0;
}

} //end namespace

#include "manager.moc"
