import string
import sys
import traceback
import SCons.Node
import SCons.Errors
class Task:
    def __init__(self, tm, targets, top, node):
        self.tm=tm
        self.targets=targets
        self.top=top
        self.node=node
    def display(self, message):
        pass
    def prepare(self):
        self.tm.exception_raise()
        if self.tm.message:
            self.display(self.tm.message)
            self.tm.message=None
        for t in self.targets:
            t.prepare()
            for s in t.side_effects:
                s.prepare()
    def execute(self):
        try:
            everything_was_cached=1
            for t in self.targets:
                if not t.retrieve_from_cache():
                    everything_was_cached=0
                    break
            if not everything_was_cached:
                self.targets[0].build()
        except KeyboardInterrupt:
            raise
        except SystemExit:
            exc_value=sys.exc_info()[1]
            raise SCons.Errors.ExplicitExit(self.targets[0], exc_value.code)
        except SCons.Errors.UserError:
            raise
        except SCons.Errors.BuildError:
            raise
        except:
            exc_type, exc_value, exc_traceback=sys.exc_info()
            raise SCons.Errors.BuildError(self.targets[0], "Exception", exc_type, exc_value, exc_traceback)
    def get_target(self):
        return self.node
    def executed(self):
        if self.targets[0].get_state()==SCons.Node.executing:
            for t in self.targets:
                for side_effect in t.side_effects:
                    side_effect.set_state(None)
                t.set_state(SCons.Node.executed)
                t.built()
        else:
            for t in self.targets:
                t.visited()
        self.tm.executed(self.node)
    def failed(self):
        self.fail_stop()
    def fail_stop(self):
        for t in self.targets:
            t.set_state(SCons.Node.failed)
        self.tm.failed(self.node)
        self.tm.stop()
    def fail_continue(self):
        for t in self.targets:
            def set_state(node): node.set_state(SCons.Node.failed)
            t.call_for_all_waiting_parents(set_state)
        self.tm.executed(self.node)
    def mark_targets(self, state):
        for t in self.targets:
            t.set_state(state)
    def mark_targets_and_side_effects(self, state):
        for t in self.targets:
            for side_effect in t.side_effects:
                side_effect.set_state(state)
            t.set_state(state)
    def make_ready_all(self):
        self.out_of_date=self.targets[:]
        self.mark_targets_and_side_effects(SCons.Node.executing)
    def make_ready_current(self):
        self.out_of_date=[]
        for t in self.targets:
            if not t.current():
                self.out_of_date.append(t)
        if self.out_of_date:
            self.mark_targets_and_side_effects(SCons.Node.executing)
        else:
            self.mark_targets(SCons.Node.up_to_date)
    make_ready=make_ready_current
    def postprocess(self):
        for t in self.targets:
            t.postprocess()
    def exc_info(self):
        return self.tm.exception
    def exc_clear(self):
        self.tm.exception_clear()
    def exception_set(self):
        self.tm.exception_set()
def order(dependencies):
    return dependencies
class Taskmaster:
    def __init__(self, targets=[], tasker=Task, order=order):
        self.targets=targets
        self.candidates=targets[:]
        self.candidates.reverse()
        self.executing=[]
        self.pending=[]
        self.tasker=tasker
        self.ready=None
        self.order=order
        self.exception_clear()
        self.message=None
    def _find_next_ready_node(self):
        if self.ready:
            return
        while self.candidates:
            node=self.candidates[-1]
            state=node.get_state()
            if state != None and state != SCons.Node.stack:
                self.candidates.pop()
                continue
            node.set_state(SCons.Node.stack)
            try:
                children=node.children()
            except SystemExit:
                exc_value=sys.exc_info()[1]
                e=SCons.Errors.ExplicitExit(node, exc_value.code)
                self.exception_set((SCons.Errors.ExplicitExit, e))
                self.candidates.pop()
                self.ready=node
                break
            except KeyboardInterrupt:
                raise
            except:
                self.exception_set()
                self.candidates.pop()
                self.ready=node
                break
            def in_stack(node): return node.get_state()==SCons.Node.stack
            cycle=filter(in_stack, children)
            if cycle:
                nodes=filter(in_stack, self.candidates) + cycle
                nodes.reverse()
                desc="Dependency cycle: " + string.join(map(str, nodes), " -> ")
                raise SCons.Errors.UserError, desc
            try:
                def derived_nodes(node): return node.is_derived() or node.is_pseudo_derived()
                derived=filter(derived_nodes, children)
            except KeyboardInterrupt:
                raise
            except:
                self.exception_set()
                self.candidates.pop()
                self.ready=node
                break
            if not derived and node in self.targets:
                alt, message=node.alter_targets()
                if alt:
                    self.message=message
                    self.candidates.pop()
                    self.candidates.extend(alt)
                    continue
            def unbuilt_nodes(node): return node.get_state()==None
            not_built=filter(unbuilt_nodes, derived)
            if not_built:
                def add_to_waiting_parents(child, parent=node):
                    child.add_to_waiting_parents(parent)
                map(add_to_waiting_parents, not_built)
                not_built.reverse()
                self.candidates.extend(self.order(not_built))
                continue
            cont=0
            for side_effect in node.side_effects:
                if side_effect.get_state()==SCons.Node.executing:
                    self.pending.append(node)
                    node.set_state(SCons.Node.pending)
                    self.candidates.pop()
                    cont=1
                    break
            if cont: continue
            if node.depends_on(self.executing) or node.depends_on(self.pending):
                self.pending.append(node)
                node.set_state(SCons.Node.pending)
                self.candidates.pop()
                continue
            self.candidates.pop()
            self.ready=node
            break
    def next_task(self):
        self._find_next_ready_node()
        node=self.ready
        if node is None:
            return None
        try:
            tlist=node.builder.targets(node)
        except AttributeError:
            tlist=[node]
        self.executing.extend(tlist)
        self.executing.extend(node.side_effects)
        
        task=self.tasker(self, tlist, node in self.targets, node)
        try:
            task.make_ready()
        except KeyboardInterrupt:
            raise
        except:
            self.exception_set()
        self.ready=None
        return task
    def is_blocked(self):
        self._find_next_ready_node()
        return not self.ready and (self.pending or self.executing)
    def stop(self):
        self.candidates=[]
        self.ready=None
        self.pending=[]
    def failed(self, node):
        try:
            tlist=node.builder.targets(node)
        except AttributeError:
            tlist=[node]
        for t in tlist:
            self.executing.remove(t)
        for side_effect in node.side_effects:
            self.executing.remove(side_effect)
    def executed(self, node):
        try:
            tlist=node.builder.targets(node)
        except AttributeError:
            tlist=[node]
        for t in tlist:
            self.executing.remove(t)
        for side_effect in node.side_effects:
            self.executing.remove(side_effect)
        for node in self.pending:
            node.set_state(None)
        self.pending.reverse()
        self.candidates.extend(self.pending)
        self.pending=[]
    def exception_set(self, exception=None):
        if exception is None:
            exception=sys.exc_info()
        self.exception=exception
        self.exception_raise=self._exception_raise
    def exception_clear(self):
        self.exception=(None, None, None)
        self.exception_raise=self._no_exception_to_raise
    def _no_exception_to_raise(self):
        pass
    def _exception_raise(self):
        exc_type, exc_value=self.exception[:2]
        raise exc_type, exc_value
