<?php
/**
 * Contains functions related to managing
 * Access Control Lists.
 *
 * $Horde: imp/lib/ACL.php,v 1.3 2003/02/07 23:16:46 jan Exp $
 *
 * Copyright 2003 Chris Hastie <imp@oak-wood.co.uk>
 *
 * See the enclosed file COPYING for license information (GPL). If you
 * did not receive this file, see http://www.fsf.org/copyleft/gpl.html.
 *
 * @author  Chris Hastie <imp@oak-wood.co.uk>
 * @version $Revision: 1.3 $
 * @since   IMP 4.0
 * @package imp
 */
class IMP_ACL {

    /**
     * Hash containing connection parameters.
     *
     * @var array $_params
     */
    var $_params = array();

    /**
     * Boolean indicating if the driver is supported by the server
     *
     * @var boolean $_supported
     */
    var $_supported = false;

    /**
     * Hash containing the list of possible rights and a human
     * readable description of each
     * 
     * Array (
     *     right-id => right-description   
     * )
     *
     * @var array $_rightsList
     */
    var $_rightsList = array();

    /**
     * Array containing user names that can not have their access
     * rights changed.
     *
     * @var boolean $_protected
     */
    var $_protected;

    /**
     * Constructor.
     *
     * @access public
     *
     * @param optional array $params  Hash containing connection parameters.
     */
    function IMP_ACL($params = array())
    {
        $this->_params = $params;
    }    

    /**
     * Attempts to retrieve the existing ACL for a folder from 
     * the current IMAP server. If protocol is imap/ssl, will
     * only attempt ssl connection with PHP >= 4.3
     *
     * @access public
     *
     * @param string folder  The folder to get the ACL for
     *
     * @return array  A hash containing information on the ACL
     *                Array (
     *                    user => Array (
     *                                right => 1
     *                            )
     *                )
     */
    function getACL($folder)
    {
        return false;
    }

    /**
     * Sets the ACL on an IMAP server
     *
     * @access public
     *
     * @param string $folder  The folder on which to edit the ACL
     *                           
     * @param string $share_user  The user to grant rights to
     *
     * @param array $acl  An array, the keys of which are the 
     *                    rights to be granted (see RFC 2086)
     *
     * @return mixed  True on success, false on failure unless
     *                server doesn't support ACLs, returns 'no_support'
     */
    function createACL($folder, $share_user, $acl)
    {
        return false;
    }

    /**
     * Edits an ACL on an IMAP server
     *
     * @access public
     *
     * @param string $folder  The folder on which to edit the ACL
     *                           
     * @param string $share_user  The user to grant rights to
     *
     * @param array $acl  An array, the keys of which are the 
     *                    rights to be granted (see RFC 2086)
     *
     * @return mixed  True on success, false on failure unless
     *                server doesn't support ACLs, returns 'no_support'
     */
    function editACL($folder, $share_user, $acl)
    {
        return false;
    }    

    /**
     * Can a user edit the ACL for this folder? Returns true if $user   
     * permission to edit the ACL on $folder
     *
     * @param string $folder  The folder name
     *
     * @param string $user  A user name
     *
     * @returns boolean  True if $user has 'a' right
     */
    function canEdit($folder, $user)
    {
        return true;
    }

    function getRights()
    {
        return $this->_rightsList;
    }

    function getProtected()
    {
        return $this->_protected;
    }

    function isSupported()
    {
        return $this->_supported;
    }

    /**
     * Attempts to return an ACL instance based on $driver.
     *
     * @access public
     *
     * @param string $driver          The type of concrete ACL subclass
     *                                to return.  The is based on the acl
     *                                source ($driver).  The code is
     *                                dynamically included.
     *
     * @param optional array $params  A hash containing any additional
     *                                configuration or connection parameters
     *                                a subclass might need.
     *
     * @return mixed  The newly created concrete ACL instance, or false
     *                on error.
     */
    function &factory($driver, $params = array())
    {
        $driver = basename($driver);
        require_once dirname(__FILE__) . '/ACL/' . $driver . '.php';
        $class = 'IMP_ACL_' . $driver;
        if (class_exists($class)) {
            return new $class($params);
        } else {
            return false;
        }
    }

    /**
     * Attempts to return a reference to a concrete ACL instance
     * based on $driver.  It will only create a new instance if no
     * ACL instance with the same parameters currently exists.
     *
     * This method must be invoked as: $var = &ACL::singleton()
     *
     * @access public
     *
     * @param string $driver          The type of concrete ACL subclass
     *                                to return.  The is based on the acl
     *                                source ($driver).  The code is
     *                                dynamically included.
     *
     * @param optional array $params  A hash containing any additional
     *                                configuration or connection parameters
     *                                a subclass might need.
     *
     * @return mixed  The created concrete ACL instance, or false on error.
     */
    function &singleton($driver, $params = array())
    {
        static $instances;

        if (!isset($instances)) {
            $instances = array();
        }

        $signature = serialize(array($driver, $params));
        if (!array_key_exists($signature, $instances)) {
            $instances[$signature] = &IMP_ACL::factory($driver, $params);
        }

        return $instances[$signature];
    }

}
