#include <stdio.h>
#include <math.h>

/*
 * My eternal thanks to Pat Kane for spending a week of his life researching
 * and writing this module.
 * (It probably would have taken me months and months)
 */

/*
 * Solid Noise - as describe by Perlin (1985)
 *
 * By:  P.E.Kane, 1990
 */

static char rcsid[] = "$Id: S3Noise.c,v 1.2 90/09/07 01:11:37 kane Exp $  PEK";

#if defined(sun) || defined(ultrix)
# define RMAX 2147483647
#else
# define RMAX 32767
#endif
# define RND1 ((double)rand()/(double)RMAX)

# define MAXRX  64
# define MAXRY  64
# define MAXRZ  64

struct Lattice {
    float a, b, c;   /* gradient: [a,b,c] */
    float d;         /* random value */
};

struct Lattice theLattice[MAXRX][MAXRY][MAXRZ];

static lattice_inited = 0;

InitLattice ()
{
    register int i, j, k;
    lattice_inited = 1;
    
    /*
     * fill lattice with random numbers between 0.0 and 1.0
     */
    for (i = 0; i < MAXRX; i++)
    {
	for (j = 0; j < MAXRY; j++)
	{
	    for (k = 0; k < MAXRZ; k++)
	    {
		theLattice[i][j][k].d = RND1;
	    }
	}
    }
    /*
     * compute gradients 
     */
    for (i = 0; i < MAXRX; i++)
    {
	for (j = 0; j < MAXRY; j++)
	{
	    for (k = 0; k < MAXRZ; k++)
	    {
		struct Lattice *lp = &theLattice[i][j][k];
		
		if ( i == 0 )  
		{
		    lp->a = (theLattice[i+1][j][k].d -
			     theLattice[(MAXRX-1)][j][k].d ) / 2.0;
		}
		else if ( i == (MAXRX-1) )
		{	
		    lp->a = (theLattice[0][j][k].d -
			     theLattice[i-1][j][k].d ) / 2.0;  
		}
		else
		{
		    lp->a = (theLattice[i+1][j][k].d -
			     theLattice[i-1][j][k].d ) / 2.0;
		}
		
		
		if ( j == 0 ) 
		{
		    lp->b = (theLattice[i][j+1][k].d -
			     theLattice[i][(MAXRY-1)][k].d ) / 2.0;   
		}
		else if ( j == (MAXRY-1) )
		{
		    lp->b = (theLattice[i][0][k].d -
			     theLattice[i][j-1][k].d ) / 2.0;
		}
		else
		{
		    lp->b = (theLattice[i][j+1][k].d -
			     theLattice[i][j-1][k].d ) / 2.0;
		}
		
		
		if ( k == 0 )
		{
		    lp->c = (theLattice[i][j][k+1].d -
			     theLattice[i][j][(MAXRZ-1)].d ) / 2.0;
		}
		else if ( k == (MAXRZ-1) )
		{
		    lp->c = (theLattice[i][j][0].d -
			     theLattice[i][j][k-1].d ) / 2.0;   
		}
		else
		{
		     lp->c = (theLattice[i][j][k+1].d -
			     theLattice[i][j][k-1].d ) / 2.0;
		}
	    }
	}
    }
}

#ifdef notdef
DumpLattice (nx, ny, nz)
{
    int i, j, k;
    fprintf (stderr, "First %dx%dx%d of the Lattice (%dx%dx%d):\n",
	     nx, ny,nz,
	     MAXRX, MAXRY, MAXRZ);
    for (k = 0; k < MAXRZ && k < nz; k++)
    {
	fprintf (stderr, "z = %d\n\t", k);
	for (j = 0; j < MAXRY && j < ny; j++)
	{
	    for (i = 0; i < MAXRX && i < nx ; i++)
	    {
		fprintf (stderr, "%6.2f ",
			 theLattice[i][j][k].d );
	    }
	    fprintf (stderr, "\n\t");
	}
    }
    fprintf (stderr, "\n");
}
#endif

double
Wrap (v, mod)
   double v;
   int mod;
{
   double t;
   
   if ( (t = fmod(v,(double)mod)) < 0.0)
       return (t+(double)mod);
   else
       return (t);
}


 
double
S3Noise (x, y, z)
    double x, y, z;
{
    double SInterpolate ();
    double LInterpolate ();
    double BLInterpolate ();
    struct Lattice *P0, *P1, *P2, *P3;   /* the 8 points of a sub-cube */
    struct Lattice *P4, *P5, *P6, *P7;
    double r0, r1;
    int i1, j1, k1;
    int i2, j2, k2;
    double tx, ty, tz;
    double d0, d1, d2, d3;       /* from the 4 corners of an Y-Z surface */
    double dd0, dd1;             /* from endpoints of Z segment */
    double ddd;                  /* from the point */

    x = Wrap (x, MAXRX-1);
    i1 = (int)x;
    i2 = (int)Wrap (x+1.0, MAXRX-1);
    
    y = Wrap (y, MAXRY-1);
    j1 = (int)y;
    j2 = (int)Wrap (y+1.0, MAXRY-1);
    
    z = Wrap(z, MAXRZ-1);
    k1 = (int)z;
    k2 = (int)Wrap (z+1.0, MAXRZ-1);
	
    tx = x - i1;
    /*
     ***************************************************************
     * Use 8 points of a sub-cube in the lattice to
     * get 4 new values (d0, d1, d2, d3) of an  Y-Z surface
     *
     *
     *                 d1
     *        P2 O------o--------O P3
     *          /|     ..       /|
     *         / |    . .      / |
     *        /  |   .  .     /  |
     *       /   |  .   .    /   |
     *      /   d3 .    .   /    |             Y (j)
     *  P6 O------o--------O P7  |             |
     *     |     |.     .  |     |             |
     *     |     |.  d0 .  |     |             |
     *     |  P0 O------o--|-----O P1          *------X  (i)
     *     |    / .   .    |    /             /
     *     |   /  .  .     |   /             /
     *     |  /   . .      |  /             Z (k)
     *     | /    ..       | /
     *     |/     .        |/
     *  P4 O------o--------O P5
     *            d2
     *
     *
     ***************************************************************
     */
	P0 = &theLattice[i1][j1][k1];
	P1 = &theLattice[i2][j1][k1];
	P2 = &theLattice[i1][j2][k1];
	P3 = &theLattice[i2][j2][k1];
	P4 = &theLattice[i1][j1][k2];
	P5 = &theLattice[i2][j1][k2];
	P6 = &theLattice[i1][j2][k2];
	P7 = &theLattice[i2][j2][k2];
	
	r0 = P0->a;
	r1 = P1->a;
	d0 = SInterpolate(tx,
			  P0->d, P1->d,
			  r0, r1);
	r0 = P2->a;
	r1 = P3->a;
	d1 = SInterpolate(tx,
			  P2->d, P3->d,
			  r0, r1);
	r0 = P4->a;
	r1 = P5->a;
	d2 = SInterpolate(tx,
			  P4->d, P5->d,
			  r0, r1);
	r0 = P6->a;
	r1 = P7->a;
	d3 = SInterpolate(tx,
			  P6->d, P7->d,
			  r0, r1);
     /*
     **************************************
     *  Interpolate to form Z segment
     *
     *             O d1
     *            /|
     *           / |
     *          /  |
     *         /   |
     *        /    o  dd0
     *    d3 O    .|
     *       |   . |
     *       |  .  |
     *       | .   O d0
     *       |.   / 
     *  dd1  o   /
     *       |  /   
     *       | /    
     *       |/
     *    d2 O
     *
     **************************************
     */
	
	ty = y - j1;
	
	/* 
	 * use a linear interpolation, in the
	 * x direction, to get the y-gradients.
	 */		 
	r0 = LInterpolate(tx, P0->b, P1->b); 
	r1 = LInterpolate(tx, P2->b, P3->b);
	dd0 = SInterpolate(ty,
			   d0, d1,
			   r0, r1);
	
	r0 = LInterpolate(tx, P4->b, P5->b);
	r1 = LInterpolate(tx, P6->b, P7->b);	
	dd1 = SInterpolate(ty,
			   d2, d3,
			   r0, r1);
	
     /*
     *************************************************************** 
     * Final interpolation gives value at point in XYZ latice
     *
     *                dd0
     *              O
     *             /
     *            o ddd
     *           /
     *          /
     *     dd1 O 
     ***************************************************************
     */
	tz = z - k1;
	/* 
	 * Use a bilinear interpolation, in the
	 * x & y directions, to get the z-gradients.
	 * See page 96 of "Numerical Recipes".
	 */		
	r0 = BLInterpolate(tx, ty,
			   P0->c, P1->c, P3->c, P2->c);  
	r1 = BLInterpolate(tx, ty,
			   P4->c, P5->c, P7->c, P6->c);  
	
	ddd = SInterpolate(tz,
			    dd0, dd1,
			    r0, r1);

    return ( ddd );
}


/*
 * Hermite spline interpolation
 */
double
SInterpolate (t, P1, P4, R1, R4)
    double t, P1, P4, R1, R4;
{
    double t2, t3;
    double d_ret;
    
    /*
     * The formula to fit a Hermite curve
     * (Foley and Van Dam page 517) is
     * x(t)  = P1x(2t^3 - 3t^2 + 1)
     *       + P4x(-2t^3 + 3t^2)
     *       + R1x(t^3 - 2t^2 + t)
     *       + R4x(t^3 - t^2)
     */
    t2 = t*t;
    t3 = t2*t;
    d_ret =  P1 * (  2*t3  - 3*t2 + 1 )
    +  P4 * ( -2*t3  + 3*t2     )
    +  R1 * (    t3  - 2*t2 + t )
    +  R4 * (    t3  -   t2     ) ;
    
    return (d_ret);
    
}

/*
 * Linear Interpolation
 */
double
LInterpolate (t, P1, P2)
    double t, P1, P2;
{
    return ( ((1.0 - t ) * P1) 
	    + (t * P2)
	    );    
}

/*
 * Bilinear Interpolation
 */
double
BLInterpolate (t, u, P1, P2, P3, P4)
    double t, u, P1, P2, P3, P4;
{
    return ( (1-t) * (1-u) * P1 
	    + t * (1-u) * P2
	    + t * u * P3
	    + (1-t) * u * P4
	    );
}

