/*
 * Functions to create and manage a hash table of colors allocated when
 * mapping a base-shaded image to a colormap.  Mapping a color to the
 * closest colormap value is expensive.  So we save all mappings here
 * so we don't ever have to look up the same color twice.
 */
#include <stdio.h>
#include <X11/Intrinsic.h>

#define	HASHFUNC(r, g, b)	(((r) << 4) ^ ((g) << 2) ^ (b)) & 0xfff

#define CBUCKETS	4096


struct chash_rec {
        int red, green, blue;
	int cindx;
        struct chash_rec *next;
};

struct chash_rec *CHash[CBUCKETS];
int CHcount;


/*
 * Clear out all hash pointers and zero the counting variable.
 * This really only needs to ever be called once, since freeing
 * the hash table also re-initializes it.
 */
void
InitCHash()
{
	int i;

	for (i=0; i<CBUCKETS; i++)
	{
		CHash[i] = NULL;
	}
	CHcount = 0;
}


/*
 * Free all allocated hash table entries, and re-initialize the hash table
 */
void
FreeCHash()
{
	int i;
	int depth, tmpd;
	struct chash_rec *cptr;
	struct chash_rec *cptr2;

	depth = 0;
	for (i=0; i<CBUCKETS; i++)
	{
		cptr = CHash[i];
		tmpd = 0;
		while(cptr != NULL)
		{
			cptr2 = cptr;
			cptr = cptr->next;
			XtFree(cptr2);
			tmpd++;
		}
		CHash[i] = NULL;
		if (tmpd > depth)
			depth = tmpd;
	}
/*
fprintf(stderr, "Hashed %d colors, Deepest bucket was %d\n", CHcount, depth);
*/
	CHcount = 0;
}


/*
 * Add a color, and the colormap entry it maps to to the hash table.
 */
void
AddCHash(red, green, blue, cindx)
	int red, green, blue;
	int cindx;
{
	int indx;
	struct chash_rec *cptr;

	indx = HASHFUNC(red, green, blue);
	cptr = CHash[indx];
	if (cptr == NULL)
	{
		cptr = (struct chash_rec *) XtMalloc(sizeof(struct chash_rec));
		cptr->red = red;
		cptr->green = green;
		cptr->blue = blue;
		cptr->cindx = cindx;
		cptr->next = NULL;
		CHash[indx] = cptr;
	}
	else
	{
		/*
		 * This should never happen, but just in case, eliminate
		 * duplicates.
		 */
		while(cptr->next != NULL)
		{
			if ((cptr->red == red)&&
			    (cptr->green == green)&&
			    (cptr->blue == blue))
			{
				return;
			}
			cptr = cptr->next;
		}
		if ((cptr->red == red)&&
		    (cptr->green == green)&&
		    (cptr->blue == blue))
		{
			return;
		}
		cptr->next = (struct chash_rec *)
			XtMalloc(sizeof(struct chash_rec));
		cptr = cptr->next;
		cptr->red = red;
		cptr->green = green;
		cptr->blue = blue;
		cptr->cindx = cindx;
		cptr->next = NULL;
	}
	CHcount++;
	return;
}


/*
 * Look up a color in the hash table, and return the colormap entry number
 * it maps too.  Return -1 if the color isn't in the hash table.
 */
int
FindCHash(red, green, blue)
	int red, green, blue;
{
	int indx;
	struct chash_rec *cptr;

	indx = HASHFUNC(red, green, blue);
	cptr = CHash[indx];
	while(cptr != NULL)
	{
		if ((cptr->red == red)&&
		    (cptr->green == green)&&
		    (cptr->blue == blue))
		{
			return(cptr->cindx);
		}
		cptr = cptr->next;
	}
	return(-1);
}

