/*
 * This routine lets me use solid noise to essentially carve a 3D object
 * out of granite.  The code here was reverse-engineered as best I could
 * based on the pseudo-code on page 354 of _The_RenderMan_Companion_ by
 * Steve Upstill.
 */

#include <stdio.h>
#include <sys/types.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "color.h"

#define _ABS(x)         ((x) < 0 ? -(x) : (x))


extern double S3Noise();
extern void UnRotate();

extern Display	*dsp;
extern Window	Dwindow;
extern GC	DrawGC;
extern unsigned int Width, Height;
extern XColor	color[];
extern Colormap	Cmap;
extern float	view_dist;
extern float	distance;
extern int	NumCells;
extern Bool	Nomap;
extern struct	col_rec Colors[];



/*
 * Allocate the granite colormap.  This is basically just a gray-scale.
 */
ShadeGranite()
{
	int i;

	for (i=0; i<NumCells; i++)
	{
		Colors[i].red = i;
		Colors[i].green = i;
		Colors[i].blue = i;

		color[i].red = Colors[i].red * 65536 / NumCells;
		color[i].green = Colors[i].green * 65536 / NumCells;
		color[i].blue = Colors[i].blue * 65536 / NumCells;
		color[i].flags = DoRed|DoGreen|DoBlue;
		if (Nomap == True)
		{
			XAllocColor(dsp, Cmap, &color[i]);
		}
		color[i].pixel = i;
	}
	if (Nomap == False)
	{
		XStoreColors(dsp, Cmap, color, NumCells);
	}
}




/*
 * Calculate, Shade, and Draw the triangle defined by the three points.
 */
void
graniteTriangle(x1, y1, z1, x2, y2, z2, x3, y3, z3, cos_theta)
	float x1, y1, z1;
	float x2, y2, z2;
	float x3, y3, z3;
	float cos_theta;
{
	int i;
	int tmp, dist;
	int xscale, yscale, xoffset, yoffset;
	int colornum;
	int red_v, green_v, blue_v;
	float x, y, z;
	float xp1, yp1, xp2, yp2, xp3, yp3;
	double xf, yf, zf;
	double nret;
	double sum;
	double freq;
	XPoint points[4];

	/*
	 * Note, the point must be unscaled, and unrotated before being
	 * passed to S3Noise.  Otherwise the granite pattern won't appear
	 * to rotate with the object.
	 */
	x = (x1 + x2 + x3) / 3.0;
	y = (y1 + y2 + y3) / 3.0;
	z = (z1 + z2 + z3) / 3.0;
	z = z + distance;
	UnRotate(&x, &y, &z);
	z = z - distance;
	xf = (double)(x + 1.8) * 10.0 / 2.0;
	yf = (double)(y + 1.8) * 10.0 / 2.0;
	zf = (double)(z + distance + 1.8) * 10.0 / 2.0;

	sum = 0.0;
	freq = 4.0;
	for (i=0; i<6; i++)
	{
		nret = S3Noise(freq * xf, freq * yf, freq * zf);
		nret = _ABS((double)(0.5 - nret));
		sum = sum + ((nret * 4.0) / freq);
		freq *= 2;
	}
	if (sum < 0)
		sum = 0;
	if (sum >= 1)
		sum = 0.9999;

	/*
	 * sum is the intesity of gray in the granite at this point,
	 * and cos_theta tells the intensity of the reflected light
	 * from this polygon.
	 */
	colornum = NumCells * sum * (cos_theta + 0.05);
	if (colornum > (NumCells - 1))
	{
		colornum = (NumCells - 1);
	}
	if (Nomap == True)
	{
		red_v = colornum;
		green_v = colornum;
		blue_v = colornum;
		dist = (int)sqrt((double)3 * 256 * 256);
		for (i=0; i<NumCells; i++)
		{
			tmp = (int)sqrt((double)(red_v - Colors[i].red) *
				(red_v - Colors[i].red) +
				(green_v - Colors[i].green) *
				(green_v - Colors[i].green) +
				(blue_v - Colors[i].blue) *
				(blue_v - Colors[i].blue));
			if (tmp < dist)
			{
				colornum = i;
				dist = tmp;
			}
		}
	}

	/*
	 * Scale and draw image on 2D
	 */
	xscale = xoffset = (Width) / 2;
	yscale = yoffset = (Height) / 2;

	xp1 = (x1 * view_dist) / (view_dist - z1);
	yp1 = (y1 * view_dist) / (view_dist - z1);
	xp2 = (x2 * view_dist) / (view_dist - z2);
	yp2 = (y2 * view_dist) / (view_dist - z2);
	xp3 = (x3 * view_dist) / (view_dist - z3);
	yp3 = (y3 * view_dist) / (view_dist - z3);

	points[0].x = (int)(xp1 * xscale) + xoffset;
	points[1].x = (int)(xp2 * xscale) + xoffset;
	points[2].x = (int)(xp3 * xscale) + xoffset;
	points[3].x = points[0].x;

	points[0].y = yoffset - (int)(yp1 * yscale);
	points[1].y = yoffset - (int)(yp2 * yscale);
	points[2].y = yoffset - (int)(yp3 * yscale);
	points[3].y = points[0].y;

	XSetForeground(dsp, DrawGC, colornum);
	XFillPolygon(dsp, Dwindow, DrawGC, points, 3,
		Convex, CoordModeOrigin);
}

