/*
 * This routine lets me use solid noise to essentially carve a 3D object
 * out of blue marble.  The code here was reverse-engineered as best I could
 * based on the pseudo-code on page 355 of _The_RenderMan_Companion_ by
 * Steve Upstill.
 */

#include <stdio.h>
#include <stdio.h>
#include <sys/types.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "color.h"

extern double S3Noise();
extern void UnRotate();

extern Display	*dsp;
extern Window	Dwindow;
extern GC	DrawGC;
extern unsigned int Width, Height;
extern XColor	color[];
extern Colormap	Cmap;
extern float	view_dist;
extern float	distance;
extern int	NumCells;
extern Bool	Nomap;


struct pnt_rec {
float x, y, z;
};

extern struct col_rec Colors[];

struct pnt_rec Color_pts[] = {
	{0.25, 0.25, 0.75},
	{0.25, 0.25, 0.75},
	{0.20, 0.20, 0.70},
	{0.20, 0.20, 0.70},
	{0.20, 0.20, 0.70},
	{0.25, 0.25, 0.75},
	{0.25, 0.25, 0.75},
	{0.15, 0.15, 0.66},
	{0.15, 0.15, 0.66},
	{0.10, 0.10, 0.60},
	{0.10, 0.10, 0.60},
	{0.25, 0.25, 0.75},
	{0.10, 0.10, 0.60}
};
#define NCOL_PTS	13



/*
 * A spline function that takes a value in the range [0 - 1)
 * and a sequence of [0 - N] (at least 4) points.  It uses a Hermite
 * curve fitting algorithm to make a spline of points [1 - (N - 1)]
 * Using point 0 and point N to find the tangents at point 1 and
 * point N - 1 respectively.
 */
void
spline(val, npts, pts, pt_ret)
float val;
int npts;
struct pnt_rec pts[];
struct pnt_rec *pt_ret;
{
	int pt_cnt;
	int start, end;
	float nval;

	if ((val < 0.0)||(val >= 1.0)||(npts < 1))
	{
		pt_ret->x = 0.0;
		pt_ret->y = 0.0;
		pt_ret->z = 0.0;
	}
	else if (npts < 2)
	{
		pt_ret->x = pts[0].x;
		pt_ret->y = pts[0].y;
		pt_ret->z = pts[0].z;
	}
	else if (npts < 4)
	{
		pt_ret->x = pts[1].x;
		pt_ret->y = pts[1].y;
		pt_ret->z = pts[1].z;
	}
	else
	{
		start = (int)((npts - 3) * val) + 1;
		end = start + 1;
		/*
		 * the spline is linearly divided into equal intervals
		 * and the value passed is scaled to a new value that is
		 * in the range [0 - 1) for its interval
		 */
		nval = (val - (float)(start - 1) / (npts - 3)) * (npts - 3);
		/*
		 * The formula to fit a Hermite curve
		 * (Foley and Van Dam page 517) is
		 * x(t) = P1x(2t^3 - 3t^2 + 1) + P4x(-2t^3 + 3t^2)
		 *      + R1x(t^3 - 2t^2 + t) + R4x(t^3 - t^2)
		 * where P1 = pts[1], P4 = pts[2], R1 = pts[2] - pts[0],
		 * and R4 = pts[3] - pts[1]
		 */
		pt_ret->x = (pts[start].x *
			(2 * nval * nval * nval - 3 * nval * nval + 1)) +
			(pts[end].x *
			(3 * nval * nval - 2 * nval * nval * nval)) +
			((pts[start + 1].x - pts[start - 1].x) *
			(nval * nval * nval - 2 * nval * nval + nval)) +
			((pts[end + 1].x - pts[end - 1].x) *
			(nval * nval * nval - nval * nval));
		pt_ret->y = (pts[start].y *
			(2 * nval * nval * nval - 3 * nval * nval + 1)) +
			(pts[end].y *
			(3 * nval * nval - 2 * nval * nval * nval)) +
			((pts[start + 1].y - pts[start - 1].y) *
			(nval * nval * nval - 2 * nval * nval + nval)) +
			((pts[end + 1].y - pts[end - 1].y) *
			(nval * nval * nval - nval * nval));
		pt_ret->z = (pts[start].z *
			(2 * nval * nval * nval - 3 * nval * nval + 1)) +
			(pts[end].z *
			(3 * nval * nval - 2 * nval * nval * nval)) +
			((pts[start + 1].z - pts[start - 1].z) *
			(nval * nval * nval - 2 * nval * nval + nval)) +
			((pts[end + 1].z - pts[end - 1].z) *
			(nval * nval * nval - nval * nval));
	}
}


/*
 * Create the colormap for the blue-marble surface.
 */
ShadeMarble()
{
	int i, j;
	float csp;
	struct pnt_rec point;

	for (i=0; i<17; i++)
	{
		csp = (float)i / 17.0;
		spline(csp, NCOL_PTS, Color_pts, &point);
		for (j=1; j<16; j++)
		{
			Colors[(i * 15) + j].red = point.x * 256 *
				(float)(j) / 15.0;
			Colors[(i * 15) + j].green = point.y * 256 *
				(float)(j) / 15.0;
			Colors[(i * 15) + j].blue = point.z * 256 *
				(float)(j) / 15.0;

			color[(i * 15) + j].red = Colors[(i * 15) + j].red *
				65536 / NumCells;
			color[(i * 15) + j].green = Colors[(i * 15) + j].green *
				65536 / NumCells;
			color[(i * 15) + j].blue = Colors[(i * 15) + j].blue *
				65536 / NumCells;
			color[(i * 15) + j].flags = DoRed|DoGreen|DoBlue;
			if (Nomap == True)
			{
				XAllocColor(dsp, Cmap, &color[(i * 15) + j]);
			}
			color[(i * 15) + j].pixel = (i * 15) + j;
		}
	}
	if (Nomap == False)
	{
		XStoreColors(dsp, Cmap, color, NumCells);
	}
}


/*
 * Calculate, Shade, and Draw the triangle defined by the three points.
 */
void
marbleTriangle(x1, y1, z1, x2, y2, z2, x3, y3, z3, cos_theta)
	float x1, y1, z1;
	float x2, y2, z2;
	float x3, y3, z3;
	float cos_theta;
{
	int i;
	int tmp, dist;
	int xscale, yscale, xoffset, yoffset;
	int colornum;
	int red_v, green_v, blue_v;
	float x, y, z;
	float xp1, yp1, xp2, yp2, xp3, yp3;
	float scale, pixelsize, twice, turbulence;
	float weight, csp;
	double xf, yf, zf;
	double nret;
	XPoint points[4];
	struct pnt_rec point;

	pixelsize = 0.1;
	twice = 2 * pixelsize;
	turbulence = 0;

	/*
	 * Note, the point must be unscaled, and unrotated before being
	 * passed to S3Noise.  Otherwise the marble pattern won't appear
	 * to rotate with the object.
	 */
	x = (x1 + x2 + x3) / 3.0;
	y = (y1 + y2 + y3) / 3.0;
	z = (z1 + z2 + z3) / 3.0;
	z = z + distance;
	UnRotate(&x, &y, &z);
	z = z - distance;
	xf = (double)(x + 1.8) * 5.0 / 2.0;
	yf = (double)(y + 1.8) * 5.0 / 2.0;
	zf = (double)(z + distance + 1.8) * 5.0 / 2.0;
	for (scale = 1; scale > twice; scale /= 2)
	{
		nret = S3Noise((xf / scale), (yf / scale), (zf / scale));
		turbulence = turbulence + (scale * nret);
	}
	if (scale > pixelsize)
	{
		weight = (scale / pixelsize) - 1;
		if (weight < 0)
			weight = 0;
		if (weight > 1)
			weight = 1;
		nret = S3Noise((xf / scale), (yf / scale), (zf / scale));
		turbulence = turbulence + (weight * scale * nret);
	}
	csp = 4 * turbulence - 3;
	if (csp < 0)
		csp = 0;
	if (csp >= 1)
		csp = 0.9999;
	if (Nomap == True)
	{
		spline(csp, NCOL_PTS, Color_pts, &point);
		red_v = point.x * NumCells * (cos_theta + 0.05);
		green_v = point.y * NumCells * (cos_theta + 0.05);
		blue_v = point.z * NumCells * (cos_theta + 0.05);
		if (red_v > (NumCells - 1))
		{
			red_v = (NumCells - 1);
		}
		if (green_v > (NumCells - 1))
		{
			green_v = (NumCells - 1);
		}
		if (blue_v > (NumCells - 1))
		{
			blue_v = (NumCells - 1);
		}
	
		dist = (int)sqrt((double)3 * 256 * 256);
		colornum = 0;
		for (i=0; i<NumCells; i++)
		{
			tmp = (int)sqrt((double)(red_v - Colors[i].red) *
				(red_v - Colors[i].red) +
				(green_v - Colors[i].green) *
				(green_v - Colors[i].green) +
				(blue_v - Colors[i].blue) *
				(blue_v - Colors[i].blue));
			if (tmp < dist)
			{
				colornum = i;
				dist = tmp;
			}
		}
	}
	else
	{
		colornum = (int)(17.0 * csp);
		tmp = (int)(15.0 * (cos_theta + 0.05));
		if (tmp > 14)
		{
			tmp = 14;
		}
		colornum = (colornum * 15) + tmp + 1;
	}

	/*
	 * Scale and draw image on 2D
	 */
	xscale = xoffset = (Width) / 2;
	yscale = yoffset = (Height) / 2;

	xp1 = (x1 * view_dist) / (view_dist - z1);
	yp1 = (y1 * view_dist) / (view_dist - z1);
	xp2 = (x2 * view_dist) / (view_dist - z2);
	yp2 = (y2 * view_dist) / (view_dist - z2);
	xp3 = (x3 * view_dist) / (view_dist - z3);
	yp3 = (y3 * view_dist) / (view_dist - z3);

	points[0].x = (int)(xp1 * xscale) + xoffset;
	points[1].x = (int)(xp2 * xscale) + xoffset;
	points[2].x = (int)(xp3 * xscale) + xoffset;
	points[3].x = points[0].x;

	points[0].y = yoffset - (int)(yp1 * yscale);
	points[1].y = yoffset - (int)(yp2 * yscale);
	points[2].y = yoffset - (int)(yp3 * yscale);
	points[3].y = points[0].y;

	XSetForeground(dsp, DrawGC, colornum);
	XFillPolygon(dsp, Dwindow, DrawGC, points, 3,
		Convex, CoordModeOrigin);
}

