/*
 * This routine lets me use solid noise to essentially carve a 3D object
 * out of wood.  The code here was reverse-engineered as best I could
 * based on the pseudo-code on page 351 of _The_RenderMan_Companion_ by
 * Steve Upstill.
 */

#include <stdio.h>
#include <sys/types.h>
#include <math.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include "color.h"

#define _ABS(x)         ((x) < 0 ? -(x) : (x))


extern double S3Noise();
extern void UnRotate();

extern Display	*dsp;
extern Window	Dwindow;
extern GC	DrawGC;
extern unsigned int Width, Height;
extern XColor	color[];
extern Colormap	Cmap;
extern float	view_dist;
extern float	distance;
extern int	NumCells;
extern Bool	Nomap;


struct col_pts {
double red, green, blue;
};

extern struct col_rec Colors[];


/*
struct col_pts lightwood = {0.3, 0.12, 0.03};
struct col_pts darkwood = {0.05, 0.01, 0.005};
*/
struct col_pts lightwood = {0.6, 0.24, 0.06};
struct col_pts darkwood = {0.25, 0.05, 0.025};


/*
 * The smoothstep function as described on page 312 of _The_RenderMan_Companion_
 */
double
smoothstep(min, max, val)
	double min, max, val;
{
	if (val < min)
	{
		return(0.0);
	}
	else if (val >= max)
	{
		return(1.0);
	}
	else
	{
		val = (val - min) * 1.0 / (max - min);
		val = 3.0 * val * val - 2.0 * val * val * val;
		return(val);
	}
}


/*
 * The mix function as described on page 331 of _The_RenderMan_Companion_
 */
void
mix(col1, col2, mval, rcol)
	struct col_pts *col1;
	struct col_pts *col2;
	double mval;
	struct col_pts *rcol;
{
	rcol->red = ((1.0 - mval) * col1->red) + (mval * col2->red);
	rcol->green = ((1.0 - mval) * col1->green) + (mval * col2->green);
	rcol->blue = ((1.0 - mval) * col1->blue) + (mval * col2->blue);
}


/*
 * Create the colormap for the wood surface.
 */
ShadeWood()
{
	int i, j;
	struct col_pts acolor;

	for (i=0; i<17; i++)
	{
		mix(&lightwood, &darkwood, ((double)(i) / 17.0), &acolor);
		for (j=1; j<16; j++)
		{
			Colors[(i * 15) + j].red = acolor.red * NumCells *
				(float)(j) / 15.0;
			Colors[(i * 15) + j].green = acolor.green * NumCells *
				(float)(j) / 15.0;
			Colors[(i * 15) + j].blue = acolor.blue * NumCells *
				(float)(j) / 15.0;

			color[(i * 15) + j].red = Colors[(i * 15) + j].red *
				65536 / NumCells;
			color[(i * 15) + j].green = Colors[(i * 15) + j].green *
				65536 / NumCells;
			color[(i * 15) + j].blue = Colors[(i * 15) + j].blue *
				65536 / NumCells;
			color[(i * 15) + j].flags = DoRed|DoGreen|DoBlue;
			if (Nomap == True)
			{
				XAllocColor(dsp, Cmap, &color[(i * 15) + j]);
			}
			color[(i * 15) + j].pixel = (i * 15) + j;
		}
	}
	if (Nomap == False)
	{
		XStoreColors(dsp, Cmap, color, NumCells);
	}
}




/*
 * Calculate, Shade, and Draw the triangle defined by the three points.
 */
void
woodTriangle(x1, y1, z1, x2, y2, z2, x3, y3, z3, cos_theta)
	float x1, y1, z1;
	float x2, y2, z2;
	float x3, y3, z3;
	float cos_theta;
{
	int i;
	int tmp, dist;
	int xscale, yscale, xoffset, yoffset;
	int colornum;
	int red_v, green_v, blue_v;
	float x, y, z;
	float xp1, yp1, xp2, yp2, xp3, yp3;
	double xf, yf, zf;
	double nret;
	double rad;
	double ringscale, rscale, nscale;
	XPoint points[4];
	struct col_pts thiscolor;

	ringscale = 10.0;
	rscale = 5.0;
	nscale = 2.0;

	/*
	 * Note, the point must be unscaled, and unrotated before being
	 * passed to S3Noise.  Otherwise the wood pattern won't appear
	 * to rotate with the object.
	 */
	x = (x1 + x2 + x3) / 3.0;
	y = (y1 + y2 + y3) / 3.0;
	z = (z1 + z2 + z3) / 3.0;
	z = z + distance;
	UnRotate(&x, &y, &z);
	z = z - distance;

	/*
	 * decrease the constant multiple to zoom on the grain
	 */
	xf = (double)(x + 1.8) * 5.0 / 2.0;
	yf = (double)(y + 1.8) * 5.0 / 2.0;
	zf = (double)(z + distance + 1.8) * 5.0 / 2.0;

	/*
	 * increase nscale to reduce turbulence
	 */
	nret = S3Noise(xf/nscale, yf/nscale, zf/nscale);
	xf += nret;
	yf += nret;
	zf += nret;
	rad = sqrt((yf * yf) + (zf * zf));

	rad *= ringscale;
	rad += _ABS(S3Noise(rad/rscale, rad/rscale, rad/rscale));
	rad = rad - (int)rad;

	rad = smoothstep(0.0, 0.8, rad) *
		(1.0 - smoothstep(0.83, 0.86, rad));
	if (rad < 0)
		rad = 0;
	if (rad >= 1)
		rad = 0.9999;

	if (Nomap == True)
	{
		mix(&lightwood, &darkwood, rad, &thiscolor);
		red_v = thiscolor.red * NumCells * (cos_theta + 0.05);
		green_v = thiscolor.green * NumCells * (cos_theta + 0.05);
		blue_v = thiscolor.blue * NumCells * (cos_theta + 0.05);
		if (red_v > (NumCells - 1))
		{
			red_v = (NumCells - 1);
		}
		if (green_v > (NumCells - 1))
		{
			green_v = (NumCells - 1);
		}
		if (blue_v > (NumCells - 1))
		{
			blue_v = (NumCells - 1);
		}

		dist = (int)sqrt((double)3 * 256 * 256);
		colornum = 0;
		for (i=0; i<NumCells; i++)
		{
			tmp = (int)sqrt((double)(red_v - Colors[i].red) *
				(red_v - Colors[i].red) +
				(green_v - Colors[i].green) *
				(green_v - Colors[i].green) +
				(blue_v - Colors[i].blue) *
				(blue_v - Colors[i].blue));
			if (tmp < dist)
			{
				colornum = i;
				dist = tmp;
			}
		}
	}
	else
	{
		colornum = (int)(17.0 * rad);
		tmp = (int)(15.0 * (cos_theta + 0.05));
		if (tmp > 14)
		{
			tmp = 14;
		}
		colornum = (colornum * 15) + tmp + 1;
	}

	/*
	 * Scale and draw image on 2D
	 */
	xscale = xoffset = (Width) / 2;
	yscale = yoffset = (Height) / 2;

	xp1 = (x1 * view_dist) / (view_dist - z1);
	yp1 = (y1 * view_dist) / (view_dist - z1);
	xp2 = (x2 * view_dist) / (view_dist - z2);
	yp2 = (y2 * view_dist) / (view_dist - z2);
	xp3 = (x3 * view_dist) / (view_dist - z3);
	yp3 = (y3 * view_dist) / (view_dist - z3);

	points[0].x = (int)(xp1 * xscale) + xoffset;
	points[1].x = (int)(xp2 * xscale) + xoffset;
	points[2].x = (int)(xp3 * xscale) + xoffset;
	points[3].x = points[0].x;

	points[0].y = yoffset - (int)(yp1 * yscale);
	points[1].y = yoffset - (int)(yp2 * yscale);
	points[2].y = yoffset - (int)(yp3 * yscale);
	points[3].y = points[0].y;

	XSetForeground(dsp, DrawGC, colornum);
	XFillPolygon(dsp, Dwindow, DrawGC, points, 3,
		Convex, CoordModeOrigin);
}

