/*------- LL Double linked list library: core functions ---------------- */
/*  author: G. Matas                           (g.matas@ee.surrey.ac.uk) */
/* +-------------------------------------------------------------------+ */
/* | Copyright 1992, 1993, George Matas.                               | */
/* |   Permission to use, copy, modify, and distribute this software   | */
/* |   and its documentation for any purpose and without fee is hereby | */
/* |   granted, provided that the above copyright notice appear in all | */
/* |   copies and that both that copyright notice and this permission  | */
/* |   notice appear in supporting documentation.  This software is    | */
/* |   provided "as is" without express or implied warranty.           | */
/* +-------------------------------------------------------------------+
*/

/*
      G. Matas, 30-Dec-93 v5.5
	 - long history list deleted; available in the 5.5 delta.
	   The list became redundant as LL was put under SCCS control.
*/
/*  based on a link library by Duane Morse                                  */
/*--------------------------------------------------------------------------*/

#include "LL.h"
#include "linkLL.h"
#include <stdlib.h>
#include <memory.h>
#include <assert.h>

/* common error messages */
static char * NullMall = "malloc returned NULL";

/*--------------- linking ------------------------------------------*/
void 
l_lafter(current, new)   l_list *current; l_list *new;
{
  new->forward = current->forward;
  new->backward = current;
  current->forward->backward = new;
  current->forward = new;
}

void
l_lbefore(current, new)  l_list *current;l_list  *new;
{
  new->forward = current;
  new->backward = current->backward;
  current->backward->forward = new;
  current->backward = new;
}

void
l_unlink(link)  l_list *link;

{
  link->forward->backward = link->backward;
  link->backward->forward = link->forward;
}

static void
l_linit(link)  l_list *link;
{ link->forward = link->backward = link; }


#ifndef L_BASIC_MACRO
l_list *l_nextl(l_list *link) { return link->forward;  }
l_list *l_prevl(l_list *link) { return link->backward; }
int    l_lempty(l_list *link) { return (link->forward == link); }
#endif

/*------------- LinkIn/Out an element ----------------------------*/
void *
LinkAftLL(curr, new)  void * curr;void * new;
                        { l_lafter(elm2link(curr),elm2link(new)); return new; }
void *
LinkBefLL(curr, new)  void * curr;void * new;
                        { l_lbefore(elm2link(curr),elm2link(new)); return new; }
void *
UnlinkLL(el)  void * el;
                        {l_unlink(elm2link(el)); return el; }


void *
UnlinkNeLL(el)  void * el;
{  
   void * next = NextElmLL(el);
   l_unlink(elm2link(el));
   return next;
}

void *
UnlinkPrLL(el)  void * el;
{  
   void * prev = PrevElmLL(el);
   l_unlink(elm2link(el));
   return prev;
}

/*---------------- Insert Element ----------------------------------------*/
static void *
InsLLf(li, size, data, linkin)
l_list *li; size_t size; void *data;
void (*linkin)();
{
  l_list * new;

  if (NULL==(new=(l_list*)malloc(size+sizeof(union_l_list))))
       AbortLL_M("InsLLf",NullMall);
  linkin(li,new);
  memcpy(link2elm(new),data,size);
  new->size=size;
  return (link2elm(new));
}

void *
InsBefLLf (el, size, data)  void *el; size_t size; void *data;
                        { return InsLLf(elm2link(el),size,data,l_lbefore); }
void *
InsAftLLf (el, size, data)  void * el; size_t size; void * data;
                        { return InsLLf(elm2link(el),size,data,l_lafter); }
void *
InsLastLLf (list, size, data)  t_LL list; size_t size; void * data;
                        { 
#ifdef LLDEBUG
          assert(list);
          assert(data);
#endif
return InsLLf(list2link(list),size,data,l_lbefore); }
void *
InsFirstLLf (list, size, data)  t_LL  list; size_t size; void * data;
                        { return InsLLf(list2link(list),size,data,l_lafter); }

/*---------------- Delete Element ----------------------------------------*/
void
DelElmLL  (el)  void * el;
{ 
#ifdef LLDEBUG
          assert(el);
#endif
  l_unlink( elm2link(el));
  free    ( elm2link(el));
}
void *
DelElmNeLL (el)  void * el;
{ 
 void * next ;
#ifdef LLDEBUG
          assert(el);
#endif
  next = NextElmLL(el);
  DelElmLL(el);
  return (next);
}

void *
DelElmPrLL (el)  void * el;
{ 
  void * prev ;
#ifdef LLDEBUG
          assert(el);
#endif
  prev = PrevElmLL(el);
  DelElmLL(el);
  return (prev);
}

/*----------------- Get an element ---------------------------------------*/
void *
FirstElmLL (list)  t_LL list;
        { 
#ifdef LLDEBUG
	  assert(list); 
#endif
	  return link2elm(l_nextl(list2link(list))); }

void *
LastElmLL (list)  t_LL list;
        { 
#ifdef LLDEBUG
	  assert(list); 
#endif
	  return link2elm(l_prevl(list2link(list))); }
void *
PrevElmLL (el)
void *el;
        { 
#ifdef LLDEBUG
          assert(el);
#endif
          return link2elm(l_prevl(elm2link(el))); }

void *  NextElmLL(el)
void *el;
        { 
#ifdef LLDEBUG
          assert(el);
#endif
	  return link2elm(l_nextl(elm2link(el))); }

void *  PrevCElmLL(el)
void * el;
{
  l_list * prevLink        = l_prevl(elm2link(el));
  if (prevLink->size == 0)   prevLink=l_prevl(prevLink);
  return link2elm(prevLink);
}

void *  NextCElmLL(el)
void * el;
{
  l_list * nextLink        = l_nextl(elm2link(el));
  if (nextLink->size == 0)   nextLink=l_nextl(nextLink);
  return link2elm(nextLink);
}

/* 2.1 : NthElmLL can be called with a negative value */
void *
NthElmLL (list, num)  t_LL list; t_LLsize num;
{
  l_list * link;

#ifdef LLDEBUG
          assert(list);
#endif
  link = list2link(list);
#ifdef LLDEBUG 
	  assert(link); 
#endif
  if (num >0) while (num--) link = l_nextl(link);
  else        while (num++) link = l_prevl(link); 

  return link2elm(link);
}
void *
RelNthElmLL (el, num)  void * el; t_LLsize num;
{
  l_list * link;
#ifdef LLDEBUG 
	  assert(el); 
#endif
  link = elm2link(el);
#ifdef LLDEBUG 
	  assert(link); 
#endif
  if (num >0) while (num--) link = l_nextl(link);
  else        while (num++) link = l_prevl(link); 

  return link2elm(link);
}


/*--------------------------------------------------------------------------*/
int
IsElmLL     (el)  void * el;
        { 
#ifdef LLDEBUG 
	  assert(el); 
#endif
		return elm2link(el)->size ; }
int
IsLastElmLL (el)  void *el;
        {return IsElmLL(NextElmLL(el)); }
int
IsFirstElmLL(el)  void *el;
        {return IsElmLL(PrevElmLL(el)); }

/*--------------------------------------------------------------------------*/
static char LL_id[] = "gm";
t_LL
ConsLL ()
{
  t_LL   head;

  if (NULL==(head=(t_LL) malloc(sizeof(*head))))
               AbortLL_M("CreatLL",NullMall);

  l_linit(list2link(head));
  head->id  = LL_id;
  head->links.u.ll.size= 0;
  return (head);
}

/*--------------------------------------------------------------------------*/
int
IsEmptyLL(list)  t_LL list;
        { return (l_lempty(list2link(list))); }
t_LL
EmptyLL(list)  t_LL list;
{
  l_list  * head = list2link(list);
  l_list  * link = l_nextl(head);
  l_list  * old  =link;

  while(head != link){
    old = link;
    link = l_nextl(link);
    free(old);
  }

  l_linit(head);

  return list;
}

/*--------------------------------------------------------------------------*/
void  *
DestLL(list)  t_LL list;
{
#ifdef LLDEBUG 
	  assert(list); 
#endif
  EmptyLL(list);
  free(list);
  return NULL;
}

/*--------------------------------------------------------------------------*/
void *
ApplyLL (list, apply)
t_LL list;
void *(*apply)();
{
  void * el, * ret_el, *next;

  SafeForeachLL_M (list,el,next){
   next = NextElmLL(el);
   if ((ret_el=(*apply)(el)) != NULL ) return ret_el;
  }

  return NULL;
}

/*--------------------------------------------------------------------------*/
t_LL
ReverseLL (list)  t_LL list;
{
  l_list * head = list2link(list);
  l_list * link = head; 
  l_list * temp;
 
  do{
    temp= link->forward;               /* swap */
    link->forward = link->backward;
    link->backward= temp;

    link=l_prevl(link);                /* move */
  }
  while (head != link) ;
  
  return list;
}

/*--------------------------------------------------------------------------*/
t_LLsize
SizeLL (list)  t_LL list;
{
  t_LLsize i=0;
  l_list * head = list2link(list);
  l_list * link; 

  ForeachLink_M(head,link) i++;
  
  return i;
}

/*--------------------------------------------------------------------------*/
t_LL
ConsPtrLL(src)  t_LL src;
{
  void * el ;
  t_LL dest= ConsLL();

  ForeachLL_M (src,el)
     InsLastLL(dest,el);

  return dest;
}
/*--------------------------------------------------------------------------*/
t_LL
ConsCopyLL(src)  t_LL src;
{
  void * el ;
  t_LL dest= ConsLL();
#ifdef LLDEBUG 
	  assert(src); 
#endif

  ForeachLL_M (src,el)
     InsLastLLf(dest,elm2link(el)->size, el);

  return dest;
}

 
/*--------------------------------------------------------------------------*/
/* cut what is required and paste it after dest */
static void
CutPaste(first_out, first_not_out, dest)  l_list *first_out; l_list *first_not_out;
l_list *dest;
{
  l_list *last_out = first_not_out->backward;
 
  if (first_out==first_not_out) return;

  first_out->backward->forward = first_not_out;      /* cut */
  first_not_out->backward      = first_out->backward;  
 
  last_out->forward   = dest->forward;
  first_out->backward = dest;

  dest->forward->backward = last_out;
  dest->forward           = first_out; 
  
}
/*-------------- Move List -----------------------------------------------*/
t_LL
MoveListFirstLL(dest, src)  t_LL  dest; t_LL src;
{
  CutPaste(elm2link(FirstElmLL(src)),list2link(src),list2link(dest));
  return dest;
}
t_LL
MoveListLastLL(dest, src)  t_LL  dest; t_LL src;
{
#ifdef LLDEBUG 
	  assert(src); 
	  assert(dest); 
#endif
  CutPaste(elm2link(FirstElmLL(src)),list2link(src),elm2link(LastElmLL(dest)));
  return dest;
}
void *
MoveListAftLL(el, src)  void *el;  t_LL src;
{
  CutPaste(elm2link(FirstElmLL(src)),list2link(src),elm2link(el));
  return el;
}
void *
MoveListBefLL(el, src)  void *el;  t_LL src;
{
  CutPaste(elm2link(FirstElmLL(src)),list2link(src),elm2link(PrevElmLL(el)));
  return el;
}
/*-------------- Move Head -----------------------------------------------*/
t_LL
MoveHeadFirstLL(dest, src, head)  t_LL  dest; t_LL src; void *head;
{
  CutPaste(elm2link(FirstElmLL(src)),elm2link(head),list2link(dest));
  return dest;
}
t_LL
MoveHeadLastLL(dest, src, head)  t_LL  dest; t_LL src; void *head;
{
  CutPaste(elm2link(FirstElmLL(src)),elm2link(head),elm2link(LastElmLL(dest)));
  return dest;
}
void *
MoveHeadAftLL(el, src, head)  void *el;  t_LL src; void *head;
{
  CutPaste(elm2link(FirstElmLL(src)),elm2link(head),elm2link(el));
  return el;
}
void *
MoveHeadBefLL(el, src, head)  void *el;  t_LL src; void *head;
{
  CutPaste(elm2link(FirstElmLL(src)), elm2link(head),elm2link(PrevElmLL(el)));
  return el;
}
/*-------------- Move Tail -----------------------------------------------*/
t_LL
MoveTailFirstLL(dest, src, tail)  t_LL  dest; t_LL src; void *tail;
{
  CutPaste(elm2link(tail),list2link(src),list2link(dest));
  return dest;
}

t_LL
MoveTailLastLL(dest, src, tail)  t_LL  dest; t_LL src; void *tail;
{
  CutPaste(elm2link(tail),list2link(src),elm2link(LastElmLL(dest)));
  return dest;
}

void *
MoveTailAftLL(el, src, tail)  void *el;  t_LL src; void *tail;
{
  CutPaste(elm2link(tail),list2link(src),elm2link(el));
  return el;
}

void *
MoveTailBefLL(el, src, tail)  void *el;  t_LL src; void *tail;
{
  CutPaste(elm2link(tail),list2link(src),elm2link(PrevElmLL(el)));
  return el;
}

/*-------- Create a look up table into a list for random access ------------*/
void *
LookInLL(list)  t_LL list;
{

  void * * array = (void **) malloc ((SizeLL(list)+ 1) * sizeof(void *));
	/* array has one element more then the size of the list  */
	/* so that the first element is array[1]                 */
	/* array[0] is the head of the list*/

  void * el;
  int i = 1;

  array[0]= &(list->id);
  ForeachLL_M (list,el)
     array[i++] = el;

  return array;
}

/*--------------------------------------------------------------------------*/
t_LLsize
IndexElmLL(list, ind_el)  t_LL list; void *ind_el;
{
  void *el; 
  t_LLsize i=1;

  ForeachLL_M (list,el)
   if (el==ind_el) return i;
   else i++;
  
 return -1;
}

/*  Original LL had this */ 
/*static int (*UserCompare) (const void * el1, const void * el2);*/
/* Using this for the NON-ANSI people */
static int (*UserCompare)();

/*  Original LL had this */ 
/*
static int IntCompare(const void *el1, const void*el2)
    { return (*UserCompare) (*(void *const*)el1, *(void *const*)el2);}
*/

static int IntCompare();

/*--------------------------------------------------------------------------*/
/*t_LL SysSortLL(t_LL list,  int (*compar) (const void*, const void*))*/
t_LL
SysSortLL (list, compar)
t_LL list;  int (*compar) ();
{
  int ListSize = SizeLL(list);
  l_list *      head_link;

  void * el;
  void * * array = (void **) malloc (ListSize * sizeof(void *));
  int i = 0;
  
  ForeachLL_M (list,el)
    array[i++] = el;

  UserCompare = compar;
  qsort(array,ListSize,sizeof(void *),IntCompare);

  head_link = list2link(list);
  l_linit(head_link);

  for(i=0; i<ListSize; i++)
    l_lbefore(head_link,elm2link(array[i]));
  
  free(array);
  return list;
}

/* Using this for the NON-ANSI people */

static int
IntCompare(el1, el2)
char *el1;
char *el2;
    { return (*UserCompare) (*(char *)el1, *(char *)el2);}
