/*
 *************************************************************************
 *************************************************************************
    Copyright (C) 1995   Michael J. Oehler

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


  Module Description:
    This file contains externally accessible routines.
    These routines represent "ending operations".    For example,
      Usage()             prints out valid arguments & ends the program.
      Abend()             pops up an error window & ends the program.
      Exit_Program()      ends program normally.
      Release_Cards()     release allocated card data before exiting.
      Child_Check()       checks to see if a child ended correctly.
    These routines attempt to be motif independent as much as possible.

 *************************************************************************
 *************************************************************************
*/

/* include files */
#include <string.h>
#include <stdio.h>
#include <signal.h>
#include <sys/wait.h>
#include "crd.h"



/*
 *************************************************************************
  Name:
     Child_Check

  Purpose:
     Signal Handler for child process termination
     Prevent Zombies!

  Calling Sequence:
     1.  see Print()         in file_callback.c. Establishes the signal handler.
     2.  SIGCLD             signal occurs calling Child_Termination()
     2a. SIGTERM            Although not installed as a signal handler purposely.
            The user may issue a kill command to this child & SIGCLD makes us go there.
     2b. Other signal as of yet undiscovered, may also make control go there.
     3.  Child_Termination() in mcard.c calls XtAppAddTimeOut
     4.  Ending up here. Whew!

  Description:
     1. Check to see if a child really died.
     2. Determine 
     2. Hide pop up.
     3. Check for normal termination. Make sure print job occurred

 *************************************************************************
*/
extern void Child_Check()
{

int status, return_code;


/*
 Test to see if there was a fork problem
 Really don't want to continue if there wasn't a child.
 the ifdef only makes the debugging tools happy.
*/
#ifdef DEBUG
int child_pid;

if ( ( child_pid = wait(&status) ) == -1 )
   return;
#else
if ( (  wait(&status) ) == -1 )
   return;
#endif

/*
 The lower byte contains the signal that called the handler.
 The upper byte contains the return code. ie, child called exit.
 if status = 133 or 0x85, SIGTRAP & core, means you are using a debugger!
 */
        return_code = (status & ~0xFF) >> 8;
        status &= 0xFF;

#ifdef DEBUG
	printf("Child %d terminated.\tReturn code %d\tSignal: %d\n.",child_pid,return_code,status);
#endif

/* If non-null status, there was a problem. */
	switch (status) {
        /* Print Job was submitted */
	case 0: {
          if (!return_code)
	    Create_Info_Prompt("Print_Info");
	  else
	    Create_Error_Prompt("Print_Error");
	  break;
	}

	/* SIGTERM occurs when the job is terminated by kill or print cancel button */
        case SIGTERM: {
          Create_Error_Prompt("Print_Error1");
	  break;
	}

        /* Other signals indicated a problem */
	case SIGINT:                   /* not sure how a ^C would be given to lpr */
	case SIGKILL: {
          Create_Error_Prompt("Print_Error2");
	  break;
	}
	} /* switch */

} /* End Child_Termination */



/*
 *************************************************************************
  Name:
     Exit_Program
  Purpose:
     Releases the data structures & end the program.
     Ideally, this should always be the final routine called.

  Calling Sequence:
     The user selected the file menu & then the "Exit" option. This calls:
     File_Menu_Callback() file_callback.c. A switch statement sends us here.
                               OR
     SIGINT, SIGKILL, SIGTERM, SIGQUIT signal were issued which calls Sigint_Handler()
     see mcard.c. Sigint_Handler() then directs the X Server here.

  Description:
     1. Release all cards.
     2. Free any remaining malloc'ed memory.
     3. Close X connection.
     4. Exit

 *************************************************************************
*/
extern void Exit_Program() 
{
#ifdef DEBUG
/* Some day, include a "nice exit op" ask the user to save */
        printf("Hello Mike, I made it!\n");
#endif

/* This free's the XtMalloc'ed memory for the "circular user data linked list" */
      Release_Cards();

/* ReleaseCards() lease one user card available (works with NEW command), free it */
      XtFree(head_ptr->card_data->address);
      XtFree(head_ptr->card_data->index);

/* Aftering releasing all these, a few things still remain */
      XtFree(head_ptr->file_name);
      XtFree((char *) head_ptr->card_data);
      XtFree(head_ptr->search_for);
      XtFree((char *) head_ptr);

/* Close our X server connection & end */
      XtCloseDisplay(XtDisplay(head_ptr->parent_form));
      exit(0);
} /* end Exit_Program */


/* This routine is called when somethin goes wrong */
extern void Abend(widget,i,string)
Widget widget;
int i;
char *string;
{
   XCloseDisplay(XtDisplay(widget));
   exit(i);
}

/*
 *************************************************************************
 *************************************************************************
 Name:
  Release_Cards()

 Calling Sequence:
    This routine is a "helper" routine performing a necessary function for
    many others. But not from the graphical user interface. Specifically,

  Signal_Handler.       Catches ^C interrupts.        See main() in mcard.c
  File_Menu_Callback()  Exit button.           See switch in file_callback.c
  File_Menu_Callback()            New button.  See switch in file_callback.c
  Extract_File_Name_Callback()    Open button.                    

 Purpose:
  This routine releases the user's data in the circular linked list.
  This routine does not release the VISIBLE data in the "card widget's"
  text & address fields! This is a function of the Redraw_List() procedure.
  This procedure does not rely on any Motif routines, just the X toolkit.

 Description:
     1. Clear out the (user) data in the primary card.
     2. Allocate 1 byte for the index & address fields.
     3. Place a NULL string in those fields.
     4. Free all other index & data strings.
     5. Free the data structures that pointer to these strings.
     6. Unmap any visible "card widgets"

 *************************************************************************
 *************************************************************************
*/
extern void Release_Cards()
{
Card_Content_Ptr tmp_ptr;
Card_Ptr user_data;
int i;

/* Free the data held in the primary card, but don't actually free the first card */
user_data = head_ptr->card_data;

/* Clean out the data stored in the primary card */
   XtFree(user_data->index);
   XtFree(user_data->address);

/* init rest of the structure */
     user_data->address = XtMalloc(1);
     user_data->address[0] = '\0';
     user_data->index = XtMalloc(1);
     user_data->index[0] = '\0';

user_data = user_data->next;
for ( i = head_ptr->cards; i > 1; i--) {
  /* Clean out the data stored in this card */
  XtFree(user_data->index);
  XtFree(user_data->address);
  user_data = user_data->next;
  
  /* Release the structre */
  XtFree((char *) user_data->prior);
}

/* Re-establish the number of cards in the list & clean up the circular linked list */
head_ptr->cards = 1;
head_ptr->primary_card     = head_ptr->card_data;
head_ptr->card_data->next  = head_ptr->card_data;
head_ptr->card_data->prior = head_ptr->card_data;

/* Unmap the "card widgets" */
if (head_ptr->card_view) {
    tmp_ptr = head_ptr->viewable_cards->next;
    for ( i = head_ptr->visible_spots; i > 1; i--) {
	XtUnmapWidget(tmp_ptr->form);
	(head_ptr->available_spots)++;
	tmp_ptr = tmp_ptr->next;
	head_ptr->visible_spots = 1;
    }
}
else {
	XmListDeleteAllItems(head_ptr->scroll);
	XmListSelectPos(head_ptr->scroll,1,False);
}

} /* Release_Cards */



/* **************************************************************************** */
/* *** Print the man page like instructions *********************************** */
/* **************************************************************************** */
static void Usage()
{
printf("NAME\n");
printf("\tmcard\n");

printf("SYNOPSIS\n");
printf("\tmcard [-in <input_file>][-v][-h][-bg | -background][-fg][-fn][-display]\n");

printf("DESCRIPTION\n");

printf("OPTIONS\n");
printf("This application understands all standard X Toolkit command-line options.\n");
printf("[-bg | -background]\n");
printf("[-bd \n");
printf("[-bw | borderwidth]\n");
printf("[-bordercolor]\n");
printf("[-display]\n");
printf("[-fg | -foreground]\n");
printf("[-fn | -font]\n");
printf("[-iconic]\n");
printf("[-name]\n");
printf("[+/-rv | -reverse]\n");
printf("[-title]\n");
printf("[-xrm]\n");

printf("LIMITATIONS\n");
printf("\n");

printf("EXAMPLE\n");
printf("\t\thuffman -i preamble -o my_output -v\n");
printf("\n");

} /* end Usage */


/*
 *************************************************************************
 *************************************************************************
 Name
   Parse_Arguments

 Purpose
   This routine parses the command line arguments specific to mcard.
   Each argument must be proceed by an "-".

If an option to an argument is not given, the program will exit after
printing some help.  This routine does not check whether they are valid
or that usuable options are supplied. 

More appropriate routines must check to see if the option is usuable.
**************************************************************************
**************************************************************************
*/
extern char *  Parse_Arguments(argc_ptr, argv)
int *argc_ptr;
char *argv[];
{
int i;
char * file_name= (char *) NULL;
int argc=*argc_ptr;

/* Parse the argv & argc arguments */
if (argc >= 2)
{
    for (i=1; i<argc; i++)
    {
	/* -h print man page */
	if ((strcmp(argv[i],"-h")==0) || (strcmp(argv[i],"-help")==0))
	{
	    (*argc_ptr)--;
	    Usage();
	}
	
	/* -v turn verbose on */
	if ((strcmp(argv[i],"-v")==0) || (strcmp(argv[i],"-verbose")==0)) {
	    (*argc_ptr)--;
	}
	
	/* -in Input File card file, just test to see if it exists. */
	if ((strcmp(argv[i],"-file")==0)|| (strcmp(argv[i],"-in")==0))
	{
	    *argc_ptr-=2;
	    i++;
	    file_name = argv[i];
	    if (i >= argc || argv[i][0] == '-')
	    {
		file_name = NULL;
		Usage();
	    }
	}
	
    } /* for */
} /* if */

return(file_name);
} /* end Parse_Arguments */
