/*
  ************************************************************************
  ************************************************************************
    Copyright (C) 1995   Michael J. Oehler

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  ************************************************************************
  ************************************************************************
*/

#include <stdio.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>


#include <Xm/MessageB.h>
#include <Xm/FileSB.h>
#include "crd.h"
#include "ps_program.h"

#define DIR_CHAR	"/"

#ifdef __STDC__
extern void	Release_Cards( void );
static void	Create_File_Dialog(int);
extern void	Delete_Card( void );
static void	Extract_File_Name_Callback(Widget, XtPointer, XmFileSelectionBoxCallbackStruct *);
static void	Create_Print_File(char *, char *, int, Card_Ptr);
static void	Print(char *, char *);
extern void	Child_Termination( void );
extern void Open_File(char *, int);

#else
extern void	Exit_Program();
extern void	Release_Cards();
static void	Create_File_Dialog();
static void	Extract_File_Name_Callback();
extern void	Child_Termination();
extern void	Delete_Card();
static void	Print();
static void	Create_Print_File();
extern int	Get_CRD_File();            /* located in file-io.c */
extern void Open_File();
#endif


/*
   These defines are only used to pass client_data to the callback helper
   routines. They are used when similar, but not exactly the same, functionality
   is performed by one of the callbacks. For instance, Duplicating and Adding a
   card, and Open and Merging a file, have similar functionality. Instead of
   redundant code, these defines introduce a conditional to control the minut
   differences.
*/
#define OPEN 0
#define MERGE 1


/*
  ************************************************************************
  ************************************************************************
   Name:
     File_Menu_Callback

   Calling Sequence:
     This routine is called only from the  "File" button in the main menu.
     This callback is established during the creation of the sub menu.
     See init.c "file_sub_menu = XmVaCreateSimplePulldownMenu(menu_bar,

    Description:
      The routine is passed in an integer, in item_no.
      It represents which button in the sub menu initiated this callback.
      A Switch statement then controls which function will be performed.

  ************************************************************************
  ************************************************************************
*/
extern void File_Menu_Callback(widget, item_no, call_data)
Widget widget;
int item_no;
XmAnyCallbackStruct *call_data;
{
    switch ( item_no ) {

    /* "New" command */
    case 0: {
	Release_Cards();
	Redraw_List();
	break;
    }

    /* "Open" command */
    case 1: {
	Create_File_Dialog(OPEN);
	break;
    }

    /* "Save" command */
    case 2: {
	Grab_Primary_Card_Data(widget);
	Put_CRD_File("mcard.crd");
	break;
    }


    /* "Save As" command */
    case 3: {
	Grab_Primary_Card_Data(widget);
	Create_Error_Prompt("Rabbit");
	break;
    }

    /* "Print" one card command */
    case 4: {
	Grab_Primary_Card_Data(widget);
	Create_Print_File(NULL,NULL,1, head_ptr->primary_card);
	Print("mcard.ps","-Pps");
	break;
    }

    /* "Print All" command */
    case 5: {
	Grab_Primary_Card_Data(widget);
	Create_Print_File(NULL,NULL,head_ptr->cards, head_ptr->card_data);
	Print("mcard.ps","-Pps");
	break;
    }
    
    /* "Merge" command */
    case 6: {
	Grab_Primary_Card_Data(widget);
	Create_File_Dialog(MERGE);
	break;
    }

    /* "Exit" command */
    case 7: {
     Exit_Program();
    }
}    
} /* end File_Menu_Callback */


/*
  ************************************************************************
  ************************************************************************
  Name:
     Extract_File_Name

  Calling Sequence
     1. User selected Open, Merge, or Save As, from the main menu
     2. File_Menu_Callback called Create_File_Dialog
     3. Once the user OKs on a file name
     4. XtAddCallback calls Extract_File_Name_Callback

   Description:
     1. Get the file name
     2. Clear out all cards.
     3. Destroy the file dialog
     4. Go get the cards from the disk. Use the file name
     5. Release the first card. There was no data in it.
  ************************************************************************
  ************************************************************************
*/
static void Extract_File_Name_Callback(fs, client_data, cbs)
Widget fs; /* file selection dialog box */
XtPointer client_data;
XmFileSelectionBoxCallbackStruct *cbs;
{
char *file_name;

/*
  Go convert the XmString selected by the user to a character string
  This string must be XtFree()'ed.
  The Check is to see if internal Motif error. This will never occur ;-)
*/
     if (!XmStringGetLtoR(cbs->value,XmSTRING_DEFAULT_CHARSET,&file_name))
     {
	 Create_Error_Prompt("File_Error1");
	 return; 
     }

/* Dismissed the file dialog */
     XtDestroyWidget(fs);

/*
   Go Clean Out all the cards entered. Only one card will be left.
   This lone card will have no text in it & becomes the starting
   point as new cards are read in from disk.
   Release_Cards() is called here  and not in Open_File since
   a command line argument may also call Open_File.
*/
     if ( (int) client_data == OPEN)
	Release_Cards();

/*
 The file name is kept in the master data structure in case of a "save"
 operation. This memory will be free upon exit. Free the old file name!
 & Set to the new one. We do this for "saving" purposes
*/
    XtFree(head_ptr->file_name);
    head_ptr->file_name = file_name;
    Open_File(file_name, (int) client_data);

} /*  end Extract_File_Name_Callback */

/*
  ************************************************************************
  ************************************************************************
  Name:
     Open_File

  Calling Sequence
     1. From Extract_File_Name_Callback

   Description:

  ************************************************************************
  ************************************************************************
*/
extern void Open_File(file_name, client_data)
char * file_name;
int client_data;
{
    /* Go do the motif independent stuff */
    if ( !Get_CRD_File(file_name) ) {
      Create_Error_Prompt("File_Error");
      return;
    }

/*
  This removes the first card which wasn't filled in.
  Delete_Card will update all pointers in head_ptr.
  We set the primary_card below because Get_CRD_File leaves it at the end.
  It seems good to start the display from the front of the list
*/
    head_ptr->primary_card = head_ptr->card_data;

/* Leave the cards alone if merging, Otherwise remove the blank card */
     if ( client_data == OPEN)
	Delete_Card();

/*
  Map up the "card widgets"
  Note, we take advantage of the "No governing" option to Redraw_List
 */
   Any_To_Map(head_ptr->cards);
   Redraw_List();

}  /* end Open_File */


/*
  ************************************************************************
  ************************************************************************
  Name:
         Create_File_Dialog

  Calling Sequence:

  Description:

  ************************************************************************
  ************************************************************************
*/
static void Create_File_Dialog(client_data)
int client_data;
{
Widget dialog;

    dialog = XmCreateFileSelectionDialog( head_ptr->parent_form,
	"File", NULL, 0);
    XtAddCallback(dialog, XmNokCallback, 
	(XtCallbackProc) Extract_File_Name_Callback, (XtPointer) client_data);
    XtAddCallback(dialog, XmNcancelCallback, 
	(XtCallbackProc) XtDestroyWidget, NULL);

   XtUnmanageChild(XmFileSelectionBoxGetChild(dialog,XmDIALOG_HELP_BUTTON));
   XtManageChild(dialog);

} /* end Create_File_Dialog */



/*
  ************************************************************************
  ************************************************************************
  Name:
    Create_Print_File()

  Purpose:
    Generates postscript code to nicely print the user data.

  Calling Sequence:
    1.  User pulls down the file menu
    2.  Selects "Print" or "Print All"
    3.  Switch in File_Menu_Callback() sends control here.

  Description    
    1. 

  ************************************************************************
  ************************************************************************
*/
static void Create_Print_File(file,directory,cards, ptr)
char *file;
char *directory;
int cards;
Card_Ptr ptr;
{
/* The conversion routine code goes here */
	char *fname;
	int x;
        FILE *fp;

/*  Go allocate some space to store the path name etc.  */
        if ( !( fname = (char *) XtMalloc( MAXPATHLEN) ) ) {
	  Create_Error_Prompt("File_Error");
	  return;
	}

/* If no file name or directory path was specified, supply a default */
     if (!directory) {
       if  ( !getwd(fname)  ) {
	 Create_Error_Prompt("File_Error2");
	 return;
       }
     }
     else
       strcpy(fname,directory);

/* Choose a default file name */
     if (!file) {
       strcat(fname,DIR_CHAR);
       strcat(fname,"mcard.ps");
     }
     else {
       strcat(fname,DIR_CHAR);
       strcat(fname,file);
	}

/* Open the file */
        fp = fopen(fname, "w");
        if ( !fp ) {
	  Create_Error_Prompt("File_Error");
	  XtFree(fname);
	  return ;
        }
XtFree(fname);

/* The postscript program, mostly comes from the ps_program.h as a data file */
x = strlen(postscript_prog);
fwrite(postscript_prog, 1, x, fp);

/* Insert the user data */
{
int i,page;
char *index, *address;
char *from, *to;

/* Create shome temporary storage to hold user card data */
     index   = (char *) XtMalloc( sizeof( char ) *  CARD_CHAR_WIDTH );
     address = (char *) XtMalloc( sizeof( char ) *  CARD_CHAR_WIDTH * CARD_CHAR_ROW );

/* Insert every user index & address into the postscript program  */
     for (i=cards; i>0 ;  i-=3) {
       fwrite("new_page\n",1,9,fp);

       /* Only three cards per page */
       for (page=0; i-page && page<3; page++) {

	 /* Insert index postscript commands */
	 fwrite("coordinates\nbox\ngoto_index\n",1,27,fp);

	 /* Insert actual index */
	 index[0]='(';
	 strcpy(index + 1,ptr->index);
	 strcat(index,") show\ngoto_address\n");
	 x=strlen(index);
	 fwrite(index,1,x,fp);

	 /* Insert address here, NULL addresses are skipped. */
	 from = ptr->address;
	 for ( to=ptr->address; to < ptr->address + strlen(ptr->address);  from = ++to )
	   {
	     /* begin postscript text "(" */
	     address[0]='('; 	     address[1]='\0';

	     /* Look for any Newline lines, if none, just do this one line */
	     to = strstr(from, "\n");
	     if (!to) to = from + strlen(ptr->address) -1;

	     /* Copy in the address to the postscript format */
	     x = (int) (to - from);   /* figure out how many characters between last CR */
	     strncpy(address + 1, from, x);
	     address[x+1]='\0';
	     
	     /* this will end out postscript text ")" */
	     strcat(address,") show\nnewline\n");
	     fwrite(address,1,strlen(address),fp);
	   }

	 /* Next card */
	 ptr = ptr->next;
       }
      fwrite("showpage\n",1,9,fp);
     }



XtFree(index);
XtFree(address);
} /* end local indentation */

fclose(fp);

} /* end Create_Print_File */

/*
  ************************************************************************
  ************************************************************************
  Name:
    Print()

  Calling Sequence:
    1. User selects "Print" or "Print ALL" from the "File" Menu
    2. Switch in File_Menu_Callback directs to Create_Print_File
    3. Then to Here.

  Description:
   Submits a print job by forking().
    0. Only permitt one child. ?
    1. Want notification of child termination. SIGCHLD
    2. Set up postscript program for specified card data (one card or all)
    2. Fork a child process.
    3. Exec's the "line printer request" lpr.
    4. Child lpr process is passed postscript program which gets printed.
    5. Create a cancel menu. ? not implemented
    
   WARNING, just because a print job gets submitted, does not mean it may
   print! Access, Out of Paper, Non existent print name... can stop print
  ************************************************************************
  ************************************************************************
*/
static void Print(file, printer)
char  *file;
char  *printer;
{
int x;
char *cmd="lpr";

/*
  Want notification of child termination
  Note, we only establish this once. The first time a printing operation is called.
 */
     if (signal(SIGCLD, SIG_IGN) != SIG_IGN)
       signal(SIGCLD,( void (*)() ) Child_Termination);

if ( ( x=fork() ) == -1 )
  {
    /* Fork fails */
    Create_Error_Prompt("Print_Error3");
    return;
  }
else /* Fork success */
  if ( x == 0)
    {
      /* In Child process.  exec "lpr" command */
      execlp(cmd,cmd,printer,file, (char *) 0);

      /*
       IF YOU GET HERE,  execvp just returned -1 & errno is set.
      !!!! You are running in the child process only, not the real program. !!!!!
      It does not exit the real program! Understand.
      This exit goes to Child_Termination() see mcard.c via SIGCLD
      */
      exit(errno);
    }
  else                  /* Parent process. */
    {
	/*
	 The fork should be transparent to the user.
	 Nothing is needed beyond the return.
	 */
      return;
    }
Exit_Program(); /* Never executed */
}
