/*****************************************************************************

	config.c

	Environment:    Unix R40V3/Solaris2/Linux.

	Revision history:	@(#)config.c	1.9     97/06/23


	DESCRIPTION: Part of the Mdb Application.
			Launch the Config Dialog.
			Reconfigure Layout, folders, etc.

	COPYRIGHT NOTICE:
        Permission to use,  copy,  modify,  and  distribute  this
        software  and  its    documentation   is  hereby  granted
        without fee, provided that  the  above  copyright  notice
        appear  in all copies and that both that copyright notice
        and  this  permission   notice   appear   in   supporting
        documentation.  The   author  makes  no   representations
        about   the   suitability   of   this  software  for  any
        purpose.  It  is  provided  "as  is"  without  express or
        implied warranty.

        THE AUTHOR DISCLAIMS ALL WARRANTIES WITH REGARD  TO  THIS
        SOFTWARE,    INCLUDING    ALL   IMPLIED   WARRANTIES   OF
        MERCHANTABILITY AND FITNESS, IN NO EVENT SHALL THE AUTHOR
        BE  LIABLE  FOR  ANY  SPECIAL,  INDIRECT OR CONSEQUENTIAL
        DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS  OF
        USE, DATA OR PROFITS, WHETHER IN AN ACTION  OF  CONTRACT,
        NEGLIGENCE  OR  OTHER  TORTIOUS   ACTION,   ARISING   OUT
        OF   OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS
        SOFTWARE.

******************************************************************************/
/******************************************************************************/
#pragma ident "@(#)config.c      1.9	97/06/23"

#define CONFIG
#include "mdb.h"

/*
 * Externals.
 */
extern Boolean	Debug;

/*
 * Local Variables.
 */
static Widget	PopUp;
static Widget	ItemLabel;
static Widget	DesText;
static Widget	ParText;
static Widget	ClsText;
static Widget	AddButton;
static Widget	DelButton;
static Widget	ArrowButtons;
static Widget	ArgText;
static Widget	ArgWin;
static Widget	HlpWin;
static Widget	HlpText;
static Widget	Isens[20];

static struct	item *Db;
static struct	item DbPk;
static struct	item DbSk;
static struct	fold *Fl;
static Boolean	DelAdd;
static Boolean	InUse = False;
static int	Lasti;
static int	Changed;
static char	Ttl3Lbl[40];
static char	Ttl4Lbl[40];
static char	Ttl2Lbl[40];
static char	Testfile[PATH_MAX];


/*
 * Forward declarations.
 */
static int
config1_cb(Widget w, int client_data, XmAnyCallbackStruct *cbs );
static int res_edit( Widget w, int cmd );

/*
 * The Main Action Area:
 *      LabelString:            Callback:       cbValue:        Sens:   Help:
 */
static ActionAreaItem action_items[] = {
	{ "okButton",		(XtCP)config1_cb,	OK,	0,	HCQ },
	{ "applyButton",	(XtCP)config1_cb,	APPL,	0,	HAP },
	{ "clearButton",	(XtCP)config1_cb,	CLEAR,	0,	QUICK },
	{ "addButton",		(XtCP)config1_cb,	ADDI,	0,	QUICK },
	{ "deleteButton",	(XtCP)config1_cb,	DELI,	0,	QUICK },
	{ "testButton",		(XtCP)config1_cb,	TEST,	0,	HTE },
	{ "cancelButton",	(XtCP)config1_cb,	QUIT,	0,	HCA },
	{ "helpButton",		(XtCP)config1_cb,	HELP,	0,	QUICK },
};


/*
 * Return temporary
 * test resource file.
 */
char *ConfigTestFile(void)
{
	return(  Testfile );
}


/*
 * Update all items in the db-file.
 */
/*ARGSUSED*/
static int db_update( FILE *fdo, FILE *fdn )
{

	int i, itn;
	char dbuffo[200];
	char dbuffn[sizeof(dbuffo)];

	while ( fgets( dbuffo, sizeof(dbuffo), fdo ) != NULL ) {

		if ( *dbuffo == '\t' ) {

			/*
			 * Get curent item #.
			 */
			if ( (itn = (int)atoi(&dbuffo[1])) == DELETED ) {
				(void)fputs( dbuffo, fdn );
				continue;
			}

			/*
			 * Look up new item #, if
			 * not found - deleted.
			 */
			for( i = 0; i < Lasti+1; i++ ) {
				if ( itn == Db[i].pos ) {
					itn = i;
					break;
				}
			}

			if ( i == Lasti+1 )
				itn = DELETED;

			(void)sprintf( dbuffn, "\t%03d\t", itn );
			(void)strcat( dbuffn, &dbuffo[strlen(dbuffn)] );
			(void)fputs( dbuffn, fdn );
		} else
			(void)fputs( dbuffo, fdn );

		if ( ferror( fdn ) )
			return( errno );
	}

	return(0);
}


/*
 * Return True if opened folder
 * match our current folder class.
 */
/*ARGSUSED*/
static Boolean check_class( char *fname )
{
	int cnt = 10;
	char dbclass[80];
	char dbuff[200];
	FILE *fd;

	(void)sprintf( dbclass, DBCLASS, GetFolderClass() );

	if ( (fd = fopen( fname, "r" )) == NULL )
		return(False);

	while ( fgets( dbuff, sizeof(dbuff), fd ) != NULL ) {

		if ( ! --cnt )
			break;

		if ( ! strncmp( dbclass, dbuff, strlen(dbclass) ) )
			break;
	}

	(void)fclose(fd);

	if ( ! cnt )
		return(False);

	return(True);
}


/*
 * Rebuild all folders
 * known by this session.
 */
static void db_rebuild(void)
{
	int errn;
	int tm = 5;
	int fld = 0;
	struct flock fl;
	struct stat sbuf;
	char fnameo[PATH_MAX];
	char fnamen[PATH_MAX];
	FILE *fdo, *fdn;

	if ( ! DelAdd ) {
		/*
		 * Nothing added or deleted.
		 */
		return;
	}


	(void)xpmsg( PopUp, "info: Rebuilding associated folders .." );
	Wait(2);

	while ( GetFolder(fld) != NULL ) {

		if ( !strncmp( GetFolder(fld), PRIVATE, sizeof(PRIVATE)-1 ) ) {
			/*
			 * Private folders from $HOME.
			 */
			(void)sprintf( fnameo, "%s%s/%s%s",
				_XmOSGetHomeDirName(),
				GetPrivateFolderPath(),
				GetFolder(fld), MDBEXT );
		} else {
			(void)sprintf( fnameo, "%s/%s%s",
				GetFolderPath(), GetFolder(fld), MDBEXT );
		}

		fld++;
				
		if (  stat( fnameo, &sbuf ) < 0 )
			continue;

		/*
		 * Must match this layout.
		 */
		if ( check_class( fnameo ) == False )
			continue;

		if ( ! (sbuf.st_mode & S_IWUSR) ) {
			errno = EACCES;
			(void)xpmsg( PopUp, "error: Can't back-up %s", fnameo );
			Wait(2);
			continue;
		}

		(void)sprintf( fnamen, "%s~", fnameo );

		if ( rename( fnameo, fnamen ) ) {
			(void)xpmsg( PopUp, "error: Can't back-up %s", fnameo );
			Wait(2);
			continue;
		}

		if ( (fdo = fopen( fnamen, "r" )) == NULL ) {
			(void)xpmsg( PopUp, "error: Can't open %s", fnamen );
			Wait(2);
			(void)rename( fnamen, fnameo );
			continue;
		}

		if ( (fdn = fopen( fnameo, "w" )) == NULL ) {
			(void)xpmsg( PopUp, "error: Can't create %s", fnameo );
			Wait(2);
			(void)fclose(fdo);
			(void)rename( fnamen, fnameo );
			continue;
		}

		/*
		 * Inherit mode.
		 */
		(void)fchmod( fileno(fdn), sbuf.st_mode );

		/*
		 * Lock file while rebuilding.
		 */
		bzero( &fl, sizeof(struct flock) );
		fl.l_type = F_WRLCK;

		while( fcntl( fileno(fdn), F_SETLK, &fl ) && --tm )
			Wait(1);

		if ( ! tm ) {
			(void)xpmsg( PopUp, "error: Can't access %s", fnameo );
			Wait(2);
			(void)fclose(fdo);
			(void)fclose(fdn);
			(void)rename( fnamen, fnameo );
			continue;
		}

		errn = db_update( fdo, fdn );

		bzero( &fl, sizeof( struct flock ) );
		fl.l_type = F_UNLCK;

		(void)fcntl( fileno(fdn), F_SETLKW, &fl );

		(void)fclose(fdo);
		(void)fclose(fdn);

		if ( errn ) {
			errno = errn;
			(void)xpmsg( PopUp, "error: creating %s", fnameo );
			Wait(2);
			(void)rename( fnamen, fnameo );
		}
	}
}


/*
 * Install changes into
 * the configuration file,
 * and then go on to update
 * all folders to match the
 * new Layout.
 */
/*ARGSUSED*/
static int fcopy( char *ifname, char *ofname )
{
	struct stat sbuf;
	size_t cnt;
	int errn;
	char dbuff[1024];
	char baname[PATH_MAX];
	FILE *ifd, *ofd;

	if ( Changed != CHANGED )
		return(0);


	if ( (ifd = fopen( ifname, "r" )) == NULL ) {
		(void)xpmsg( PopUp, "error: saving configuration %s", ifname );
		Wait(2);
		return(-1);
	}

	if (  stat( ofname, &sbuf ) < 0 ) {
		(void)xpmsg( PopUp, "error: saving configuration %s", ofname );
		Wait(2);
		return(-1);
	}


	if ( ! (sbuf.st_mode & S_IWUSR) ) {
		errno = EACCES;
		(void)xpmsg( PopUp, "error: saving configuration %s", ofname );
		Wait(2);
		return(-1);
	}

	(void)sprintf( baname, "%s~", ofname );

	if ( rename( ofname, baname ) ) {
		(void)xpmsg( PopUp, "error: saving configuration %s", ofname );
		Wait(2);
		return(-1);
	}

	if ( (ofd = fopen( ofname, "w" )) == NULL ) {
		(void)fclose(ifd);
		(void)xpmsg( PopUp, "error: saving configuration %s", ofname );
		Wait(2);
		(void)rename( baname, ofname );
		Changed = 0;
		return(-1);
	}

	/*
	 * Inherit mode.
	 */
	(void)fchmod( fileno(ofd), sbuf.st_mode );

	XtSetSensitive( PopUp, False );

	while ( (cnt = fread( dbuff, 1, sizeof(dbuff) , ifd )) ) {
		if ( fwrite( dbuff, 1, cnt , ofd ) != cnt ) 
			break;
	}

	errn = errno;

	(void)fclose(ifd);
	(void)fclose(ofd);

	XtSetSensitive( PopUp, True );

	if ( cnt ) {
		errno = errn;
		(void)xpmsg( PopUp, "error: saving configuration %s", ofname );
		Wait(2);
		(void)rename( baname, ofname );
		Changed = 0;
		return(-1);
	} else 
		db_rebuild();


	return(0);
}


/*
 * Check existent of LAYOUT.
 * Create if it does not exist.
 */
/*ARGSUSED*/
static void check_ly( char *clname, char *lname )
{
	struct stat sbuf;
	Boolean create = True;
	int fno = 1;
	char ly[80];
	char ofname[PATH_MAX];
	char ofname1[PATH_MAX];
	char ifname1[PATH_MAX];
	char fclass[40];
	char fname[40];
	char dbuff[200];
	char *ptr, *ifname;
	FILE *ifd, *ofd;

	(void)strcpy( ly, lname );

	ptr = strchr( ly, '(' );
	*strrchr( ptr, ')' ) = '\0';
	ptr++;

	if ( (ifname = TouchLayout( ptr, &create )) == NULL ) {
		/*
		 * Should not happen.
		 */
		return;
	}

	if ( create == False )
		return;

	if (  stat( clname, &sbuf ) < 0 )
		return;

	/*
	 * Start with the db resource file.
	 */
	if ( (ifd = fopen( clname, "r" )) == NULL )
		return;


	(void)sprintf( ofname, "%s/%s", ifname, ptr );

	if ( (ofd = fopen( ofname, "w" )) == NULL ) {
		(void)xpmsg( PopUp, "error: can't create Layout %s", ofname );
		Wait(2);
		(void)fclose(ifd);
		return;
	}

	/*
	 * Inherit mode.
	 */
	(void)fchmod( fileno(ofd), sbuf.st_mode );

	(void)strcpy( fclass, FOLDERCLASS );
	*strchr( fclass, ':' ) = '\0';
	(void)strcpy( fname, FOLDERNX );
	*strchr( fname, '%' ) = '\0';

	while( fgets( dbuff, sizeof(dbuff), ifd ) != NULL ) {

		if ( ! strncmp( dbuff, fclass, strlen(fclass) ) )
			(void)sprintf( dbuff, FOLDERCLASS, FUNDEF );

		if ( ! strncmp( dbuff, fname, strlen(fname) ) ) {
			if ( ! fno )
				continue;
			(void)sprintf( dbuff, FOLDERNX, fno, FUNDEF );
			fno = 0;
		}
		(void)fputs( dbuff, ofd );
	}

	if ( ferror( ofd ) ) {
		(void)xpmsg( PopUp, "error: Cant't create Layout %s", ofname );
		Wait(2);
		(void)fclose(ifd);
		(void)fclose(ofd);
		(void)unlink(ofname);
		return;
	}

	(void)fclose(ifd);
	(void)fclose(ofd);

	/*
	 * Continue with MDBGEN (if it not exist).
	 */
	(void)strcpy( ofname1, ofname );
	(void)strcpy( ifname1, ifname );
	(void)sprintf( ofname, "%s/%s", ifname, MDBGEN );

	if (  stat( ofname, &sbuf ) < 0 ) {

		create = False;
		if ( (ifname = TouchLayout( MDBGEN, &create )) == NULL ) {
			/*
			 * Should not happen.
			 */
			return;
		}

		if (  stat( ifname, &sbuf ) < 0 )
			return;

		if ( (ifd = fopen( ifname, "r" )) == NULL )
			return;

		if ( (ofd = fopen( ofname, "w" )) == NULL ) {
			(void)xpmsg( PopUp,
				     "error: can't create Layout %s", ofname );
			Wait(2);
			(void)fclose(ifd);
			return;
		}

		/*
		 * Inherit mode.
		 */
		(void)fchmod( fileno(ofd), sbuf.st_mode );

		while( fgets( dbuff, sizeof(dbuff), ifd ) != NULL )
			(void)fputs( dbuff, ofd );

		if ( ferror( ofd ) ) {
			(void)xpmsg( PopUp,
				     "error: Cant't create Layout %s", ofname );
			Wait(2);
			(void)fclose(ifd);
			(void)fclose(ofd);
			(void)unlink(ofname);
			(void)unlink(ofname1);
			return;
		}
		(void)fclose(ifd);
		(void)fclose(ofd);
	}

	(void)xpmsg( PopUp,
		"info: New Layout \"%s\" installed in %s", ptr, ifname1 );
		Wait(2);
}


/*
 * Read all configuration data
 * into memory. Save into temp
 * file when Save or Test is
 * in progress.
 */
/*ARGSUSED*/
static int res_edit( Widget w, int cmd )
{
	int i, itn, fln;
	int rval;
	int pkey = 0;
	int skey = 0;
	Boolean items = False;
	Boolean create = False;
	static char ifname[PATH_MAX];
	char dbuff[PATH_MAX*3];
	char *ptr, *val;
	FILE *ifd, *ofd;
	static FILE *ifdorg;
	static long enditems;

	if ( (cmd == START) || (cmd == RESTART) ) {

		if ( cmd == RESTART )
			(void)fclose(ifdorg);

		if ( (ptr = TouchLayout( GetAppClass(), &create )) == NULL ) {
			(void)xpmsg( PopUp,
				     "error: getting configuration file" );
			return(-1);
		}

		(void)strcpy( ifname, ptr );

		(void)sprintf( Testfile, "%s.config", ifname );

		/*
		 * Build the database into memory.
		 */

		if ( (ifdorg = fopen( ifname, "r" )) == NULL ) {
			(void)xpmsg( PopUp, "error: %s", ifname );
			return(-1);
		}

		if ( cmd == START ) {

			/*
			 * Items.
			 */
			Db = (struct item *)
				XtMalloc( MAXITEMS*sizeof(struct item) );
			/*
			 * Folders.
			 */
			Fl = (struct fold *)
				XtMalloc( MAXFOLDS*sizeof(struct fold) );
		}


		bzero( (char *) Db, MAXITEMS*sizeof(struct item) ); 
		bzero( (char *) Fl, MAXFOLDS*sizeof(struct fold) ); 

		for ( i = 0; i < MAXITEMS; i++ )
			Db[i].pos = DELETED;

		*Ttl2Lbl = *Ttl3Lbl = *Ttl4Lbl = '\0';
		fln = 1;
		itn = 0;

		/*
		 * Process all lines until ENDRECORD.
		 */
		while( fgets( dbuff, sizeof(dbuff), ifdorg ) != NULL ) {

			dbuff[strlen(dbuff)-1] = '\0';

			if ( (val = strrchr( dbuff , '\t')) == NULL )
				if ( (val=strrchr( dbuff , ' ')) == NULL )
					continue;
			val++;


			if ( !strncmp(dbuff, ENDRECORD, sizeof(ENDRECORD)-4)) {
				enditems = ftell( ifdorg );
				break;
			}

			/*
			 * If numbered item..
			 */
			if ( ! strncmp( dbuff, "*Item", 5 ) ) {

				itn = (int)strtol( &dbuff[5], &ptr, 0 );

				Db[itn].pos = itn;

				if ( !strncmp( ptr, LABEL, sizeof(LABEL)-1 ) ) {
					(void)strcpy( Db[itn].des, val );
					continue;
				}

				if ( !strncmp( ptr, PAREN, sizeof(PAREN)-1 ) ) {
					(void)strcpy( Db[itn].par, val );
					continue;
				}

				if ( !strncmp( ptr, CLASS, sizeof(CLASS)-1 ) ) {
					if ( *Db[itn].cls )
						continue;
					(void)strcpy( Db[itn].cls, val );
					continue;
				}

				if ( !strncmp( ptr, IHELP, sizeof(IHELP)-1 ) ) {
					(void)strcpy( Db[itn].hlp, val );
					continue;
				}

				if ( !strncmp( ptr, FONTL, sizeof(FONTL)-1 ) ) {
					(void)strcpy( Db[itn].fnt, val );
					continue;
				}

				if ( !strncmp( ptr, ROWS, sizeof(ROWS)-1 ) ) {
					(void)strcpy( Db[itn].row, val );
					(void)sprintf( Db[itn].cls, "%s%s",
						       TEXTS, val );
					continue;
				}

				if ( !strncmp( ptr, COLS, sizeof(COLS)-1 ) ) {
					(void)strcpy( Db[itn].col, val );
					(void)sprintf( Db[itn].cls, "%s%s",
						       TEXTL, val );
					continue;
				}

				if ( !strncmp( ptr, ORI, sizeof(ORI)-1 ) ) {
					(void)strcpy( Db[itn].ori, val );
					continue;
				}

				if ( !strncmp( ptr, HTM, sizeof(HTM)-1 ) ) {
					(void)strcpy( Db[itn].htm, val );
					(void)sprintf( Db[itn].cls, "%s%s",
						       EMALL, val );
					continue;
				}

				if ( !strncmp( ptr, ROPT, sizeof(ROPT)-1 ) ) {
					(void)strcpy( Db[itn].opt, val );
					continue;
				}

				if ( !strncmp( ptr, BACKG, sizeof(BACKG)-1 ) ) {
					(void)strcpy( Db[itn].bgr, val );
					continue;
				}
			} 

			/*
			 * Un-numbered resources.
			 */
			if ( !strncmp( dbuff, PKEYI, sizeof(PKEYI)-1 ) ) {
				pkey = i = (int)atoi( val );
				(void)strcpy( Db[i].cls, PRIM );
				Db[i].key = True;
				continue;
			}

			if ( !strncmp( dbuff, SKEYI, sizeof(SKEYI)-1 ) ) {
				if ( ! (i = (int)atoi( val )) )
					continue;
				skey = i;
				(void)strcpy( Db[i].cls, SPRIM );
				Db[i].skey = True;
				continue;
			}

			if ( !strncmp( dbuff, SEL2L, sizeof(SEL2L)-1 ) ) {
				(void)strcpy( Ttl2Lbl, val );
				continue;
			}

			if ( !strncmp( dbuff, SEL3L, sizeof(SEL3L)-1 ) ) {
				(void)strcpy( Ttl3Lbl, val );
				continue;
			}

			if ( !strncmp( dbuff, SEL4L, sizeof(SEL4L)-1 ) ) {
				(void)strcpy( Ttl4Lbl, val );
				continue;
			}

			if ( !strncmp( dbuff, FOLD, sizeof(FOLD)-1 ) ) {
				(void)strcpy( Fl[fln].name, val );
				Fl[fln++].type = dbuff[sizeof(FOLD)-1];
				continue;
			}
		}

		Lasti = itn;

		/*
		 * Handle Radio buttons.
		 */
		ptr = RADIOB; ptr++;
		for( i = 0; i < Lasti+1; i++ ) {
			if ( ! strcmp( Db[i].cls, ptr )) {
				if ( Db[i].ori[2] == 'V' )
					(void)sprintf( Db[i].cls, "%s%s",
						       VRADIO, Db[i].opt );
				if ( Db[i].ori[2] == 'H' )
					(void)sprintf( Db[i].cls, "%s%s",
						       HRADIO, Db[i].opt );
			}
		}

		/*
		 * Make a copy of pkey
		 * and skey, fnt, bg and help.
		 */
		(void)strcpy( DbPk.fnt, Db[pkey].fnt );
		(void)strcpy( DbPk.bgr, Db[pkey].bgr );
		(void)strcpy( DbPk.hlp, Db[pkey].hlp );

		(void)strcpy( DbSk.fnt, Db[skey].fnt );
		(void)strcpy( DbSk.bgr, Db[skey].bgr );
		(void)strcpy( DbSk.hlp, Db[skey].hlp );

		return(0);
	}

	if ( (cmd == SAVE) || (cmd == TEST) ) {

		/*
		 * Dump from memory to temp file
		 * to be used by Test or for permanent Save.
		 */
		if ( (ifd = fopen( ifname, "r" )) == NULL ) {
			(void)xpmsg( PopUp, "error: %s", ifname );
			return(-1);
		}

		if ( (ofd = fopen( Testfile, "w" )) == NULL ) {
			(void)xpmsg( PopUp, "error: %s", Testfile );
			(void)fclose(ifd);
			return(-1);
		}
		fln = 1;

		while( fgets( dbuff, sizeof(dbuff), ifd ) != NULL ) {

			dbuff[strlen(dbuff)-1] = '\0';

			/*
			 * Scan for the keys must be first.
			 */
			if ( !strncmp( dbuff, PKEYI, sizeof(PKEYI)-1 ) ) {
				for ( i = 0; i < Lasti+1; i++ ) {
					if ( Db[i].key == True ) {
						(void)fprintf( ofd,
							"%s\t\t\t%d\n",
							PKEYI, i ); 
						break;
					}
				}
				continue;
			}

			if ( !strncmp( dbuff, SKEYI, sizeof(SKEYI)-1 ) ) {
				for ( i = 0; i < Lasti+1; i++ ) {
					if ( Db[i].skey == True ) {
						(void)fprintf( ofd,
							"%s\t\t\t%d\n",
							SKEYI, i ); 
						break;
					}
				}

				/*
				 * Skey optional but must exist
				 * with 0 value.
				 */
				if ( i == Lasti+1 ) {
					(void)fprintf( ofd, "%s\t\t\t0\n",
						SKEYI ); 
				}
				continue;
			}

			if ( !strncmp( dbuff, SEL2L, sizeof(SEL2L)-1 ) ) {
				(void)fprintf( ofd, "%s%s\t\t%s\n",
					       SEL2L, LABEL, Ttl2Lbl );
				continue;
			}

			if ( !strncmp( dbuff, SEL3L, sizeof(SEL3L)-1 ) ) {
				(void)fprintf( ofd, "%s%s\t\t%s\n",
					       SEL3L, LABEL, Ttl3Lbl );
				continue;
			}

			if ( !strncmp( dbuff, SEL4L, sizeof(SEL4L)-1 ) ) {
				(void)fprintf( ofd, "%s%s\t\t%s\n",
					       SEL4L, LABEL, Ttl4Lbl );
				continue;
			}

			if ( !strncmp( dbuff, FOLD, sizeof(FOLD)-1 ) ) {
				static int last = 0;
				switch( dbuff[sizeof(FOLD)-1] )  {
					case 'C': (void)fprintf( ofd,
							FOLDERCLASS,
							Fl[fln++].name );
						break;
					case 'I': (void)fprintf( ofd,
							FOLDERINFO,
							Fl[fln++].name );
						break;
					case 'G': (void)fprintf( ofd,
							GFOLDERPATH,
							Fl[fln++].name );
						break;
					case 'P': (void)fprintf( ofd,
							PFOLDERPATH,
							Fl[fln++].name );
						break;
					case 'N': while (Fl[fln].type == 'N') {
							(void)fprintf( ofd,
								FOLDERNX,
								fln-4,
								Fl[fln].name);
								fln++;
							}
							last = fln-1;
						break;
					case 'L': while (Fl[fln].type == 'L') {
							(void)fprintf( ofd,
								FOLDERLX,
								fln-last,
								Fl[fln].name);

							   if ( cmd == SAVE ) {
							       check_ly( ifname,
								Fl[fln].name );
							   }
							   fln++;
							}
						break;
				}
				continue;
			}

			/*
			 * Save  comments until we reach
			 * the item area.
			 */
			if ( ((*dbuff == '!') && (items == True )) ) {
				(void)fprintf( ofd, "%s\n", dbuff );
				continue;
			}

			/*
			 * A hack to skip ItemX.. stuff
			 */
			if ( !strncmp( dbuff, "*Item", 5 ) ) {
				if ( dbuff[5] != 'X' ) {
					items = True;
					break;
				}
			}
			(void)fprintf( ofd, "%s\n", dbuff );
		}

		/*
		 * Empty the memory to file.
		 */
		for( i = 0; i < Lasti+1; i++ ) {

			/*
			 * Remove the "-option stuff
			 * we  have added to items.
			 */
			(void)strcpy( dbuff, Db[i].cls );
			if ( (ptr = strrchr( dbuff, '-' )) != NULL )
				*ptr = '\0';

			(void)fprintf( ofd,"*Item%d%s\t\t%s\n",
				       i, LABEL, Db[i].des );
			(void)fprintf( ofd, "*Item%d%s\t\t\t%s\n",
				       i, CLASS, dbuff );
			(void)fprintf( ofd, "*Item%d%s\t\t\t%s\n",
				       i, PAREN, Db[i].par );

			if ( *Db[i].opt ) {
				(void)fprintf( ofd, "*Item%d%s\t\t\t%s\n",
					       i, ROPT, Db[i].opt );
			}

			if ( *Db[i].hlp ) {
				(void)fprintf( ofd, "*Item%d%s\t\t\t%s\n",
					       i, IHELP, Db[i].hlp );
			}

			if ( *Db[i].row ) {
				(void)fprintf( ofd, "*Item%d%s\t\t\t%s\n",
					       i, ROWS, Db[i].row );
			}

			if ( *Db[i].col ) {
				(void)fprintf( ofd, "*Item%d%s\t\t%s\n",
					       i, COLS, Db[i].col );
			}

			if ( *Db[i].fnt ) {
				(void)fprintf( ofd, "*Item%d%s\t\t%s\n",
					       i, FONTL, Db[i].fnt );
			}

			if ( *Db[i].bgr ) {
				(void)fprintf( ofd, "*Item%d%s\t\t%s\n",
					       i, BACKG, Db[i].bgr );
			}

			if ( *Db[i].ori ) {
				(void)fprintf( ofd, "*Item%d%s\t\t%s\n",
					       i, ORI, Db[i].ori );
			}

			if ( *Db[i].htm ) {
				(void)fprintf( ofd, "*Item%d%s\t\t%s\n",
					       i, HTM, Db[i].htm );
			}

			(void)fprintf( ofd, "!\n" );
		}

		(void)fprintf( ofd, ENDRECORD );

		/*
		 * Print whatever left in input file.
		 */
		(void)fseek( ifdorg, enditems, SEEK_SET );
		while( fgets( dbuff, sizeof(dbuff), ifdorg ) != NULL )
			(void)fprintf( ofd, dbuff );

		(void)fclose(ofd);

		return(0);
	}

	if ( cmd == OK ) {

		if ( config1_cb( PopUp, SAVE, NULL ) )
			return(-1);

		(void)fclose(ifdorg);

		/*
		 * Copy folders and
		 * update attached dbs.
		 */
		rval = fcopy( Testfile, ifname );

		if ( ! Debug )
			(void)unlink( Testfile );

		return(rval);
	}

	if ( cmd == QUIT ) {
		(void)fclose(ifdorg);
		if ( ! Debug )
			(void)unlink( Testfile );
		return(0);
	}

	return(-1);
}


/*
 * Split & cat args from the
 * format 'Radio-horz=' type
 * argument as exposed on the
 * Item button.
 */
/*ARGSUSED*/
static void split_args( Widget w, char *cls )
{
	char buff[80];
	char *ptr;

	if ( (ptr = strchr( cls, '-')) == NULL ) {
		XmTextSetString( ArgText, "" );
		XmTextSetString( w, cls );
		return;
	}

	XmTextSetString( ArgText, ++ptr );


	(void)strcpy( buff, cls );
	*strchr( buff, '-' ) = '\0';

	XmTextSetString( w, buff );

}

/*ARGSUSED*/
static void cat_args( Widget w, char *cls )
{
	char buff[80];
	char *ptr;

	(void)strcpy( cls, ptr = XmTextGetString( w ) );
	XtFree(ptr);

	if ( *(ptr = XmTextGetString( ArgText )) ) {
		(void)sprintf( buff, "-%s", ptr );
		(void)strcat( cls, buff );
	}
	XtFree(ptr);
}


/*
 * Check and correct consistency
 * of all fields.
 */
/*ARGSUSED*/
static int fvalid( int itn, int ttl )
{
	int i;
	char *ptr;
	char cls[80], par[80], des[80];
	char catbuff[80];

	(void)strcpy( des, ptr = XmTextGetString( DesText ) ); XtFree(ptr);
	(void)strcpy( cls, ptr = XmTextGetString( ClsText ) ); XtFree(ptr);
	(void)strcpy( par, ptr = XmTextGetString( ParText ) ); XtFree(ptr);

	if ( !(strncmp( des, NULLI, sizeof(NULLI)-2)) && (*cls == '0') ) {
		XtVaSetValues( DesText, XmNeditable,  False, NULL );
		XtSetSensitive( ArgWin, False );
		XtSetSensitive( HlpWin, False );
		*Db[itn].hlp = *Db[itn].htm = '\0';
		*Db[itn].fnt = *Db[itn].bgr = '\0';
		*Db[itn].opt = *Db[itn].ori = '\0';
		*Db[itn].row = *Db[itn].col = '\0';
		return(0);
	}

	if ( ttl == EV_FOLD ) { 
		if ( ! *des ) {
			errno = 0;
			(void)xpmsg( PopUp, "error: no folder info" );
			return(-1);
		}

		if ( ! strcmp( FCLASSLY , ptr = XmTextGetString( ClsText ) ) ) {
			if ( (strchr( des, '(' ) == NULL) ||
					    (strchr( des, ')' )) == NULL ) {
					errno = 0;
					(void)xpmsg( PopUp,
					"error: syntax:  name(TITLE)" );
					XtFree(ptr);
					return(-1);
			}
		}

		if ( ! strncmp( des, FUNDEF, sizeof(FUNDEF)-1 ) )
			(void)xpmsg( PopUp, "info: Please define a folder" );

		XtFree(ptr);
		return(0);
	}

	if ( ! *par ) {
		errno = 0;
		(void)xpmsg( PopUp, "error: no parent" );
		return(-1);
	}

	if ( (atoi( par ) <1) || (atoi( par ) >5) ) {
		errno = 0;
		(void)xpmsg( PopUp, "error: valid parents are 1 to 5" );
		return(-1);
	}

	if ( ! *cls ) {
		errno = 0;
		(void)xpmsg( PopUp, "error: no Item Type" );
		return(-1);
	}

	if (  ttl )  {
		switch ( ttl ) {
			case 2: (void)strcpy( Ttl2Lbl,
					     ptr = XmTextGetString( DesText ) );
				XtFree(ptr);
				break;
			case 3: (void)strcpy( Ttl3Lbl,
					     ptr = XmTextGetString( DesText ) );
				XtFree(ptr);
				break;
			case 4: (void)strcpy( Ttl4Lbl,
					     ptr = XmTextGetString( DesText ) );
				XtFree(ptr);
				break;
		}
		return(0);
	}

	for( i = 0; i < Lasti+1; i++  ) {
		/*
		 * Description must be unique.
		 */
		if ( i == itn )
			continue;

		if ( ! strcmp( Db[i].des, NULLI ) )
			continue;

		if ( ! strcmp( Db[i].des, des ) ) {
			errno = 0;
			(void)xpmsg( PopUp,
			  "error: Multiple defined description, item %d", i );
			return(-1);
		}
	}

	cat_args( ClsText, catbuff );
	XtSetSensitive( ArgWin, True );
	XtSetSensitive( HlpWin, True );

	if ( !strncmp( catbuff, PRIM, sizeof(PRIM)-1 ) ) {
		Boolean chg = True;

		for( i = 0; i < Lasti+1; i++ ) {
			if ( i == itn ) {
				if ( Db[i].key == True )
					chg = False;
				Db[i].key = True;
				(void)strcpy( Db[i].cls, PRIM );
			} else {
				if ( Db[i].key == True ) {
					*strchr( Db[i].cls, '-' ) = '\0';
					*Db[i].hlp = *Db[i].htm = '\0';
					*Db[i].fnt = *Db[i].bgr = '\0';
					*Db[i].opt = *Db[i].ori = '\0';
					*Db[i].row = *Db[i].col = '\0';
				}
				Db[i].key = False;
			}
		}
		if ( chg == True ) {
			(void)strcpy( Db[itn].hlp, DbPk.hlp );
			(void)strcpy( Db[itn].fnt, DbPk.fnt );
			(void)strcpy( Db[itn].bgr, DbPk.bgr );
			*Db[itn].opt = *Db[itn].ori = '\0';
			*Db[itn].row = *Db[itn].col = '\0';
			*Db[itn].htm = '\0';
		}
		XtVaSetValues( ArgText, XmNeditable, False, NULL );
		return(0);
	}

	if ( !strncmp( catbuff, SPRIM, sizeof(SPRIM)-1 ) ) {
		Boolean chg = True;

		for( i = 0; i < Lasti+1; i++ ) {
			if ( i == itn ) {
				if ( Db[i].skey == True )
					chg = False;
				Db[i].skey = True;
				(void)strcpy( Db[i].cls, SPRIM );
			} else {
				if ( Db[i].skey == True ) {
					*strchr( Db[i].cls, '-' ) = '\0';
					*Db[i].hlp = *Db[i].fnt = '\0';
					*Db[i].bgr = *Db[i].htm = '\0';
					*Db[i].opt = *Db[i].ori = '\0';
					*Db[i].row = *Db[i].col = '\0';
				}
				Db[i].skey = False;
			}
		}
		if ( chg == True ) {
			(void)strcpy( Db[itn].hlp, DbSk.hlp );
			(void)strcpy( Db[itn].fnt, DbSk.fnt );
			(void)strcpy( Db[itn].bgr, DbSk.bgr );
			*Db[itn].opt = *Db[itn].ori = '\0';
			*Db[itn].row = *Db[itn].col = '\0';
			*Db[itn].htm = '\0';
		}

		if ( ! *Db[itn].bgr ) {
			/*
			 * A fresh skey is to be defined.
			 * Inherit highlight from pkey.
			 */
			(void)strcpy( Db[itn].fnt, DbPk.fnt );
			(void)strcpy( Db[itn].bgr, DbPk.bgr );
			*Db[itn].opt = *Db[itn].ori = '\0';
			*Db[itn].row = *Db[itn].col = '\0';
			*Db[itn].htm = '\0';
		}
		XtVaSetValues( ArgText, XmNeditable, False, NULL );
		return(0);
	}

	XtVaSetValues( ArgText, XmNeditable, True, NULL );
	Db[itn].key = False;
	Db[itn].skey = False;

	if ( !strncmp( catbuff, TEXTS, sizeof(TEXTS)-2 ) ) {
		int rows;
		if ( (ptr = strchr( catbuff, '=' )) == NULL ) {
			errno = 0;
			(void)xpmsg( PopUp, "error: define '=No of rows'" );
			return(-1);
		}
		ptr++;
		if ( (*ptr == '\0') || ((rows = (int)atoi( ptr )) == 0)  ) {
			errno = 0;
			(void)xpmsg( PopUp,
				     "error: define '%sNo of rows'", catbuff );
			return(-1);
		}

		if ( *par != '5' )
			(void)xpmsg( PopUp, "info: forced to parent 5" );
		(void)sprintf( Db[itn].row, "%d", rows );
		(void)strcpy( Db[itn].par, "5" );
		XmTextSetString( ParText, Db[itn].par );
		*Db[itn].col = *Db[itn].htm = '\0';
		*Db[itn].bgr = *Db[itn].fnt = '\0';
		*Db[itn].opt = *Db[itn].ori = '\0';
		return(0);
	}

	if ( !strncmp( catbuff, EMALL, sizeof(EMALL)-2 ) ) {
		int found = 0;

		if ( (ptr = strchr( catbuff, '=' )) == NULL ) {
			errno = 0;
			(void)xpmsg( PopUp, "error: define 3 items" );
			return(-1);
		}
		ptr++;
		if ( isdigit(*ptr) )
			found++;
		for ( i = 0; i < strlen(ptr); i++ ) {
			if ( (isspace(ptr[i])) && (isdigit(ptr[i+1])) )
				found++;
		}

		if ( found < 3 ) {
			errno = 0;
			(void)xpmsg( PopUp, "error: define 3 items" );
			return(-1);
		}
		found = 0;

		if ( *par != '1' )
			(void)xpmsg( PopUp, "info: forced to parent 1" );
		(void)strcpy( Db[itn].par, "1" );
		XmTextSetString( ParText, Db[itn].par );
		(void)strcpy( Db[itn].htm, ptr );
		*Db[itn].bgr = *Db[itn].fnt = '\0';
		*Db[itn].row = *Db[itn].col = '\0';

		for( i = 0; i < Lasti+1; i++  ) {
			if ( *Db[i].htm )
				if ( ++found >= 2 ) {
				  errno = 0;
				  (void)xpmsg( PopUp,
				  "error: only one Email item must be defined");
				  return(-1);
				}
		}
		return(0);
	}

	if ( !strncmp( catbuff, TEXTL, sizeof(TEXTL)-2 ) ) {
		int cols;
		if ( (ptr = strchr( catbuff, '=' )) == NULL ) {
			errno = 0;
			(void)xpmsg( PopUp, "error: define '=No of columns'" );
			return(-1);
		}
		/*
		 * if 0, make it an ordinary
		 * Text item.
		 */
		ptr++;
		if ( (*ptr == '\0') || ((cols = (int)atoi( ptr )) == 0)  ) {
			if ( (ptr = strchr( Db[itn].cls, '-' )) != NULL )
				*ptr = '\0';
			*Db[itn].col = '\0';
			XmTextSetString( ClsText, Db[itn].cls );
			XmTextSetString( ArgText, "" );
			XtSetSensitive( ArgWin, False );
		} else
			(void)sprintf( Db[itn].col, "%d", cols );

		*Db[itn].bgr = *Db[itn].fnt = '\0';
		*Db[itn].opt = *Db[itn].ori = '\0';
		*Db[itn].row = *Db[itn].htm = '\0';
		return(0);
	}

	if ( !strncmp( catbuff, TEXTN, sizeof(TEXTN)-2 ) ) {
		XtSetSensitive( ArgWin, False );
		*Db[itn].bgr = *Db[itn].fnt = '\0';
		*Db[itn].opt = *Db[itn].ori = '\0';
		*Db[itn].row = *Db[itn].col = '\0';
		*Db[itn].htm = '\0';
		return(0);
	}

	if ( !strncmp( catbuff, VRADIO, sizeof(VRADIO)-2 ) ) {
		if ( (ptr = strchr( catbuff, '=' )) == NULL ) {
			errno = 0;
			(void)xpmsg( PopUp, "error: define '=item itemx..'" );
			return(-1);
		}
		if ( strlen( ptr ) < 2 ) {
			errno = 0;
			(void)xpmsg( PopUp,
				     "error: define '%sitem itemx..'", catbuff);
			return(-1);
		}
		(void)strcpy( Db[itn].ori, VERT );
		(void)strcpy( Db[itn].opt, ++ptr );
		*Db[itn].bgr = *Db[itn].fnt = '\0';
		*Db[itn].row = *Db[itn].col = '\0';
		*Db[itn].htm = '\0';
		return(0);
	}

	if ( !strncmp( catbuff, HRADIO, sizeof(HRADIO)-2 ) ) {
		if ( (ptr = strchr( catbuff, '=' )) == NULL ) {
			errno = 0;
			(void)xpmsg( PopUp, "error: define '=item itemx..'" );
			return(-1);
		}
		if ( strlen( ptr ) < 2 ) {
			errno = 0;
			(void)xpmsg( PopUp,
				     "error: define '%sitem itemx..'", catbuff);
			return(-1);
		}
		(void)strcpy( Db[itn].ori, HORI );
		(void)strcpy( Db[itn].opt, ++ptr );
		*Db[itn].bgr = *Db[itn].fnt = '\0';
		*Db[itn].row = *Db[itn].col = '\0';
		*Db[itn].htm = '\0';
		return(0);
	}

	if ( !strncmp( catbuff, TOGGLE, sizeof(TOGGLE)-1 ) ) {
		XtSetSensitive( ArgWin, False );
		if ( (atoi( par ) <2) || (atoi( par ) >4) ) {
			errno = 0;
			(void)xpmsg( PopUp,
				     "error: Toggles for parent 2-4 only" );
			return(-1);
		}
		*Db[itn].bgr = *Db[itn].fnt = '\0';
		*Db[itn].opt = *Db[itn].ori = '\0';
		*Db[itn].row = *Db[itn].col = '\0';
		*Db[itn].htm = '\0';
		return(0);
	}

	(void)xpmsg( PopUp, "error: missing fields" );
	errno = 0;

	return(-1);
}


/*
 * Kepp cursors at the
 * end of text.
 */
static void setcur(void)
{
	char *ptr;

	XtVaSetValues( DesText, XmNcursorPosition,
			strlen( ptr = XmTextGetString( DesText) ), NULL );
	XtFree(ptr);

	XtVaSetValues( ArgText, XmNcursorPosition,
			strlen( ptr = XmTextGetString( ArgText) ), NULL );
	XtFree(ptr);

	XtVaSetValues( HlpText, XmNcursorPosition,
			strlen( ptr = XmTextGetString( HlpText) ), NULL );
	XtFree(ptr);
}


/*
 * Main callbcak. Also called
 * Recursively and will therefor
 * return status.
 */
/*ARGSUSED*/
static int
config1_cb(Widget w, int client_data, XmAnyCallbackStruct *cbs )
{

	static int itn, fln, ttl;
	int i, cnt;
	int sens = 0;
	char *src, *dst, *ptr;
	static char dbuff[100];
	XmString str = NULL;

	if ( client_data == START ) {

		itn = 0;
		fln = 1;
		ttl = 0;

		/*
		 * Expose the first item
		 * when started.
		 */
		split_args( ClsText,  Db[itn].cls );
		XmTextSetString( DesText, Db[itn].des );
		XmTextSetString( ParText, Db[itn].par );
		XmTextSetString( HlpText, Db[itn].hlp );

		(void)sprintf( dbuff, "%d/%d", itn, Lasti+1 );
		str = XmStringCreateSimple( dbuff );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);
		(void)fvalid( 0, 0 );

		XtSetSensitive( DelButton, True );
		XtSetSensitive( AddButton, True );

		setcur(); return(0);
	}

	xpmsg_close();

	XtVaSetValues( DesText, XmNeditable,  True, NULL );

	if ( (client_data >= EV_ITEMS) && (client_data < EV_TITLES) ) {

		/*
		 * Items
		 */
		ttl = 0;

		XtVaSetValues( DesText, XmNeditable, True, NULL );
		XtVaSetValues( ParText, XmNeditable, True, NULL );
		XtVaSetValues( ArgText, XmNeditable, True, NULL );
		XtVaSetValues( HlpText, XmNeditable, True, NULL );
		XtSetSensitive( ArgWin, True );
		XtSetSensitive( HlpWin, True );
		XtSetSensitive( ItemLabel, True );
		while( Isens[sens] ) XtSetSensitive( Isens[sens++], True );
		XtSetSensitive( Isens[0], False );
		XtSetSensitive( ArrowButtons, True );
		XtSetSensitive( ArgWin, True );
		XtSetSensitive( DelButton, True );
		XtSetSensitive( AddButton, True );

	}

	if ( (client_data >= EV_TITLES) && (client_data <= EV_FOLDERS) ) {

		/*
		 * Titles & Folders
		 */
		XtVaSetValues( ClsText, XmNeditable, False, NULL );
		XtVaSetValues( ParText, XmNeditable, False, NULL );
		XtSetSensitive( ItemLabel, False );
		XmTextSetString( ArgText, "" );
		XtSetSensitive( ArgWin, False );
		XmTextSetString( HlpText, "" );
		XtSetSensitive( HlpWin, False );
		while( Isens[sens] ) XtSetSensitive( Isens[sens++], False );
		XtSetSensitive( Isens[0], True );
		if ( client_data < EV_FOLD ) {
			XtSetSensitive( ArrowButtons, False );
			XtSetSensitive( DelButton, False );
			XtSetSensitive( AddButton, False );
		} else {
			XtSetSensitive( DelButton, True );
			XtSetSensitive( AddButton, True );
			XtSetSensitive( ArrowButtons, True );
		}
	}

	if ( (client_data == QUIT) || (client_data == OK) ) {

		if ( ! res_edit( PopUp, client_data ) ) {

			InUse = False;
			DestroyDialog(PopUp);

			XtFree( (char *)Db );
			XtFree( (char *)Fl );

			if ( (client_data == OK) && (Changed == CHANGED) ) {
				(void)layout_cb( NULL, CHANGED, NULL );
				/* NOTREACHED */
			}

			XtSetSensitive( GetTopWidget(), True );
			return(0);
		}

		/*
		 * Things went wrong.
		 */
		Changed = 0;
		DelAdd = False;

		if ( res_edit( PopUp, RESTART ) ) {

			/*
			 * Double fault.
			 * Keep messages on screen for a while.
			 */
			InUse = False;
			Wait(2);
			DestroyDialog(PopUp);

			XtFree( (char *)Db );
			XtFree( (char *)Fl );
			XtSetSensitive( GetTopWidget(), True );
		}
		return(0);
	}

	if ( (client_data == HELP) || (client_data == GHELP) ) {

		/*
		 * Launch help in accordance with
		 * the current item type on display.
		 */
		if ( client_data == GHELP ) {
			help( PopUp, HINDX, HCN );
			return(0);
		}

		if ( ! ttl ) {
			if ( (Db[itn].key) || (Db[itn].skey) ) {
				help( PopUp, HINDX, HDI );
				return(0);
			}

			(void)strcpy( dbuff, Db[itn].cls );

			if ( ! strncmp( dbuff, RADIO, sizeof(RADIO)-1 ) ) {
				help( PopUp, HINDX, HRA );
				return(0);
			}
			if ( ! strncmp( dbuff, TOGGLE, sizeof(TOGGLE)-1 ) ) {
				help( PopUp, HINDX, HTG );
				return(0);
			}
			if ( ! strncmp( dbuff, TEXTL, sizeof(TEXTL)-1 ) ) {
				help( PopUp, HINDX, HTL );
				return(0);
			}

			if ( ! strncmp( dbuff, TEXTN, sizeof(TEXTN)-1 ) ) {
				help( PopUp, HINDX, HXT );
				return(0);
			}
			if ( ! strncmp( dbuff, NULLI, sizeof(NULLI)-1 ) ) {
				help( PopUp, HINDX, HNL );
				return(0);
			}
			if ( ! strncmp( dbuff, TEXTS, sizeof(TEXTS)-1 ) ) {
				help( PopUp, HINDX, HSR );
				return(0);
			}
			if ( ! strncmp( dbuff, EMALL, sizeof(EMALL)-1 ) ) {
				help( PopUp, HINDX, HEM );
				return(0);
			}
			if ( ! strncmp( dbuff, "0", 1 ) ) {
				help( PopUp, HINDX, HNL );
				return(0);
			}
		}

		if ( ttl == EV_FOLD ) {

			switch( Fl[fln].type ) {
				case 'C': help( PopUp, HINDX, HCC );
					break;
				case 'I': help( PopUp, HINDX, HCI );
					break;
				case 'G': help( PopUp, HINDX, HCG ); 
					break;
				case 'P': help( PopUp, HINDX, HCM );
					break;
				case 'N': help( PopUp, HINDX, HFO );
					break;
				case 'L': help( PopUp, HINDX, HLA );
					break;
			}
			return(0);
		}

		if ( ttl )
			help( PopUp, HINDX, HCT );

		return(0);
	}

	if ( client_data == CLEAR ) {
		/*
		 * Clear all fields.
		 */
		XmTextSetString( DesText, "" );
		XmTextSetString( ArgText, "" );
		XmTextSetString( HlpText, "" );
		if ( ! ttl  ) {
			XmTextSetString( ClsText, "" );
			XmTextSetString( ParText, "" );
		}
		setcur(); return(0);
	}


	if ( client_data == APPL ) {
		/*
		 * Apply changes.
		 */
		src = XmTextGetString( ArgText );
		dst = strchr( PRIM, '-' );
		if ( (Db[itn].key == True ) &&(strcmp( src, ++dst)) &&(!ttl) ) {
			XtFree(src);
			errno = 0;
			(void)xpmsg( PopUp,
				     "error: define a new Primary Key first" );
			XmTextSetString( DesText, Db[itn].des );
			XmTextSetString( ParText, Db[itn].par );
			XmTextSetString( HlpText, Db[itn].hlp );
			split_args( ClsText,  Db[itn].cls );
			XmTextSetString( ArgText, dst );
			XtVaSetValues( ArgText, XmNeditable, False, NULL );
			XtSetSensitive( ArgWin, True );
			setcur(); return(-1);
		}
		XtFree(src);

		if ( fvalid( itn, ttl ) )
			return(-1);


		if ( ! ttl ) {

			(void)strcpy( Db[itn].cls,
				      ptr = XmTextGetString( ClsText ) );
			XtFree(ptr);
			cat_args( ClsText, Db[itn].cls );
			(void)strcpy( Db[itn].des,
				      ptr = XmTextGetString( DesText ) );
			XtFree(ptr);
			(void)strcpy( Db[itn].par,
				      ptr = XmTextGetString( ParText ) );
			XtFree(ptr);
			(void)strcpy( Db[itn].hlp,
				      ptr = XmTextGetString( HlpText ) );
			XtFree(ptr);
		}

		if ( ttl == EV_FOLD ) {
			(void)strcpy( Fl[fln].name,
				      ptr = XmTextGetString( DesText ) );
			XtFree(ptr);
		}

		if ( cbs != NULL /* Button pressed */ )
			Changed = CHANGED;
		setcur(); return(0);
	}

	if ( client_data == TEST ) {
		/*
		 * Test changes by launching
		 * a test version of mdb.
		 */
		if ( config1_cb( PopUp, APPL, NULL ) )
			return(0);

		if ( ! res_edit( PopUp, TEST ) )
			layout_cb( NULL, TEST, NULL );

		return(0);
	}


	if ( (client_data == DELI) && (ttl == EV_FOLD) ) {
		/*
		 * Delete folder.
		 */
		char old;
		old = Fl[fln].type;

		Changed = CHANGED;

		if ( Fl[fln+1].type != Fl[fln].type ) {
			if ( Fl[fln-1].type != Fl[fln].type )
				return(0);
		}



		cnt = (MAXFOLDS - (fln+1)) * sizeof( struct fold );
		src = (char *)&Fl[fln+1];
		dst = (char *)&Fl[fln];

		for( i = 0; i < cnt; i++ )
			*dst++ = *src++;

		if ( old != Fl[fln].type )
			--fln;

		(void)config1_cb( PopUp, EV_FOLD, NULL );

		setcur(); return(0);
	}

	if ( client_data == DELI ) {
		/*
		 * Delete Item.
		 */
		if ( Lasti-1 < 0 )
			return(0);

		if ( Db[itn].key == True ) {
			errno = 0;
			(void)xpmsg( PopUp,
				     "error: define a new Primary Key first" );
				return(-1);
		}

		if ( itn == Lasti ) {
			Lasti--;
			itn--;
		} else {

			cnt = ((MAXITEMS-1) - itn) * sizeof(struct item);
			dst = (char *)&Db[itn];
			src = (char *)&Db[itn+1];

			for ( i = 0; i < cnt; i++ )
				*dst++ = *src++;

			Lasti--;
		}

		split_args( ClsText,  Db[itn].cls );
		XmTextSetString( DesText, Db[itn].des );
		XmTextSetString( ParText, Db[itn].par );
		XmTextSetString( HlpText, Db[itn].hlp );

		(void)sprintf( dbuff, "%d/%d", itn, Lasti+1 );
		str = XmStringCreateSimple( dbuff );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);

		DelAdd = True;
		Changed = CHANGED;

		setcur(); return(fvalid( itn, ttl ));
	}

	if ( (client_data == ADDI) && (ttl == EV_FOLD) ) {
		/*
		 * Add folder.
		 */

		cnt = ((MAXFOLDS-1) -fln) * sizeof( struct fold );
		src = (char *)&Fl[MAXFOLDS-1];
		dst = (char *)&Fl[MAXFOLDS];

		for( i = 0; i < cnt; i++ )
			*--dst = *--src;

		(void)strcpy( Fl[fln].name , ptr = XmTextGetString( DesText ) );
		XtFree(ptr);

		Changed = CHANGED;

		setcur(); (void)fvalid( itn, ttl); return(0);
	}

	if ( client_data == ADDI ) {
		/*
		 * Add item.
		 */
		cnt = ((MAXITEMS-1) - itn) * sizeof(struct item );
		src = (char *)&Db[MAXITEMS-1];
		dst = (char *)&Db[MAXITEMS];

		for ( i = 0; i < cnt; i++ )
			*--dst = *--src;

		if ( fvalid( itn, ttl ) ) {
			/*
			 * Undo.
			 */
			dst = (char *)&Db[itn];
			src = (char *)&Db[itn+1];

			for ( i = 0; i < cnt; i++ )
				*dst++ = *src++;

			split_args( ClsText,  Db[itn].cls );
			XmTextSetString( DesText, Db[itn].des );
			XmTextSetString( ParText, Db[itn].par );
			XmTextSetString( HlpText, Db[itn].hlp );
			setcur(); return(0);
		} 

		if ( Lasti < MAXITEMS-1 )
			Lasti++;
		else 
			(void)xpmsg( PopUp,
				     "info: Mdb full. Last item discarded" );

		cat_args( ClsText, Db[itn].cls );
		(void)strcpy( Db[itn].des, ptr = XmTextGetString( DesText ) );
		XtFree(ptr);
		(void)strcpy( Db[itn].par, ptr = XmTextGetString( ParText ) );
		XtFree(ptr);
		(void)strcpy( Db[itn].hlp, ptr = XmTextGetString( HlpText ) );
		XtFree(ptr);
		/*
		 * Mark as unused for now.
		 */
		Db[itn].pos = DELETED;
		DelAdd = True;
		Changed = CHANGED;

		(void)sprintf( dbuff, "%d/%d", itn, Lasti+1 );
		str = XmStringCreateSimple( dbuff );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);

		setcur(); return(0);
	}

	if ( client_data == SAVE ) {
		/*
		 * Save current state into temp file.
		 */
		if ( config1_cb( PopUp, APPL, NULL ) )
			return(-1);

		if ( !ttl ) {
			cat_args( ClsText, Db[itn].cls );
			(void)strcpy( Db[itn].des,
				      ptr = XmTextGetString( DesText ) );
			XtFree(ptr);
			(void)strcpy( Db[itn].par,
				      ptr = XmTextGetString( ParText ) );
			XtFree(ptr);
			(void)strcpy( Db[itn].hlp,
				      ptr = XmTextGetString( HlpText ) );
			XtFree(ptr);
		}

		if ( ttl == EV_FOLD ) {
			(void)strcpy( Fl[fln].name,
				      ptr = XmTextGetString( DesText ) );
			XtFree(ptr);
		}

		(void)res_edit( PopUp, SAVE );

		return(0);
	}

	if ( (client_data == UP) && (ttl == EV_FOLD ) ) {
		/*
		 * Up-button.
		 */
		if ( Fl[fln - 1].type ) {
			--fln;
			(void)config1_cb( PopUp, EV_FOLD, NULL );
		}
		return(0);
	}

	if ( (client_data == UP) && (!ttl) ) {
		/*
		 * Up-button.
		 */
		if ( --itn < 0 ) {
			itn = Lasti;
		}

		split_args( ClsText,  Db[itn].cls );
		XmTextSetString( DesText, Db[itn].des );
		XmTextSetString( ParText, Db[itn].par );
		XmTextSetString( HlpText, Db[itn].hlp );

		(void)sprintf( dbuff, "%d/%d", itn, Lasti+1 );
		str = XmStringCreateSimple( dbuff );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);

		setcur(); return(fvalid( itn, ttl ));
	}

	if ( (client_data == DN) && (ttl == EV_FOLD ) ) {
		/*
		 * Down-button.
		 */
		if ( Fl[fln + 1].type ) {
			fln++;
			(void)config1_cb( PopUp, EV_FOLD, NULL );
		}
		setcur(); return(0);
	}

	if ( (client_data == DN) && (!ttl) ) {
		/*
		 * Down-button.
		 */
		if ( ++itn > Lasti ) {
			itn = 0;
		}

		split_args( ClsText,  Db[itn].cls );
		XmTextSetString( DesText, Db[itn].des );
		XmTextSetString( ParText, Db[itn].par );
		XmTextSetString( HlpText, Db[itn].hlp );

		(void)sprintf( dbuff, "%d/%d", itn, Lasti+1 );
		str = XmStringCreateSimple( dbuff );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);

		setcur(); return(fvalid( itn, ttl ));
	}

	XmTextSetString( DesText, "" );
	XmTextSetString( ClsText, "" );
	XmTextSetString( ParText, "" );
	XmTextSetString( ArgText, "" );
	XmTextSetString( HlpText, "" );

	if ( client_data == EV_NULLI ) {
		/*
		 * Apply a NULL item.
		 */
		XtVaSetValues( DesText, XmNeditable,  False, NULL );
		XmTextSetString( DesText, NULLI );
		XmTextSetString( ClsText, "0" );
		XmTextSetString( ParText, "0" );
		XmTextSetString( ArgText, "" );
		XmTextSetString( HlpText, "" );
		XtSetSensitive( ArgWin, False );
		XtSetSensitive( HlpWin, False );
		return(0);
	}

	if ( client_data == EV_TEXTN ) {
		/*
		 * Apply a Text item.
		 */
		split_args( ClsText,  TEXTN );
		XmTextSetString( ArgText, "" );
		XtSetSensitive( ArgWin, False );
		setcur(); return(0);
	}

	if ( client_data == EV_TEXTL ) {
		/*
		 * Apply a Limited Text item.
		 */
		split_args( ClsText, TEXTL );
		setcur(); return(0);
	}

	if ( client_data == EV_TEXTS ) {
		/*
		 * Apply a Scrolled Test item.
		 */
		XmTextSetString( ParText, "5" );
		split_args( ClsText, TEXTS );
		setcur(); return(0);
	}

	if ( client_data == EV_EMAIL ) {
		/*
		 * Apply an Email item.
		 */
		XmTextSetString( ParText, "1" );
		split_args( ClsText, EMALL );
		setcur(); return(0);
	}

	if ( client_data == EV_HRADIO ) {
		/*
		 * Apply a Horizontal Radio item.
		 */
		split_args( ClsText, HRADIO );
		setcur(); return(0);
	}

	if ( client_data == EV_VRADIO ) {
		/*
		 * Apply a Vertical Radio item.
		 */
		split_args( ClsText, VRADIO );
		setcur(); return(0);
	}

	if ( client_data == EV_PRIM ) {
		/*
		 * Apply a Pimary Key item.
		 */
		split_args( ClsText, PRIM );
		XtVaSetValues( ArgText, XmNeditable, False, NULL );
		setcur(); return(0);
	}

	if ( client_data == EV_SPRIM ) {
		/*
		 * Apply a Secondary Pimary Key item.
		 */
		split_args( ClsText, SPRIM );
		XtVaSetValues( ArgText, XmNeditable, False, NULL );
		setcur(); return(0);
	}

	if ( client_data == EV_TOGGLE ) {
		/*
		 * Apply a Toggle item.
		 */
		split_args( ClsText, TOGGLE );
		XmTextSetString( ArgText, "" );
		XtSetSensitive( ArgWin, False );
		setcur(); return(0);
	}

	if ( client_data == EV_TTL2 ) {
		/*
		 * Apply a Title.
		 */
		ttl = 2;
		XmTextSetString( ClsText, TTL2 );
		XmTextSetString( DesText, Ttl2Lbl );
		XmTextSetString( ParText, "2" );
		str = XmStringCreateSimple( "" );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);
		setcur(); return(0);
	}

	if ( client_data == EV_TTL3 ) {
		/*
		 * Apply a Title.
		 */
		ttl = 3;
		XmTextSetString( ClsText, TTL3 );
		XmTextSetString( DesText, Ttl3Lbl );
		XmTextSetString( ParText, "3" );
		str = XmStringCreateSimple( "" );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);
		setcur(); return(0);
	}

	if ( client_data == EV_TTL4 ) {
		/*
		 * Apply a Title.
		 */
		ttl = 4;
		XmTextSetString( ClsText, TTL4 );
		XmTextSetString( DesText, Ttl4Lbl );
		XmTextSetString( ParText, "4" );
		str = XmStringCreateSimple( "" );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);
		setcur(); return(0);
	}

	if ( client_data == EV_FOLD ) {
		/*
		 * Apply a Folder.
		 */
		static int last;
		ttl = EV_FOLD;

		if ( fln < 5 ) {
			XtSetSensitive( AddButton, False );
			XtSetSensitive( DelButton, False );
		} else {
			XtSetSensitive( AddButton, True );
			XtSetSensitive( DelButton, True );
			XtSetSensitive( ItemLabel, True );
		}

		switch( Fl[fln].type ) {
			case 'C':
				XmTextSetString(ClsText, "Folder CLASS");
				str = XmStringCreateSimple( "" );
				break;
			case 'I':
				XmTextSetString(ClsText, "Folder TITLE");
				str = XmStringCreateSimple( "" );
				break;
			case 'G':
				XmTextSetString(ClsText,"Folder Global PATH");
				str = XmStringCreateSimple( "" );
				break;
			case 'P':
				XmTextSetString(ClsText, "Folder Private PATH");
				str = XmStringCreateSimple( "" );
				break;
			case 'N':
				XmTextSetString(ClsText,"Folder NAME");
				(void)sprintf( dbuff, "%d", fln-4 );
				str = XmStringCreateSimple( dbuff );
				last = fln;
				break;
			case 'L':
				XmTextSetString(ClsText, FCLASSLY );
				(void)sprintf( dbuff, "%d", fln-last );
				str = XmStringCreateSimple( dbuff );
				break;
		}

		XmTextSetString( DesText, Fl[fln].name);
		XmTextSetString( ParText, "" );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);
		setcur(); (void)fvalid( itn, ttl ); return(0);
	}

	if ( client_data == EV_ITEM ) {
		/*
		 * Go back to db items.
		 */
		split_args( ClsText, Db[itn].cls );
		XmTextSetString( DesText, Db[itn].des );
		XmTextSetString( ParText, Db[itn].par );
		XmTextSetString( HlpText, Db[itn].hlp );

		(void)sprintf( dbuff, "%d/%d", itn, Lasti+1 );
		str = XmStringCreateSimple( dbuff );
		XtVaSetValues( ItemLabel, XmNlabelString, str, NULL );
		XmStringFree(str);
		setcur(); return(0);
	}
	return(0);
}

/*
 * External close request.
 */
/*ARGSUSED*/
void config_close(Widget top)
{
	/*
	 * External close request.
	 */
	if ( (InUse && ! top) || (InUse && (PopUp == top)))
		(void)config1_cb( NULL, QUIT, NULL );
}


/*
 * Launch the Dialog.
 */
/*ARGSUSED*/
void config( Widget parent )
{

	Widget paned_w, form_w;
	Widget frame_w, rowcol_w;
	Widget rowcol1_w, rowcol2_w;
	Widget rowcol3_w, OptM_w;
	Widget tmp_w, *wptr;
	Arg args[20];
	int n;
	int sens = 0;

	if ( InUse == True ) {
		(void)xpmsg( PopUp, "info: in use" );
		return;
	}

	bzero( Isens, sizeof(Isens) );
	/*
	 * Launch the Dialog.
	 */
	PopUp = CreateDialog( parent, "configWin" );
	InUse = True;

	XtAddCallback( PopUp,
		XmNhelpCallback, (XtCP)config1_cb, (XtPointer)GHELP );

	paned_w = CreateRowColumn( PopUp, "panedWin", XmVERTICAL, 0 );
	XtVaSetValues( paned_w, XmNspacing, 3 , NULL );

	form_w = XmCreateForm( paned_w, "formWin", NULL, 0 );

	frame_w = XtVaCreateManagedWidget( "frameCnfgWin",
		xmFrameWidgetClass,	form_w,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNtopOffset,		15,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNrightAttachment,	XmATTACH_FORM,
		XmNrightOffset,		10,
		XmNshadowType,		XmSHADOW_ETCHED_IN,
		NULL );

	rowcol_w = CreateRowColumn( frame_w, "rowcolWin", XmVERTICAL, 0 );
	XtManageChild(rowcol_w);
	ArrowButtons = rowcol_w;

	(void)CreateLabel( rowcol_w, "itemLabel" );

	tmp_w = XtVaCreateManagedWidget( "upArrow",
		xmArrowButtonGadgetClass,	rowcol_w,
		XmNarrowDirection,		XmARROW_UP,
		XmNhighlightThickness,		1,
		XmNhighlightOnEnter,		True,
		NULL );
	XtAddCallback( tmp_w, XmNactivateCallback,
			(XtCP)config1_cb, (XtPointer)UP );


	tmp_w = XtVaCreateManagedWidget( "dnArrow",
		xmArrowButtonGadgetClass,	rowcol_w,
		XmNarrowDirection,		XmARROW_DOWN,
		XmNhighlightThickness,		1,
		XmNhighlightOnEnter,		True,
		NULL );
	XtAddCallback( tmp_w, XmNactivateCallback,
			(XtCP)config1_cb, (XtPointer)DN );

	XtVaSetValues( tmp_w = CreateLabel( form_w, "descLabel" ),
		XmNtopAttachment,		XmATTACH_FORM,
		XmNtopOffset,			5,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNrecomputeSize,		False,
		XmNbottomAttachment,		XmATTACH_POSITION,
		XmNbottomPosition,		48,
		XmNwidth,			135,
		NULL );

	XtVaSetValues( DesText = CreateText( form_w, "descLabel", QUICK ),
		XmNtopAttachment,		XmATTACH_FORM,
		XmNtopOffset,			5,
		XmNleftAttachment,		XmATTACH_WIDGET,
		XmNleftWidget,			tmp_w,
		XmNrightAttachment,		XmATTACH_POSITION,
		XmNrightPosition,		80,
		XmNbottomAttachment,		XmATTACH_POSITION,
		XmNbottomPosition,		48,
		XmNcolumns,			GetItemXTcolumns(),
		XmNhighlightOnEnter,		False,
		XmNtraversalOn,			False,
		NULL );

	XtVaSetValues( rowcol1_w = CreateRowColumn( form_w,
			"classRc", XmHORIZONTAL, 0 ),
		XmNtopAttachment,		XmATTACH_POSITION,
		XmNtopPosition,			52,
		XmNbottomAttachment,		XmATTACH_FORM,
		XmNbottomOffset,		5,
		XmNleftAttachment,		XmATTACH_FORM,
		NULL );

	tmp_w = XmCreatePulldownMenu( rowcol1_w, "classMenu", NULL, 0 );
	n = 0;
	XtSetArg (args[n], XmNsubMenuId,	tmp_w ); n++;
	OptM_w = XmCreateOptionMenu( rowcol1_w, "classOptions", args, n );
	XtManageChild( OptM_w );

	Isens[sens++] = CreatePushButton( tmp_w, ITEMS,
				(XtCP)config1_cb, EV_ITEM, HCN );
	Isens[sens++] = CreatePushButton( tmp_w, TEXTN,
				(XtCP)config1_cb, EV_TEXTN, HXT );
	Isens[sens++] = CreatePushButton( tmp_w, PRIM,
				(XtCP)config1_cb, EV_PRIM, HDI );
	Isens[sens++] = CreatePushButton( tmp_w, SPRIM,
				(XtCP)config1_cb, EV_SPRIM, HDI );
	Isens[sens++] = CreatePushButton( tmp_w, TEXTL,
				(XtCP)config1_cb, EV_TEXTL, HTL );
	Isens[sens++] = CreatePushButton( tmp_w, TEXTS,
				(XtCP)config1_cb, EV_TEXTS, HSR );
	Isens[sens++] = CreatePushButton( tmp_w, EMALL,
				(XtCP)config1_cb, EV_EMAIL, HEM );
	Isens[sens++] = CreatePushButton( tmp_w, VRADIO,
				(XtCP)config1_cb, EV_VRADIO, HRA );
	Isens[sens++] = CreatePushButton( tmp_w, HRADIO,
				(XtCP)config1_cb, EV_HRADIO, HRA );
	Isens[sens++] = CreatePushButton( tmp_w, TOGGLE,
				(XtCP)config1_cb, EV_TOGGLE, HTG );
	Isens[sens++] = CreatePushButton( tmp_w, NULLI,
				(XtCP)config1_cb, EV_NULLI, HNL );

	XtSetSensitive( Isens[0], False );

	(void)CreateSeparator( tmp_w, 0 );

	(void)CreatePushButton( tmp_w, TTL2, (XtCP)config1_cb, EV_TTL2, HCT );
	(void)CreatePushButton( tmp_w, TTL3, (XtCP)config1_cb, EV_TTL3, HCT );
	(void)CreatePushButton( tmp_w, TTL4, (XtCP)config1_cb, EV_TTL4, HCT );

	(void)CreateSeparator( tmp_w, 0 );

	(void)CreatePushButton( tmp_w, TFOLD, (XtCP)config1_cb, EV_FOLD, HCC );

	XtVaSetValues( ItemLabel = CreateLabel( form_w,	"itemNoLabel" ),
		XmNtopAttachment,		XmATTACH_POSITION,
		XmNtopPosition,			52,
		XmNbottomAttachment,		XmATTACH_FORM,
		XmNbottomOffset,		5,
		XmNrightAttachment,		XmATTACH_WIDGET,
		XmNrightWidget,			frame_w,
		XmNrightOffset,			10,
		XmNwidth,			35,
		XmNrecomputeSize,		False,
		XmNalignment,			XmALIGNMENT_BEGINNING,
		NULL );

	XtVaSetValues( ParText = CreateText( form_w, "parLabel", NULL ),
		XmNtopAttachment,		XmATTACH_POSITION,
		XmNtopPosition,			52,
		XmNbottomAttachment,		XmATTACH_FORM,
		XmNbottomOffset,		5,
		XmNrightAttachment,		XmATTACH_POSITION,
		XmNrightPosition,		80,
		XmNcolumns,			2,
		NULL );

	XtVaSetValues( tmp_w = CreateLabel( form_w, "parLabel" ),
		XmNtopAttachment,		XmATTACH_POSITION,
		XmNtopPosition,			52,
		XmNbottomAttachment,		XmATTACH_FORM,
		XmNbottomOffset,		5,
		XmNrightAttachment,		XmATTACH_WIDGET,
		XmNrightWidget,			ParText,
		XmNrightOffset,			5,
		XmNwidth,			45,
		NULL );

	XtVaSetValues( ClsText = CreateText( form_w, "classLabel", QUICK ),
		XmNtopAttachment,		XmATTACH_POSITION,
		XmNtopPosition,			52,
		XmNbottomAttachment,		XmATTACH_FORM,
		XmNbottomOffset,		5,
		XmNleftAttachment,		XmATTACH_WIDGET,
		XmNleftWidget,			rowcol1_w,
		XmNrightAttachment,		XmATTACH_WIDGET,
		XmNrightWidget,			tmp_w,
		XmNcursorPositionVisible,	False,
		XmNeditable,			False,
		XmNrightOffset,			35,
		XmNcolumns,			10,
		NULL );

	XtVaSetValues( rowcol2_w = CreateRowColumn( paned_w,
				"rowcolWin", XmHORIZONTAL, 0 ),
		XmNspacing, 25, NULL );

	rowcol3_w = CreateRowColumn( rowcol2_w, "rowcolWin", XmVERTICAL, 0 );

	ArgWin = CreateRowColumn( rowcol3_w, "rowcolWin", XmHORIZONTAL, 0 );
	XtManageChild(ArgWin);

	XtVaSetValues( tmp_w = CreateLabel( ArgWin, "argsLabel" ),
		XmNwidth,			135,
		XmNrecomputeSize,		False,
		XmNalignment,			XmALIGNMENT_CENTER,
		NULL );

	XtVaSetValues( ArgText = CreateText( ArgWin, "argsLabel", QUICK ),
		XmNcolumns, 40, NULL );

	HlpWin = CreateRowColumn( rowcol3_w, "rowcolWin", XmHORIZONTAL, 0 );
	XtManageChild(HlpWin);

	XtVaSetValues( tmp_w = CreateLabel( HlpWin, "helpLabel" ),
		XmNwidth,			135,
		XmNrecomputeSize	,	False,
		XmNalignment,			XmALIGNMENT_CENTER,
		NULL );

	XtVaSetValues( HlpText = CreateText( HlpWin, "helpLabel", QUICK ),
		XmNcolumns, 40, NULL );

	n = 0;
	XtSetArg(args[n], XmNlabelType, GetMdbPixmap()? XmPIXMAP:XmSTRING); n++;
	XtSetArg(args[n], XmNlabelPixmap,  GetMdbPixmap()); n++;
	XtManageChild( XmCreateLabel( rowcol2_w, "MDB", args, n ) );

	(void)CreateSeparator( paned_w , 0 );

	wptr = (Widget *)XtMalloc( XtNumber(action_items) * sizeof(Widget) );

	(void)CreateActionArea( paned_w, action_items,
		XtNumber(action_items), NULL, NULL, wptr, False );
	/*
	 * Pick out Add & Del buttons.
	 */
	AddButton = wptr[3];
	DelButton = wptr[4];
	XtFree( (char *)wptr );

	XtManageChild(rowcol1_w);
	XtManageChild(rowcol2_w);
	XtManageChild(rowcol3_w);

	XtManageChild(form_w);
	XtManageChild(paned_w);

	MapDialog( parent, PopUp, 0, 200 );

	Changed = 0;
	DelAdd = False;

	if ( res_edit( PopUp, START ) ) {
		Wait(5);
		(void)config1_cb( PopUp, QUIT, NULL );
		return;
	}

	XtSetSensitive( GetTopWidget(), False );

	(void)config1_cb( PopUp, START, NULL );
}
