////////////////////////////////////////////////////////////////////////////////
//
//	Class : mxMailMsg
//
//	Author : Andy Jefferson
//
//	Description :
//		mxMailMsg is a representation of a mail message.
//		It builds on the mxMsgSummary class and adds the idea of
//		a body text as well as attachments. It provides the support
//		for MIME as a consequence.
//
//	Public Methods :
//		Methods are available for setting and retrieving all of the
//		parameters that make up a mail message.
//
//	Limitations :
//		Currently only copes with MIME Version 1.0 tags and only
//		with MIME-specified 'subtypes'.
//
////////////////////////////////////////////////////////////////////////////////
#ifndef mxMailMsg_h
#define mxMailMsg_h
#ifdef __cplusplus

#include <std/string.h>
#include <iostream.h>
#include <fstream.h>
#include <bool.h>
#include <vector.h>
#include "mxMsgSummary.h"
#include "mxMailAttachment.h"

class mxMailMsg : public mxMsgSummary
{
	public:

		// Constructor, assignment operator, etc

		mxMailMsg()
		  {_mime=FALSE;};
		mxMailMsg(const mxMailMsg&);
		mxMailMsg& operator=(const mxMailMsg&);

		// Access functions

		int			noOfAttachments() const;
		const mxMailAttachment&	attachment(int no) const;
		string			bodyText() const;
		bool			mime() const;

		// Update function - setting all of message from message text

		void		setMessage(string);

		// Update functions - setting message bit by bit

		void		setBodyText(string);
		void		addAttachment(mxMailAttachment&);
		void		removeAttachments();

		bool		writeToFile(string filename,
					    bool append,
					    bool sendheaders) const;

		// Control functions

		friend ostream&	operator<< (ostream &s,mxMailMsg& msg);

	private:

		// Message data items

		string				_bodyText;
		vector<mxMailAttachment>	_attachments;
		bool				_mime;
		string				_mimeSeparator;

		bool		convertPartIntoAttachment(string msg_part);
		void		writeMsgToStream(ofstream& outfile,
						 bool sendheaders) const;
		string		splitAddressees(string prefix,string addresses) const;
};

inline
string	mxMailMsg::bodyText() const
	{return _bodyText;}
inline
bool	mxMailMsg::mime() const
	{return _mime;}
inline
int	mxMailMsg::noOfAttachments() const
	{return _attachments.size();}
inline
const mxMailAttachment&	mxMailMsg::attachment(int no) const
	{return _attachments[no];}
inline
void	mxMailMsg::setBodyText(string text)
	{_bodyText = text;}
inline
ostream&        operator<<(ostream& s,mxMailMsg& msg)
{
	return s << endl
		 << "Message-Text:" << endl << msg._bodyText << endl
		 << "No of Attachments=" << msg._attachments.size() << endl;
}

#endif
#endif
