////////////////////////////////////////////////////////////////////////////////
//
//	Class : akProcess
//
//	Author : Andy Jefferson
//
//	Description :
//		The akProcess class provides an interface to running a UNIX
//		sub-process. It takes care of setting up of and closing of
//		pipes to the processes standard-input (cin),
//		standard-output (cout), and standard-error (cerr) streams.
//		The user provides a callback function that is called whenever
//		data (a string) is returned from the process (on cout or cerr).
//		The command string passed to the start function is the UNIX
//		command with all arguments (e.g 'ls -ial /usr/lib')
//		If the string passed back has size = 0 then the command has
//		terminated itself.
//
//		The user can start and stop the process whenever required and
//		has an active function to detect if the process is active.
//		The user can pass input to the process via a string
//
//	Public Methods :
//		(constructor) requires no arguments.
//		start()		Function to start the process up.
//		stop()		Function to stop the process.
//		active()	Function to give the process status.
//		input()		Used to pass input to the process.
//
//	Limitations :
//		The only input is via a string - would have liked to provide
//		a stream, but wanted to protect the inputStream.
//		Not doing anything about buffering of streams - depends on the
//		buffering strategy of the invoked UNIX command.
//
////////////////////////////////////////////////////////////////////////////////
#ifndef _akProcess_h
#define _akProcess_h
#ifdef __cplusplus

#include <std/string.h>
#include <bool.h>
#include <iostream.h>
#include <fstream.h>

typedef	void	(*akProcessCB)(XtPointer,string);
////////////////////////////////////////////////////////////////////////////////
class akProcess
{
	public:

		// Constructor/destructor

		akProcess();
		virtual	~akProcess();

		// Access functions
		virtual	bool	start(string command,
				      akProcessCB callback,
				      XtPointer clientData);
		virtual bool	stop();
		virtual	void	input(string input);
		virtual	bool	active();


		// Class name
		virtual const char *const className(void)
			{return ("akProcess");}

	protected:

		// The streams connected to the process - hidden from user
		ofstream		_inputStream;
		ifstream		_outputStream;
		ifstream		_errorStream;

	private:

		akProcessCB		_userCB;
		XtPointer		_userData;

		XtInputId		_outputId;
		XtInputId		_errorId;
		bool			_active;
		int			_pid;

		static void	receiveOutputCB(XtPointer,int*,XtInputId*);
};

inline
bool	akProcess::active()
	{return _active;}
inline
void	akProcess::input(string the_input)
{
	if (_inputStream)
	  _inputStream << the_input << endl;
}

#endif
#endif
