/* zdbm.h -- Zombie Database Manager header file.
 * Chris Hyde
 */
/* Modifications:
 *      Fixed prototype for file_size() so it matched function. DC 11/20/93
 *	Added PtsRoot  DC 2/23/94
 *	Added include of "util.h" so new library will be used. DC 3/11/94
 *	Renamed MAXNAMLEN as MAXANAMELEN due to name conflict. DC 5/8/94
 *      Increased FNAMELEN by one as per Tom Ploeghmakers and Jeremy
 *        Stark. DC 10/8/94
 *	Added priority fields to index_record and problem_record.
 *	  Other misc. mods. related to priority. DC 11/21/94
 */


#include <sys/param.h>
#include <limits.h>

#include "pts.h"
#include "util.h"

/* Error numbers returned, use the constant names not their values */

#define BADTYPE		256	/* the type passed into a function is not
				 * defined */
#define NOTFOUND	257	/* the problem record asked for was not
				 * found */
#define DOUBLEREP	258	/* the reporter is already in the list of 
				 * reporters */
#define DELETED		259	/* the record has been deleted */
#define ISSOLVED	260	/* the problem has been solved */
#define ISUNSOLVED	261	/*  the problem is UNSOLVED */
#define NOTOWNER        262	/* a request to end an edit has been made by 
				 * someone that did not start it */
#define NOREP		263	/* either there is no reporters file or no
				 * old reporters file */
/* defined in errno.h 
 * ETXTBSY -  the file that was requested for opening is busy.
 * EWOULDBLOCK - the file that was requested to be locked is already locked.
 */
#define EXISTS		264	/* the problem already has been requested for
				 * reopening */
#define LINKERR		265	/* the problem requested for deletion is
				 * an original link */
#define REMOVEERROR	266     /* a record was being removed when an error
				 * occured.  Returned from confirm_reopen()
				 * when the reopen index entry is being
				 * removed. */
#define STRINGERROR     267     /* a string operation failed.  Usually returned
				 * from new_prid(). */
#define ROLLEDOVER	268	/* an index search has gone through the index
				 * once without finding what it was looking 
				 * for. */
#define SAMELEAF	269	/* an attempt to move or link a problem to 
				 * the leaf in which it resides */

#define BEGIN_OF_ZDBM_ERROR_CODES 256/* the lowest value for zdbm error codes */
#define END_ZDBM_ERROR_CODES    269 /* the maximum value for zdbm error codes */

#ifdef NOSTRERROR
extern char *sys_errlist[];
#define strerror(n) (sys_errlist[n])
#endif /*NOSTRERROR*/

/* The following are used in the records for field widths */

#define MAXPRID		13			/* length of a problem ID */
#define MAXANAMELEN	41			/* length of an account name */
#define DBPATHLEN	257			/* length of a database path
						 * including NULL */
#define PADLEN		26			/* the amount of padding needed
						 * in the index records */
#define FNAMELEN	(MAXPRID + 9)
#define INAMELEN	(DBPATHLEN + 6)		/* length of the name of an
						 * index file */

/* PATH_MAX is defined as 1023 in limits.h */
#ifndef PATH_MAX
#define PATH_MAX 1023
#endif

#define COMPLETEPATHLEN (PATH_MAX + 1)		/* length of a complete path */
#define MAXSUMLEN	81			/* length of the short desc. */
#define MAXREC		999999			/* maximum # of records in an
						 * index file */

/* #define MAXHOSTNAMELEN 64 defined in sys/param.h */
#ifndef MAXHOSTNAMELEN
#define MAXHOSTNAMELEN 64
#endif

/* values for parameters */

#define SOLVED		135	/* the problem has been solved */
#define LINKED		134	/* the problem is a link */
#define ORGLINK		139	/* the problem is an original link */
#define UNSOLVED	143	/* problem is unsolved */
#define GONE		133	/* the problem is deleted */
#define PARTICULAR	144	/* want to find a particular problem */
#define START		145	/* start an edit session */
#define END		146	/* end an edit session */
#define DELETEOK	147	/* it is ok to delete a problem */
#define DELETENOTOK	148	/* it is not ok to delete an original link */
#define INDEX           149     /* the file to open is an index file */
#define REP             150     /* the file to open is not an index file */
#define MOVED           151     /* the record has been moved to another leaf */
#define REOPENED	152	/* problem has been requested for reopening */
#define LINKS		  1	/* want to follow links in a problem search */
#define NOLINKS		  0	/* don't want to follow links in a search */
#define FULLRECORD	  1	/* user wants a full problem record from 
				 * get_problem() */
#define PARTIALRECORD	  0	/* user wants a partial problem record from
				 * get_problem() */
#define TRUE              1
#define FALSE             0
#define SUCESS		  1
#define FAIL		  0
#define ERROR		(-1)

#ifndef STD_BLK			/* gives the size of a disk I/O block */
#define STD_BLK		1024	/* our default */
#endif

/* Constant character strings for the reopened directory and temproary
 * file names.
 */

#define REOPENDIR	"REOPEN"	/* the name of the reopened dir. */

#define TEMPFILE	"PTS.XXXXXX"	/* the name of a temporary file used
					 * when deleting an index record from
					 * the REOPEN index. */
#define TEMPORARY	"TPTS.XXXXXX"	/* the name of the temp. file used
					 * while changing (editing) the problem
					 * logs. */

#ifndef PTSROOT				/* the root of the problem tree, if
					 * not defined will be here. */
#define PTSROOT	"/usr/local/lib/PTS/DB/"
#endif


/* This priority stuff was added 11/19/94 by DC */
#define DEF_PRIORITY        0		/* The default priority.  */
#define MAX_PRIORITY	  127		/* maximum. (signed char.)*/
#define MIN_PRIORITY	(-128)		/* minumum. (ditto.)	  */


/* permission flags for files - read, write permission for owner and read
 * permission for group and others (found in sys/stat.h).
 */

#define FILE_PERMS	S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH

/* permission flags for directories - read, write and execute for owner,
 * read and execute for group and other (found in sys/stat.h).
 */

#define DIR_PERMS	S_IRUSR | S_IWUSR | S_IXUSR | S_IRGRP | S_IXGRP\
			| S_IROTH | S_IXOTH

/* a helpful macro, since strncpy() won't append a NULL if s2 is longer
 * than s1, this macro takes care of the problem.
 */

#define nl_strncpy(s1, s2, n)	{strncpy(s1, s2, n); (s1)[n-1]='\0';}

/* structure for an index record, the structure is set up to be 512 bytes
 * in size.  This is an integral number of bytes of a disk block, on the
 * development system (Apollo Domain) the size of a disk block is 1024
 * bytes, so two index records will fit in a disk block.  It is unlikely
 * that this (the size of an index record) can be lowered and still allow
 * the database to function as designed.  It may however be increased
 * in the future to add flexability or more features.
 * It is implied that greater priority values indicate problems of
 * greater importance.  The default priority should be 0.
 */

typedef struct
{
   char prid[MAXPRID];              /* the problem id */
   int  mrd;                        /* problem has been moved, reopened,
				     * linked or deleted */
   int  sf;                         /* problem is SOLVED or UNSOLVED */
   int  ef;                         /* problem is/is not being edited */
   char account_name[MAXANAMELEN];   /* name of the user doing the edit */
   char node_name[MAXHOSTNAMELEN];  /* the machine being used for the edit */
   int  pid;                        /* the process id of the edit */
   char ln_prid[MAXPRID];           /* prid of the original problem, NULL
				     * unless problem has been moved or
				     * linked */
   char ln_path[DBPATHLEN];	    /* the database path to the leaf containing
				     * the original problem */
   char sd[MAXSUMLEN];              /* short description of the problem,
				     * the path to the original problem if
				     * this ones been moved or linked */
   signed char priority;	    /* ...-2, -1, 0, 1, 2..., with 0 default */
   char fr[PADLEN];	            /* used to pad out to 512 bytes and for 
				     * future expansion */
 } index_record;

/* structure for a reporter record */

typedef struct
{
   char account_name[MAXANAMELEN];  /* the name of the reporter */
   char node_name[MAXHOSTNAMELEN]; /* the name of the machine the problem
			            * is on */
   char date_time[MAXANAMELEN];     /* the date and time the report was made */
} reporter_record;

/* structure for a problem record */

typedef struct 
{
   char prid[MAXPRID];			/* the problem id of this problem */
   int  status;				/* the status of the problem, SOLVED,
					 * UNSOLVED or REOPENED */
   signed char priority;		/* Priority, 0 being "normal".*/
   char short_description[MAXSUMLEN];	/* the short description for this 
					 * problem */ 
   int  num_rep;			/* the number of reporter records */
   reporter_record   *reporter_list;	/* an array of reporter records */
   int		     num_old_rep;	/* the number of old reporter records */
   reporter_record   *old_reporter_list; 
   char              *log_file;		/* the log file */
   void		     *reop_rec;		/* points to the reopened record, if
					 * the problem has been reopened, NULL
					 * otherwise */
} problem_record;

/* structure for a reopend record, what is returned by view_reopen() */

typedef struct 
{
   char prid[MAXPRID];         /* the problem id of the reopened record */
   char db_path[DBPATHLEN];    /* the db_path to the problem that is to be 
				* reopened */
   char *reasons_log;	       /* the name of the file containing
				* the reasons */
   reporter_record rep_rec;    /* see above */
   problem_record *pro_rec;    /* the problem record of the reopened 
				* problem */
} reopened_record;

/* structure for the edit information */

typedef struct
{
   int  ef;				/* whether the problem is currently
					 * being edited (TRUE or FALSE) */
   char account_name[MAXANAMELEN];	/* the name of the user doing the
					 * edit */
   char node_name[MAXHOSTNAMELEN];	/* the name of the machine the PTS
					 * is running on */
   int pid;				/* the process id of the PTS running 
					 * the edit */
} edit_record;

/* the only GLOBAL variable, used to hold the error number of an error
 * that occured when the database is accessed.  It is NOT reset if the
 * access was sucessful, only if it failed.
 */

extern int db_errorno;

/* OK, I added one more. The top of the database tree. DC 2/23/94 */
extern char PtsRoot[COMPLETEPATHLEN] ;

/* Database Operations, Function Prototypes */

/* functions in get_problem.c */

problem_record *get_problem(char *, char *, int, int, int, int);
index_record *next_solved(FILE *, int, int, index_record *);
index_record *next_unsolved(FILE *, int, int, index_record *);
index_record *find_problem(FILE *, char *, index_record *);
index_record *search_index(FILE *, char *, index_record *, long);
problem_record *create_record(char *, index_record *, problem_record *);
problem_record *partial_record(char *, index_record *, problem_record *);
index_record *check_index(FILE *, index_record *);
#ifdef DEBUG /* these functions are only used if we are debugging the
	      * database. */
void print_index(index_record *);
void print_prob(problem_record *);

#endif

/* function in update_problem.c */

int solve_problem(char *, char *);
int solve_record(FILE *, index_record *, char *, char *);
int delete_problem(char *, char *, int);
char *add_problem(reporter_record *, char *, char *, char *, char);
char *add_reporter(char *, char *, reporter_record *);
int append_reporter(char *, char *, reporter_record *);
int double_reporter(FILE *, reporter_record *);

/* functions in reopen_problem.c */

int reopen_request(char *, char *, char *, reporter_record *rep_rec);
int double_problem(FILE *, char *, char *);
reopened_record *view_reopen(char *, int);
char *confirm_reopen(char *, reporter_record *, char *, char *);
int deny_reopen(char *);
int delete_record(char *);
index_record *find_reopen(FILE *, index_record *, char *);
int bad_problem(index_record *, FILE *);
int is_reopened(char *, char *);
 
/* functions in move_problem.c */

int move_problem(char *, char *, char *);
char *move_record(char *, char *, index_record *);
int link_problem(char *, char *, char *);

/* functions in edit_problem.c */

int edit_log(char *, char *, char *, int, signed char, char *);/*mod11/23/94DC*/
int remove_edit(char *, char *);
edit_record *edit_info(char *, char *);
int lock_for_edit(FILE *, index_record *);
int unlock_from_edit(FILE *, index_record *, char *, char *,
		     signed char, char*); /*mod 11/23/94 DC*/

/* functions in zdbm_io.c */

int exists(char *, char *, char *, char *);
int append(char *, char *, char *, char *, char *);
FILE *open_file(char *, char *, char *, int);
long file_size(FILE *fp, long record_size); /*Modified, 11/20/93, RDC */
int lock_file(FILE *);
int unlock_file(FILE *);
int make_branch(char *);
FILE *create_file(char *, char *, char *, int);
int remove_file(char *, char *, char *);
char *new_prid(long);
int trunc_file(char *, char *, long);
int rename_file(char *, char *, char *, char *, char *, char *);
int copy_file(char *, char *, char *, char *, char *, char *);
int write_log(char *, char *, char *);
char *zdbm_strerror(int);

