/* problems.c - By Luke Sheneman.  1992 */

/* Modified: 2/14/93, DC.
 * Added FULLRECORD/PARTIALRECORD field to all view_reopen() and
 * get_problem() calls.
 *
 * Modified: 2/12/94, Dean Collins.
 * Modifications recommended by magnush@isy.liu.se (Magnus Hammerin).
 * Changed mktemp() usage (added temp strings since mktemp overwrites
 * it's argument.)  Added one byte to a malloc that was misbehaving.
 *
 * Modified: 2/12/94, Dean Collins.
 * Changed definition of filename[] strings in several functions.
 * Were defined as FNAMELEN, should have been COMPLETEPATHLEN in size.
 *
 * Modified: Dean Collins Thu Feb 24 10:18:51 1994
 * Fixed problem with PrintProblem() not knowing about old_reporter_list.
 * Made changes having to do with mail.
 * 
 * Modified: Dean Collins Sat Mar 12 17:23:24 1994
 * Changed to use SendMail() rather than call Mailer directly.
 *
 * Modified: Dean Collins Sun Mar 13 01:59:42 1994
 * Replaced MAXNAME with MAXANAMELEN from zdbm.h.
 *
 * Modified: Dean Collins Fri Apr 08 18:58:38 1994
 * Fixed problem with printing REOPENED problems.  Printing should
 * work correctly for all three cases.
 *
 * Modified: Dean Collins Wed Nov 16 12:32:42 PST 1994
 * Added priority to WriteProblem().
 *
 * Modified: Dean Collins Mon Jan  2 16:39:12 PST 1995
 * Changed all printfs in DEBUG sections to fprintf(stderr...).
 *
 * Modified: Dean Collins Sun May 21 13:59:00 PDT 1995
 * Changed argument 4 of all calls to get_problem() from NULL to 0.
 * It expects an int, and NULL is often defined as a pointer.
 * Changed numerous return(NULL)s to return(False) or return(0)
 * as apppropriate for the same reason.
 *
 * Modified: Dean Collins Sat May 27 19:56:03 PDT 1995
 * Added ReportAgainPRID(), which is like ReportAgain() except
 * it takes a different parameter list.  Added ReadProblemPRID()
 * which, similarly, is just like ReadProblem() but takes a
 * different parameter list.
 */

/*
 * Copyright (c) 1995,1994,1993 Dean Collins.
 * Copyright (c) 1992 Luke Sheneman.
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the names of the University of Idaho
 * Dean Collins and Luke Sheneman not be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission.  The University of Idaho, Luke Sheneman and Dean Collins
 * make no representations about the suitability of this software for
 * any purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO, LUKE SHENEMAN, AND DEAN COLLINS  DISCLAIM ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY OF IDAHO
 * LUKE SHENEMAN OR DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */


/*
 * 
 *	problem.c  - this source file includes the following functions:
 *
 *	FormatProblemTypes() - Returns an array of strings containing the 
 *		paths which can be followed when traversing the problem tree
 *		at that point in the tree.
 *      ReadProblem() - Reads a problem from the database.  Designed for
 *		use with the X user interface, which accounts for it's
 *		odd parameter list.  See ReadProblemPRID() for another
 *		way to accomplish the same thing.
 *	ReadProblemPRID() - Just like ReadProblem, but with different args.
 *	LockProblem() and UnlockProblem() - Used to insure that only one person
 * 		is editing a problem at a time.  Lock needs to return success
 *		or failure if it can/can't lock a file (after a couple of 
 * 		seconds).
 *	SolveProblem() - Marks a problem as solved.
 *	ReopenProblem() - Records a request by a user to reopen a solved 
 * 		problem.
 *	ReopenRequest() - Marks a problem as being requested by a user for 
 *		reopening.
 *	DenyReopen() - Sends a "reason text" message to the user who requested
 * 		a problem be reopened.
 * 	ReportAgain() - Allows a user to add themselves to the list of reporters
 *		for an unsolved problem.
 *	ReportAgainPRID() - Like ReportAgain() only it takes a different
 *		parameter list.
 *	DeleteProblem() - Removes a problem from the database.
 *	MoveProblem() - Moves a problem from one location to another in the 
 * 		problem tree.
 *	LinkProblem() - Makes a problem appear to be in two places in the 
 * 		problem tree at once.
 *	PrintProblem() - Print a problem.
 * 	WriteProblem() - Creates a new problem entry in the problem database.
 *	AppendProblem() - Appends a string to an existing problem log.
 * 	IsSysop() - determines if the effective group id of the user places
 *		that user in the SYSOPGID (the group of sysops)
 *	CopyProblemRecord() - Copies a problem record recieved from the 
 *		zdbm functions into a newly allocated problem record.
 *	FixTime() - Removes the offending newline from calling ctime().
 *	IsReopened() - Calls a zdbm function to see if a desired problem is 
 *   		requested to be reopened or not.
 */

#include <sys/types.h>
#include <stdio.h>
#include "zdbm.h"
#include "cloud.h"
#include "clouderror.h"


	/* The command to mail files will be stored here. DC 2/24/94 */
char Mailer[COMPLETEPATHLEN]=MAILER ;

	/* The command-line switch used to indicate the subject of
	 * a mail message will be stored here. DC 2/24/94 
	 */
char MailerOpts[COMPLETEPATHLEN]=MAILER_OPTS ;

	/* If not NULL, this is a NULL-terminated list of sysops.
	 * If empty (NULL) use GID of user. 
	 **/
StringListRec *SysopList=NULL ;





int FormatProblemTypes(tree,path,problem,control,types)
ProblemTree *tree;
char *path;
int problem;
int control;
char *types[];
{
	int loop;
	char newpath[DBPATHLEN];

	cld_errno=0;

	zdebug1("Path before fixing in FormatProblemTypes() = %s\n",path);

	newpath[0]='\0';
	strcpy(newpath,FixPath(path));
	if(!strcmp(newpath,""))
		sprintf(path,"ROOT");
	else
		sprintf(path,"ROOT/%s",newpath);

	zdebug1("Path after fixing in FormatProblemTypes() = %s\n",path);
	zdebug1("PATH IN FORMATPROBLEMTYPES() = ***%s***\n",path);

        switch(control)
        {
                case ABS_TYPE:
		{	
			ProblemTree *subhead,*t;

			zdebug3("Just before call to SubTree() tree=%p path=%s pathptr=%p\n",tree,path,path);

			t=SubTree(tree,path);

			zdebug("Just AFTER call to SubTree()\n");

			if(!t)
			{
				cld_errno=CLD_ERR10;
				return(-1);
			}
			else
			{

				subhead=t->subhead;
				if(!subhead)
					return(0);

				loop=0;
				while( (subhead) && (loop<MAXBRANCHES) )
				{
					strcpy(types[loop],subhead->path);
					subhead=subhead->next;	

					zdebug1("-->%s\n",types[loop]) ;

					loop++;
				}
				types[loop][0]='\0';
			}	
                        break;
		}
                case INC_TYPE:
		{
			ProblemTree *subhead,*t;	
			int pos;

			t=SubTree(tree,path);
			if(!t)
			{
				cld_errno=CLD_ERR10;
				return(-1);
			}
			else
			{
				subhead=t->subhead;
				if(!subhead)
				{
					cld_errno=CLD_ERR11;
					return(-1);
				}
				pos=0;
				loop=0;
				while( (subhead) && (pos!=problem) )
				{
					subhead=subhead->next;
					pos++;
				}
				if(!subhead)	/* desired subhead not found */
				{
					cld_errno=CLD_ERR12;
					return(-1);			
				}

				strcpy(path,ROOTSTR);
				strcat(path,subhead->fullpath);
				if(pos<problem)
				{
					cld_errno=CLD_ERR13;
					return(-1);
				}
				else
				{
					subhead=subhead->subhead;
					if(!subhead)
						return(0);	/* NO CHILDREN */
					while( (subhead) && (loop<MAXBRANCHES) )
					{
						strcpy(types[loop],subhead->path);
						subhead=subhead->next;
						loop++;
					}
					types[loop][0]='\0';
				}
			}

                        break;
		}
                case PREV_TYPE:
		{
			ProblemTree *subhead,*t;
			char *parts[MAXBRANCHES];
			char parent[DBPATHLEN];
			int end;


			BreakUpPath(path,parts);	
			end=0;
			while(strcmp(parts[end++],""));
			parent[0]='\0';
			for(loop=0;loop<end-2;loop++)
				sprintf(parent,"%s/%s",parent,parts[loop]);	
			if(!strcmp(parent,"/ROOT"))			/*  KLUDGE-O-RAMA  */
				strcpy(parent,ROOTSTR);
			strcpy(path,parent);
			if(!strcmp(path,""))
			{
				FreeParts(parts);
				return(0);	/* no previous types - at ROOT */
			}

			t=SubTree(tree,path);
			if(!t)
			{
				cld_errno=CLD_ERR10;
				FreeParts(parts);
				return(-1);
			}
			else
			{
				subhead=t->subhead;
				if(!subhead)
				{
					FreeParts(parts);
					return(0);	/* NO CHILDREN */
				}
				loop=0;
				while( (subhead) && (loop<MAXBRANCHES) )
				{
					strcpy(types[loop],subhead->path);
					subhead=subhead->next;	
					loop++;
				}
				types[loop][0]='\0';
			}	
			
                        break;
		}
                default:
		{
			cld_errno=CLD_ERR13;	
			return(-1);
		}
        }
	return(True);
}



problem_record *ReadProblem(tree,path,problem,summarylist)
ProblemTree *tree;
char *path;
int problem;
Summary *summarylist;
{
	Summary *ts;
	int counter;
	problem_record *res;
	static problem_record *realres=NULL;
	char newpath[DBPATHLEN];
	int loop;

	cld_errno=0;

	zdebug2("In ReadProblem.  path=%s   problem#=%d\n",path,problem);

	strcpy(newpath,FixPath(path));

#ifdef DEBUG
	ts=summarylist;
	zdebug("******  Traversing summarylist.\n");
	while(ts)
	{
		zdebug1("SUMMARY: PRID=%s\n",ts->prid);
		zdebug1("	 PATH=%s\n",ts->path);
		zdebug1("      SUMMARY=%s\n",ts->summary);
		zdebug1("      STATUS=%d\n",ts->status);
		ts=ts->next;
	}
#endif /* DEBUG */

	counter=0;
	ts=summarylist;
	while( (ts) && (counter<problem) )
	{
#ifdef DEBUG
		zdebug1("In while() loop. Traversing summarylist.  ts->summary=%s\n",ts->summary); 
#endif /* DEBUG */
		counter++;
		ts=ts->next;
	}
	if(!ts)
	{
		cld_errno=CLD_ERR14;	
		return(NULL);	/* ERROR! */
	}

	res=get_problem(newpath,ts->prid,PARTICULAR,0,LINKS,FULLRECORD);/*Modified: 2/14/93, DC*/
	if(!res)
	{
		cld_errno=CLD_ERR15;	
		return(NULL);	
	}
	else
	{
		if(realres)
		{
			zdebug("In ReadProblem() - About to free a problem record.\n");

			FreeProblemRecord(realres);
		}
		realres=(problem_record *)malloc(sizeof(problem_record));
		CopyProblemRecord(realres,res);
		return(realres);
	}
}


	/* Just like ReadProblem() only it takes a different param. list */
problem_record *ReadProblemPRID(path,prid)
char *path;
char *prid;
{
	problem_record *res=NULL ;
	static problem_record *realres=NULL;

	res = get_problem(FixPath(path),prid,PARTICULAR,0,LINKS,FULLRECORD) ;
	if(!res)
	{
		cld_errno=CLD_ERR15;
		return(NULL);
	}
	else
	{
		if(realres)
		{
			zdebug("In ReadProblemPRID() - About to free a problem record.\n");

			FreeProblemRecord(realres);
		}
		realres=(problem_record *)malloc(sizeof(problem_record));
		CopyProblemRecord(realres,res);
		return(realres);
	}
}



int LockProblem(path,problem)
char *path;
problem_record *problem;
{
	int res;
	char newpath[DBPATHLEN];

	cld_errno=0;

	zdebug1("The path at beginning of LockProblem() = %s\n",path);

	strcpy(newpath,FixPath(path));

	zdebug1(" In LockProblem() - newpath = %s\n",newpath);

	res=edit_log(newpath,problem->prid,NULL,START,0,NULL);
	if(!res)
	{
		cld_errno=CLD_ERR16;
		return(False);
	}
	else
		return(True);
}




int UnlockProblem(path,problem)
char *path;
problem_record *problem;
{
	int res;
	char newpath[DBPATHLEN];

	cld_errno=0;

	strcpy(newpath,FixPath(path));

	zdebug1(" In UnlockProblem() - newpath = %s\n",newpath);

	res=edit_log(newpath,problem->prid,NULL,END,problem->priority,problem->short_description);
	if(!res)
	{
		cld_errno=CLD_ERR17;
		return(False);
	}
	else
		return(True);
}





int SolveProblem(tree,path,problem,solution,userinfo)
ProblemTree *tree;
char *path;
problem_record *problem;
char *solution;
UserInfo *userinfo;
{
	int res,loop;
	char newpath[DBPATHLEN];
	char filename[COMPLETEPATHLEN];   /*Temp filename*/
	char tfilename[COMPLETEPATHLEN];  /*Temp filename for mktemp() use.*/
					  /*added DC 2/12/94 */
	char reporter[MAXANAMELEN];
	char node[MAXHOSTNAMELEN];
	FILE *fp;

	cld_errno=0;

	zdebug("At very beginning of SolveProblem()\n");

	strcpy(newpath,FixPath(path));

	zdebug1("In SolveProblem() - newpath = %s\n",newpath);

	res=AppendProblem(tree,path,problem,userinfo,solution);

	zdebug1("In SolveProblem(), just returned from AppendProblem()(res=%d)\n",res) ;

	res=solve_problem(newpath,problem->prid);
	if(!res)  
	{
		cld_errno=CLD_ERR18;
		return(False);
	}
	else
	{
		zdebug("In SolveProblem(), just returned from solve_problem()\n") ;

		for (loop=0; loop < problem->num_rep; loop++)
		{
			strcpy(tfilename, TEMPLATE) ; /*DC,2/12/94*/
			strcpy(filename,(char *)mktemp(tfilename)); /*DC,2/12/94*/

			fp=fopen(filename,"w");
			if(!fp)
			{	
				cld_errno=CLD_ERR21;	/* fix this */
				return(False);
			}
			else
			{
				strcpy(reporter,((problem->reporter_list)[loop]).account_name);
				strcpy(node,((problem->reporter_list)[loop]).node_name);

				fprintf(fp,"Dear %s,\n\n",reporter);
				fprintf(fp,"	A problem in the PTS database has been solved in which you were\n");
				fprintf(fp,"listed as a reporter.  You are hereby being notified that this problem has\n");	
				fprintf(fp,"been solved by the support staff.\n\n");
				fprintf(fp,"The problem log for this problem is as follows:\n\n");
				fprintf(fp,"_____________________________________________________________________\n");
				fprintf(fp,"%s\n\n",problem->log_file);
				fprintf(fp,"_____________________________________________________________________\n");
				fprintf(fp,"Thank you for your support.\n\n");
				fclose(fp);
				SendMail(Mailer,MailerOpts,SOLVED_SUBJ,reporter,node,filename); /* Added 3/12/24 DC */
				unlink(filename);
			}
		}
		return(True);
	}
}



problem_record *ReopenProblem(tree,path,problem,sysop_string)
ProblemTree *tree;
char *path;
problem_record *problem;
char *sysop_string;
{
	char *prid;	
	problem_record *res;
	static problem_record *realres=NULL;

	cld_errno=0;
	
	prid=confirm_reopen(((reopened_record *)(problem->reop_rec))->prid,
		&(((reopened_record *)(problem->reop_rec))->
		rep_rec),sysop_string,((reopened_record *)(problem->reop_rec))->
		reasons_log);

	if(!prid)
	{
		cld_errno=CLD_ERR19;
		return(NULL);
	}
	else
	{
		res=get_problem( ((reopened_record *)(problem->reop_rec))->db_path,prid,PARTICULAR,
			0,LINKS,FULLRECORD);/*Modified: 2/14/93, DC*/
		if(!res)
		{
			cld_errno=CLD_ERR35;
			return(NULL);
		}
		else
		{
			if(realres)
				FreeProblemRecord(realres);
			realres=(problem_record *)malloc(sizeof(problem_record));
			CopyProblemRecord(realres,res);
			return(realres);
		}
	}
}



int ReopenRequest(path,problem,reason,user)
char *path;
problem_record *problem;
char *reason;
UserInfo *user;
{
	int res,t;
	reporter_record report;
	char hostname[MAXHOSTNAMELEN];
	char newpath[DBPATHLEN];

	cld_errno=0;

	strcpy(newpath,FixPath(path));

	zdebug1("In ReopenRequest: newpath=%s\n",newpath);

	gethostname(hostname,MAXHOSTNAMELEN);
	strcpy(report.node_name,hostname);
	strcpy(report.account_name,user->username);
	t=time(0);
	strcpy(report.date_time,FixTime((char *)ctime((time_t *)&t)));

	res=reopen_request(newpath,problem->prid,reason,&report);
	if(!res)
	{
		cld_errno=CLD_ERR20;
		return(False);
	}
	else
		return(True);
}





int DenyReopen(tree,path,problem,user,deny)
ProblemTree *tree;
char *path;
problem_record *problem;
UserInfo *user;
char *deny;
{
	char username[MAXANAMELEN];
	char realname[MAXANAMELEN];
	char reporter[MAXANAMELEN];
        char node[MAXHOSTNAMELEN];
	/*char filename[FNAMELEN];*/
	char filename[COMPLETEPATHLEN];	  /*Temporary filename, DC 2/12/94 */
	char tfilename[COMPLETEPATHLEN];  /*Temp filename for mktemp() use.*/
					  /*added DC 2/12/94 */
	char sys[100];
	FILE *fp;
	int res;
	
	cld_errno=0;

	strcpy(username,user->username);
	strcpy(realname,user->realname);
	strcpy(reporter,((reopened_record *)(problem->reop_rec))->rep_rec.account_name);
        strcpy(node,    ((reopened_record *)(problem->reop_rec))->rep_rec.node_name);

	strcpy(tfilename, TEMPLATE) ; /*DC,2/12/94*/
	strcpy(filename,(char *)mktemp(tfilename)); /*DC,2/12/94 */


	fp=fopen(filename,"w");
	if(!fp)
	{
		cld_errno=CLD_ERR21;	
		return(False);
	}
	else
	{
		fprintf(fp,"Dear %s,\n\n",reporter);
		fprintf(fp,"      Your request to reopen a problem in the XPTS database was denied.\n\n");
		fprintf(fp,"Request Denied By: %s  (%s)\n",username,realname);
		fprintf(fp,"Reason For Denial:\n\n\n");
		fprintf(fp,"%s",deny);
		fflush(fp);
                SendMail(Mailer, MailerOpts, REOPEN_SUBJ, reporter,node,
                         filename); /* Added 3/12/24 DC */

		fclose(fp);
		unlink(filename);	
		
		res=deny_reopen(((reopened_record *)(problem->reop_rec))->prid);
		if(!res)
		{
			cld_errno=CLD_ERR22;
			return(False);
		}
		else
			return(True);
	}
}





problem_record *ReportAgain(path,userinfo,problem)
char *path;
UserInfo *userinfo;
problem_record *problem;
{
	int res;
	reporter_record report;
	static problem_record *realres=NULL;
	problem_record *problem_rec;
        char hostname[MAXHOSTNAMELEN];
	char newpath[DBPATHLEN];
	int t;

	cld_errno=0;
	
	strcpy(newpath,FixPath(path));

	zdebug1("ReportAgain() - newpath = %s\n",newpath);

        gethostname(hostname,MAXHOSTNAMELEN);
        strcpy(report.node_name,hostname);
        strcpy(report.account_name,userinfo->username);
        t=time(0);
        strcpy(report.date_time,FixTime((char *)ctime((time_t *)&t)));

	res=(int)add_reporter(newpath,problem->prid,&report);
	if(!res)
	{
		cld_errno=CLD_ERR23;		
		return(NULL);
	}
	else
	{
		problem_rec=get_problem(newpath,problem->prid,PARTICULAR,0,LINKS,FULLRECORD);/*Modified: 2/14/93, DC*/
		if(!problem_rec)
		{
			cld_errno=CLD_ERR23;
			return(NULL);
		}
		else
		{
			if(realres)
				FreeProblemRecord(realres);	
			realres=(problem_record *)malloc(sizeof(problem_record));
			CopyProblemRecord(realres,problem_rec);
			return(realres);
		}
	}
}



	/* Almost the same as ReportAgain() above, only different. */
int ReportAgainPRID(path,prid,email,hostname)
char *path;
char *prid;
char *email;
char *hostname;
{
	int res;
	reporter_record report;
	static problem_record *realres=NULL;
	problem_record *problem_rec;
	char newpath[DBPATHLEN];
	int t;

	cld_errno=0;

	strcpy(newpath,FixPath(path));

	zdebug1("ReportAgainPRID() - newpath = %s\n",newpath);

	strcpy(report.node_name,hostname);
	strcpy(report.account_name,email);
	t=time(0);
	strcpy(report.date_time,FixTime((char *)ctime((time_t *)&t)));

	res=(int)add_reporter(newpath,prid,&report);
	if(!res)
	{
		cld_errno=CLD_ERR23;
		return(0);
	}
	else
	{
		problem_rec=get_problem(newpath,prid,PARTICULAR,0,LINKS,FULLRECORD);
		if(!problem_rec)
		{
			cld_errno=CLD_ERR23;
			return(0);
		}
		else
		{
			if(realres)
				FreeProblemRecord(realres);
			realres=(problem_record *)malloc(sizeof(problem_record)) ;
			CopyProblemRecord(realres,problem_rec);
/*		      return(realres); */
			return(1) ;
		}
	}
}

		

int DeleteProblem(tree,path,problem)
ProblemTree *tree;
char *path;
problem_record *problem;
{
	int res;
	char newpath[DBPATHLEN];

	cld_errno=0;

	strcpy(newpath,FixPath(path));

	zdebug1("DeleteProblem() newpath = %s\n",newpath);

	res=delete_problem(newpath,problem->prid,DELETEOK);
	if(!res)
	{
		cld_errno=CLD_ERR24;
		return(False);
	}
	else
		return(True);
}



int MoveProblem(tree,path1,problem,path2)
ProblemTree *tree;
char *path1;
problem_record *problem;
char *path2;
{
	int res;
	char newpath1[DBPATHLEN],newpath2[DBPATHLEN];
	char temp_path[DBPATHLEN];
	ProblemTree *temp_sub_tree;

	cld_errno=0;

	strcpy(newpath1,FixPath(path1));/* Make Sure Paths Are Usable. */
	strcpy(newpath2,FixPath(path2));


	sprintf(temp_path,"ROOT/%s",newpath1);	/* Check To See If Paths Are Valid. */
	temp_sub_tree=SubTree(tree,temp_path);
	if( (!temp_sub_tree) || (temp_sub_tree->subhead) )
	{
		cld_errno=CLD_ERR25;		
		return(False);
	}

	sprintf(temp_path,"ROOT/%s",newpath2);
	temp_sub_tree=SubTree(tree,temp_path);
	if( (!temp_sub_tree) || (temp_sub_tree->subhead) )
	{
		cld_errno=CLD_ERR26;
		return(False);
	}


	zdebug2("This is in MoveProblem() - path1=%s  path2=%s\n",newpath1,newpath2);

	res=move_problem(newpath1,newpath2,problem->prid);
	if(!res)	
	{
		cld_errno=CLD_ERR27;
		return(False);
	}
	else
		return(True);
}



int LinkProblem(tree,path1,problem,path2)
ProblemTree *tree;
char *path1;
problem_record *problem;
char *path2;
{
	int res;
	char newpath1[DBPATHLEN],newpath2[DBPATHLEN];
	char temp_path[DBPATHLEN];
	ProblemTree *temp_sub_tree;

	cld_errno=0;

	strcpy(newpath1,FixPath(path1));
	strcpy(newpath2,FixPath(path2));


	sprintf(temp_path,"ROOT/%s",newpath1);	/* Check To See If Paths Are Valid. */
	temp_sub_tree=SubTree(tree,temp_path);
	if( (!temp_sub_tree) || (temp_sub_tree->subhead) )
	{
		cld_errno=CLD_ERR28;
		return(False);
	}

	sprintf(temp_path,"ROOT/%s",newpath2);
	temp_sub_tree=SubTree(tree,temp_path);
	if( (!temp_sub_tree) || (temp_sub_tree->subhead) )
	{
		cld_errno=CLD_ERR29;
		return(False);
	}

	zdebug2("This is in LinkProblem() - path1=%s  path2=%s\n",newpath1,newpath2);

	res=link_problem(newpath1,newpath2,problem->prid);
	if(!res)
	{
		cld_errno=CLD_ERR30;
		return(False);
	}
	else
		return(True);
}




int PrintProblem(tree,path,problem)
ProblemTree *tree;
char *path;
problem_record *problem;
{
	/*char filename[FNAMELEN];*/
	char filename[COMPLETEPATHLEN];   /*Temp filename DC 2/12/94 */
	char tfilename[COMPLETEPATHLEN];  /*Temp filename for mktemp() use.*/
					  /*added DC 2/12/94 */
	char sys[100];
	FILE *fp;
	reporter_record *reporter;
	int i,j,buflen;
	char newpath[DBPATHLEN];
	int maxname=MAXANAMELEN-8;
	int numrep=0 ; 		/* Number of reporters in reporter list */
	reopened_record *reop_rec ;   /*The reopened record.  Added 4/8/94 DC*/
	reporter_record rep_rec ;     /*A reporter record.  Added 4/8/94 DC*/

	zdebug("Entering PrintProblem()\n") ;

	cld_errno=0;

	strcpy(newpath,FixPath(path));

	strcpy(tfilename, TEMPLATE) ; /*DC,2/12/94*/
	strcpy(filename,(char *)mktemp(tfilename)); /*DC,2/12/94 */

	fp=fopen(filename,"w");
	if(!fp)
	{
		cld_errno=CLD_ERR31;
		zdebug("Leaving PrintProblem()\n") ;
		return(False);
	}
	else
	{
		fprintf(fp,"PROBLEM TYPE:\t%s\n",newpath);
		fprintf(fp,"STATUS:\t\t");
		switch(problem->status)
		{
			case UNSOLVED:
				fprintf(fp,"Unsolved\n");
				break;
			case SOLVED:
				fprintf(fp,"Solved\n");
				break;
			case REOPENED:
				fprintf(fp,"Reopen Request\n");
				break;
			default:
				break;
		}
		
#ifdef DO_PRIORITY
		fprintf(fp, "PRIORITY:\t") ;            /* DC 11/21/94 */
		fprintf(fp, sctoa(problem->priority)) ;
#endif


		fprintf(fp,"\n\nSUMMARY:\n");
		fprintf(fp,"%s\n\n",problem->short_description);
		
		fprintf(fp,"%-*s%-*s%s\n",maxname,"REPORTER",FNAMELEN,"NODE","DATE/TIME");
		fprintf(fp,"%-*s%-*s%s\n",maxname,"--------",FNAMELEN,"----","---------");


			/* switch added 3/18/94 DC */
		switch(problem->status)
		{
			case UNSOLVED:
				reporter = problem->reporter_list;
				numrep   = problem->num_rep;
				break;
			case SOLVED:
				reporter = problem->old_reporter_list;
				numrep   = problem->num_old_rep ;
				break;
			case REOPENED:
				/* This is rather convoluted. what's new. DC.*/
				reop_rec = problem->reop_rec ;
				rep_rec  = reop_rec->rep_rec ;
				reporter = &rep_rec ;
				numrep   = 1 ;
				break;
			default:
				break;
		}

		zdebug2("PrintProblem() -- reporter=%p, numrep=%d\n",
			reporter,numrep) ;

		if(reporter)
		{
			for(i=0;i<numrep;++i)
			{
				fprintf(fp,"%s",(reporter[i]).account_name);
				buflen=strlen((reporter[i]).account_name);
				for(j=buflen;j<maxname;++j)
					fprintf(fp," ");

				fprintf(fp,"%s",(reporter[i]).node_name);
				buflen=strlen((reporter[i]).node_name);
				for(j=buflen;j<FNAMELEN;++j)
					fprintf(fp," ");

				fprintf(fp,"%s",(reporter[i]).date_time);	

				fprintf(fp,"\n");
			}
		}
		fprintf(fp,"\n");

		fprintf(fp,"PROBLEM DESCRIPTION:\n");
		fprintf(fp,"%s",problem->log_file);
		fprintf(fp,"\n"); 
					

		fclose(fp);
		sprintf(sys,"%s %s %s",PrinterCmd,PrinterOpts,filename);
		system(sys);
		unlink(filename);

		zdebug("Leaving PrintProblem()\n") ;

		return(True);
	}
}



int WriteProblem(tree,path,userinfo,description,summary,node,priority)
ProblemTree *tree;
char *path;
UserInfo *userinfo;
char *description;
char *summary;
char *node;
char priority; /* 11/16/94 DC*/
{
	int res,t;
        reporter_record report;
	char newpath[DBPATHLEN];

	cld_errno=0;

	strcpy(newpath,FixPath(path));

	zdebug1("In WriteProblem().  newpath=%s\n",newpath);

        strcpy(report.node_name,node);
        strcpy(report.account_name,userinfo->username);
        t=time(0);
        strcpy(report.date_time,FixTime((char *)ctime((time_t *)&t)));

	zdebug1("path=%s in WriteProblem() after FixPath() call\n",path);
	zdebug1("newpath=%s in WriteProblem() after FixPath() call\n",newpath);

	res=(int)add_problem(&report,description,summary,newpath,priority);
	if(!res)
	{
		cld_errno=CLD_ERR32;	
		return(False);
	}
	else
		return(True);
}






int AppendProblem(tree,path,problem,userinfo,text)
ProblemTree *tree;
char *path;
problem_record *problem;
UserInfo *userinfo;
char *text;
{
	int res;
	char newpath[DBPATHLEN];

	cld_errno=0;

	strcpy(newpath,FixPath(path));

	zdebug1("AppendProblem() - newpath = %s\n",newpath);

	res=edit_log(newpath,problem->prid,text,END,problem->priority,problem->short_description);

	zdebug1("AppendProblem() - res = %d\n",res);

	if(!res)
	{
		cld_errno=CLD_ERR33;	
		return(False);
	}
	else
		return(True);

}






/* IsSysop looks at the UserInfo structure that was passed to it from the ui.
   It must be called with a UserInfo structure that has already been 
   initialized by calling Setup().  IsSysop returns a positive integer if 
   the user's effective group id (gid) is currently the same as the sysop's, 
   as defined in the constant SYSOPGID.  If the user's effective group id
   is NOT the SYSOPGID, then a zero (0) is passed back.

   Luke Sheneman.  	Wed Jul 22 12:02:22 PDT 1992
*/

int IsSysop(userinfo)
UserInfo *userinfo;
{
	cld_errno=0;
	return( userinfo->sysopflag );
}





/* Expects realres to be allocated, and res to be allocated and filled  */

void CopyProblemRecord(realres,res)
problem_record *realres;
problem_record *res;
{
	int loop;

	cld_errno=0;

	strcpy(realres->prid,res->prid);
	strcpy(realres->short_description,res->short_description);
	realres->num_rep=res->num_rep;

	realres->reporter_list=(reporter_record *)malloc(sizeof(reporter_record)*realres->num_rep);
	for(loop=0;loop<realres->num_rep;loop++)
	{
		strcpy(realres->reporter_list[loop].account_name,res->reporter_list[loop].account_name);
		strcpy(realres->reporter_list[loop].node_name,res->reporter_list[loop].node_name);
		strcpy(realres->reporter_list[loop].date_time,res->reporter_list[loop].date_time);
	}

	realres->num_old_rep=res->num_old_rep;
	realres->old_reporter_list=(reporter_record *)malloc(sizeof(reporter_record)*realres->num_old_rep);
	for(loop=0;loop<realres->num_old_rep;loop++)
	{
		strcpy(realres->old_reporter_list[loop].account_name,res->old_reporter_list[loop].account_name);
		strcpy(realres->old_reporter_list[loop].node_name,res->old_reporter_list[loop].node_name);
		strcpy(realres->old_reporter_list[loop].date_time,res->old_reporter_list[loop].date_time);
	}

	realres->log_file=(char *)malloc(sizeof(char)*strlen(res->log_file)+1); /*DC,2/12/94*/
	strcpy(realres->log_file,res->log_file);

	realres->status=res->status;
	realres->priority=res->priority;	/* DC 11/21/94 */
}


char *FixTime(timestr)
char *timestr;
{
	cld_errno=0;

	timestr[strlen(timestr)-1]='\0';
	return(timestr);
}


int IsReopened(path,problem)
char *path;
problem_record *problem;
{
	cld_errno=0;
	return(is_reopened(path,problem->prid));
}


void FreeProblemRecord(rec)
problem_record *rec;
{
	int loop;

	if(!rec)
		return;

	zdebug("In FreeProblemRecord()\n");

/* This needs to work in order to free this properly.  */

/*
	for(loop=0;loop<(rec->num_rep);loop++)
		if( (rec->reporter_list)[loop] )
			free( (rec->reporter_list)[loop] );
*/

	zdebug("In FreeProblemRecord().  Just freed reporter's list.\n");

/*
	for(loop=0;loop<(rec->num_old_rep);loop++)
		if(((rec->old_reporter_list)[loop])!=NULL)
			free( (rec->old_reporter_list)[loop] );
*/

	zdebug("In FreeProblemRecord().  Just freed old reporter's list.\n");

	if(rec)
		free(rec);

	zdebug("In FreeProblemRecord().  Just freed the reporter record.\n");

}
