/* newprob.c -- command-line utility to add a problem to the PTS database.
 * Dean Collins
 * created: Sat Jun 11 19:43:46 1994
 *
 * Modified 10/8/94, DC
 *	Added initialization for newprob.next pointer, as recommended
 *	by Torsten Lierman.  Added if() check to stop calling strcpy()
 *	and strlen() with a NULL pointer, as recommended by Oliver Schersand.
 */

/*
/* Copyright (c) 1995,1994 Dean Collins
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notices appear in all copies and
 * that both those copyright notices and this permission notice appear in
 * supporting documentation, and that the name of Dean Collins not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  Dean Collins makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 *
 * DEAN COLLINS DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 * NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION
 * WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdlib.h>		/* General C utilities header file */
#include <stdio.h>              /* Standard input/output utilities hdr. file */
#include <sys/param.h>          /* System parameters header file */
#include "pts.h"
#include "zdbm.h"               /* Zombie Database Manager header file */
#include "cloud.h"              /* Nebulous Cloud header file */
#include "clouderror.h"         /* Nebulous Cloud error rtn. header file */
#include "config.h"             /* Config. file parser header file */
#include "patchlevel.h"		/* For version info. */


	/* Needed for or getopt() */
#include <unistd.h>
extern char *optarg;
/*extern int optind, opterr, optopt;*/


	/* Customize these prompts as desired. */
#ifndef ENTER_PATH_PROMPT
#define ENTER_PATH_PROMPT	"Problem tree path: "
#endif /*ENTER_PATH_PROMPT*/

#ifndef ENTER_SUMMARY_PROMPT
#define ENTER_SUMMARY_PROMPT	"One-line summary: "
#endif /*ENTER_SUMMARY_PROMPT*/

#ifndef ENTER_DESC_PROMPT
#define ENTER_DESC_PROMPT	\
"Enter a full problem description (Ctrl-D to complete, Ctrl-C to cancel):\n"
#endif /*ENTER_DESC_PROMPT*/

/* The maximum # of characters on a line of input or in a single
 * command-line option argument.  Really shouldn't be more that about
 * 80 characters or so.
 */
#ifndef MAXLINELEN
#define MAXLINELEN 1024
#endif /*MAXLINELEN*/


/******************** MODULES *************************************/
/*
______________________________________________________________________
main()

FUNCTION DESCRIPTION:

	This function is the main routine for the newprob utility.
It allows the user to add a problem to the database via command-line
options.  Any values not provided by the user on the command-line will
be prompted for.
	(There are entirely too many exit() calls in this function.)
______________________________________________________________________
UNIT TESTING:
     This function has been tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  6/11/94
     Modified: Dean Collins	Date:  10/8/94
	Added if() check to stop doing strlen() & strcpy() of a NULL string.
______________________________________________________________________
*/


int
main(int   argc,
     char *argv[])

     /* Interface description:
         INPUT:
          argc  - Number of command line arguments, including command itself.
          argv  - An array of command-line arguments.
         OUTPUT:
          None.
         RETURN VALUE:
          int   - Zero if successful completion, non-zero otherwise.
     */

{
     /* Internal variables: */
          /* Major */
   ProblemTree *ProbTree = NULL ;	/* The tree of problem types. */
   UserInfo UserInfoRec ;		/* Information about the user. */
   struct Full_path pathlist ;
   char *Path = NULL ;			/* The tree leaf for the new problem. */
   char *Summary = NULL ;		/* The summary    "   "   "     "     */
   char *Desc = NULL ;		 	/* The description "  "   "     "     */
   char Node[MAXHOSTNAMELEN+1] ;	/* The hostname   "   "   "     "     */
   char Reporter[MAXMAILADDR+1] ;	/* The reporter   "   "   "     "     */
   signed char Priority=0 ;		/* The priority   "   "   "     "     */

          /* Minor */
   int Status = 0 ;			/* A return value */
   int c ;				/* A command-line option character */
   char Str[MAXLINELEN] ; /* Maximum amound of data on a single line of input */
   char *Buffer ;			/* A temporary buffer of characters */
   

   /*---------------------- Start of main() routine -----------------------*/

   pathlist.next = NULL ;  /* Much-needed initialization added 10/8/94, DC */

   Node[0]='\0' ;
   Reporter[0]='\0' ;

     /* Get information about the user and read the problem hierarchy
      * from the config file.
      */
#ifdef DEBUG
   fprintf(stderr,"calling Setup()...\n") ;
#endif
/*CLD*/   if (Setup(&UserInfoRec, &ProbTree) == False)
   {  fprintf(stderr, "Error in configuration file %s !\n", CONFIGFILE) ;
      exit(1);
   }
#ifdef DEBUG
   fprintf(stderr,"returned from Setup()...\n") ;
   fprintf(stderr,"\tUserInfoRec.sysopflag = %d\n", UserInfoRec.sysopflag);
   fprintf(stderr,"\tUserInfoRec.username  = %s\n", UserInfoRec.username) ;
   fprintf(stderr,"\tUserInfoRec.realname  = %s\n", UserInfoRec.realname) ;
#endif

   gethostname(Node, MAXHOSTNAMELEN) ; /* Can be over-ridden below...*/

	/* Get the problem tree into a list of leaves more easily printed.*/
/*CLD*/  TraverseLeaves(ProbTree, &pathlist) ;

	/* Parse the command line.  */
   while ( (c = getopt(argc, argv, "p:s:d:n:r:c:")) != -1 )
   {  switch(c)
      { 
         case 'p':		/* PATH */
#               ifdef DEBUG
		fprintf(stderr,"p option found ('%s')\n", optarg) ;
#		endif /*DEBUG*/
			/* Determine arg size, allocate space, and copy it */
		Path = malloc (strlen(optarg) + 1) ;
		strcpy(Path, optarg) ;
		break ;

         case 's':		/* SUBJECT */
#               ifdef DEBUG
		fprintf(stderr,"s option found ('%s')\n", optarg) ;
#		endif /*DEBUG*/
			/* Determine arg size, allocate space, and copy it */
		Summary = malloc (strlen(optarg) + 1) ;
		strcpy(Summary, optarg) ;
		break ;
			
         case 'd':		/* DESCRIPTION */
#               ifdef DEBUG
		fprintf(stderr,"d option found ('%s')\n", optarg) ;
#		endif /*DEBUG*/
			/* Determine arg size, allocate space, and copy it */
		Desc = malloc (strlen(optarg) + 1) ;
		strcpy(Desc, optarg) ;
		break ;

         case 'n':		/* HOST */
#               ifdef DEBUG
		fprintf(stderr,"n option found ('%s')\n", optarg) ;
#		endif /*DEBUG*/
			/* Determine arg size, allocate space, and copy it */
		/* This one doesn't need malloc()*/
		strcpy(Node, optarg) ;
		break ;

         case 'r':		/* REPORTER */
#               ifdef DEBUG
		fprintf(stderr,"r option found ('%s')\n", optarg) ;
#		endif /*DEBUG*/
			/* Determine arg size, allocate space, and copy it */
		/* This one doesn't need malloc()*/
		if (IsSysop(&UserInfoRec))
		   strcpy(Reporter, optarg) ;
                else 
		   fprintf(stderr,"newprob: Warning -- not a sysop.  "
                                  "Ignoring 'r' option.\n") ;
		break ;

         case 'c':		/* PRIORITY (Criticality) */
#               ifdef DEBUG
		fprintf(stderr,"c option found ('%s')\n", optarg) ;
#		endif /*DEBUG*/
		Priority=atoi(optarg) ;
		break ;

	 default:		/* Invalid or misused options, '-?', etc. */
		fprintf(stderr, 
		      "PTS newprob utility, Version %s.%s\n"
		      "Usage: \n"
		      "  %s [-p path] [-s 'summary'] [-d 'description']"
		      " {-n node} {-r reporter}\n",
		      VERSION, PATCHLEVEL, argv[0]) ;
		exit(-1) ;
      }
   }


	/* Prompt for any values not present on the command-line. */

   while (Path == NULL)
   {  printf (ENTER_PATH_PROMPT) ;
      gets(Str) ;
      if (strcmp(Str,"?") != 0)
      {  Path = malloc (strlen(Str) + 1) ;
         strcpy(Path, Str) ;
      }
      else
      {
		/* Print the list of problem types (leaves only.)
		 * This should really be piped through more.
		 * However, since this program must run set-uid,
		 * that should be done carefully.
		 */
/*CLD*/  TraverseList(stdout,&pathlist) ;

      }
   }
      

   if (Summary == NULL)
   {  printf (ENTER_SUMMARY_PROMPT) ;
      gets(Str) ;
      Summary = malloc (strlen(Str) + 1) ;
      strcpy(Summary, Str) ;
   }
      

   if (Desc == NULL)
   {  printf (ENTER_DESC_PROMPT) ;

		/* Desc is multi-line, so read each line into Str and
		 * append the string to Desc.
		 *
		 * Should really call vi, or $EDITOR, but again,
		 * since this is running set-uid, this should be
		 * done very carefully.
		 *
		 * "Good enough for now..."
		 */
      while (gets(Str) != NULL)
      {
			/* "3" comes from the two '\0' chars. plus
			 * a '\n' which will be added.
			 */
         if (Desc)  /* if() check added 10/8/94, DC */
         {  Buffer = malloc (strlen(Str) + strlen(Desc) + 3) ;
            strcpy(Buffer, Desc) ;
            strcat(Buffer, Str) ;
         }
         else 
         {  Buffer = malloc (strlen(Str) + 3) ;
            strcpy(Buffer, Str) ;
         }

         strcat(Buffer, "\n") ;

         if (Desc) 
            free(Desc) ;

         Desc = Buffer ;
      }
   }

   if (strlen(Reporter) == 0)
   {  strcpy(Reporter,UserInfoRec.username) ;
#     ifdef DEBUG
      fprintf(stderr,"No reporter specified on command-line.  Using current.\n") ;
#     endif
   }

	/* make sure Path contains no space/tabs/etc at head or end. */
   if (Path != NULL)
      strcpy (Path, FixStr(Path)) ;

	/* make sure Summary contains no space/tabs/etc at head or end. */
   if (Summary != NULL)
      strcpy (Summary, FixStr(Summary)) ;

	/* make sure Node contains no space/tabs/etc at head or end. */
   if (Node!= NULL)
      strcpy (Node, FixStr(Node)) ;

	/* make sure Reporter contains no space/tabs/etc at head or end. */
   if (Reporter!= NULL)
      strcpy (Reporter, FixStr(Reporter)) ;


#  ifdef DEBUG
   fprintf(stderr,"Problem:\n") ;
   fprintf(stderr,"\tPath: %s\n", Path) ;
   fprintf(stderr,"\tSummary: %s\n", Summary) ;
   fprintf(stderr,"\tReporter: %s\n", Reporter) ;
   fprintf(stderr,"\tNode: %s\n", Node) ;
   fprintf(stderr,"\tDesc: %s\n", Desc) ;
#  endif /*DEBUG*/

		/* If we've got everything we need to add the problem,
		 * then wait no more... do it!
		 */
   if (Path != NULL && Summary != NULL && Node != NULL && Desc != NULL)
   {
		/* Is Path a valid leaf in the problem tree??? */
/*CLD*/ if ( ! IsLeaf (ProbTree, Path))
        { fprintf(stderr, "Error: problem-tree path is not a leaf: %s\n", Path);
          exit (-1) ;
        }

/*A BIT OF A HACK... (In case we're not who we are.) */
        strcpy (UserInfoRec.username, Reporter) ;

		/* Add the problem! */
/*CLD*/      Status = WriteProblem(ProbTree, FixPath(Path), &UserInfoRec,
				   Desc, Summary, Node, Priority) ;

      if (Status == (int)NULL)	/* Error! */
      {  fprintf(stderr,
		"Error writing problem!\n\nPossible error cause:\n%s\n", 
/*CLD*/		CloudStrerror(cld_errno)) ;
          exit (-1) ;
      }
      else			/* Success! */
      {   
#        ifdef DEBUG
         fprintf(stderr,"success\n") ;
#        endif /* DEBUG */
         exit(0) ;	/* The "good" exit. */
      }

   } else	/* Not all values were successfully determined.  Yikes...*/
   {  fprintf(stderr, "Error: Not all required values present.\n") ;
      if (Path == NULL)    fprintf(stderr, "Path was NULL\n") ;
      if (Summary == NULL) fprintf(stderr, "Summary was NULL\n") ;
      if (Node == NULL)    fprintf(stderr, "Node was NULL\n") ;
      if (Desc == NULL)    fprintf(stderr, "Description was NULL\n") ;
      exit (-1) ;
    }

   exit(-1) ;  /* Shouldn't ever get here. */

}  /*-------------------------- End of main() -----------------------------*/

/* end of newprob.c */

