/* ptsager.c -- The Problem Tracking System problem-ager/auto-mailer.
 * Authors: Brian Goff & Dean Collins
 *
 * Modified: Sat Mar 27 15:20:27 PST 1993
 * 
 * Modified: Dean Collins Thu Feb 24 10:37:15 1994
 * Removed mutant use of ../xpts/util.c (since FixStr, StripLeading &
 * StripTrailing were moved to neb_cld, plus they're not even used anymore).
 * Changed MAIL_ADM_COMMAND to use Mailer & MAIL_ADM_OPTS.
 * An undocumented #define had to be modified for ptsager to work on
 * systems that used a mailer other than /usr/ucb/Mail.  It now
 * uses the mailprog: listed in the config file, or the default MAILPROG.
 * Options still need to be defined in MAIL_ADM_OPTS, but that can be
 * changed in the Imakefile.
 *
 * Modified: Dean Collins Sat Mar 12 18:21:03 1994
 * Removed MailAdm() and replaced with SendMail() call.  Removed
 * MAIL_ADM_OPTS.  Added MAIL_ADM_SUBJ.  Adjusted for renaming
 * of MailInfo.username as MailInfo.useraddr.
 */

/* Copyright 1994 Dean Collins.
 * Copyright 1993 Brian Goff, Dean Collins.
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notices appear in all copies and
 * that both those copyright notices and this permission notice appear in
 * supporting documentation, and that neither the name of the University of
 * Idaho nor the name of Dean Collins nor the name of Brian Goff
 * be used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission from both parties.
 * Neither The University of Idaho nor Dean Collins nor Brian Goff make
 * any representations about the suitability of this software for any
 * purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO, DEAN COLLINS AND BRIAN GOFF DISCLAIM ALL
 * WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE
 * UNIVERSITY OF IDAHO OR DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT
 * OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS
 * OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
 * OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE
 * OR PERFORMANCE OF THIS SOFTWARE.
 */

#include <stdlib.h>
#include <stdio.h>              /* Standard input/output utilities hdr. file */
#include <sys/param.h>          /* System parameters header file */
#include <sys/types.h>		/* Standard types header file */
#include <sys/time.h>		/* System time header file */
#include <time.h>               /* time.h - ANSI C, DATE and TIME header file.*/
#include <limits.h>		/* Sizes, limits, etc. */
#include "pts.h"
#include "zdbm.h"               /* Zombie Database Manager header file */
#include "cloud.h"              /* Nebulous Cloud header file */
#include "clouderror.h"         /* Nebulous Cloud error rtn. header file */
#include "config.h"             /* Config. file parser header file */


#define MAIL_HEADER \
"\nNOTICE: These PTS problems are unsolved and are %d days old:\n\n"

#ifndef MAIL_ADM_SUBJ
#define MAIL_ADM_SUBJ "PTS Unsolved Problems Notification"
#endif

#define DAYLEN  3
#define YEARLEN 4

/* This may be necessary on SunOS and HP-UX */
#ifndef ARG_MAX
#ifdef _POSIX_ARG_MAX
#define ARG_MAX _POSIX_ARG_MAX
#else
#define ARG_MAX 4096
#endif /*_POSIX_ARG_MAX*/
#endif /*ARG_MAX*/

/* This may be necessary on SunOS and HP-UX */
#ifndef FILENAME_MAX
#ifdef _POSIX_PATH_MAX
#define FILENAME_MAX  _POSIX_PATH_MAX
#else
#define FILENAME_MAX 255
#endif /*_POSIX_PATH_MAX*/
#endif /*FILENAME_MAX*/



/*
 * MailSysAdm[i].TmpFName is the temporary mail file for person i.
 * MailSysAdm[i].Flag says whether temporary mail file for person i
 * has been written to (other than header).
 */
typedef struct
{       int Flag ;
        char TmpFName[FILENAME_MAX] ;
} MailSysAdmRec ;


/* FUNCTION PROTOTYPES */
int main(int argc, char *argv[]) ;
void WriteHeader(char *fname, char *header, int days) ;
int CountRecs(MailInfo *list) ;
void PrintSummaries(Summary *CloudUnSummaries, char **UnsolvedList) ;
void GrabSysDate(int *yr, int *day) ;
void ParseDate(char *prid, int *yr, int *day) ;
void AppendProb(char *fname, char *path, char *summary) ;


/*
______________________________________________________________________
main()

FUNCTION DESCRIPTION:

	This is the main function for the PTS problem-ager/
auto-mailer.  It finds problems that are X days old and sends
mail about them to user Y.  X & Y come from the PTS config file.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/11/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/

int
main(int argc,
     char *argv[])

     /* Interface description:
         INPUT:
          argc  - The number of command-line arguments. (UNUSED)
          argv  - An array of command-line arguments. (UNUSED)
         OUTPUT:
          None.
         RETURN VALUE:
          int   - Status flag.  Non-zero is error.
     */
      
{
     /* Internal variables: */
          /* Major */
   char **UnsolvedList = NULL ;		/* The list of unsolved problems */
   Summary *CloudUnSummaries = NULL ;	/* A ptr. to the unsolved problems
					 * linked list (used by neb_cld).
					 */
   Summary *s = NULL ;			/* The list of problem summaries */
   ProblemTree *ProbTree ;		/* The problem types tree */
   UserInfo UserInfoRec ;		/* Information about the user */
   problem_record *ProblemRecord ;	/* A full problem record */
   MailInfo *MailList=NULL ;		/* A list of people to send mail to. */
   MailSysAdmRec *MailSysAdm=NULL ;	/* A list of mail filenames */
   int NumAdmRecs=0 ;			/* Counter of # of MailSysAdm records*/
   int CurYear, CurDay ;		/* current year and day of year */
   int ProbYear, ProbDay ;		/* year and day of year a problem
					 * was reported
					 */

          /* Minor */
   MailInfo *p=NULL ;			/* A temporary mail list pointer */
   int age ;				/* The age of a problem */
   int i ;				/* Misc counter. */
   StringListRec *plist ;


   /*---------------------- Start of main() routine -----------------------*/

#ifdef DEBUG
   freopen("/tmp/debug", "w", stderr) ;
#endif

     /* Get information about the user and read the problem hierarchy. */

   zdebug("calling Setup()...\n") ;
   zdebug("loading config file...\n") ;

/*CLD*/   if (Setup(&UserInfoRec, &ProbTree) == False)
   {  fprintf(stderr, "Error in configuration file %s !\n", CONFIGFILE) ;
      exit(1);
   }

   zdebug("returned from Setup()...\n") ;
   zdebug1("\tUserInfoRec.sysopflag = %d\n", UserInfoRec.sysopflag);
   zdebug1("\tUserInfoRec.username  = %s\n", UserInfoRec.username) ;
   zdebug1("\tUserInfoRec.realname  = %s\n", UserInfoRec.realname) ;

     /* Read the mail list from the config file. */
   zdebug("Calling config_mail()\n") ;
   config_mail(&MailList) ;
#ifdef DEBUG
   zdebug1("Returned from config_mail(), MailList=%p\n",MailList) ;
   p=MailList ;
   while (p)
   {  zdebug2("name=%s\tdays=%d\tpaths=",p->useraddr,p->days) ;
      if (p->pathlist)
      {  for (plist = p->pathlist; plist; plist=plist->next)
            zdebug1("%s ", plist->string) ;
      }
      zdebug("\n") ;

      p=p->next ;
   }
#endif


     /* Read all unsolved problems */
   zdebug("searching for all unsolved problems...\n") ;

   CloudUnSummaries = ReadUnsolvedSummaries(ProbTree, &UnsolvedList) ;

#ifdef DEBUG
   zdebug("Checking for NULL values\n") ;
   if (ProbTree == NULL)	     zdebug("ProbTree is NULL\n") ;
   if (CloudUnSummaries== NULL)      zdebug("CloudUnSummaries is NULL\n") ;
   if (UnsolvedList == NULL)	     zdebug("UnsolvedList is NULL\n") ;
   else if (UnsolvedList[0] == NULL) zdebug("UnsolvedList[0] is NULL\n") ;
#endif

   if (!CloudUnSummaries)
   {  printf("No unsolved problems found.\n") ;
      exit (0);
   }

     /* Count number of people in MailList and allocate an array of
      * MailSysAdm records.  These contain TmpFName and Flag.
      * MailSysAdm[i].TmpFName is the temporary mail file for person i.
      * MailSysAdm[i].Flag says whether temporary mail file for person i
      * has been written to (other than header).
      */
   NumAdmRecs = CountRecs(MailList) ;

     /* Allocate memory for array of MailSysAdm records.  This should have 
      * enough room for NumAdmRecs records.
      */
   MailSysAdm = calloc(NumAdmRecs+1, sizeof(MailSysAdmRec)) ;

     /* Fill in MailSysAdm records, setting Flag fields to 0 and
      * fill in the TmpFName fields with names created by tmpnam() .
      * Write temp file headers for each filename in MailSysAdm array.
      */
   for (i=0, p=MailList ; (i<NumAdmRecs) && (p != NULL); ++i, p=p->next)
   {   MailSysAdm[i].Flag = 0 ;
       strcpy(MailSysAdm[i].TmpFName, tmpnam(NULL)) ;

       /* Write temp file headers for each filename in MailSysAdm array */
       zdebug2("writing header for (%s,%d)\n",MailSysAdm[i].TmpFName,p->days) ;

       WriteHeader(MailSysAdm[i].TmpFName, MAIL_HEADER, p->days) ;
   }


   /* Grab System Date */
   GrabSysDate(&CurYear, &CurDay) ;


      /* For each unsolved problem
       * 	Grab date for the problem (parse out fields)
       *	Compare date to current date (get back number of DaysOld)
       *	
       *	For each mail person
       *		Compare DaysOld to their day value
       *		If same,
       *		    Compare ProbType to their problist
       *		    If in list or list is empty,
       *                      Append that problem to their MailSysAdm mail file
       *		      Set their MailSysAdm flag to TRUE
       *		    Endif
       *		Endif
       *	Done
       * Done
       */
   for (s=CloudUnSummaries; s != NULL; s=s->next)
   {
      ParseDate(s->prid,&ProbYear,&ProbDay) ;

      age = DaysOld(ProbYear,ProbDay,CurYear,CurDay) ;

      zdebug1("age is %d\n",age) ;


      for (i=0, p=MailList ; (i<NumAdmRecs) && (p != NULL); ++i, p=p->next)
      {   if (age == p->days)
          {
#ifdef DEBUG
             zdebug1("p->pathlist = %p\n", p->pathlist) ;
             if (p->pathlist)
                zdebug1("p->pathlist->string = %s\n", p->pathlist->string) ;
#endif
	  				/* Following if() added 3/1/94 DC*/
             if (p->pathlist == NULL || 
                    InStringList(s->path, p->pathlist, PREFIXCMP))
             {  AppendProb(MailSysAdm[i].TmpFName, s->path, s->summary) ;
                MailSysAdm[i].Flag = TRUE ;
             }
          }
      }
   }



   /*
    * For each mail person
    *	If their MailSysAdm flag is TRUE
    *	    mail that person's MailSysAdm file to them
    *	    remove the temporary mail file
    *	Endif
    * Done
    */
   for (i=0, p=MailList; (i<NumAdmRecs) && (p != NULL); ++i, p=p->next)
   {
       if (MailSysAdm[i].Flag)
          SendMail (Mailer, MailerOpts, MAIL_ADM_SUBJ, p->useraddr, "", 
                    MailSysAdm[i].TmpFName) ;

       unlink(MailSysAdm[i].TmpFName) ;

       zdebug1("Unlinked %s\n",MailSysAdm[i].TmpFName) ;
   }


}  /*-------------------------- End of main() -----------------------------*/

/*
______________________________________________________________________
CountRecs()

FUNCTION DESCRIPTION:
	Count the number of records in a MailInfo linked list.
If list is NULL it will return zero.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/



int CountRecs(MailInfo *list)

     /* Interface description:
         INPUT:
          list  -  the MailInfo list to count
         OUTPUT:
          None.
         RETURN VALUE:
          int   -  then number of records in the list
     */
      
{
     /* Internal variables: */
          /* Major */
          /* NONE */

          /* Minor */
   int count=0 ;
   MailInfo *p ;


   /*------------------- Start of CountRecs() routine --------------------*/

     /* For each record in the MailInfo list, increment count */
   p=list ;
   while (p)
   {  ++count ;
      p=p->next ;
   }

   zdebug1("CountRecs returning: %d\n",count) ;


   return(count) ;

}  /*----------------------- End of CountRecs() --------------------------*/

/*
______________________________________________________________________
WriteHeader()

FUNCTION DESCRIPTION:
	Write the header for a mail file.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/



void
WriteHeader(char *fname, char *header, int days)

     /* Interface description:
         INPUT:
          fname   - filename of mail file
          header  - header to write to fname (should contain one "%s" for days)
          days    - "days old" value to print in header
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   FILE *fp ;

          /* Minor */


   /*------------------- Start of WriteHeader() routine --------------------*/

   zdebug2("Writing header '%s' to file '%s'\n",header, fname) ;

     /* Open the file fname for writing */
   if ((fp=fopen(fname, "w"))== NULL)
   {  fprintf(stderr,"Error opening file %s\n",fname) ;
      exit(1) ;
   }


     /* Write the header */
   fprintf(fp, header, days) ;

     /* Close the file */
   fclose(fp) ;

}  /*----------------------- End of WriteHeader() --------------------------*/



/*
______________________________________________________________________
GrabSysDate()

FUNCTION DESCRIPTION:

	Obtain the system date.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/



void GrabSysDate(int *yr, int *day)

     /* Interface description:
         INPUT:
          yr    - ptr to current system year
          day   - ptr to current system day of year (starting at 1)
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   struct timeval curdate;        /* current system date */
   struct timezone tz;		  /* current time zone */
   struct tm *localdate ;

          /* Minor */


   /*------------------ Start of GrabSysDate() routine -------------------*/

   gettimeofday(&curdate, &tz) ;

   localdate=localtime(&(curdate.tv_sec)) ;

   *yr = 1900 + localdate->tm_year ;

	/* tm_yday is 0 to 365.  We need to compare with strftime's 
	 * %j, which is 1 to 366.  ho hum.
	 */
   *day = localdate->tm_yday + 1 ; 

   zdebug2("Local Time (day/yr): %d/%d\n",*day,*yr) ;

}  /*----------------------- End of GrabSysDate() -----------------------*/

/*
______________________________________________________________________
ParseDate()

FUNCTION DESCRIPTION:

	Obtain the date associated with a problem.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/



void ParseDate(char *prid, int *yr, int *day)

     /* Interface description:
         INPUT:
          prid  - ptr. to a problem ID
         OUTPUT:
          yr    - ptr. to the year the problem was reported
          day   - ptr. to the day of the year the problem was reported
			(starting at 1)
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   char daystr[MAXPRID] ;
   char yrstr[MAXPRID] ;
   char *p ;

          /* Minor */


   /*------------------- Start of ParseDate() routine --------------------*/

     /* Copy the day of the year from the problem ID into daystr */
   strncpy(daystr,prid,DAYLEN) ;
   daystr[DAYLEN] = '\0' ;

     /* Copy the the year from the problem ID into yrstr */
   p = &(prid[DAYLEN]) ;
   strncpy(yrstr,p,YEARLEN) ;
   yrstr[YEARLEN] = '\0' ;

     /* Convert to integers and store in yr & day */
   *yr = atoi(yrstr) ;
   *day = atoi(daystr) ;

   zdebug5("ParseDate: daystr='%s', yrstr='%s, day=%d, yr=%d', prid=%s\n",
           daystr,yrstr,*day,*yr,prid) ;

}  /*----------------------- End of ParseDate() --------------------------*/



/*
______________________________________________________________________
DaysOld()

FUNCTION DESCRIPTION:

	Determines the difference (in days) between two dates.
Differences of more than 365 days are ignored.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/

int DaysOld(int ProbYear, int ProbDay, int CurYear, int CurDay)

     /* Interface description:
         INPUT:
          ProbYear - The year associated with a problem
          ProbDay  - The day of the year associated with a problem
          CurYear  - The current year
          CurDay   - The current day of the year
         OUTPUT:
          None.
         RETURN VALUE:
          int      - Difference (in days) between the two dates.
			zero is returned if the difference is greater
			then 365 days.
     */
      
{
     /* Internal variables: */
          /* Major */
   int delta ;		/* Difference (in days) between the two dates. */

          /* Minor */


   /*---------------------- Start of DaysOld() routine --------------------*/

     /* If the two years are the same,
      *		delta = CurDay - Probday ;
      * If the two days differ by one,
      *         delta = YearLength(ProbYear) - ProbDay + CurDay ;
      * Otherwise,
      * 	delta = 0 ;
      */

   switch(CurYear - ProbYear)
   {
      case 0:   delta = CurDay - ProbDay ;
                break ;

      case 1:   delta = YearLength(ProbYear) - ProbDay + CurDay ;
                break ;

      default:  delta = 0 ;
   }

   return(delta) ;

}  /*-------------------------- End of DaysOld() --------------------------*/


/*
______________________________________________________________________
YearLength()

FUNCTION DESCRIPTION:
	Computes the number of days in a given year, accounting
for leap years.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/



int YearLength(int year)

     /* Interface description:
         INPUT:
          year  - The year in question.
         OUTPUT:
          None.
         RETURN VALUE:
          int   - The number if days in the year.
     */
      
{
     /* Internal variables: */
          /* Major */
   int length ;

          /* Minor */


   /*------------------- Start of YearLength() routine --------------------*/

     /* If the year is not evenly divisible by four,
      *		length = 365 ;
      * else if the year is not evenly divisible by 400 (is a leap year),
      *          length = 365 ;
      * else length = 366 ;
      */


   zdebug1("year%%4=%d\n",year%4) ;

   if (year % 4) length = 365 ;

   else if (year % 400) length = 365 ;
        else length = 366 ;

     /* Yes, I know there's more to this... Care to contribute?... */

   zdebug2("YearLength(): Year = %d, Length = %d\n", year, length) ;

   return (length) ;

}  /*----------------------- End of YearLength() --------------------------*/

/*
______________________________________________________________________
AppendProb()

FUNCTION DESCRIPTION:

	Appends a problem summary (and it's problem type) to a file.
______________________________________________________________________
REVISION HISTORY:
     Authors:  Dean Collins & Brian Goff      Date:  2/28/93
     Documentation Enhancement: DC		     3/27/93
______________________________________________________________________
*/


void AppendProb(char *fname, char *path, char *summary)

     /* Interface description:
         INPUT:
          fname   - The name of the file to append to.
          path    - The problem tree path for the problem.
          summary - The summary for the problem.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   FILE *fp ;

          /* Minor */


   /*------------------- Start of AppendProb() routine --------------------*/

   zdebug("Entering AppendProb() \n") ;

     /* Open the file to write to. */
   if ((fp=fopen(fname, "a"))== NULL)
   {  fprintf(stderr,"Error opening file %s for append\n",fname) ;
      exit(1) ;
   }

     /* Print the summary and path to the file */
   fprintf(fp, "Summary:  %s\n", summary) ;
   fprintf(fp, "Path:     %s\n\n", path) ;

     /* Closwe the file */
   fclose(fp) ;

   zdebug("Leaving AppendProb() \n") ;
   
}  /*----------------------- End of AppendProb() --------------------------*/

/* End of ptsager.c */
