/*
 * misc.c -- Miscellaneous functions for the X user interface
 *             to the Problem Tracking System (PTS) database.
 * Dean Collins 
 *
 * created:		Wed Aug  5 09:31:11 PDT 1992
 * Version 1.0:		Sat Jan  9 10:44:49 PST 1993
 * Version 1.01:	Wed Mar  7 00:00:00 PST 1993
 * Version 1.03:	Fri Nov 26 16:13:59 PST 1993
 * Version 1.03b:	Sat Feb 12 17:23:17 PST 1994
 * Version 1.04:	Mon Mar  7 18:07:14 PST 1994
 *
 * Modified: Dean Collins Sat Feb 12 17:16:41 1994
 * Modification inspired by Jaap Haalboom (haalboom@prl.philips.nl).
 * Added Xlibint.h include for _XExtension.  Added for all architectures,
 * not just __hppa.
 *
 * Modified: Dean Collins Mon Mar  7 18:03:35 PST 1994
 * Modification inspired by Ove Hansen.
 * Added USESYSDEFERR to ignore my not-so-portable error handler
 * Define this if you're getting compiler errors for the function
 * XPrintDefaultError(). 
 *
 * Modified: Dean Collins Fri May 06 19:47:40 1994
 * Relocated the inclusion of <X11/Xlibint.h> so that it's after
 * <X11/Intrinsic.h>, which was causing grief.
 */

/*
 * Copyright (c) 1995,1994,1993,1992 Dean Collins.
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notices appear in all copies and
 * that both those copyright notices and this permission notice appear in
 * supporting documentation, and that neither the name of the University of
 * Idaho nor the name of Dean Collins be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission from both parties.  Neither The University of Idaho
 * nor Dean Collins make any representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 * 
 * THE UNIVERSITY OF IDAHO AND DEAN COLLINS DISCLAIM ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY OF IDAHO
 * OR DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#include <stdlib.h>		/* General C utilities header file */
#include <stdio.h>		/* Standard input/output utilities hdr. file */
#include <errno.h>		/* Error utilities header file */

#include <X11/Intrinsic.h>	/* Include standard Toolkit Header file */
#include <X11/StringDefs.h>	/* Include StringDefs header file */
#include <X11/Shell.h>		/* Include the Shell header file for popups */

#ifndef NOEXTVAR
#include <X11/Xlibint.h>	/* Include internal support routines hdr. file*/
#endif /*NOEXTVAR*/

#include <X11/Xaw/AsciiText.h>	/* Include the ascii Text widget hdr. file */
#include <X11/Xaw/Box.h>	/* Include the Box widget header file */
#include <X11/Xaw/Command.h>	/* Include the Command widget header file */
#include <X11/Xaw/Dialog.h>	/* Include the Dialog widget header file */
#include <X11/Xaw/Form.h>	/* Include the Form widget header file */
#include <X11/Xaw/List.h>	/* Include the List widget header file */
#include <X11/Xaw/Paned.h>	/* Include the Paned widget header file */
#include <X11/Xaw/Scrollbar.h>	/* Include the Scrollbar widget header file */
#include <X11/Xaw/Viewport.h>	/* Include the Viewport widget header file */

#include <X11/Xaw/Cardinals.h>	/* Definitions of ZERO, ONE, etc... */

#include <X11/Xmu/Error.h>	/* Error handler functions */

#include "zdbm.h"		/* Zombie Database Manager header file */
#include "cloud.h"		/* Nebulous Cloud header file */
#include "clouderror.h"		/* Nebulous Cloud error rtn. header file */
#include "xpts.h"		/* Xpts header file */


extern bool NoQuit ;		/* True if user started PTS in "no-quit" mode, */
				/* false otherwise. */

      /* The current problem record.  This ideally would
       * be part of a new widget.  The widget would have the
       * resources listed in ProblemType.  It would have functions
       * to display, read and write the problem.
       * Next version, maybe...
       */
extern problem_record * ProblemRecord ;

      /* The current list of problem summaries.  Similar
       * comment to that above.
       */
extern String *SummaryItems ;

        /* The problem types tree. */
extern ProblemTree *ProbTree ;

        /* Information about the user. */
extern UserInfo UserInfoRec ;

        /* The path for the problem types tree. */
extern char Path[] ;

extern char Node[] ;
extern char ProblemText[] ;

extern String *ProblemItems ;

/* The following string will be displayed by the "About PTS" button
 * on the Main Menu: 
 */
extern const char PTSInfoText[] ;

extern Summary *CloudUnSummaries ;
extern Summary *CloudSummaries ;

extern Widget GlobalNewProblemButton ;
extern Widget GlobalPreviousButton ;
extern Widget GlobalMainMenuWidget ;

extern int WarningFlag ;

#ifndef NOSIGNALS
extern SigType SigRec ;
#endif

/******************** MODULES *************************************/

/*
______________________________________________________________________
ChangeList()

FUNCTION DESCRIPTION:

     This function will allow the user to select a sub-type of the current
problem type.  Selecting a sub-type which contains no further sub-types
causes a list of problem summaries for that sub-type to be displayed.
There are two ways to call this function.  First, you can specify
a complete problem type.  Second, you can specify an incomplete problem
type and indicate the child of that type you have selected.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  2/20/92
     Version 1.0 finalized: Dean Collins	Date: 1/9/93
______________________________________________________________________
*/


void
ChangeList(String	type, 
           int		item,
           int		flag,
           XtPointer	client_data)

     /* Interface description:
         INPUT:
          type		- The specified problem type to display, OR
                             the parent of the problem type to display.
          item		- The index of the selected child of the
                             specified problem type if flag=INC_TYPE, OR 
                             NULL otherwise.
          flag		- Equal to ABS_TYPE if type is the
                             complete problem type to display, OR
                             equal to INC_TYPE if type is the parent of
                             the problem to display and item is the
                             index of the selected child of this type, OR
                             equal to PREV_TYPE if item is a child
                             of the desired problem type path.
          client_data	- This is an array of pointers to widgets.
                             It is defined in the following manner:

                                 client_data[0] = list widget
                                 client_data[1] = dialog widget
                                 client_data[2] = parent widget
                                 client_data[3] = quitButton widget
                                 client_data[4] = previousButton widget
                           	 client_data[5] = newProblem widget
				 client_data[6] = pickProblem popup widget

         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */

          /* Major */
   Widget list ;
   Widget dialog ;
   Widget parent ;
   Widget quitButton ;
   Widget previousButton ;
   Widget newProblem ;
   Widget pickProblem ;
   Widget panes ;
   XtPointer *ptrlist ;
   XtPointer *ptrlist2 ;

          /* Minor */
   Arg args[3] ;
   int status=0 ;
   char OldPath[DBPATHLEN+1] ;
   Dimension width=0, height=0 ;
   Dimension pwidth=0, pheight=0 ;
   Dimension border ;
   static int insummaries=FALSE ;  /* Whether we're currently in the problem
				    * summaries list or not. 
				    */
     

   /*----------------- Start of ChangeList() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(misc.c): Entering ChangeList()\n") ;
#endif

        /* Make local widgets for each of the items in client_data */
   ptrlist2 = (XtPointer *)client_data ;
   list		   = (Widget) (ptrlist2[0]) ;
   dialog	   = (Widget) (ptrlist2[1]) ;
   parent	   = (Widget) (ptrlist2[2]) ;
   quitButton	   = (Widget) (ptrlist2[3]) ;
   previousButton  = (Widget) (ptrlist2[4]) ;
   newProblem	   = (Widget) (ptrlist2[5]) ;
   pickProblem	   = (Widget) (ptrlist2[6]) ;

   panes = XtParent(XtParent(list)) ;

#ifdef STUPIDXSERVER 
     /* The list widget keeps getting mangled, so lets unmanage it
      * for the time being.  Once everything is ready, manage it again.
      * This problem needs to be tracked down.  (XtUnmangleChild() ! :-)
      */
   XtUnmanageChild(list) ;
#endif


   AllocProblemItems() ; /* Allocate ProblemItems strings */

#ifdef DEBUG
   printf("XPTS(misc.c): Calling FormatProblemTypes with Path=%s\n",Path) ;
#endif
   /* Attempt to change to the new problem type. */
/*CLD*/   status = FormatProblemTypes(ProbTree,Path,item,flag,ProblemItems) ;

#ifdef DEBUG
   printf("XPTS(misc.c): Returned from FormatProblemTypes()...Path=%s\n",Path) ;
   printf("XPTS(misc.c): FormatProblemTypes status is %d\n", status) ;
   printf("XPTS(misc.c): ChangeList(): Removing extra ProblemItems strings\n") ;
#endif
   FreeItems(ProblemItems, MAXBRANCHES) ;/* Remove unused ProblemItems strings*/



     /* If the problem type doesn't exist in the problem tree,
      *   then call ErrorPopup() with an appropriate error message,
      *   and return.
      */
   if (status < 0)  	/* Error in problem-type path */
   {   /* Might be an error, might be a summary selection.
        * Call ReadProblem to decide.  If it returns NULL there
        * was an error.
        * Make a copy of Path, since ReadProblem will probably mess it up.
        */
#ifdef DEBUG
      printf("XPTS(misc.c): FormatProblemTypes() returned an error,\t(OK, if already at a leaf).\n") ;
#endif
      strcpy(OldPath, Path) ; /* Save the Path */
#ifdef DEBUG
      printf("XPTS(misc.c): Calling ReadProblem() in cloud...\n") ;
#endif
/*CLD*/      ProblemRecord=ReadProblem(ProbTree, Path, item, CloudSummaries) ;
#ifdef DEBUG
      printf("XPTS(misc.c): Returned from ReadProblem()...ProblemRecord=%p\n",
             ProblemRecord) ;
#endif

      if (ProblemRecord == NULL)
      {  
#ifdef DEBUG
         printf("XPTS(misc.c): ReadProblem() returned a NULL\n") ;
#endif
         ErrorPopup("error",pickProblem,"Problem nonexistent!\n") ;
         strcpy(Path, OldPath) ; /* Restore the Path */
         XtManageChild(list) ;
         return ;
      }
      else
      {  strcpy(Path, OldPath) ; /* Restore the Path */
	 /* Call ViewProblemActivate() with ANYPROB and item */
         ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
         ptrlist[0] = (XtPointer) pickProblem ;
         ptrlist[1] = (XtPointer) ANYPROB ;
#ifdef DEBUG
         printf("XPTS(misc.c): Calling ViewProblemActivate()...\n"
		"\tptrlist=%p,ptrlist[0]=%p,ptrlist[1]=%p,list=%p,Path=%p\n",
		ptrlist,ptrlist[0],ptrlist[1],list,Path) ;
#endif
         ViewProblemActivate(list, (XtPointer)ptrlist, (XtPointer)item) ;
      }

   } 



	/**** NOT  A  LEAF ****/

   if (status > 0)	/* Child types exist, returned in ProblemItems */
   {
     /* If the problem type specified is NOT a leaf, 
      *   Generate a printable list of problem types.
      *   Change the list widget's text.
      *   Change the list to be two column.
      *   Get rid of the newProblem button, if present.
      *   Change the dialog widget's text to be the new problem type.
      */

      XtSetArg(args[0], XtNdefaultColumns, TWO) ;
      XtSetValues(list, args, ONE) ; 

      if (insummaries)   /* Halve height & width of popup */
      {  insummaries=FALSE ;
         XtUnmanageChild(panes) ;
         XtSetArg(args[0], XtNwidth, &width) ;
         XtSetArg(args[1], XtNheight, &height) ;
         XtGetValues(pickProblem, args, TWO) ;
#ifdef DEBUG
         printf("XPTS(misc.c): width=%d, height=%d.  Halving popup size...\n",
		(int)width, (int)height) ;
#endif
	 width=width/2 ;  height=height/2 ;
	 XawPanedAllowResize(XtParent(list), True) ;
         XtSetArg(args[0], XtNwidth, width) ;
         XtSetArg(args[1], XtNheight, height) ;
         XtSetValues(pickProblem, args, TWO) ;

		/* Shrink panes */
         XtSetArg(args[0], XtNborderWidth, &border) ;
         XtGetValues(panes, args, ONE) ;
	 pwidth  = width  - 2 * border ; 
	 pheight = height - 2 * border ; 
         XtSetArg(args[0], XtNwidth, pwidth) ;
         XtSetArg(args[1], XtNheight, pheight) ;
#ifdef DEBUG
         printf("XPTS(misc.c): "
		"Setting -- panewidth=%d, paneheight=%d. (border=%d)\n",
		(int)pwidth, (int)pheight, (int)border) ;
#endif
         XtSetValues(panes, args, TWO) ;
         XtSetArg(args[0], XtNwidth, &pwidth) ;
         XtSetArg(args[1], XtNheight, &pheight) ;
         XtGetValues(panes, args, TWO) ;
#ifdef DEBUG
         printf("XPTS(misc.c): Result  -- panewidth=%d, paneheight=%d\n",
		(int)pwidth, (int)pheight) ;
         if (pwidth > width || pheight > height)
            printf("XPTS(misc.c): Paned widget resize failure.  Darn.\n") ;
#endif
      }

      XawListChange(list, ProblemItems, ZERO, ZERO, TRUE) ;
      XtSetSensitive(list, True) ;
      XtUnmanageChild(newProblem) ;
   }



	/**** IS  A  LEAF ****/

   if (status == 0)	/* No child types, only summaries exist */
   {
     /* If the problem type specified IS a leaf,
      *   Read the problem summaries for the problem type
      *         from the database.
      *   Change the list widget's text.
      *   Change the list to be one column only.
      *   Add the newProblem button to the button box.
      */
      if (!insummaries) 
         XtUnmanageChild(panes) ;

      XtSetArg(args[0], XtNdefaultColumns, ONE) ;
      XtSetValues(list, args, ONE) ; 

      if (!insummaries)   /* Double the height & width for the summaries list*/
      {  insummaries=TRUE ;
         XtSetArg(args[0], XtNwidth, &width) ;
         XtSetArg(args[1], XtNheight, &height) ;
         XtGetValues(pickProblem, args, TWO) ;
#ifdef DEBUG
         printf("XPTS(misc.c): width=%d, height=%d.  Doubling popup size...\n",
		(int)width, (int)height) ;
#endif
         width=width*2 ;  height=height*2 ;
	 XawPanedAllowResize(XtParent(list), True) ;
         XtSetArg(args[0], XtNwidth, width) ;
         XtSetArg(args[1], XtNheight, height) ;
#ifdef DEBUG
   printf("ChangeList() pickProblem=%p\n", (XtPointer)pickProblem) ;
#endif
         XtSetValues(pickProblem, args, TWO) ;

                /* Stretch panes */
         XtSetArg(args[0], XtNborderWidth, &border) ;
         XtGetValues(panes, args, ONE) ;
         pwidth  = width  - 2 * border ;
         pheight = height - 2 * border ;
         XtSetArg(args[0], XtNwidth, pwidth) ;
         XtSetArg(args[1], XtNheight, pheight) ;
#ifdef DEBUG
         printf("XPTS(misc.c): "
                "Setting -- panewidth=%d, paneheight=%d. (border=%d)\n",
                (int)pwidth, (int)pheight, (int)border) ;
#endif
         XtSetValues(panes, args, TWO) ;
         XtSetArg(args[0], XtNwidth, &pwidth) ;
         XtSetArg(args[1], XtNheight, &pheight) ;
         XtGetValues(panes, args, TWO) ;
#ifdef DEBUG
         printf("XPTS(misc.c): Result  -- panewidth=%d, paneheight=%d\n",
                (int)pwidth, (int)pheight) ;
         if (pwidth != width-2*border || pheight != height-2*border)
            printf("XPTS(misc.c): Paned widget resize failure.  Darn.\n") ;
#endif
      }

      ReadList(pickProblem, pickProblem) ; /* Fill in the summaries list. */
      XtManageChild(newProblem) ;
   }
   


     /* If the problem type is ROOTSTR, the problem tree root, 
      *   unrealize the previousButton widget and
      *   realize the quitButton widget.
      * else
      *   unrealize the quitButton widget and
      *   realize the previousButton widget.
      */
#ifdef DEBUG
   printf("XPTS(misc.c): ChangeList(): "
          "About to compare ROOTSTR (%s) to Path (%s)...\n",
 	  ROOTSTR, Path) ;
#endif
/*   if (strcmp(ROOTSTR, Path) == 0)*/
/*CLD*/   if (strcmp(FixPath(Path), "") == 0) /*FixPath because ROOT = ROOT/ */
   {  XtManageChild(quitButton) ;
      XtUnmanageChild(previousButton) ;
   }
   else
   {  XtManageChild(previousButton) ;
      XtUnmanageChild(quitButton) ;
   }

#ifdef STUPIDXSERVER 
   XtManageChild(list) ;
#endif


   XtManageChild(panes) ;

#ifdef DEBUG
   printf("XPTS(misc.c): Leaving ChangeList()\n") ;
#endif

}  /*--------------------- End of ChangeList() ------------------------*/


/*
______________________________________________________________________
DoDeleteProblem()

FUNCTION DESCRIPTION:
     This function is called when the user has indicated that they
do want to delete a problem.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  6/23/92
     Version 1.0 finalized: Dean Collins	Date: 1/9/93
______________________________________________________________________
*/



void
DoDeleteProblem(Widget		w,
		XtPointer	client_data,
		XtPointer	call_data)

     /* Interface description:
         INPUT:
          w           - The widget for which this callback is registered.
          client_data - Specifies arbitrary client-supplied data
                        that the widget w should pass back to the
                        client when the widget executes the client's
                        callback procedure.  In this callback,
                        this is the problem viewer popup widget.
          call_data   - Specifies any callback-specific data the widget w
                        wants to pass to the client.
         OUTPUT:
           None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   Widget warning ;
   Widget probViewer ;
   Widget list ;
          /* Minor */
   char errorstr[MAXERROR] ;


   /*------------- Start of DoDeleteProblem() routine --------------*/
#ifdef DEBUG
   printf("XPTS(misc.c): Entering DoDeleteProblem()\n") ;
#endif

     /* Make sure only sysops are executing this function. */
   if (!IsSysop(&UserInfoRec))
   { 
#ifdef DEBUG
        fprintf(stderr,
         "XPTS(DoDeleteProblem()): Illegal access to DoDeleteProblem() "
	 "by non-sysop.\n") ;
#endif
     return ;
   }

     /* Get values for the warning and list widgets. */
   probViewer = (Widget)client_data ;
   warning = XtNameToWidget(probViewer, "warning") ;
   list = XtNameToWidget(XtParent(probViewer), "*list") ;

#ifdef DEBUG
   printf("XPTS(DoDeleteProblem()): probViewer=%p, parent=%p, warning=%p,\n"
          "                         list=%p.",
          probViewer, XtParent(probViewer), warning, list) ;
#endif

     /* Are there any reopen requests for this problem? 
      * if so, call SuperWarningPopup().
      */
   if (IsReopened(Path, ProblemRecord) > 0)
   {  sprintf(errorstr,
             "WARNING: This problem has reopen requests.  Deleteing this\n"
             "problem will delete the reopen requests as well.  You should\n"
             "probably accept or deny each reopen request before deleting.\n") ;
#ifdef DEBUG
      printf("XPTS(misc.c): Calling SuperWarningPopup!\n") ;
#endif
      XtSetSensitive(probViewer, False) ;
      XFlush(XtDisplay(probViewer)) ;
      SuperWarningPopup("warning", warning, errorstr) ;
      XtSetSensitive(probViewer, True) ;
      if (!WarningFlag) return ;
#ifdef DEBUG
      printf("XPTS(misc.c): SuperWarningPopup returned True! Continuing...\n") ;
#endif
   }

     /* Delete the problem. */
/*CLD*/   if (DeleteProblem(ProbTree, Path, ProblemRecord) == (int)NULL)
   {  /* An error occured */
      sprintf(errorstr,
                "Error deleting problem!\n\nPossible error cause:\n%s\n",
/*CLD*/         CloudStrerror()) ;
      ErrorPopup("error", warning, errorstr) ;
      return ;
   }

        /* Update the list of summaries for the parent popup.
         * This might be the pickProblem popup or the unsolvedPopup popup.
         */
   ReadList(XtParent(probViewer), probViewer) ;


     /* Destroy the problem viewer widget.  Realize it's parent,
      * the problem tree navigation widget.
      */

#ifdef DEBUG
   printf("XPTS(DoDeleteProblem()): setting the parent to be sensitive again.\n") ;
#endif
   XtSetSensitive(XtParent(probViewer), True) ;

#ifdef DEBUG
   printf("XPTS(DoDeleteProblem()): Destroying the warning popup '%s'...\n"
	  "                         warning popup window-ID=%d\n",
          XtName(warning), XtWindow(warning)) ;
#endif
/*   XtDestroyWidget(warning) ;Don't do, since destroying probViewer will
 *                             also destroy the warning popup
 */

#ifdef DEBUG
   printf("XPTS(DoDeleteProblem()): probViewer=%p, parent=%p.\n",
          probViewer, XtParent(probViewer)) ;
   printf("XPTS(misc.c): Destroying the problem viewer "
          "for the deleted problem, '%s'...\n", XtName(probViewer)) ;
#endif
   XtDestroyWidget(probViewer) ;


#ifdef DEBUG
   printf("XPTS(misc.c): Leaving DoDeleteProblem()\n") ;
#endif

}  /*------------ End of DoDeleteProblem() --------------------*/


/*
______________________________________________________________________
DoMoveProblem1()

FUNCTION DESCRIPTION:
     This function is called when the user has indicated what problem
type to move the problem to.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  6/23/92
     Version 1.0 finalized: Dean Collins	Date: 1/9/93
______________________________________________________________________
*/


void
DoMoveProblem1(Widget    w,
              XtPointer  client_data,
              XtPointer  call_data)

     /* Interface description:
         INPUT:
          w           - The widget for which this callback is registered.
          client_data - Specifies arbitrary client-supplied data
                        that the widget w should pass back to the
                        client when the widget executes the client's
                        callback procedure.  In this callback,
                        this is the parent widget.
          call_data   - Specifies any callback-specific data the widget w
                        wants to pass to the client.
         OUTPUT:
           None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* Minor */
          /* NONE */


   /*------------------ Start of DoMoveProblem1() routine -------------------*/
#ifdef DEBUG
   printf("XPTS(misc.c): Entering DoMoveProblem1()\n") ;
#endif

     /* Make sure only sysops are executing this function. */
   if (!IsSysop(&UserInfoRec))
   { 
#ifdef DEBUG
        fprintf(stderr,
        "XPTS(misc.c): Illegal access to DoMoveProblem1() by non-sysop.\n") ;
#endif
     return ;
   }

     /* Display a warning using WarningPopup(). */

     /* If the user responds to the warning with "continue",
      * use DoMoveProblem2() to move the problem from one place in
      * the database to another.
      */
/* for now, don't do the warning...cut to the chase. */
DoMoveProblem2(w, client_data, call_data) ;

#ifdef DEBUG
   printf("XPTS(misc.c): Leaving DoMoveProblem1()\n") ;
#endif
}  /*----------------- End of DoMoveProblem1() -------------------------*/


/*
______________________________________________________________________
DoMoveProblem2()

FUNCTION DESCRIPTION:
     This function moves a problem from one type to another.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  6/23/92
     Version 1.0 finalized: Dean Collins	Date: 1/9/93
______________________________________________________________________
*/



void
DoMoveProblem2(Widget     w,
              XtPointer  client_data,
              XtPointer  call_data)

     /* Interface description:
         INPUT:
          w           - The widget for which this callback is registered.
          client_data - Specifies arbitrary client-supplied data
                        that the widget w should pass back to the
                        client when the widget executes the client's
                        callback procedure.  In this callback,
                        this is an array of pointers to widgets,
			defined like this:
				client_data[0] = move problem popup widget
				client_data[1] = dialog widget containing
                                                 new problem type.

          call_data   - Specifies any callback-specific data the widget w
                        wants to pass to the client.
         OUTPUT:
           None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   XtPointer *ptrlist ;
   XtPointer *ptrlist2 ;
   Widget parent ;
   Widget probViewer ;

          /* Minor */
   String newtype ;
   Arg args[5] ;
   char errorstr[MAXERROR] ;


   /*--------------- Start of DoMoveProblem2() routine ----------------*/
#ifdef DEBUG
   printf("XPTS(misc.c): Entering DoMoveProblem2()\n") ;
#endif

     /* Make sure only sysops are executing this function. */
   if (!IsSysop(&UserInfoRec))
   { 
#ifdef DEBUG
        fprintf(stderr,
        "XPTS(misc.c): Illegal access to DoMoveProblem2() by non-sysop.\n") ;
#endif
     return ;
   }

   if (ProblemRecord == NULL)
   {
#ifdef DEBUG
      printf("XPTS(select.c): ProblemRecord is null !!!  Returning.\n") ;
#endif
      return ;
   }


   ptrlist = (XtPointer *)client_data ;
   newtype = XawDialogGetValueString((Widget)(ptrlist[1])) ;
   parent = (Widget) (ptrlist[0]) ;
   probViewer = XtParent(parent) ;

#ifdef DEBUG
      printf("XPTS(misc.c): DoMoveProblem2():  Moving problem to type %s\n", newtype) ;
#endif

     /* Use MoveProblem to move the problem from one place in
      * the database to another.
      */
/*CLD*/   if (MoveProblem(ProbTree, Path, ProblemRecord, newtype) == (int)NULL)
   {
     /* If an error occurs, use ErrorPopup() to show a message
      * and quit.
      */
      sprintf(errorstr, "Error moving problem!\n\nPossible error cause:\n%s\n",
/*CLD*/         CloudStrerror()) ;
      ErrorPopup("error", parent, errorstr) ;
      return ;
   }

     /* Update the screen.  */

     /* Change problem viewer "Problem Type" dialog box
      * to reflect the new problem type.
      */

     /* Get rid of the "move problem" popup.  */
   ptrlist2 = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist2[0] = ptrlist[0] ;
   ptrlist2[1] = NULL ;
   DestroyPopup(w, (XtPointer)ptrlist2, (XtPointer)NULL) ;


     /* Update the list of summaries for the parent popup.
      * This might be the pickProblem popup or the unsolvedPopup popup.
      */
   ReadList(XtParent(probViewer), probViewer) ;

     /* Destroy the problem viewer widget.  Realize it's parent,
      * the problem tree navigation widget.
      */

#ifdef DEBUG
   printf("XPTS(DoMoveProblem()): setting the parent to be sensitive again.\n"
) ;
#endif
   XtSetSensitive(XtParent(probViewer), True) ;

#ifdef DEBUG
   printf("XPTS(misc.c): Destroying the problem viewer "
          "for the moved problem, '%s'...\n", XtName(probViewer)) ;
#endif
   XtDestroyWidget(probViewer) ;


#ifdef DEBUG
   printf("XPTS(misc.c): Leaving DoMoveProblem2()\n") ;
#endif
}  /*--------------------- End of DoMoveProblem2() -------------------*/


/*
______________________________________________________________________
GotoProblemType()

FUNCTION DESCRIPTION:

     This function will change the list widget so that it represents the
specified problem type.  (This function is not implemented.)
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  2/25/92
______________________________________________________________________
*/


void
GotoProblemType(Widget          w,
                XtPointer       client_data,
                XtPointer       call_data)

     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          client_data    - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback, 
                           this is information to be passed to
                           ChangeList().
          call_data      - Specifies any callback-specific data the widget w
                           wants to pass to the client.  In this callback,
                           this is a pointer to a record containing
                           information about the list selection.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
          /* NONE */
          /* Minor */
          /* NONE */



   /*----------------- Start of GotoProblemType() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(misc.c): Entering gotoProblemType()\n") ;
#endif

     /* Convert what the user has entered into the dialog box
      * into a problem type string.
      *   Ideally, this would:
      *   1. convert to lower-case.
      *   2. interpret ".." as super-type.
      *   3. ignore ".".
      *   4. interpret problem-type paths not beginning with "/"
      *         to be relative to current problem path.
      *   5. (what else?)
      *    For this version, only 1 & 4 are to be implemented.
      */

     /* If the input path begins with "/", send this path 
      * to ChangeList().
      */

     /* If the input path doesn't begin with "/", append
      * it to the current path and send this to ChangeList().
      */

#ifdef DEBUG
   printf("XPTS(misc.c): Leaving GotoProblemType()\n") ;
#endif

}  /*--------------------- End of GotoProblemType() ------------------------*/


/*
______________________________________________________________________
Syntax()

FUNCTION DESCRIPTION:

     This function prints an error message indicating there was a syntax
error on the command-line and exits the application.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  2/20/92
     Version 1.0 finalized: Dean Collins	Date: 1/9/93
______________________________________________________________________
*/


void
Syntax(XtAppContext   app_con,
       char           *call)

     /* Interface description:
         INPUT:
          app_con    - The application context
          call       - The name of the program, from the command line
         OUTPUT:
          None.
         RETURN VALUE:
          None. (exits application with a value of 1.)
     */
      
{
     /* Internal variables: */
          /* Major */
          /* NONE  */      
          /* Minor */
          /* NONE  */

   /*---------------------- Start of Syntax() routine -----------------------*/
#ifdef DEBUG
   printf("XPTS(misc.c): Entering Syntax()\n") ;
#endif

     /* Destroy the X application context, which will close any and all
      * windows using XtDestroyApplicationContext().
      */
   XtDestroyApplicationContext(app_con) ;

     /* Print an error message indicating there was a syntax error on 
      * the command-line.
      */
    fprintf(stderr, "Usage: %s [-noquit]\n", call) ;
    fprintf(stderr,
	"     (Standard X Toolkit options are acceptable, also.)\n") ;

     /* exit the application with a non-zero value, indicating 
      * there was an error.
      */
   exit(1) ;

}  /*-------------------------- End of Syntax() -----------------------------*/


/*
______________________________________________________________________
NoOp()

FUNCTION DESCRIPTION:

     This function does nothing, but it does it well.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  10/29/92
     Version 1.0 finalized: Dean Collins	Date: 1/9/93
______________________________________________________________________
*/


XtActionProc
NoOp(void)

     /* Interface description:
         INPUT:
          None.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
	  /* NONE  */
          /* Minor */
	  /* NONE  */

   /*----------------- Start of NoOp() routine ------------------*/

#ifdef DEBUG
   printf ("NoOp() called\n") ;
#endif

   return ;

}  /*--------------------- End of NoOp() ------------------------*/


/*
______________________________________________________________________
AllocProblemItems()

FUNCTION DESCRIPTION:

     This function allocates Strings for the ProblemItems array.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/14/92
     Version 1.0 finalized: Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


void
AllocProblemItems(void)

     /* Interface description:
         INPUT:
          None.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
   int i ;

   /*-------------- Start of AllocProblemItems() routine ---------------*/

#ifdef DEBUG
   printf("XPTS(misc.c): AllocProblemItems(): Allocating ProblemItems strings.\n") ;
#endif

   if (ProblemItems == NULL)
   {  /* Allocate the needed number of strings for ProblemItems */
      ProblemItems = (String *)XtCalloc(MAXBRANCHES, sizeof(String)) ;
      for (i=0; i<MAXBRANCHES; ++i)
      {  ProblemItems[i]=(String)XtMalloc(DBPATHLEN) ;
         ProblemItems[i][0]='\0' ;
      }
   }
   else for (i=0; i<MAXBRANCHES; ++i)  /* Allocate any needed strings */
   {  if (ProblemItems[i] == NULL)
         ProblemItems[i]=(String)XtMalloc(DBPATHLEN) ;
      ProblemItems[i][0]='\0' ;
   }

#ifdef DEBUG
   printf("XPTS(popup.c): AllocProblemItems(): Done allocating ProblemItems strings.\n") ;
#endif
} /*-------------- End of AllocProblemItems() routine ---------------*/


/*
______________________________________________________________________
FreeItems()

FUNCTION DESCRIPTION:

     This function Frees items from an array of Strings that are
empty.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/14/92
     Version 1.0 finalized: Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


void
FreeItems(char **Items, int MaxItems)

     /* Interface description:
         INPUT:
          Items		- Array of strings to free.
	  MaxItems	- Maximum number of strings to free.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
   int i ;

   /*----------------- Start of FreeItems() routine ------------------*/

#ifdef DEBUG
   printf("XPTS(misc.c): FreeItems(): Removing extra Items strings with XtFree().\n") ;
#endif

   /* Remove extra Items strings with XtFree().*/
   if (Items != NULL)
      for (i=0; i<MaxItems; ++i)
      {  if (Items[i] != NULL)
            if (Items[i][0] == '\0')
            {  XtFree(Items[i]) ;
               Items[i]=NULL ;
            }
      }

#ifdef DEBUG
   printf("XPTS(misc.c): FreeItems(): Done removing extra Items strings with XtFree().\n") ;

   for (i=0; Items[i] != NULL; ++i)
   { printf("XPTS(misc.c): Items[%d]=%p\n",i, Items[i]) ;
      printf("XPTS(misc.c): Items[%d]='%s'\n", i, Items[i]) ;
   }
   printf("XPTS(misc.c): Items[%d] is NULL\n", i) ;

#endif

}  /*--------------------- End of FreeItems() ------------------------*/


/*
______________________________________________________________________
AppMainLoop()

FUNCTION DESCRIPTION:

     This function is the X application main loop, and is patterned
after XtAppMainLoop().  It is used to aid debugging efforts, since
it's difficult do some (debugging) things once you call XtAppMainLoop().
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/16/92
     Version 1.0 finalized: Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


void
AppMainLoop(XtAppContext app_con)

     /* Interface description:
         INPUT:
          app_con	- The application context structure.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   XEvent event ;
   Arg args[5] ;

          /* Minor */
   String tmpstr ;
   int lasttype=0 ; 


   /*----------------- Start of AppMainLoop() routine ------------------*/

   forever()
   {  
      XtAppNextEvent(app_con, &event) ;

/* This spews like crazy. */
/*
#ifdef DEBUG
      if (lasttype != event.type)
      {  lasttype=event.type ;
         printf("XPTS(AppMainLoop()) -- Event: <%d> ", event.type) ;
         printf("\n") ;
      }
#endif
*/

#ifndef NOSIGNALS
      if (SigRec.count) DoSignal(SigRec.signal) ;/* Deal with pending signals*/
#endif

      XtDispatchEvent(&event) ;

#ifndef NOSIGNALS
      if (SigRec.count) DoSignal(SigRec.signal) ;/* Deal with pending signals*/
#endif
   }
}  /*--------------------- End of AppMainLoop() ------------------------*/


/*
______________________________________________________________________
PrintNewProblemLabel()

FUNCTION DESCRIPTION:

	This function is used for debugging.  It prints the label
associated with the "New Problem" button, if defined.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/20/92
     Version 1.0 finalized: Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


void
PrintNewProblemLabel(void)

     /* Interface description:
         INPUT:
          None.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
   String tmpstr ;
   Arg args[5] ;

          /* Minor */
   int i ;


   /*------------ Start of PrintNewProblemLabel() routine -------------*/

   if (GlobalNewProblemButton != NULL)
     if (XtIsWidget(GlobalNewProblemButton))
      {  XtSetArg(args[0], XtNlabel, &tmpstr) ;
         XtGetValues(GlobalNewProblemButton, args, ONE) ;
         printf("-- label = `%s' ", tmpstr) ;
      }

}  /*---------------- End of PrintNewProblemLabel() -------------------*/


/*
______________________________________________________________________
LockedProblem()

FUNCTION DESCRIPTION:

        This function will set or reset the flag indicating that there
is a locked problem, or query the current state of the flag.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/21/92
     Version 1.0 finalized: Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


int
LockedProblem(int newflag)
     /* Interface description:
         INPUT:
	  newflag	- The new value of the flag, TRUE or FALSE, or
			  QUERY, to just return the current flag.
         OUTPUT:
          None.
         RETURN VALUE:
          int		- True if the flag is true, false if it is false.
     */

{
     /* Internal variables: */
          /* Major */
   static int flag=FALSE ;
          /* Minor */
   int tmpflag ;

   /*----------------- Start of LockedProblem() routine ------------------*/
 
#  ifdef DEBUG
   printf("XPTS(misc.c): LockedProblem() -- flag = %d; newflag = %d\n", flag, newflag) ;
#  endif /*DEBUG*/

   if (newflag == QUERY) return(flag) ;

   tmpflag=flag ;
   flag=newflag ;
   return(tmpflag) ;

}  /*--------------------- End of LockedProblem() ------------------------*/


/*
______________________________________________________________________
PTS_XIOErrorHandler()

FUNCTION DESCRIPTION:

        This function handles fatal IO errors for X.
______________________________________________________________________
REVISION HISTORY:
     Author:  X Library authors			Date:  1989

     Revised:  Dean Collins			Date:  11/28/92
	Added call to ShutDownDB().
	Changed to modern ANSI function declaration.

     Version 1.0 finalized:  Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


int
PTS_XIOErrorHandler(Display *display)

     /* Interface description:
         INPUT:
          display	- The X display pointer.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{ 
     /* Internal variables: */
          /* Major */
   char msg[80] ;
          /* Minor */
          /* NONE  */

   /*-------------- Start of PTS_XIOErrorHandler() routine ---------------*/

#  ifdef DEBUG
   fprintf(stderr, "XPTS(misc.c): Entering PTS_XIOErrorHandler()...\n") ;
#endif

   fprintf(stderr, "Fatal IO error detected!\n"
      "XPTS is unlocking any locked problems and exiting immediatly...\n") ;
   

     /* Unlock any locked problems... */

   ShutDownDB(FALSE) ;


     /* Print error message */
   fprintf (stderr, "XIO:  fatal IO error %d (%s) on X server \"%s\"\r\n",
           errno, strerror(errno), DisplayString (display)) ;
   fprintf (stderr, "      after %lu requests (%lu known processed) "
		    "with %d events remaining.\r\n",
            NextRequest(display) - 1, LastKnownRequestProcessed(display),
	    QLength(display)) ;

   if (errno == EPIPE)
   {
      fprintf (stderr, "      The connection was probably broken by a server "
   		       "shutdown or KillClient.\r\n") ;
   }

#  ifdef DEBUG
   fprintf(stderr, "XPTS(misc.c): Leaving PTS_XIOErrorHandler().\n") ;
#endif

   exit (1) ;

}  /*------------------ End of PTS_XIOErrorHandler() ---------------------*/


/*
______________________________________________________________________
PTS_XErrorHandler()

FUNCTION DESCRIPTION:

        This function handles fatal X errors.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  X Library authors			Date:  1989

     Revised:  Dean Collins			Date:  11/28/92
        Added call to ShutDownDB().
	Changed to modern ANSI function declaration.
     Revised:  Dean Collins			Date:  3/7/94
	Added USESYSDEFERR.  See comment for XPrintDefaultError().

     Version 1.0  finalized:  Dean Collins	Date:  1/9/93
     Version 1.04 finalized:  Dean Collins	Date:  3/7/94
______________________________________________________________________
*/


int
PTS_XErrorHandler(Display	*dpy,
		  XErrorEvent	*event) 

     /* Interface description:
         INPUT:
	  dpy		- The X display pointer.
	  event		- The X error event record.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
          /* NONE  */

   /*-------------- Start of PTS_XErrorHandler() routine ---------------*/

#   ifdef DEBUG
    fprintf(stderr, "XPTS(misc.c): Entering PTS_XErrorHandler()...\n") ;
#   endif

     /* Use XPrintDefaultError() or XmuPrintDefaultErrorMessage() to print
      * the error message.
      */
#ifndef USESYSDEFERR
    if (XPrintDefaultError (dpy, event, stderr) == 0) return 0 ;
#else
    if (XmuPrintDefaultErrorMessage (dpy, event, stderr) == 0) return 0 ;
#endif /*USESYSDEFERR*/

     /* OK, Here's my other change... Call ShutDownDB() before exiting.
      * This is the whole reason for having this otherwise duplicate code.
      */
   fprintf(stderr, "X error detected!\n"
      "XPTS is unlocking any locked problems and exiting immediatly...\n") ;

    ShutDownDB(FALSE) ;

    exit(1) ;

}  /*------------------ End of PTS_XErrorHandler() ---------------------*/


/*
______________________________________________________________________
XPrintDefaultError()

FUNCTION DESCRIPTION:

        This function prints an error message for a fatal X error.
______________________________________________________________________
REVISION HISTORY:
     Author:  X Library authors		Date:  1989

     Revised:  Dean Collins      	Date:  11/28/92
	Changed function declaration to ANSI syntax.  
     Revised:  Dean Collins      	Date:  12/6/92
	Added BadGC hack to ignore mysterious protocol error.  (yuck.)
	(Somebody please figure out what is causing this...)
     Revised:  Dean Collins		Date:  3/7/93
	Changed from ignoring BadGC to printing a special message.
	Please help me track down this problem!
     Revised:  Dean Collins		Date:  3/7/94
	Added USESYSDEFERR flag for systems where this function
	breaks (it uses variables that are not necessarily available
	on all systems unless they have "true" X11R4/5 from MIT.)
	If they're using a vendor's version it may well break,
	so just call the default error handler instead.
	No catching of BadGC, however, if this is done.

     Version 1.0  finalized:  Dean Collins	Date:  1/9/93
     Version 1.01 finalized:  Dean Collins	Date:  3/7/93
     Version 1.04 finalized:  Dean Collins	Date:  3/7/94 (a whole year?!)
______________________________________________________________________
*/


int
XPrintDefaultError(Display *dpy,
		   XErrorEvent *event,
		   FILE *fp)

     /* Interface description:
         INPUT:
	  dpy		- The X display pointer.
	  event		- The X error event structure.
	  fp		- The file to print the error message to.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
    char buffer[BUFSIZ] ;
    char mesg[BUFSIZ] ;
    char number[32] ;
    char *mtype = "XlibMessage" ;
#ifndef NOEXTVAR
    register _XExtension *ext = (_XExtension *)NULL ; 
#endif

          /* Minor */
          /* NONE  */


   /*-------------- Start of XPrintDefaultError() routine ---------------*/

#ifdef USESYSDEFERR
    XmuPrintDefaultErrorMessage (dpy, event, fp) ;
#else

/* Borrowed, verbatum, from R4 Xlib _PrintDefaultError() function. All I 
 * changed (originally) was the declaration, which is now modern ANSI.
 * I've added some #ifdefs and a variable initialization.
 *
 * _XExtension is private to the library, so this routine may break.
 * If it does, eliminate all references to _XExtension and the variable 'ext'
 * by definining NOEXTVAR.
 */

    buffer[0] = '\0' ; /* Added DC, 5/6/94*/
    XGetErrorText(dpy, event->error_code, buffer, BUFSIZ) ;
    XGetErrorDatabaseText(dpy, mtype, "XError", "X Error", mesg, BUFSIZ) ;
    (void) fprintf(fp, "%s:  %s\n  ", mesg, buffer) ;
    XGetErrorDatabaseText(dpy, mtype, "MajorCode", "Request Major code %d",
        mesg, BUFSIZ) ;
    (void) fprintf(fp, mesg, event->request_code) ;
    if (event->request_code < 128) {
        sprintf(number, "%d", event->request_code) ;
        XGetErrorDatabaseText(dpy, "XRequest", number, "", buffer, BUFSIZ) ;
    }
      else {
#ifndef NOEXTVAR
        for (ext = dpy->ext_procs ;
             ext && (ext->codes.major_opcode != event->request_code) ;
             ext = ext->next)
          ;
        if (ext)
            strcpy(buffer, ext->name) ;
        else
            buffer[0] = '\0' ;

#endif /*NOEXTVAR*/
    }
    (void) fprintf(fp, " (%s)\n  ", buffer) ;
    XGetErrorDatabaseText(dpy, mtype, "MinorCode", "Request Minor code %d",
        mesg, BUFSIZ) ;
    (void) fprintf(fp, mesg, event->minor_code) ;
#ifndef NOEXTVAR
    if (ext) {
        sprintf(mesg, "%s.%d", ext->name, event->minor_code) ;
        XGetErrorDatabaseText(dpy, "XRequest", mesg, "", buffer, BUFSIZ) ;
        (void) fprintf(fp, " (%s)", buffer) ;
    }
#endif
    fputs("\n  ", fp) ;
    XGetErrorDatabaseText(dpy, mtype, "ResourceID", "ResourceID 0x%x",
        mesg, BUFSIZ) ;
    (void) fprintf(fp, mesg, event->resourceid) ;
    fputs("\n  ", fp) ;
    XGetErrorDatabaseText(dpy, mtype, "ErrorSerial", "Error Serial #%d",
        mesg, BUFSIZ) ;
    (void) fprintf(fp, mesg, event->serial) ;
    fputs("\n  ", fp) ;
    XGetErrorDatabaseText(dpy, mtype, "CurrentSerial", "Current Serial #%d",
        mesg, BUFSIZ) ;
/*    (void) fprintf(fp, mesg, dpy->request);*/
    fputs("\n", fp) ;
    if (event->error_code == BadImplementation) return 0 ;

    if (event->error_code == BadGC)	/* Does this still happen??? */ 
    {	fputs("\nNOTICE: You have uncovered the mysterious BadGC error.\n",fp) ;
	fputs("Please send a description of EXACTLY what you were doing\n",fp) ;
	fputs("when this occured to \"dean@halcyon.com\".  Include all\n",fp) ;
	fputs("error information possible.  You assistance in tracking\n",fp) ;
	fputs("down this error will be greatly appreciated.\n",fp) ;
    }

#endif /*USESYSDEFERR*/

    return 1 ;


}  /*------------------ End of XPrintDefaultError() ---------------------*/


/*
______________________________________________________________________
ReadList()

FUNCTION DESCRIPTION:

        This function reads a list of summaries and fills in the list
for the problem selection or the unsolved problems windows.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/29/92
     Version 1.0 finalized:  Dean Collins	Date:  1/9/93
______________________________________________________________________
*/

int
ReadList(Widget popup,
         Widget probViewer)

     /* Interface description:
         INPUT:
	  popup		- The popup that contains the list to update.
	  probViewer	- The parent widget of any error message popups.
         OUTPUT:
          None.
         RETURN VALUE:
          int		- 0 means success, non-zero means error.
     */

{
     /* Internal variables: */
          /* Major */
   static String EmptySummariesList[] = { EMPTY_SUMMARIES, NULL } ;
   static String EmptyUnsolvedList[]  = { EMPTY_UNSOLVED,  NULL } ;
   Widget list=NULL ;
   Widget parent=NULL ;
   static String *UnsolvedList=NULL ;
   char errorstr[MAXERROR] ;
   String *NewList=NULL ;
   Widget warning=NULL ;

          /* Minor */
   int i ;
   int flag ;
   Arg args[5] ;


   /*----------------- Start of ReadList() routine ------------------*/

     /* Assign local list widget a value */
   list = XtNameToWidget(popup, "*list") ;


     /* Re-read the problem summaries for the current problem type,
      * using ChangeList, giving it the current problem type as
      * argument one, and an XtPointer array as argument two,
      * where the array contains the list, dialog and parent widgets.
      */
#ifdef DEBUG
   printf("XPTS(misc.c): Updating popup's summary list...\n") ;
   printf("XPTS(misc.c): popup's name='%s'\n",
          XtName(popup)) ;
#endif


                /* IS POPUP pickProblem? */
   if (strcmp(XtName(popup), "pickProblem") == 0)
   {
     /**** Popup is pickProblem popup ****/

#ifdef DEBUG
      
      printf("XPTS(misc.c): Calling ReadSummaries()...") ;
#endif
     /* Have to store the new list in NewList and not directly in
      * SummaryItems since the list widget requires that its list resource
      * only be changed through XawListChange() or XtSetValues(),
      * and it must exist unchanged otherwise.
      */
/*CLD*/ CloudSummaries = ReadSummaries(ProbTree, Path, &NewList) ;
#ifdef DEBUG
      printf("XPTS(misc.c): Returned from ReadSummaries()...\n") ;
#endif

#ifdef DEBUG
      if (NewList != NULL)
      {  for (i=0; NewList[i] != NULL; ++i)
         {  printf("XPTS(misc.c): NewList[%d]=%p\n",i, NewList[i]) ;
            printf("XPTS(misc.c): NewList[%d]='%s'\n", i, NewList[i]) ;
         }
         printf("XPTS(misc.c): NewList[%d] is NULL\n", i) ;
      } else printf("XPTS(misc.c): NewList is NULL\n") ;
#endif

      flag=0 ;
      if (NewList == NULL) flag=1 ;
      else if ((int) (NewList[0]) == (int)NULL) flag=1 ;

      if (flag)
      {  /* NO PROBLEMS! */
         /* if an empty list was returned, set the value of list to
          * be EmptySummariesList.
          */

#ifdef DEBUG
         printf("XPTS(misc.c): Setting list to be EMPTY_SUMMARIES.\n") ;
#endif


         XawListChange(list, EmptySummariesList, ONE, ZERO, TRUE) ;
         XtSetSensitive(list, False) ;
      }
      else
      {
         /* if an empty list was not returned, set the value of list to
          * be NewList.
          */
         XawListChange(list, NewList, ZERO, ZERO, TRUE) ;
         XtSetSensitive(list, True) ;
      }
      if (SummaryItems) FreeList(SummaryItems) ;
      SummaryItems=NewList ;
   }



   else  /**** Popup is unsolvedPopup ****/
   {

#ifdef DEBUG
      printf("XPTS(misc.c): Updating UnsolvedList...\n") ;
#endif
     /* Set the value of list to be the list of unsolved problem summaries. */

     /* Have to store the new list in NewList and not directly in
      * SummaryItems since the list widget requires that its list resource
      * only be changed through XawListChange() or XtSetValues(),
      * and it must exist unchanged otherwise.
      */
/*CLD*/   CloudUnSummaries = ReadUnsolvedSummaries(ProbTree, &NewList) ;
/* <= means error, 0 means none, positive is ptr to linked list. */
#ifdef DEBUG
      printf("XPTS(misc.c): CloudUnSummaries = %p\n", CloudUnSummaries) ;
#endif

      if ((int)CloudUnSummaries < 0)
      {
         sprintf(errorstr, "Error reading problem summaries!\n\n"
                "Possible error cause:\n%s\n",
/*CLD*/         CloudStrerror()) ;
         ErrorPopup("error", probViewer, errorstr) ;
         return (-1) ;
      }

#ifdef DEBUG
      if (NewList != NULL)
      {  for (i=0; NewList[i] != NULL; ++i)
         {  printf("XPTS(misc.c): NewList[%d]=%p\n",i, NewList[i]) ;
            printf("XPTS(misc.c): NewList[%d]='%s'\n", i, NewList[i]) ;
         }
         printf("XPTS(misc.c): NewList[%d] is NULL\n", i) ;
      } else printf("XPTS(misc.c): NewList is NULL\n") ;
#endif

      flag=0 ;
      if (NewList == NULL) flag=1 ;
      else if ((int) (NewList[0]) == (int)NULL) flag=1 ;

      if (flag)
      {  /* NO UNSOLVED PROBLEMS!  EXCELLENT! */
         /* if an empty list was returned, set the value of list to
          * be EmptyUnsolvedList.
          */
         XawListChange(list, EmptyUnsolvedList, ZERO, ZERO, TRUE) ;
         XtSetSensitive(list, False) ;
      }
      else
      {  /* if an empty list was not returned, set the value of list to
          * be NewList.
          */
         XawListChange(list, NewList, ZERO, ZERO, TRUE) ;
         XtSetSensitive(list, True) ;
      }
      if (UnsolvedList) FreeList(UnsolvedList) ;
      UnsolvedList=NewList ;

   }

   return (0) ;

}  /*--------------------- End of ReadList() ------------------------*/


/*
______________________________________________________________________
ShutDownDB()

FUNCTION DESCRIPTION:

        This function unlocks any currently locked problems.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/21/92
     Version 1.0 finalized:  Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


void
ShutDownDB(int AllowPopups)

     /* Interface description:
         INPUT:
          AllowPopups		- Flag for whether to warn/inform with popups.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
          /* NONE  */

   /*----------------- Start of ShutDownDB() routine ------------------*/

#  ifdef DEBUG
   printf("XPTS(misc.c): ShutDownDB() is making sure the database "
          "is shut down nicely...\n") ;
   printf("XPTS(misc.c): LockedProblem() returns: %d\n",
	  LockedProblem(QUERY)) ;
#  endif


   if (LockedProblem(FALSE))
   {
#     ifdef DEBUG
      printf("XPTS(misc.c): Calling UnlockProblem()...\n") ;
#     endif
        /* Unlock the problem with UnlockProblem() since we're done
         * editing it.
         */
/*CLD*/ if (UnlockProblem(Path, ProblemRecord) == (int)NULL)
      {
         /* An error occured; call ErrorPopup if allowed */

         if (AllowPopups) 
         {  char errorstr[MAXERROR] ;
            sprintf(errorstr, "Error unlocking problem!\n"
                              "\nPossible error cause:\n%s\n",
/*CLD*/                       CloudStrerror()) ;
            ErrorPopup("error", GlobalMainMenuWidget, errorstr) ;
            return ;
         }
         else fprintf(stderr, "xpts: Error unlocking problem!\n") ;
         return ;
      }
#     ifdef DEBUG
      printf("XPTS(misc.c): Returned without an error from UnlockProblem().\n") ;
#     endif
   }
#  ifdef DEBUG
   printf("XPTS(misc.c): LockedProblem() was FALSE.\n") ;
#  endif

}  /*--------------------- End of ShutDownDB() ------------------------*/



/*
______________________________________________________________________
SuperWarningPopup()

FUNCTION DESCRIPTION:

        This function prompts the user for a True/False selection.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/15/92
     Version 1.0 finalized:  Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


int
SuperWarningPopup(String     name,
                  Widget      parent,
                  String      message)

     /* Interface description:
         INPUT:
          name           - The resource name of the popup.
          parent         - The parent of the popup widget.
          message        - The specific error message to display.
         OUTPUT:
          None.
         RETURN VALUE:
          int		 - True/False user selection.
     */

{
     /* Internal variables: */
          /* Major */
   char btnname[80] ;
   Widget warningPopup=NULL ;
   Widget quitButton=NULL ;
   Widget continueButton=NULL ;
   XtPointer *ptrlist=NULL ;
   XEvent event ;
   XtAppContext app_con ;

          /* Minor */
   int done=0 ;
   int ToggledSensitive=FALSE ;

   /*-------------- Start of SuperWarningPopup() routine ---------------*/

#ifdef DEBUG
   printf("XPTS(misc.c): Entering SuperWarningPopup.\n") ;
#endif


   if (XtIsSensitive(parent) == True)
   {  XtSetSensitive(parent,False) ;
      ToggledSensitive=TRUE ;
#ifdef DEBUG
      printf ("XPTS(misc.c): Parent was sensitive.\n") ;
#endif
   }
#ifdef DEBUG
   else
      printf ("XPTS(misc.c): Parent was insensitive.\n") ;
#endif

     /* Use an ErrorPopup widget to start with, even though it's not
      * REALLY an error.
      */
   ErrorPopup(name, parent, message) ;

	/* Have the quit button also call SuperWarningFalse()  */
   sprintf(btnname, "*%s*quitButton", name) ;
#ifdef DEBUG
   printf("XPTS(signal.c): SuperWarningPopup() converting name '%s' to widget...\n", btnname) ;
#endif
   quitButton = XtNameToWidget(parent, btnname) ;
   XtAddCallback(quitButton, XtNcallback, (XtCallbackProc)SuperWarningFalse,
                 parent) ;

	/* Have the continue button also call SuperWarningTrue()  */
   continueButton = XtCreateManagedWidget("continueButton", commandWidgetClass,
                                          XtParent(quitButton), NULL, ZERO) ;
   XtAddCallback(continueButton, XtNcallback, (XtCallbackProc)SuperWarningTrue,
		 parent) ;
   
   warningPopup = XtNameToWidget(parent, name) ;
   ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) warningPopup ;
   if (ToggledSensitive == TRUE)
        ptrlist[1] = (XtPointer) parent ;
   else ptrlist[1] = NULL ;
#ifdef DEBUG
   printf("XPTS(popup.c): ptrlist[1]=%p\n", ptrlist[1]) ;
#endif
   XtAddCallback(continueButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;


     /* Loop, watching for the DestroyNotify event */
   app_con = XtWidgetToApplicationContext(warningPopup) ;
   while (!done)
   {  XtAppNextEvent(app_con, &event) ;
      if (event.type == DestroyNotify)
      {
#ifdef DEBUG
         printf("XPTS(signal.c): Found a 'DestroyNotify' event.\n") ;
#endif
         done=1 ;
      }
      XtDispatchEvent(&event) ;
   }

#ifdef DEBUG
   printf("XPTS(signal.c): Leaving SuperWarningPopup.\n") ;
#endif

   return (WarningFlag) ;

}  /*------------------ End of SuperWarningPopup() ---------------------*/


/*
______________________________________________________________________
SuperWarningTrue()

FUNCTION DESCRIPTION:

        This function is called if the user selects the continue button
for the warning popup.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/15/92
     Version 1.0 finalized:  Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


int
SuperWarningTrue(   Widget        w,
                 XtPointer        client_data,
                 XtPointer        call_data)
                           
     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          client_data    - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback,
                           this is ignored.
          call_data      - Specifies any callback-specific data the widget w
                           wants to pass to the client.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{
     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
          /* NONE  */

   /*-------------- Start of SuperWarningTrue() routine ---------------*/

#ifdef DEBUG
   printf("XPTS(signal.c): Called SuperWarningTrue!!!!!\n") ;
#endif
   return(WarningFlag=TRUE) ;

}  /*------------------ End of SuperWarningTrue() ---------------------*/


/*
______________________________________________________________________
SuperWarningFalse()

FUNCTION DESCRIPTION:

        This function is called if the user selects the quit button
for the warning popup.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins			Date:  11/15/92
     Version 1.0 finalized:  Dean Collins	Date:  1/9/93
______________________________________________________________________
*/


int
SuperWarningFalse(   Widget        w,
                 XtPointer        client_data,
                 XtPointer        call_data)

     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          client_data    - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback,
                           this is ignored.
          call_data      - Specifies any callback-specific data the widget w
                           wants to pass to the client.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */

{

     /* Internal variables: */
          /* Major */
          /* NONE  */
          /* Minor */
          /* NONE  */

   /*-------------- Start of SuperWarningFalse() routine ---------------*/

#ifdef DEBUG
   printf("XPTS(signal.c): Called SuperWarningFalse!!!!!\n") ;
#endif
   return(WarningFlag=FALSE) ;

}  /*------------------ End of SuperWarningFalse() ---------------------*/

/* end of misc.c */
