/*
 * popup.c -- Popup window functions for the X user interface
 *            to the Problem Tracking System (PTS) database.
 * Dean Collins 
 * created: Fri Apr 24 10:11:55 PDT 1992
 */

/*
 * Copyright (c) 1995,1994,1993,1992 Dean Collins.
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 * 
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notices appear in all copies and
 * that both those copyright notices and this permission notice appear in
 * supporting documentation, and that neither the name of the University of
 * Idaho nor the name of Dean Collins be used in advertising or publicity
 * pertaining to distribution of the software without specific, written
 * prior permission from both parties.  Neither The University of Idaho
 * nor Dean Collins make any representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 * 
 * THE UNIVERSITY OF IDAHO AND DEAN COLLINS DISCLAIM ALL WARRANTIES WITH
 * REGARD TO THIS SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS.  IN NO EVENT SHALL THE UNIVERSITY OF IDAHO
 * OR DEAN COLLINS BE LIABLE FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL
 * DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR
 * PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS
 * ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF
 * THIS SOFTWARE.
 */

#include <stdlib.h>             /* General C utilities header file */
#include <stdio.h>		/* Standard input/output utilities hdr. file */

#include <X11/Intrinsic.h>      /* Include standard Toolkit Header file. */
#include <X11/StringDefs.h>     /* Include StringDefs header file */
#include <X11/Shell.h>          /* Include the Shell header file for popups */

#include <X11/Xaw/AsciiText.h>  /* Include the ascii Text widget hdr. file */
#include <X11/Xaw/Box.h>        /* Include the Box widget header file */
#include <X11/Xaw/Command.h>    /* Include the Command widget header file */
#include <X11/Xaw/Dialog.h>     /* Include the Dialog widget header file */
#include <X11/Xaw/Label.h>      /* Include the Label widget header file */
#include <X11/Xaw/Form.h>       /* Include the Form widget header file */
#include <X11/Xaw/List.h>       /* Include the List widget header file */
#include <X11/Xaw/Paned.h>      /* Include the Paned widget header file */
#include <X11/Xaw/Viewport.h>   /* Include the Viewport widget header file */

#include <X11/Xaw/Cardinals.h>  /* Definitions of ZERO, ONE, etc... */

#include <X11/Xmu/SysUtil.h>	/* System utility functions */

#include "zdbm.h"               /* Zombie Database Manager header file */
#include "cloud.h"		/* Nebulous Cloud header file */
#include "clouderror.h"         /* Nebulous Cloud error rtn. header file */
#include "xpts.h"               /* Xpts header file */


extern bool NoQuit ;		/* True if user started PTS in "no-quit" mode, */
				/* false otherwise. */

      /* The current problem record.  This ideally would
       * be part of a new widget.  The widget would have the
       * resources listed in ProblemType.  It would have functions
       * to display, read and write the problem.
       * Next version, maybe...
       */
extern problem_record * ProblemRecord ;

      /* The current list of problem summaries.  Similar
       * comment to that above.
       */
extern String *SummaryItems ;

        /* The problem types tree. */
extern ProblemTree *ProbTree ;

        /* Information about the user. */
extern UserInfo UserInfoRec ;

        /* The path for the problem types tree. */
extern char Path[] ;
extern char Node[] ;
extern char ProblemText[] ;

extern String *ProblemItems ;

/* The following string will be displayed by the "About PTS" button
 * on the Main Menu: 
 */
extern const char PTSInfoText[] ;

extern Summary *CloudUnSummaries ;
extern Summary *CloudSummaries ;

extern Widget GlobalNewProblemButton ;
extern Widget GlobalPreviousButton ;



/******************** MODULES *************************************/
/*
______________________________________________________________________
EditMOTDPopup()

FUNCTION DESCRIPTION:

     This function will create a popup showing the Message of the Day file
and allow the sysop user to edit it and save changes.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/25/92
______________________________________________________________________
*/


void
EditMOTDPopup(Widget       w, 
              Widget       parent)

     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          parent         - This is the parent widget.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget popup=NULL ;          /* The help popup to create. */
   Widget text=NULL ;           /* The text widget to put in the popup. */
   Widget quitButton=NULL ;     /* The help popup quit button. */
   Widget saveButton=NULL ;     /* The help popup "save & quit" button. */
   Widget buttonBox=NULL ;
   Widget ppanes=NULL ;
          /* Minor */
   static XtPointer *ptrlist=NULL ;
   Arg args[3] ;
   char errorstr[MAXERROR] ;

   /*----------------- Start of EditMOTDPopup() routine ------------------*/

#ifdef DEBUG
   printf("XPTS(popup.c): Entering EditMOTDPopup()\n") ;
   printf("XPTS(popup.c): Setting parent to be not sensitive.\n") ;
#endif

      /* The parent's widgets keep getting mangled when changing the
       * list widget, so unmanage them for now.
       */
   if (ppanes = XtNameToWidget(parent, "*panes"))
      XtUnmanageChild(ppanes) ;

   XtSetSensitive(parent,False) ;

     /* Check that the MOTD file exists.  Return with error if it doesn't. */
   if (!FileExists(MOTDFILE))
   {  if (ppanes) XtManageChild(ppanes) ;
      sprintf(errorstr, "Message of the Day file %s does not exist!", MOTDFILE) ;
      ErrorPopup("error", parent, errorstr) ;
      return ;
   }

     /* Check that the MOTD file is writable by the user.  Return with error
      * if it doesn't.
      */
   if (!IsWritable(MOTDFILE))
   {  if (ppanes) XtManageChild(ppanes) ;
      sprintf(errorstr,
		"The Message of the Day file '%s' is not writable by you!\n"
		"Check file permissions.", MOTDFILE) ;
      ErrorPopup("error", parent, errorstr) ;
      return ;
   }

     /* Create the popup using ViewTextPopup() */
   ViewTextPopup("motdPopup", &popup, parent, &text, &buttonBox, &quitButton) ;
   (void)XawTextDisplayCaret(text, TRUE) ;

   XtSetArg(args[0], XtNstring, MOTDFILE) ;
   XtSetArg(args[1], XtNtype, XawAsciiFile) ;
   XtSetValues(text, args, TWO) ;

     /* Add another button used save the MOTD. */
   saveButton = XtCreateManagedWidget("saveButton", commandWidgetClass,
                                      buttonBox, NULL, ZERO) ;

     /* Add a callback for the quit button that destroys the popup
      * using DestropPopup(). 
      */
/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) popup ;
   ptrlist[1] = (XtPointer) parent ;
   /*ptrlist[1] = NULL ;*/
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

     /* Add a callback for the save button that saves the MOTD file
      * using SaveMOTDButtonSelect().
      */
   XtAddCallback(saveButton, XtNcallback, SaveMOTDButtonSelect,
                 (XtPointer)text) ;

     /* Enable the text widget cursor (enabled by default with the
      * resources specifications.)
      */

     /* Draw the popup */
   XtPopup(popup, XtGrabExclusive) ;

   if (ppanes) XtManageChild(ppanes) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving EditMOTDPopup()\n") ;
#endif

}  /*--------------------- End of EditMOTDPopup() ------------------------*/


/*
______________________________________________________________________
HelpPopup()

FUNCTION DESCRIPTION:

     This function will create a popup containing the PTS help file and create
a callback which destroys the help popup when the popup's acknowledgement
button is selected by the user.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
HelpPopup(   Widget        w, 
          XtPointer        client_data, 
          XtPointer        call_data)

     /* Interface description:
         INPUT:
          w             - The widget for which this callback is registered.
          client_data   - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback, 
                           this is the parent widget.
          call_data     - Specifies any callback-specific data the widget w
                           wants to pass to the client.
           
         OUTPUT:
           
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget popup=NULL ;     /* The help popup to create. */
   Widget text=NULL ;      /* The text widget to place in the popup */
   Widget quitButton=NULL ;/* The help popup quit button. */
   Widget buttonBox=NULL ;
   Widget parent=NULL ;
   Widget ppanes=NULL ;
          /* Minor */
   XtPointer *ptrlist=NULL ;
   Arg args[3] ;



   /*----------------- Start of HelpPopup() routine -----------------------*/

#ifdef DEBUG
   printf("XPTS(popup.c): Entering HelpPopup()\n") ;
#endif

   parent = (Widget) client_data ;

      /* The parent's widgets keep getting mangled when changing the
       * list widget, so unmanage them for now.
       */
   if (ppanes = XtNameToWidget(parent, "*panes"))
      XtUnmanageChild(ppanes) ;

   XtSetSensitive(parent,False) ;

     /* Create the help popup using ViewTextPopup() */
   ViewTextPopup("helpPopup", &popup, parent, &text,
                 &buttonBox, &quitButton) ;

     /* Add a callback for the quit button that destroys the popup
      * when selected.
      */
   ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) popup ;
   ptrlist[1] = (XtPointer) parent ;
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

     /* Set resources for the text widget so it displays the help file. */

     /* Draw the popup */
   XtPopup(popup, XtGrabExclusive) ;

   if (ppanes) XtManageChild(ppanes) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving HelpPopup()\n") ;
#endif

}  /*--------------------- End of HelpPopup() -----------------------------*/


/*
______________________________________________________________________
InfoPopup()

FUNCTION DESCRIPTION:

     This function will display information about PTS, such as the current
version, the PTS project sponsor, the name of the design team and the design
team member names.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
InfoPopup(Widget     w, 
          XtPointer  client_data, 
          XtPointer  call_data)

     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          client_data    - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback, 
                           this is the parent widget.
          call_data      - Specifies any callback-specific data the widget w
                           wants to pass to the client.
           
         OUTPUT:
           
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget popup=NULL ;          /* The help popup to create. */
   Widget text=NULL ;           /* The text widget to put in the popup. */
   Widget quitButton=NULL ;     /* The help popup quit button. */
   bool Done=FALSE ;            /* Flag indicating that a button was pressed. */
   Widget buttonBox=NULL ;
   Widget parent=NULL ; 
   Widget ppanes=NULL ; 

          /* Minor */
   XtPointer *ptrlist=NULL ;
   Arg args[3] ;      	   	/* Used for event loop argument passing. */


   /*----------------- Start of InfoPopup() routine -----------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering InfoPopup()\n") ;
   printf("XPTS(popup.c): Setting parent to be not sensitive.\n") ;
#endif

   parent = (Widget) client_data ;

      /* The parent's widgets keep getting mangled when changing the
       * list widget, so unmanage them for now.
       */
   if (ppanes = XtNameToWidget(parent, "*panes"))
      XtUnmanageChild(ppanes) ;

   XtSetSensitive(parent,False) ;

     /* Create the help popup using ViewTextPopup() */
   ViewTextPopup("infoPopup", &popup, parent, &text, &buttonBox, &quitButton) ;

     /* Add a callback for the quit button that destroys the popup
      * when selected.
      */
   ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;  
   ptrlist[0] = (XtPointer) popup ;
   ptrlist[1] = (XtPointer) parent ;
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

     /* Set resources for the text widget so it displays PTSInfoText[][]. */
   XtSetArg(args[0], XtNtype, XawAsciiString) ;
   XtSetArg(args[1], XtNstring, PTSInfoText) ;
   XtSetValues(text, args, TWO) ;

     /* Realize the popup. */
   XtPopup(popup, XtGrabExclusive) ;

   if (ppanes) XtManageChild(ppanes) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving InfoPopup()\n") ;
#endif

}  /*--------------------- End of InfoPopup() -----------------------------*/


/*
______________________________________________________________________
PickProblemPopup()

FUNCTION DESCRIPTION:

     This function will popup a problem tree navigation window and add will
allow the user to navigate through the problem tree, report a problem, or quit
to the previous screen or menu.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
PickProblemPopup(Widget     w, 
                 Widget     parent)

     /* Interface description:
         INPUT:
          w        - The widget that caused this function to be called,
                     typically a button or list item.
          parent   - The parent widget of the popup widget to be created.             
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */

          /* Major */
   Widget pickProblem=NULL ;    /* The widget used to select problems. */
   Widget list=NULL ;           /* The list widget in pickProblem. */
   Widget buttonBox=NULL ;      /* A box containing buttons in pickProblem. */
   Widget quitButton=NULL ;     /* The quit button in the buttonBox widet. */
   Widget previousButton=NULL ; /* The button which is used to return to the */
                                /*   previous (parent) group of problems. */
   Widget dialogBox=NULL ;
   Widget newProblem=NULL ;
   Widget panes=NULL ;
   Widget ppanes=NULL ;

          /* Minor */
   static XtPointer *ptrlist=NULL ;
   static XtPointer *ptrlist2=NULL ;
   Arg args[3] ;
   int i=0 ;


   /*------------ Start of PickProblemPopup() routine -------------*/

#ifdef DEBUG
   printf("XPTS(popup.c): Entering PickProblemPopup()\n") ;
   printf("XPTS(popup.c): Setting parent to be not sensitive.\n") ;
#endif

      /* The parent's widgets keep getting mangled when changing the
       * list widget, so unmanage them for now.
       */
/*   if (ppanes = XtNameToWidget(parent, "*panes")) 
      XtUnmanageChild(ppanes) ;
*/
   XtSetSensitive(parent,False) ;

     /* Create a list popup with two columns. This popup should
      * not be realized by ListPopup().
      */
   ListPopup("pickProblem", TWO, &pickProblem, parent,
          &list, &buttonBox, &quitButton) ;

     /* Insert a dialog box between the window labels and the list */
/* This doesn't work.  Leave it commented out for now. */
/*   panes = XtParent(XtParent(list)) ;
   XtManageChild(list) ;
   XtManageChild(buttonBox) ;
   dialogBox = XtCreateManagedWidget("dialogBox", dialogWidgetClass, panes,
				  NULL, ZERO) ;
   XtManageChild(list) ;
   XtManageChild(buttonBox) ;
*/

     /* Set the value of the list to the highest level of items
      * in the problem tree.
      */
   strcpy (Path, ROOTSTR) ; /* Reset the Path */


     /* Add a button to the button-box that will be used to return
      * to a previous group of problems.  This will not be realized now,
      * but will be when an item is selected.  When the user has finally
      * returned to the root group, the quit button will be used instead
      * of the previousButton.
      */

   previousButton = XtCreateWidget("previousButton", commandWidgetClass,
				   buttonBox, NULL, ZERO) ;
   newProblem = XtCreateWidget("newProblem", commandWidgetClass,
			       buttonBox, NULL, ZERO) ;
   GlobalNewProblemButton = newProblem ;
   GlobalPreviousButton = previousButton ;

     /* Set the callback for the quitButton to be DestroyPopup().
      * Set the callback for the list widget to be PickProblemActivate().
      * Set the callback for the previousButton to be PreviousMenuSelect().
      * Set the callback for the dialogBox to be GotoProblemType(). 
      * Set the callback for the new problem button to be ProblemEntryPopup().
      */

/*   if(ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) pickProblem ;
   ptrlist[1] = (XtPointer) parent ;
#ifdef DEBUG
   printf ("XPTS(popup.c): ptrlist=%p,ptrlist[0]=%p,ptrlist[1]=%p\n"
           "pickProblem=%p,parent=%p\n",
           ptrlist,ptrlist[0],ptrlist[1],pickProblem,parent) ;
#endif
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

/*   if(ptrlist2) XtFree(ptrlist2) ;*/
   ptrlist2 = (XtPointer *) XtCalloc(7, sizeof(XtPointer)) ;
   ptrlist2[0] = (XtPointer) list ;
   ptrlist2[1] = (XtPointer) dialogBox ;
   ptrlist2[2] = (XtPointer) parent ;
   ptrlist2[3] = (XtPointer) quitButton ;
   ptrlist2[4] = (XtPointer) previousButton ;
   ptrlist2[5] = (XtPointer) newProblem ;
   ptrlist2[6] = (XtPointer) pickProblem ;

#ifdef DEBUG
   printf("\tclient_data=%p\n"
          "\tclient_data[0]=%p\n"
          "\tptrlist2=%p\n"
          "\tptrlist2[0]=%p\n"
          "\tptrlist2[1]=%p\n"
          "\tptrlist2[2]=%p\n"
          "\tptrlist2[3]=%p\n"
          "\tptrlist2[4]=%p\n"
          "\tptrlist2[5]=%p\n"
          "\tptrlist2[6]=%p\n",
          ptrlist2, ptrlist2[0], ptrlist2[1], ptrlist2[2],
          ptrlist2[3], ptrlist2[4], ptrlist2[5], ptrlist2[6]) ;
#endif
   XtAddCallback(list, XtNcallback, PickProblemActivate, (XtPointer) ptrlist2) ;
   XtAddCallback(previousButton, XtNcallback, PreviousMenuSelect,
		(XtPointer) ptrlist2) ;


     /* Now that we have the pointer list we need we can call ChangeList().
      * This may have made more sense to do above.  Oh, well.
      */
   ChangeList(Path, (int)NULL, ABS_TYPE, (XtPointer)ptrlist2) ;


/*   XtAddCallback(dialogBox , XtNcallback, GotoProblemType, NULL) ;*/

   XtAddCallback(newProblem, XtNcallback, ProblemEntryPopup,
		(XtPointer) pickProblem) ;

     /* Draw the popup */
#ifdef DEBUG
   printf("XPTS(popup.c): Calling XtPopup()\n") ;
#endif
   XtPopup(pickProblem, XtGrabExclusive) ; 

/*   if (ppanes) XtManageChild(ppanes) ;*/

#ifdef DEBUG
   printf("XPTS(popup.c): Returned from XtPopup() & XtPopupdown()\n") ;
   printf("XPTS(popup.c): Leaving PickProblemPopup()\n") ;
#endif

}  /*---------------- End of PickProblemPopup() -------------------*/


/*
______________________________________________________________________
ProblemEntryPopup()

FUNCTION DESCRIPTION:

     This function will create a popup that will allow the user to enter
information about a new problem to be entered into the PTS database.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
ProblemEntryPopup(Widget     w, 
                  XtPointer  client_data, 
                  XtPointer  call_data)

     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          client_data    - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback, 
                           this is the parent.
          call_data      - Specifies any callback-specific data the widget w
                           wants to pass to the client.
           
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */

          /* Major */
   Widget probEntry=NULL ;   /* The problem entry popup widget. */
   Widget panes=NULL ;       /* The popup window panes widget. */
   Widget wintitle1=NULL ;   /* The primary title for the popup. */
   Widget wintitle2=NULL ;   /* The secondary title for the popup. */
   Widget dialogBox=NULL ;   /* The dialog box for the popup. */
   Widget type=NULL ;        /* The problem type dialog box. */
   Widget summary=NULL ;     /* The problem summary dialog box. */
   Widget node=NULL ;        /* The problem node dialog box. */
   Widget priority=NULL ;    /* The problem priority dialog box. Added 11/16/94 DC*/
   Widget typeLabel=NULL ;   /* The problem type dialog box label. */
   Widget summaryLabel=NULL ;/* The problem summary dialog box label. */
   Widget nodeLabel=NULL ;   /* The problem node dialog box label. */
   Widget priorityLabel=NULL ;/* The problem priority Label. Added 11/16/94 DC*/
   Widget parent=NULL ;
   Widget text=NULL ;
   Widget helpLine=NULL ;
   Widget helpLine2=NULL ;
   Widget buttonBox=NULL ;
   Widget continueButton=NULL ;
   Widget cancelButton=NULL ;
   Widget ppanes=NULL ;
   Widget higherButton=NULL ;/* The button that increases the priority */
   Widget lowerButton=NULL ; /* The button that decreases the priority */

          /* Minor */
   static XtPointer *ptrlist=NULL ;
   static XtPointer *ptrlist2=NULL ;
   Arg args[3] ;
   static char tmppath[DBPATHLEN+1] ;
   Widget tmpwidget ;



   /*----------------- Start of ProblemEntryPopup() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering ProblemEntryPopup()\n") ;
#endif
   parent = (Widget) client_data ;
#ifdef DEBUG
   printf("XPTS(popup.c): Setting parent to be not sensitive.\n") ;
#endif

      /* The parent's widgets keep getting mangled when changing the
       * list widget, so unmanage (unmangle) them for now.
       */
   if (ppanes = XtNameToWidget(parent, "*panes"))
      XtUnmanageChild(ppanes) ;

   XtSetSensitive(parent,False) ;


     /* Create the popup widget, The popup widgets parent should
      * be the widget "parent".
      */
   probEntry = XtCreatePopupShell("probEntry", applicationShellWidgetClass,
                                  parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(popup.c): Created popup '%s' as a child of '%s'\n",
		XtName(probEntry), XtName(XtParent(probEntry))) ;
#endif

     /* Add a paned widget to the popup widget, dividing the popup into
      * sections.
      */
   panes = XtCreateManagedWidget("panes", panedWidgetClass, probEntry,
                                 NULL, ZERO) ;

     /* Add a two title bars to the paned widget using XtCreateManagedWidget().
      */
   wintitle1 = XtCreateManagedWidget("wintitle1", labelWidgetClass, panes,
                                     NULL, ZERO) ;
   wintitle2 = XtCreateManagedWidget("wintitle2", labelWidgetClass, panes,
                                     NULL, ZERO) ;


    dialogBox = XtCreateManagedWidget("dialogBox", formWidgetClass, panes,
                                      NULL, ZERO) ;

    typeLabel = XtVaCreateManagedWidget(
		"typeLabel", labelWidgetClass, dialogBox,
		XtNborderWidth,	0,
		XtNwidth,	20,
		NULL) ;
    type = XtVaCreateManagedWidget(
		"type", asciiTextWidgetClass, dialogBox,
		XtNfromHoriz,	typeLabel,
		NULL) ;

    summaryLabel = XtVaCreateManagedWidget(
		"summaryLabel", labelWidgetClass, dialogBox,
		XtNfromVert,	type,
		XtNwidth,	20,
		NULL) ;
    summary = XtVaCreateManagedWidget(
		"summary", asciiTextWidgetClass, dialogBox,
		XtNfromVert,	type,
		XtNfromHoriz,	summaryLabel,
		NULL) ;

    nodeLabel = XtVaCreateManagedWidget(
		"nodeLabel", labelWidgetClass, dialogBox,
		XtNfromVert,	summary,
		XtNwidth,	20,
		NULL) ;
    node = XtVaCreateManagedWidget(
		"node", asciiTextWidgetClass, dialogBox,
		XtNfromVert,	summary,
		XtNfromHoriz,	nodeLabel,
		NULL) ;


#ifdef DO_PRIORITY
    priorityLabel = XtVaCreateManagedWidget(
		"priorityLabel", labelWidgetClass, dialogBox,
		XtNfromVert,	node,
		XtNwidth,	20,
		XtNheight,	45,
		NULL) ;
    priority = XtVaCreateManagedWidget(
		"priority",	labelWidgetClass, dialogBox,
		XtNfromVert,	node,
		XtNfromHoriz,	priorityLabel,
		NULL) ;
    higherButton = XtVaCreateManagedWidget(
		"higherButton", commandWidgetClass, dialogBox, 
		XtNfromVert,	node,
		XtNfromHoriz,	priority,
		NULL) ;
    lowerButton = XtVaCreateManagedWidget(
		"lowerButton", commandWidgetClass, dialogBox,
		XtNfromVert,	higherButton,
		XtNfromHoriz,	priority,
		NULL) ;
#endif

     /* Set the values of the text entry windows.  */

	/* The problem type */
   strcpy(tmppath, FixPath(Path)) ;
   XtVaGetValues(type, XtNtextSource, &tmpwidget, NULL) ;
   XtVaSetValues(tmpwidget,
		XtNstring,	tmppath,
		XtNtype,	XawAsciiString,
		NULL) ;
   XawTextDisplayCaret(type, FALSE) ;  /* Can't edit this one...*/

	/* The Host */
/*    gethostname(Node, MAXHOSTNAMELEN) ;*/
    XmuGetHostname(Node, MAXHOSTNAMELEN) ;
   XtVaGetValues(node, XtNtextSource, &tmpwidget, NULL) ;
   XtVaSetValues(node,
		XtNstring,	Node,
		XtNtype,	XawAsciiString,
		NULL) ; 
   XawTextSetInsertionPoint(node, (long)strlen(Node)) ;

	/* Set the priority label to be something reasonable. */
#ifdef DO_PRIORITY
   SetPriorityValue(priority,DEF_PRIORITY) ;
#endif


     /* Add a label indicating that the full problem description
      * should be entered in the text window (below).
      */
   helpLine2 = XtCreateManagedWidget("helpLine2", labelWidgetClass, panes,
                                     NULL, ZERO) ;

     /* Add a text widget to the paned widget using XtCreateManagedWidget().  
      */
   text = XtCreateManagedWidget("text", asciiTextWidgetClass,
                                panes, NULL, ZERO) ;

     /* Add a brief help line indicating what actions are acceptable. */
   helpLine = XtCreateManagedWidget("helpLine", labelWidgetClass, panes,
                                    NULL, ZERO) ;

     /* Add the button-box to the panes widget. */
   buttonBox = XtCreateManagedWidget("buttonBox",  boxWidgetClass, panes,
                                     NULL, ZERO) ;

     /* Create the continue and cancel buttons, adding to the
      *  buttonbox.
      */
    cancelButton = XtCreateManagedWidget("cancelButton", commandWidgetClass,
                                         buttonBox, NULL, ZERO) ;
    continueButton = XtCreateManagedWidget("continueButton", commandWidgetClass,
                                           buttonBox, NULL, ZERO) ;

     /* Add callbacks for the buttons.  Have continueButton
      * cause AddProblemSelect() to be called, and have cancelButton
      * cause DestroyPopup() to be called.
      */
/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *)XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) probEntry ;
   ptrlist[1] = (XtPointer) parent ;
   XtAddCallback(cancelButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

/*   if (ptrlist2) XtFree(ptrlist2) ;*/
   ptrlist2 = (XtPointer *)XtCalloc(5, sizeof(XtPointer)) ;
   ptrlist2[0] = (XtPointer) type ;
   ptrlist2[1] = (XtPointer) summary ;
   ptrlist2[2] = (XtPointer) node ;
   ptrlist2[3] = (XtPointer) priority ;	/* Added, 11/16/94 DC*/
   ptrlist2[4] = (XtPointer) text ;
   ptrlist2[5] = (XtPointer) ptrlist ;
   XtAddCallback(continueButton, XtNcallback, AddProblemSelect,
                  (XtPointer)ptrlist2) ;


#ifdef DO_PRIORITY
   XtAddCallback(higherButton, XtNcallback, PriorityIncSelect, priority) ;
   XtAddCallback(lowerButton,  XtNcallback, PriorityDecSelect, priority) ;
#endif


     /* Unrealize the parent widget. */

     /* Draw the popup. */
   XtPopup(probEntry, XtGrabExclusive) ;

   if (ppanes) XtManageChild(ppanes) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving ProblemEntryPopup()\n") ;
#endif

}  /*--------------------- End of ProblemEntryPopup() ------------------------*/


/*
______________________________________________________________________
ProblemViewerPopup()

FUNCTION DESCRIPTION:

     This function creates a popup to view a full problem log.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
These include:
	Tested each button; verified that each called the correct call-back
	function.
	Tested that the appropriate buttons are managed depending on whether
	the problem is solved or unsolved and whether the user is a sysop
	or not.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
ProblemViewerPopup(String	name,
		   Widget      *probViewerp,
		   Widget       parent,
                   Widget      *typep,
                   Widget      *summaryp,
                   Widget      *priorityp,
                   Widget      *textp,
                   Widget      *editp)

     /* Interface description:
         INPUT:
          name           - The name of the problem viewer popup. 
          probViewerp    - The problem viewer popup widget. 
          parent         - The parent of the popup. 
          typep		 - Ptr to the problem type widget.
          summaryp       - Ptr to the problem summary widget.
          priorityp      - Ptr to the problem priority widget.
          textp          - The problem text widget. 
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
		/* Declare one or two widgets... :-) */
Widget wintitle1=NULL ;   /* The primary title for the popup. */
Widget wintitle2=NULL ;   /* The secondary title for the popup. */
Widget panes=NULL ;       /* The popup window panes widget. */
Widget dialogBox=NULL ;   /* The dialog box for the popup. */
Widget quitButton=NULL ;
Widget buttonBox=NULL ;
Widget buttonBox2=NULL ;
Widget buttonBox3=NULL ;
Widget buttonBox4=NULL ;
Widget reopenButton=NULL ;
Widget editButton=NULL ;
Widget saveButton=NULL ;
Widget saveButton2=NULL ;
Widget saveButton3=NULL ;
Widget cancelButton=NULL ;
Widget cancelButton2=NULL ;
Widget cancelButton3=NULL ;
Widget reopenAccept=NULL ;
Widget reopenDeny=NULL ;
Widget helpLine=NULL ;
Widget helpLine2=NULL ;
Widget helpLine3=NULL ;
Widget helpLine4=NULL ;
Widget moveButton=NULL ;
Widget solveButton=NULL ;
Widget reportAgain=NULL ;
Widget printProblem=NULL ;
Widget deleteProblem=NULL ;
Widget typeLabel=NULL ;    /* The problem type dialog box label. */
Widget summaryLabel=NULL ; /* The problem summary dialog box label. */
Widget priorityLabel=NULL ;/* The problem priority Label. Added 11/16/94 DC*/
Widget higherButton=NULL ; /* The button that increases the priority */
Widget lowerButton=NULL ;  /* The button that decreases the priority */
static XtPointer *ptrlist=NULL ;
static XtPointer *ptrlist2=NULL ;
static XtPointer *ptrlist3=NULL ;
static XtPointer *ptrlist4=NULL ;
static XtPointer *ptrlist5=NULL ;
static XtPointer *ptrlist6=NULL ;

          /* Minor */
int	status=0 ;



   /*------------ Start of ProblemViewerPopup() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering ProblemViewerPopup()\n") ;
#endif

     /* Create the popup widget.  The popup widgets parent should
      * be the widget "parent".
      */
   *probViewerp = XtCreatePopupShell(name, applicationShellWidgetClass,
                                     parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(popup.c): Created popup '%s' as a child of '%s'\n",
                XtName(*probViewerp), XtName(XtParent(*probViewerp))) ;
#endif

     /* Add a paned widget to the popup widget, dividing the popup into
      * sections.
      */
   panes = XtCreateManagedWidget("panes", panedWidgetClass, *probViewerp,
				 NULL, ZERO) ;

     /* Add a two title bars to the paned widget using XtCreateManagedWidget().
      */
   wintitle1 = XtCreateManagedWidget("wintitle1", labelWidgetClass, panes,
                                     NULL, ZERO) ;
   wintitle2 = XtCreateManagedWidget("wintitle2", labelWidgetClass, panes,
                                     NULL, ZERO) ;

     /* Add the dialog box form to the panes widget.  Add three dialog
      * boxes to the dialog box, the second offset far enough to allow
      * room for the first, the third offset for the second.
      */

   dialogBox = XtCreateManagedWidget("dialogBox", formWidgetClass, panes,
                                      NULL, ZERO) ;

   typeLabel = XtVaCreateManagedWidget(
		"typeLabel", labelWidgetClass, dialogBox,
		XtNborderWidth,	0,
		XtNwidth,	20,
		NULL) ;
   *typep = XtVaCreateManagedWidget(
		"type", asciiTextWidgetClass, dialogBox,
		XtNfromHoriz,	typeLabel,
		NULL) ;

   summaryLabel = XtVaCreateManagedWidget(
		"summaryLabel", labelWidgetClass, dialogBox,
		XtNfromVert,	*typep,
		XtNwidth,	20,
		NULL) ;
   *summaryp = XtVaCreateManagedWidget(
		"summary", asciiTextWidgetClass, dialogBox,
		XtNfromVert,	*typep,
		XtNfromHoriz,	summaryLabel,
		NULL) ;

#ifdef DO_PRIORITY
   priorityLabel = XtVaCreateManagedWidget(
		"priorityLabel", labelWidgetClass, dialogBox,
		XtNfromVert,	*summaryp,
		XtNwidth,	20,
		XtNheight,	45,
		NULL) ;
   *priorityp = XtVaCreateManagedWidget(
		"priority", labelWidgetClass, dialogBox,
		XtNfromVert,	*summaryp,
		XtNfromHoriz,	priorityLabel,
		XtNlabel,	sctoa(DEF_PRIORITY),
		XtNwidth,	5,
		XtNheight,	45,
		NULL) ;

   higherButton = XtVaCreateManagedWidget(
		"higherButton", commandWidgetClass, dialogBox,
		XtNfromVert,	*summaryp,
		XtNfromHoriz,	*priorityp,
		NULL) ;
   lowerButton = XtVaCreateManagedWidget(
		"lowerButton", commandWidgetClass, dialogBox,
		XtNfromVert,	higherButton,
		XtNfromHoriz,	*priorityp,
		NULL) ;
#endif


     /* For those dialog boxes which are currently read-only,
      * set the resources so this is so.  Set the values of
      * the dialog boxes.
      */
   XawTextDisplayCaret(*typep, FALSE) ;              /* Can't edit this one...*/
   XawTextDisplayCaret(*summaryp, FALSE) ;     /* Can't edit this one _yet_...*/

/*
   XtVaSetValues(*typep, XtNborderWidth, 0, NULL) ;
   XtVaSetValues(*summaryp, XtNborderWidth, 0, NULL) ;
*/

#ifdef DO_PRIORITY
   XtSetSensitive(higherButton, FALSE) ;
   XtSetSensitive(lowerButton, FALSE) ;
#endif


     /* Add a text widget to the paned widget using XtCreateManagedWidget().  
      */
   *textp = XtCreateManagedWidget("text", asciiTextWidgetClass,
                                  panes, NULL, ZERO) ;
   XawTextDisplayCaret(*textp, FALSE) ; 

   *editp = XtCreateWidget("edit", asciiTextWidgetClass,
                           panes, NULL, ZERO) ;
   XawTextDisplayCaret(*editp, TRUE) ;


     /* Add a brief help line indicating what actions are acceptable. */
   helpLine = XtCreateManagedWidget("helpLine", labelWidgetClass, panes,
                                    NULL, ZERO) ;



     /**** Add the button-box to the panes widget.  ****/
   buttonBox = XtCreateManagedWidget("buttonBox",  boxWidgetClass, panes,
                                     NULL, ZERO) ;

     /* Add the quit button to the button-box widget. */
   quitButton = XtCreateManagedWidget("quitButton", commandWidgetClass,
                                      buttonBox, NULL, ZERO) ;

     /* Create the reopen, edit, reopenAccept, reopenDeny,
      * and moveButton buttons,
      * adding to the buttonBox. 
      */
    reopenButton = XtCreateWidget("reopenButton", commandWidgetClass,
                                  buttonBox, NULL, ZERO) ;
    editButton = XtCreateWidget("editButton", commandWidgetClass,
                                buttonBox, NULL, ZERO) ;
    reopenAccept = XtCreateWidget("reopenAccept", commandWidgetClass,
                                  buttonBox, NULL, ZERO) ;
    reopenDeny = XtCreateWidget("reopenDeny", commandWidgetClass,
                                 buttonBox, NULL, ZERO) ;
    moveButton = XtCreateWidget("moveButton", commandWidgetClass,
                                 buttonBox, NULL, ZERO) ;
    reportAgain = XtCreateWidget("reportAgain", commandWidgetClass,
                                 buttonBox, NULL, ZERO) ;
    printProblem = XtCreateWidget("printProblem", commandWidgetClass,
                                  buttonBox, NULL, ZERO) ;
    deleteProblem = XtCreateWidget("deleteProblem", commandWidgetClass,
                                   buttonBox, NULL, ZERO) ;



     /**** Add the second button-box to the panes widget.***  
      * This one is managed when the user edits the problem.
      */
   buttonBox2 = XtCreateWidget("buttonBox2",  boxWidgetClass, panes,
                               NULL, ZERO) ;

     /* Add the cancel button to the second button-box widget. */
   cancelButton = XtCreateManagedWidget("cancelButton", commandWidgetClass,
                                        buttonBox2, NULL, ZERO) ;

     /* Add the save button to the second button-box widget. */
   saveButton = XtCreateManagedWidget("saveButton", commandWidgetClass,
                                      buttonBox2, NULL, ZERO) ;

     /* Add the solve button to the second button-box widget.
      * This is like the saveButton, except it also marks it
      * as solved.
      */
    solveButton  = XtCreateManagedWidget("solveButton", commandWidgetClass,
                                         buttonBox2, NULL, ZERO) ;

     /* Add a brief help line indicating what actions are acceptable for
      * buttonBox2.
      */
   helpLine2 = XtCreateWidget("helpLine2", labelWidgetClass, panes,
                              NULL, ZERO) ;



     /**** Add the third button-box to the panes widget. ***
      * This one is managed when the user request to reopen the problem.
      */
   buttonBox3 = XtCreateWidget("buttonBox3",  boxWidgetClass, panes,
                               NULL, ZERO) ;

     /* Add the cancel button to the third button-box widget. */
   cancelButton2 = XtCreateManagedWidget("cancelButton2", commandWidgetClass,
                                         buttonBox3, NULL, ZERO) ;

     /* Add the save button to the third button-box widget. */
   saveButton2 = XtCreateManagedWidget("saveButton2", commandWidgetClass,
                                       buttonBox3, NULL, ZERO) ;

     /* Add a brief help line indicating what actions are acceptable for
      * buttonBox3.
      */
   helpLine3 = XtCreateWidget("helpLine3", labelWidgetClass, panes,
                              NULL, ZERO) ;



     /**** Add the fourth button-box to the panes widget. ***
      * This one is managed when the sysop refuses to reopen the problem.
      */
   buttonBox4 = XtCreateWidget("buttonBox4",  boxWidgetClass, panes,
                               NULL, ZERO) ;

     /* Add the cancel button to the fourth button-box widget. */
   cancelButton3 = XtCreateManagedWidget("cancelButton3", commandWidgetClass,
                                         buttonBox4, NULL, ZERO) ;

     /* Add the save button to the fourth button-box widget. */
   saveButton3 = XtCreateManagedWidget("saveButton3", commandWidgetClass,
                                       buttonBox4, NULL, ZERO) ;

     /* Add a brief help line indicating what actions are acceptable for
      * buttonBox4.
      */
   helpLine4 = XtCreateWidget("helpLine4", labelWidgetClass, panes,
                              NULL, ZERO) ;


     /* Add callbacks for the buttons.
      *   have the quit button call DestroyPopup()
      *   have the reopen button call ReopenSelect()
      *   have the edit button call EditProblemSelect()
      *   have the solve button call SolveProblemSelect()
      *   have the save button call SaveProblemSelect()
      *   have the cancel button call CancelSaveSelect()
      *   have the second save button call ReopenRequestSelect()
      *   have the second cancel button call CancelReopenSelect()
      *   have the third save button call ReopenAcceptSelect()
      *   have the third cancel button call ReopenDenySelect()
      *   have the delete problem button call DeleteProblemSelect()
      */
/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *)XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) *probViewerp ;
   ptrlist[1] = (XtPointer) parent ;
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

/*   if (ptrlist6) XtFree(ptrlist6) ;*/
   ptrlist6 = (XtPointer *)XtCalloc(10, sizeof(XtPointer)) ;
   ptrlist6[0] = (XtPointer) *editp ;
   ptrlist6[1] = (XtPointer) buttonBox ;
   ptrlist6[2] = (XtPointer) buttonBox2 ;
   ptrlist6[3] = (XtPointer) helpLine ;
   ptrlist6[4] = (XtPointer) *textp ;
   ptrlist6[5] = (XtPointer) helpLine2 ;
   ptrlist6[6] = (XtPointer) higherButton ;
   ptrlist6[7] = (XtPointer) lowerButton ;
   ptrlist6[8] = (XtPointer) *priorityp ;
   ptrlist6[9] = (XtPointer) *summaryp ;
   XtAddCallback(editButton, XtNcallback, EditProblemSelect,
                 (XtPointer)ptrlist6) ;
   XtAddCallback(saveButton, XtNcallback, SaveProblemSelect,
                 (XtPointer)ptrlist6) ;
   XtAddCallback(cancelButton, XtNcallback, CancelSaveSelect,
                 (XtPointer)ptrlist6) ;
   XtAddCallback(solveButton, XtNcallback, SolveProblemSelect,
                 (XtPointer)ptrlist6) ;

/*   if (ptrlist2) XtFree(ptrlist2) ;*/
   ptrlist2 = (XtPointer *)XtCalloc(6, sizeof(XtPointer)) ;
   ptrlist2[0] = (XtPointer) *editp ;
   ptrlist2[1] = (XtPointer) buttonBox ;
   ptrlist2[2] = (XtPointer) buttonBox2 ;
   ptrlist2[3] = (XtPointer) helpLine ;
   ptrlist2[4] = (XtPointer) *textp ;
   ptrlist2[5] = (XtPointer) helpLine2 ;
   XtAddCallback(reportAgain, XtNcallback, ReportAgainSelect,
                 (XtPointer)ptrlist2) ;
   XtAddCallback(printProblem, XtNcallback, PrintProblemSelect,
                 (XtPointer)ptrlist2) ;

/*   if (ptrlist4) XtFree(ptrlist4) ;*/
   ptrlist4 = (XtPointer *)XtCalloc(6, sizeof(XtPointer)) ;
   ptrlist4[0] = (XtPointer) *editp ;
   ptrlist4[1] = (XtPointer) buttonBox ;
   ptrlist4[2] = (XtPointer) buttonBox3 ;
   ptrlist4[3] = (XtPointer) helpLine ;
   ptrlist4[4] = (XtPointer) *textp ;
   ptrlist4[5] = (XtPointer) helpLine3 ;
   XtAddCallback(reopenButton, XtNcallback, ReopenSelect,
                 (XtPointer)ptrlist4) ;
   XtAddCallback(cancelButton2, XtNcallback, CancelReopenSelect,
                 (XtPointer)ptrlist4) ;
   XtAddCallback(saveButton2, XtNcallback, ReopenRequestSelect,
                 (XtPointer)ptrlist4) ;

/*   if (ptrlist5) XtFree(ptrlist5) ;*/
   ptrlist5 = (XtPointer *)XtCalloc(6, sizeof(XtPointer)) ;
   ptrlist5[0] = (XtPointer) *editp ;
   ptrlist5[1] = (XtPointer) buttonBox ;
   ptrlist5[2] = (XtPointer) buttonBox4 ;
   ptrlist5[3] = (XtPointer) helpLine ;
   ptrlist5[4] = (XtPointer) *textp ;
   ptrlist5[5] = (XtPointer) helpLine4 ;
   XtAddCallback(reopenAccept, XtNcallback, ReopenAcceptSelect,
                 (XtPointer)ptrlist5) ;
   XtAddCallback(reopenDeny,   XtNcallback, ReopenDenySelect,
                 (XtPointer)ptrlist5) ;
   XtAddCallback(saveButton3, XtNcallback, ReopenDenyMailSelect,
                 (XtPointer)ptrlist5) ;
   XtAddCallback(cancelButton3, XtNcallback, ReopenDenyCancelSelect,
                 (XtPointer)ptrlist5) ;

/*   if (ptrlist3) XtFree(ptrlist3) ;*/
   ptrlist3 = (XtPointer *)XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist3[0] = (XtPointer) *probViewerp ;
/*   ptrlist3[0] = (XtPointer) moveButton ;*/
   ptrlist3[1] = NULL ;
   XtAddCallback(moveButton, XtNcallback, MoveProblemSelect,
                 (XtPointer)ptrlist3) ;

   XtAddCallback(deleteProblem, XtNcallback, DeleteProblemSelect,
                 (XtPointer) *probViewerp) ;

	/* The PRIORITY buttons "HIGHER" and "LOWER" */
#ifdef DO_PRIORITY
   XtAddCallback(higherButton, XtNcallback, PriorityIncSelect, *priorityp) ;
   XtAddCallback(lowerButton,  XtNcallback, PriorityDecSelect, *priorityp) ;
#endif


#ifdef DEBUG
   printf("XPTS(ProblemViewerPopup()): *probViewerp=%p, "
          "parent=%p.\n", *probViewerp, parent) ;
#endif


     /* Manage appropriate buttons.  Note that managing these buttons
      * does not necessarily make them visible.  It simply enables them
      * to be displayed when appropriate.  If the dialog-box that a button
      * belongs to is managed the button will appear.  If not, it will not.
      * 
      * If the user is a sysop, 
      *   If the problem is solved,
      *      manage the reopen, move, delete and print buttons.
      *   else if the problem is unsolved,
      *      manage the edit, move, solve, reportAgain, delete
      *                 and print buttons.
      *   else the problem is a reopen request, so
      *      manage reopenAccept and reopenDeny.
      * else the user is not a sysop, so
      *   If the problem is solved, 
      *      manage the reopen and print buttons.
      *   else if the problem is unsolved,
      *      manage the reportAgain and print buttons.
      *   else the problem is a reopen request, which should just be displayed.
      */

   status = ProblemRecord->status ;
#ifdef DEBUG
   printf("XPTS(popup.c): ProblemViewerPopup() -- ProblemRecord->status = %d\n",status) ;
#endif

   if (IsSysop(&UserInfoRec))
   {
      if (status == SOLVED)		/* Problem is solved */
      {  XtManageChild(reopenButton) ;
         XtManageChild(moveButton) ;
         XtManageChild(deleteProblem) ;
         XtManageChild(printProblem) ;
      }
      else if (status == UNSOLVED)	/* Problem is unsolved */
      {
         XtManageChild(editButton) ;
         XtManageChild(moveButton) ;
         XtManageChild(reportAgain) ;
         XtManageChild(deleteProblem) ;
         XtManageChild(printProblem) ;
      }
      else if (status == REOPENED)	/* Problem is a reopen request */
      {  XtManageChild(reopenAccept) ;
         XtManageChild(reopenDeny) ;
         XtManageChild(printProblem) ;
      }
      else	/* Shouldn't get here.  */
      {  XtManageChild(printProblem) ;
#ifdef DEBUG
         printf("XPTS(popup.c): Error in problem status!\n") ;
#endif
      }
   }
   else /* Not a sysop */
   {
      if (status == SOLVED)		/* Problem is solved */
      {  XtManageChild(reopenButton) ;
         XtManageChild(printProblem) ;
      }
      else if (status == UNSOLVED)	/* Problem is unsolved */
      {  XtManageChild(reportAgain) ;
         XtManageChild(printProblem) ;
      }
      else if (status == REOPENED)	/* Problem is a reopen request */
      {   /* normal users can't do much of anything to a reopen request. */
         XtManageChild(printProblem) ;
      }
      else	/* Shouldn't get here.  */
      {  XtManageChild(printProblem) ;
#ifdef DEBUG
         printf("XPTS(popup.c): Error in problem status!\n") ;
#endif
      }
   }

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving ProblemViewerPopup()\n") ;
#endif

}  /*------------- End of ProblemViewerPopup() ---------------------*/


/*
______________________________________________________________________
ShowAllUnsolvedPopup()

FUNCTION DESCRIPTION:

     This function will popup a list of short summaries for all the currently
unsolved problems in the PTS database.  It will add a callback to allow the
user to select a problem that they wish to see the entire problem log for.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
ShowAllUnsolvedPopup(Widget     w, 
                     Widget     parent)

     /* Interface description:
         INPUT:
          w        - The widget that caused this function to be called,
                     typically a button or list item.
          parent   - The parent widget of the popup widget to be created.             
         OUTPUT:
          Nope.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget unsolvedPopup=NULL ;  /* The popup being created. */
   Widget list=NULL ;           /* The list widget in unsolvedPopup. */
   Widget buttonBox=NULL ;      /* A box containing buttons in unsolvedPopup. */
   Widget quitButton=NULL ;     /* The quit button in the buttonBox widget. */
   Widget ppanes=NULL ;		/* The parent's panes widget */
   String *UnsolvedList=NULL ;
	/* This needs to be static, not const, since it needs to stick
	 * around after we've left ShowAllUnsolvedPopup().
	 */
   static String EmptyList[]= { EMPTY_UNSOLVED, NULL } ;

          /* Minor */
   static XtPointer *ptrlist=NULL ;
   static XtPointer *ptrlist2=NULL ;


   /*------------ Start of ShowAllUnsolvedPopup() routine -------------*/

#ifdef DEBUG
   printf("XPTS(popup.c): Entering ShowAllUnsolvedPopup()\n") ;
   printf("XPTS(popup.c): Setting parent to be not sensitive.\n") ;
#endif

      /* The parent's widgets keep getting mangled when changing the
       * list widget, so unmanage them for now.
       */
   if (ppanes = XtNameToWidget(parent, "*panes"))
      XtUnmanageChild(ppanes) ;

   XtSetSensitive(parent,False) ;

     /* Create a list popup with two columns. This popup should
      * not be realized by ListPopup().
      */
   ListPopup("unsolvedPopup", ONE, &unsolvedPopup, parent,
          &list, &buttonBox, &quitButton) ;

     /* Fill out the list widget for the ListPopup with unsolved summaries */
   ReadList(unsolvedPopup, unsolvedPopup) ;

     /* Set the callback for the list widget to be ViewProblemActivate().
      * Set the callback for the quit button to be DestroyPopup(). 
      */
/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *)XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) unsolvedPopup ;
   ptrlist[1] = (XtPointer) parent ;
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Making a callback for the list widget with a parent of %p\n",(XtPointer)unsolvedPopup) ; 
#endif
/*   if (ptrlist2) XtFree(ptrlist2) ;*/
   ptrlist2 = (XtPointer *)XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist2[0] = (XtPointer) unsolvedPopup ;
   ptrlist2[1] = (XtPointer) UNSOLVEDPROB ;
   XtAddCallback(list, XtNcallback, ViewProblemActivate, 
	         (XtPointer)ptrlist2) ;

     /* Draw the popup. */
   XtPopup(unsolvedPopup, XtGrabNone) ; 

   if (ppanes) XtManageChild(ppanes) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving ShowAllUnsolvedPopup()\n") ;
#endif
}  /*---------------- End of ShowAllUnsolvedPopup() ------------------*/


/*
______________________________________________________________________
SysopMenuPopup()

FUNCTION DESCRIPTION:

     This function will popup the sysop menu and add callbacks to process
sysop menu selections.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
SysopMenuPopup(Widget     w,
               Widget     parent)

     /* Interface description:
         INPUT:
          w          - The widget that caused this function to be called,
                       typically a button or list item.
          parent     - The parent widget of the popup widget to be created.             
         OUTPUT:
          None. 
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget sysopMenu=NULL ;      /* The popup being created. */
   Widget list=NULL ;           /* The list widget in unsolvedPopup. */
   Widget buttonBox=NULL ;      /* A box containing buttons in unsolvedPopup. */
   Widget quitButton=NULL ;     /* The quit button in the buttonBox widget. */
   Widget helpButton=NULL ;     /* The help button in the buttonBox widget. */
   Widget ppanes=NULL ;		/* The parent widget's panes widget. */
        /* This needs to be static, not const, since it needs to stick
         * around after we've left SysopMenuPopup().
         */
   static String SysopMenuItems[] = {
       "    Select an Unsolved Problem to Edit or Solve",
       "    Update the System Message of the Day",
       "    PTS Database Maintenance",
       NULL
   } ;

          /* Minor */
   static XtPointer *ptrlist=NULL ;


   /*----------------- Start of SysopMenuPopup() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering SysopMenuPopup()\n") ;
#endif

     /* Make sure only sysops are executing this function. */
   if (!IsSysop(&UserInfoRec))
   { 
#ifdef DEBUG
        fprintf(stderr,
        "XPTS(popup.c): Illegal access to SysopMenuPopup() by non-sysop.\n") ;
#endif
     return ;
   }

#ifdef DEBUG
   printf("XPTS(popup.c): Setting parent to be not sensitive.\n") ;
#endif

      /* The parent's widgets keep getting mangled when changing the
       * list widget, so unmanage them for now.
       */
   if (ppanes = XtNameToWidget(parent, "*panes"))
      XtUnmanageChild(ppanes) ;

   XtSetSensitive(parent,False) ;

     /* Create a list popup with one columns. This popup should
      * not be realized by ListPopup().  Set the value of the list
      * to be SysopMenuItems.
      */
   ListPopup("sysopMenu", ONE, &sysopMenu, parent,
          &list, &buttonBox, &quitButton) ;
   XawListChange(list, SysopMenuItems, ZERO, ZERO, TRUE) ;

     /* Add the sysop help button to the button-box. */
   helpButton = XtCreateManagedWidget("helpButton", commandWidgetClass,
                                      buttonBox, NULL, ZERO) ;

     /* Set the callback for the list widget to be SysopMenuActivate().
      * Set the callback for the help button to be HelpPopup().
      * Set the callback for the quit button to be DestroyPopup(). 
      */
   XtAddCallback(list, XtNcallback, SysopMenuActivate, (XtPointer)sysopMenu) ;
   XtAddCallback(helpButton, XtNcallback, HelpPopup, (XtPointer)sysopMenu) ;

/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *)XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) sysopMenu ;
   /*ptrlist[1] = NULL ;*/
   ptrlist[1] = (XtPointer) parent ;
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;

     /* Draw the popup */
   XtPopup(sysopMenu, XtGrabExclusive) ;

   if (ppanes) XtManageChild(ppanes) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving SysopMenuPopup()\n") ;
#endif
}  /*--------------------- End of SysopMenuPopup() ------------------------*/


/*
______________________________________________________________________
ButtonPopup()

FUNCTION DESCRIPTION:

     This function will create a popup containing a single button.  The button
can contain either a bitmap or a text string.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
ButtonPopup(String     name, 
            Widget    *popupp, 
            Widget     parent, 
            Widget    *buttonp)

     /* Interface description:
         INPUT:
          name		- The resource name of the popup.            
          popupp	- The popup widget being created.
          parent	- The parent of the popup widget.
          buttonp	- The button to place in the popup.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
	  /* NONE  */
          /* Minor */
	  /* NONE  */


   /*----------------- Start of ButtonPopup() routine -----------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering ButtonPopup()\n") ;
#endif

     /* Create the popup widget, of class applicationShellWidgetClass */
   *popupp = XtCreatePopupShell(name, applicationShellWidgetClass,
                                parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(popup.c): Created popup '%s' as a child of '%s'\n",
                XtName(*popupp), XtName(XtParent(*popupp))) ;
#endif

     /* Add the button to the popup. */
   *buttonp = XtCreateManagedWidget("button", commandWidgetClass, *popupp,
				    NULL, ZERO) ;


     /* Note that the widgets here have not been realized yet.
      * The popup widget must be realized by the calling procedure.
      */

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving ButtonPopup()\n") ;
#endif
}  /*--------------------- End of ButtonPopup() -----------------------------*/


/*
______________________________________________________________________
DestroyPopup()

FUNCTION DESCRIPTION:

     This function is a callback which will destroy the specified popup and
all it's children.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
DestroyPopup(Widget  w,
          XtPointer  client_data,
          XtPointer  call_data)

     /* Interface description:
         INPUT:
          w              - The widget for which this callback is registered.
          client_data    - Specifies arbitrary client-supplied data
                           that the widget w should pass back to the
                           client when the widget executes the client's
                           callback procedure.  In this callback, 
                           this an array defined like this:

                           client_data[0] - is the widget to destroy.
                           client_data[1] - is the parent widget 
                                      to realize (if not NULL).

          call_data      - Specifies any callback-specific data the widget w
                           wants to pass to the client.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
     Widget popup=NULL ;    /* The popup to destroy. */
     Widget parent=NULL ;   /* The parent of the popup, which should be */
			    /*  popped up.*/
     Widget ppanes=NULL ;
     XtPointer *ptrlist=NULL ;

          /* Minor */
     Arg args[5] ;


   /*----------------- Start of DestroyPopup() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering DestroyPopup()\n") ;
#endif

   ptrlist = (XtPointer *)client_data ;
   popup  = (Widget)(ptrlist[0]) ;
   parent = (Widget)(ptrlist[1]) ;
#ifdef DEBUG
   printf("XPTS(popup.c): parent=%p\n", parent) ;
#endif

  
#ifdef DEBUG
   printf ("XPTS(popup.c): Destroying popup '%s', a child of '%s'\n",
           XtName(popup), XtName(XtParent(popup))) ;
#endif
   if (strcmp(XtName(popup), "pickProblem") == 0)
   {  GlobalNewProblemButton = NULL ;
      GlobalPreviousButton = NULL ;
   }

     /* Destroy the widget specified in client_data. */
   XtDestroyWidget(popup) ;


     /* Popup the parent widget, if specified.  Make sure it's set
      * to be sensitive.
      */
   if (parent)
   {
      if (XtIsShell(parent))
      {
#ifdef DEBUG
         printf ("XPTS(popup.c): Popping up popup '%s', a child of '%s'\n",
                XtName(parent), XtName(XtParent(parent))) ;
#endif
         /* Should also figure out how to make the parent de-iconify here. */
         RaiseWidget(parent) ;
         XtPopup(parent, XtGrabNone) ;
      }
#ifdef DEBUG
         printf ("XPTS(popup.c): Setting popup '%s' (a child of '%s') "
		 "to be sensitive.\n",
                XtName(parent), XtName(XtParent(parent))) ;
#endif
      XtSetSensitive(parent, True) ;
   }


#ifdef DEBUG
   printf("XPTS(popup.c): Leaving DestroyPopup()\n") ;
#endif
}  /*--------------------- End of DestroyPopup() ------------------------*/


/*
______________________________________________________________________
ErrorPopup()

FUNCTION DESCRIPTION:

     This function creates a popup to inform the user of a significant error
that has occured.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
ErrorPopup(String     name, 
          Widget      parent, 
          String      message)

     /* Interface description:
         INPUT:
          name           - The resource name of the popup.
          parent         - The parent of the popup widget.
          message        - The specific error message to display.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget popup=NULL ;
   Widget panes=NULL ;        /* The window-panes widget within the popup. */
   Widget wintitle=NULL ;
   Widget buttonBox=NULL ;    /* The button-box added to the popup. */
   Widget quitButton=NULL ;   /* The quit button added to the button-box. */
   Widget viewport=NULL ;     /* The viewport widget, for scrollbars */
   Widget helpLine=NULL ;
   Widget text=NULL ;
   Widget ppanes=NULL ;
          /* Minor */
   Arg args[3] ;
   static XtPointer *ptrlist=NULL ;



   /*----------------- Start of ErrorPopup() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering ErrorPopup()\n") ;
#endif

     /* Create the popup widget.  The popup widgets parent should
      * be the widget "parent".
      */
   popup = XtCreatePopupShell(name, applicationShellWidgetClass,
                              parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(popup.c): Created popup '%s' as a child of '%s'=='%s'\n",
                XtName(popup), XtName(XtParent(popup)), XtName(parent)) ;
#endif

      /* The parent's widgets keep getting mangled when changing the
       * widget, so unmanage them for now.
       */
/*   if (ppanes = XtNameToWidget(parent, ".panes"))
      XtUnmanageChild(ppanes) ;
*/

   XtSetSensitive(parent, False) ;

     /* Add a paned widget to the popup widget, dividing the popup into
      * sections.
      */
   panes = XtCreateManagedWidget("panes", panedWidgetClass, popup, NULL, ZERO) ;

     /* Add a title bar to the paned widget using XtCreateManagedWidget(). */
   wintitle = XtCreateManagedWidget("wintitle", labelWidgetClass, panes,
                                    NULL, ZERO) ;

     /* Add an ascii text widget to the viewport widget.  Have it display
      * the error message.
      */
   text = XtCreateManagedWidget("text", asciiTextWidgetClass, panes,
                                  NULL, ZERO) ;
   XtSetArg(args[0], XtNstring, message) ;
   XtSetValues(text, args, ONE) ;

     /* Add a brief help line indicating what actions are acceptable. */
   helpLine = XtCreateManagedWidget("helpLine", labelWidgetClass, panes,
                                    NULL, ZERO) ;

     /* Add the button-box to the panes widget. */
   buttonBox = XtCreateManagedWidget("buttonBox", boxWidgetClass, panes,
                                NULL, ZERO) ;

     /* Add the quit button to the button-box widget. */
   quitButton = XtCreateManagedWidget("quitButton", commandWidgetClass,
                                 buttonBox, NULL, ZERO) ;

     /* Add a callback for the quit button, which destroys 
      * the error popup.
      */
/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) popup ;
   ptrlist[1] = (XtPointer) parent ;
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;
   

     /* Draw the popup. */
   XtPopup(popup, XtGrabExclusive) ;

/*   if (ppanes) XtManageChild(ppanes) ;*/

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving ErrorPopup()\n") ;
#endif

}  /*--------------------- End of ErrorPopup() ------------------------*/


/*
______________________________________________________________________
ListPopup()

FUNCTION DESCRIPTION:

     This function creates a popup containing a list.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
ListPopup(String  name,
	  int     columns,
          Widget *popupp,
	  Widget  parent,
	  Widget *listp,
          Widget *buttonBoxp,
	  Widget *quitButtonp)

     /* Interface description:
         INPUT:
          name		- The resource name of the popup.
          columns	- The number of columns in the list widget.
          popupp	- The popup widget being created.
          parent	- The parent of the popup widget.
          listp         - The list widget placed in the popup.
          buttonBoxp    - The button-box widget placed in the popup.
          quitButtonp   - The quit button placed in the button-box.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget panes=NULL ;       /* The window panes widget place in the popup. */
   Widget wintitle1=NULL ;   /* The primary popup title. */
   Widget wintitle2=NULL ;   /* The secondary popup title. */
   Widget viewport=NULL ;    /* The viewport widget, for scrollbars */
   Widget helpLine=NULL ;    /* The line of help for this window. */
          /* Minor */
   Arg args[3] ; 


   /*----------------- Start of ListPopup() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering ListPopup()\n") ;
   printf("XPTS(popup.c): name=%s,%p\n",name,name) ;
#endif


     /* Create the popup widget.  The popup widgets parent should
      * be the widget "parent".
      */
   *popupp = XtCreatePopupShell(name, applicationShellWidgetClass,
				parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(popup.c): Created popup '%s' as a child of '%s'\n",
                XtName(*popupp), XtName(XtParent(*popupp))) ;
#endif


     /* Add a paned widget to the popup widget, dividing the popup into
      * sections.  Do this using XtCreateManagedWidget().
      */
   panes = XtCreateManagedWidget("panes", panedWidgetClass, *popupp,
				 NULL, ZERO) ;

     /* Add two title bars to the paned widget using XtCreateManagedWidget(). */
   wintitle1 = XtCreateManagedWidget("wintitle1", labelWidgetClass, panes,
                                     NULL, ZERO) ;
   wintitle2 = XtCreateManagedWidget("wintitle2", labelWidgetClass, panes,
                                     NULL, ZERO) ;

     /* Add a viewport to the paned widget using XtCreateManagedWidget().
      * This will put scrollbars around the viewports child widget.
      */

   viewport = XtCreateManagedWidget ("viewport", viewportWidgetClass, panes,
                              NULL, ZERO) ;


     /* Make the list widget a child of the viewport.  Setting forceColumns
      * to true and setting defaultColumns to the value of "columns".
      */
   XtSetArg(args[0], XtNdefaultColumns, columns) ;
   XtSetArg(args[1], XtNforceColumns, TRUE) ;
   *listp = XtCreateManagedWidget("list", listWidgetClass, viewport,
			          args, TWO) ;

     /* Add a brief help line indicating what actions are acceptable. */
   helpLine = XtCreateManagedWidget("helpLine", labelWidgetClass, panes,
				    NULL, ZERO) ; 

     /* Add the button-box to the panes widget. */
   *buttonBoxp = XtCreateManagedWidget("buttonBox", boxWidgetClass, panes,
                                NULL, ZERO) ;

     /* Add the quit button to the button-box widget. */
   *quitButtonp = XtCreateManagedWidget("quitButton", commandWidgetClass,
                                 *buttonBoxp, NULL, ZERO) ;

     /* Note that the widgets here HAVE NOT BEEN REALIZED YET.
      * The popup widget must be realized by the calling procedure.
      * Also, the list widget is empty.  The popup widget must set this 
      * before realizing the popup.
      */

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving ListPopup()\n") ;
#endif
}  /*---------------------- End of ListPopup() -------------------------*/


/*
______________________________________________________________________
StringEntryPopup()

FUNCTION DESCRIPTION:

     This function will create a popup that prompts the user to input a
one-line text string.  Buttons are added for accepting the input string and
aborting input.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
StringEntryPopup(String     name, 
                 Widget    *popupp, 
                 Widget     parent, 
                 String     prompt,
                 XtPointer  successFunc)

     /* Interface description:
         INPUT:
          name        - The resource name of the popup.
          popup       - Ptr. to the popup widget being created.
          parent      - The parent of the popup widget.
          prompt      - The string to prompt for input.
          successFunc - The function to call if the user accepts
			the string.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget dialog=NULL ;   /* The dialog box used to enter the string. */
   static XtPointer *ptrlist=NULL ;

          /* Minor */
   Arg args[5] ;


   /*-------------- Start of StringEntryPopup() routine ---------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering StringEntryPopup()\n") ;
#endif


     /* Create the popup widget.  The popup widgets parent should
      * be the widget "parent".
      */
   *popupp = XtCreatePopupShell(name, applicationShellWidgetClass,
                                 parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(popup.c): Created popup '%s' as a child of '%s'\n",
                XtName(*popupp), XtName(XtParent(*popupp))) ;
#endif


     /* Add a dialog box to the popup. */
   dialog = XtCreateManagedWidget("dialog", dialogWidgetClass, *popupp,
				  NULL, ZERO) ;
   XtSetArg(args[0], XtNlabel, prompt) ; 
   XtSetArg(args[1], XtNvalue, "") ;
   XtSetValues(dialog, args, TWO) ;

/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *)XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) *popupp ;
   ptrlist[1] = (XtPointer) dialog ;
   XawDialogAddButton(dialog, "cancelButton", DestroyPopup,
                      (XtPointer) ptrlist) ;
   XawDialogAddButton(dialog, "acceptButton", successFunc,
                      (XtPointer) ptrlist) ;


     /* Note that the widgets here have not been realized yet.
      * The popup widget must be realized by the calling procedure.
      */
     
#ifdef DEBUG
   printf("XPTS(popup.c): Leaving StringEntryPopup()\n") ;
#endif
}  /*------------------ End of StringEntryPopup() ---------------------*/


/*
______________________________________________________________________
ViewTextPopup()

FUNCTION DESCRIPTION:

     This function creates a popup used to view a text file or text string.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins	Date:  2/21/92
______________________________________________________________________
*/


void
ViewTextPopup(String     name, 
              Widget    *popupp, 
              Widget     parent, 
              Widget    *textp,
              Widget    *buttonBoxp,
              Widget    *quitButtonp)

     /* Interface description:
         INPUT:
          name             - The resource name of the popup.
          popupp           - The popup widget being created.
          parent           - The parent of the popup widget.
          textp            - The AsciiText widget to put in the popup.
          buttonBox        - The button-box to add to the panes widget.
          quitButtonp      - The quit button to add to the popup.
         OUTPUT:
           None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget panes=NULL ;         /* The window panes widget place in the popup. */
   Widget wintitle1=NULL ;     /* The primary popup title. */
   Widget wintitle2=NULL ;     /* The secondary popup title. */
   Widget viewport=NULL ;      /* The viewport widget, for scrollbars */
   Widget helpLine=NULL ;      /* The line of help for this window. */

          /* Minor */
          /* NONE */


   /*------------------ Start of ViewTextPopup() routine -------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering ViewTextPopup()\n") ;
#endif


     /* Create the popup widget.  The popup widgets parent should
      * be the widget "parent".
      */
   *popupp = XtCreatePopupShell(name, applicationShellWidgetClass,
                                parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(popup.c): Created popup '%s' as a child of '%s'\n",
                XtName(*popupp), XtName(XtParent(*popupp))) ;
#endif


     /* Add a paned widget to the popup widget, dividing the popup into
      * sections.
      */
   panes = XtCreateManagedWidget("panes", panedWidgetClass, *popupp,
				 NULL, ZERO) ;

     /* Add two title bars to the paned widget using XtCreateManagedWidget(). */
   wintitle1 = XtCreateManagedWidget("wintitle1", labelWidgetClass, panes,
                                     NULL, ZERO) ;
   wintitle2 = XtCreateManagedWidget("wintitle2", labelWidgetClass, panes,
                                     NULL, ZERO) ;

     /* Make the text widget a child of the viewport.  Disable the insert point.
      */
   *textp = XtCreateManagedWidget("text", asciiTextWidgetClass, panes,
         			  NULL, ZERO) ;
                                  
   XawTextDisplayCaret(*textp, FALSE) ;

     /* Add a brief help line indicating what actions are acceptable. */
   helpLine = XtCreateManagedWidget("helpLine", labelWidgetClass, panes,
                                    NULL, ZERO) ;

     /* Add the button-box to the panes widget. */
   *buttonBoxp = XtCreateManagedWidget("buttonBox", boxWidgetClass, panes,
                                       NULL, ZERO) ;

     /* Add the quit button to the button-box widget. */
   *quitButtonp = XtCreateManagedWidget("quitButton", commandWidgetClass,
                                        *buttonBoxp, NULL, ZERO) ;


     /* Note that the widgets here HAVE NOT BEEN REALIZED YET.
      * The popup widget must be realized by the calling procedure.
      */
     
#ifdef DEBUG
   printf("XPTS(popup.c): Leaving ViewTextPopup()\n") ;
#endif
}  /*---------------------- End of ViewTextPopup() -------------------------*/


/*
______________________________________________________________________
WarningPopup()

FUNCTION DESCRIPTION:

     This function creates a popup to inform the user of a potentially
destructive action they have selected, and allow them to either continue or
abort.  It creates callbacks for continuing and aborting.
______________________________________________________________________
UNIT TESTING:
     This function will be tested by using a combination of white-box and 
black-box tests.
______________________________________________________________________
REVISION HISTORY:
     Author:  Dean Collins      Date:  2/21/92
______________________________________________________________________
*/


void
WarningPopup(String     name, 
             Widget     parent, 
             String     message,
             XtPointer  ok_func,
             XtPointer  ok_client_data)

     /* Interface description:
         INPUT:
          name           - The resource name of the popup.
          parent         - The parent of the popup widget.
          message        - The specific error message to display.
          ok_func        - The call-back function to call if the continue
				button is pressed.
          ok_client_data - The client_data to pass to the callback ok_func.
         OUTPUT:
          None.
         RETURN VALUE:
          None.
     */
      
{
     /* Internal variables: */
          /* Major */
   Widget panes=NULL ;         /* The window-panes widget within the popup. */
   Widget buttonBox=NULL ;     /* The button-box added to the popup. */
   Widget continueButton=NULL ;/* The continue button added to the button-box.*/
   Widget quitButton=NULL ;    /* The quitbutton added to the button-box. */
   Widget viewport=NULL ;      /* The viewport widget, for scrollbars */
   bool Done=FALSE ;           /* Flag indicating that a button was pressed. */
   Widget popup=NULL ;
   Widget wintitle=NULL ;
   Widget helpLine=NULL ;
   Widget text=NULL ;

          /* Minor */
   Arg args[3] ;
   static XtPointer *ptrlist=NULL ;


   /*----------------- Start of WarningPopup() routine ------------------*/
#ifdef DEBUG
   printf("XPTS(popup.c): Entering WarningPopup()\n") ;
#endif


     /* Create the popup widget.  The popup widgets parent should
      * be the widget "parent".
      */
   popup = XtCreatePopupShell(name, applicationShellWidgetClass,
                                parent, NULL, ZERO) ;
#ifdef DEBUG
   printf ("XPTS(WarningPopup()): Created popup '%s' as a child of '%s'\n",
                XtName(popup), XtName(XtParent(popup))) ;
   printf("XPTS(WarningPopup()): probViewer=%p, parent=%p, warning=%p.\n",
	  XtParent(popup), XtParent(XtParent(popup)), popup) ;
#endif


     /* Add a paned widget to the popup widget, dividing the popup into
      * sections.
      */
   panes = XtCreateManagedWidget("panes", panedWidgetClass, popup, NULL, ZERO) ;

     /* Add a title bar to the paned widget using XtCreateManagedWidget(). */
   wintitle = XtCreateManagedWidget("wintitle", labelWidgetClass, panes,
                                    NULL, ZERO) ;


     /* Add an ascii text widget to the panes widget.  Have it display
      * the error message.
      */
   text = XtCreateManagedWidget("text", asciiTextWidgetClass, panes,
                                  NULL, ZERO) ;

     /* A NULL message would allow the text to be specified in the resources
      * database, rather than hard-coding the message into the program.
      */
   if (message != NULL)
   {  XtSetArg(args[0], XtNstring, message) ;
      XtSetValues(text, args, ONE) ;
   }

     /* Add a brief help line indicating what actions are acceptable. */
   helpLine = XtCreateManagedWidget("helpLine", labelWidgetClass, panes,
                                    NULL, ZERO) ;

     /* Add the button-box to the panes widget. */
   buttonBox = XtCreateManagedWidget("buttonBox", boxWidgetClass, panes,
                                NULL, ZERO) ;

     /* Add the continue and quit buttons to the button-box widget. */
   quitButton = XtCreateManagedWidget("quitButton", commandWidgetClass,
                                        buttonBox, NULL, ZERO) ;
   continueButton = XtCreateManagedWidget("continueButton",
                                          commandWidgetClass,
                                          buttonBox, NULL, ZERO) ;

     /* Add a callback for the continue button that will call the
      * ok_func callback function with the client_data argument
      * set to ok_client_data.
      */
   XtAddCallback(continueButton, XtNcallback, ok_func, ok_client_data) ;

     /* Add callbacks for the quit and continue buttons that will destroy
      * the warning popup.
      */
/*   if (ptrlist) XtFree(ptrlist) ;*/
   ptrlist = (XtPointer *) XtCalloc(2, sizeof(XtPointer)) ;
   ptrlist[0] = (XtPointer) popup ;
   ptrlist[1] = NULL ;
   XtAddCallback(quitButton, XtNcallback, DestroyPopup, (XtPointer)ptrlist) ;
/*   XtAddCallback(continueButton, XtNcallback, DestroyPopup, 
		 (XtPointer)ptrlist) ;
*/

     /* Draw the popup. */
   XtPopup(popup, XtGrabExclusive) ;

#ifdef DEBUG
   printf("XPTS(popup.c): Leaving WarningPopup()\n") ;
#endif
}  /*--------------------- End of WarningPopup() ------------------------*/

/* end of popup.c */
