/*
 * get_prob.c -- 
 * Chris Hyde
 * created: 
 *
 * Modified: Dean Collins Sat Feb 12 18:48:34 1994
 * Added unistd.h inclusion for SunOS as requested by
 * magnush@isy.liu.se (Magnus Hammerin).
 * Also fixed several typos.
 *
 * Modified: Dean Collins Wed Feb 23 17:00:42 1994
 * Changed PTSROOT (a constant) to PtsRoot (a variable).
 *
 * Modified: Dean Collins Mon Nov 21 16:07:11 PST 1994
 * Modifications required for new "priority" field.
 *
 * Modified: Chris Hyde 1/10/95
 * Added new problem search types to get_problem().
 * Removed next_solved() and next_unsolved() and replaced with next_problem()
 *   which will search for a problem of the given type (solved, unsolved,
 *   priority, date).
 * Removed include <stdio.h>, since it's now included in zdbm.h.
 * Made changes to I/O to reflect zdbm names (zread(), zwrite(), etc.)
 * Replaced direct references to the index records with functions/macros
 *   defined for that purpose.
 *
 * Modified: Dean Collins Thu Jun  8 13:31:27 PDT 1995
 * Replaced #ifdef DEBUG method of debug output with calls to
 * the zdebug macros.
 *
 */

/*
 * Copyright (c) 1992,1993,1994,1995 Christopher G. Hyde
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of Idaho
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  The University
 * of Idaho makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE UNIVERSITY OF IDAHO BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include <errno.h>
#include <stdlib.h>
#ifdef SUNOS
#include <unistd.h>
#endif /*SUNOS*/

#include "zdbm.h"

problem_record *get_problem(char *db_path, char *search_string, int type,
                            int direction, int links, int whole_record)
                            
{
 static problem_record pro_rec;
 problem_record *errptr;
 static char current_db[DBPATHLEN];
 static int old_type=0;
 index_record index,
              *errpointer;
 int new=FALSE;
 ZFILE *db_pointer;
  
 zdebug("\nIn get_problem()\n");
 zdebug1("\tLeaf to search - %s\n", db_path);
 zdebug1("\tString to search for - %s\n", search_string);
 zdebug2("\tType of problem - %d\n\tDirection of search - %d\n",
               type, direction);
 zdebug1("\tFollow links - %d\n", links);
 zdebug1("\tWhole record - %d\n", whole_record);

 db_errorno=0;

/* We want to start a new search if either the leaf to search changes, or
 * the type of search changes.
 */

 if ((strcmp(current_db, db_path)!=0) || (old_type!=type))
 {
    nl_strncpy(current_db, db_path, DBPATHLEN);
    old_type=type;
    new=TRUE;

    zdebug("get_problem() - Searching new leaf, or using new type.\n");

 }
#ifdef DEBUG
 else
    zdebug("get_problem() - Searching previous leaf and type.\n");
#endif

 if ((db_pointer=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(NULL);
    
 switch (type) /* get the type of problem, or particular problem wanted */
 {
    case SOLVED:         errpointer=next_problem(db_pointer, new, links,
                                                 &index, NULL, is_solved);
                         break;
 
    case UNSOLVED:       errpointer=next_problem(db_pointer, new, links,
                                                 &index, NULL, is_unsolved);
                         break;
 
    case EQ_PRIORITY:    errpointer=next_problem(db_pointer, new, links,
                                                 &index, search_string,
                                                 eq_priority);
                         break;
 
    case GE_PRIORITY:    errpointer=next_problem(db_pointer, new, links,
                                                 &index, search_string,
                                                 ge_priority);
                         break;
 
    case LE_PRIORITY:    errpointer=next_problem(db_pointer, new, links,
                                                 &index, search_string,
                                                 le_priority);
                         break;
 
    case EQ_DATE:        errpointer=next_problem(db_pointer, new, links,
                                                 &index, search_string,
                                                 eq_date);
                         break;
 
    case GE_DATE:        errpointer=next_problem(db_pointer, new, links,
                                                 &index, search_string,
                                                 ge_date);
                         break;
 
    case LE_DATE:        errpointer=next_problem(db_pointer, new, links,
                                                 &index, search_string,
                                                 le_date);
                         break;
 
    case PARTICULAR:     errpointer=find_problem(db_pointer, search_string,
                                                 &index);
                         break;
 
    default:             db_errorno=BADTYPE;
                         errpointer=NULL;
 } /* end switch */
 
 zclose(db_pointer);
 if (errpointer==NULL) /* didn't find the problem, or error */
    return(NULL);

 zdebug("get_problem() - found index of desired problem.\n");

 if (whole_record==FULLRECORD)
 {
    zdebug("get_problem() - caller wants full problem record.\n");

    errptr=create_record(db_path, &index, &pro_rec);
 } /* end full record */
 else
 {
    zdebug("get_problem() - call wants partial problem record.\n");

    errptr=partial_record(db_path, &index, &pro_rec);
 } /* end else partial record */
 
 if (errptr==NULL)
    return(NULL); /* problem creating the problem record */
 else
 {
#ifdef DEBUG
    zdebug("get_problem() - created problem record.\n");
    zdebug("\tproblem gotten is:\n\n");
    print_prob(&pro_rec);
#endif

    return(&pro_rec);
 } /* end else created problem record */
} /* end get_problem() */

index_record *next_problem(ZFILE *db_pointer, int new, int links,
                           index_record *index, char *ss, int (*compfunc)())
{
 static long start;
 long current;
 int none_found;
 
#ifdef DEBUG
 zdebug("In next_problem().\n");
 if (new)
    zdebug("\tNEW leaf\n");
 else
    zdebug("\tPREVIOUS leaf.\n");
#endif

 if (new) /* searching a new database leaf */
    current=start=0;
 else
 {
    zseek(db_pointer, start, SEEK_SET);
    current=start;
 }
 none_found=TRUE;
 
 while(none_found)
 {
    if (zread(index, sizeof(index_record), 1, db_pointer) < 1)
    { /* problem reading database index */
       if (zerror(db_pointer)) /* was there an error reading? */
       {
          db_errorno=errno;
          break; /* get out of loop */
       }
       else /* must have been an EOF */
       {
          zdebug("next_problem() - at end-of-file.\n");

          rewind(db_pointer);
          current=0;
          if (start==0) /* we started the current search at the top */
             break; /* of the file, so we did not find an appropriate prob */
          else
             continue; /* go to the top of the loop */
       } /* end else it's an EOF */
    }
    else /* no problem reading the index file */
    {
       current+=sizeof(index_record); /* increment the current file position */
#ifdef DEBUG
       zdebug1("next_problem() - starting file location = %d\n", start);
       zdebug1("next_problem() - current file location = %d\n", current);
       print_index(index);
#endif
    } /* end else no problem */
    
    if ((is_linked(index)) && links) /* problem was linked to current leaf */
    {                                  /* and we are following links */

       zdebug("next_problem() - problem is linked.\n");

       if (check_index(db_pointer, index)==NULL)
          break; /* error updating index entry, so break out of loop */
    } /* end if LINKED problem */
    
/* if we've found a problem that meets the search criteria that hasn't been
 * deleted or moved, then we've gotten what we came for, so set the flag and
 * lets go. */
 
    zdebug1("next_problem() - compfunc sez %d\n", compfunc(index,ss));

    if ((compfunc(index, ss)) && (!is_gone(index)) && (!is_moved(index)))
    {

       zdebug("next_problem() - found a good problem.\n");

       if (!is_linked(index) || (is_linked(index) && links))
          none_found=FALSE;
#ifdef DEBUG
 else
    zdebug("\tproblem is linked and we're not following links.\n");
#endif
    } /* end if !is_gone() and !is_moved() */
#ifdef DEBUG
 else
    zdebug("next_problem() - problem is not what we are looking for.\n");
#endif
 } /* end while */
 
 if (none_found) /* we either rolled the index over once or there */
 {               /* was an error */
    if (current==start)
    {
       zdebug("next_problem() - no good problems found.\n");
       db_errorno=ROLLEDOVER;
    }
    return(NULL);
 } /* end if none_found */
 start=current;
 return(index);
} /* end next_problem() */

index_record *find_problem(ZFILE *db_pointer, char *prid, index_record *index)
{

  zdebug("\nIn find_problem()\n");
  zdebug1("\tPRID of problem to find = %s\n", prid);

  if (search_index(db_pointer, prid, index, 0L)==NULL)
     return(NULL); /* error locating problem desired */

  zdebug("find_problem() - Found index entry of desired problem\n");

  if (is_linked(index)) /* problem was found and it was linked to this */
  { /* leaf, so update the entry */

     zdebug("find_problem() - problem given is linked\n");

     if (check_index(db_pointer, index)==NULL)
    return(NULL); /* problem updating index entry */
  }
  if (is_gone(index))
  {
     zdebug("find_problem() - Problem has been deleted\n");

     db_errorno=DELETED;
     return(NULL);
  }

/* else everything worked so return the index record full of info. */
  return(index);
} /* end find_problem */

 
index_record *search_index(ZFILE *db_pointer, char *prid, index_record *index,
                long offset)
{
  long bottom,     /* of current search space */
       top,        /* of current search space */
       location;   /* current location in search space */
  int diff,
      not_found;

  zdebug("\nIn search_index()\n");
  zdebug1("\tprid - %s\n", prid);

  top=0;
  location=0;
  bottom=file_size(db_pointer, sizeof(index_record)); /* get the # of records 
                    * in the index file to be searched */
  not_found=TRUE;

  while (not_found)
  {     /* get the next index record */
     if (top>bottom) /* we've gone past the end of the file */
     {
    db_errorno=NOTFOUND;
    break; /* out of loop */
     }

     /* we're still in the file */
     location=((top+bottom)/2); /* find middle of search space */
/* go to the desired record */
     zseek(db_pointer, (location*(sizeof(index_record)))+offset, SEEK_SET);

     zdebug("search_index() - After Seeking, Before Reading\n");
     zdebug1("\tFile position = %d\n", ftell(db_pointer));
     zdebug1("\tLocation variable = %d\n", location);

     if (zread(index, sizeof(index_record), 1, db_pointer) < 1)
     {  /* there was a problem getting the record */
      if (zerror(db_pointer)) /* error reading file */
       db_errorno=errno;
        else /* the end of the file was found before the problem */
       db_errorno=NOTFOUND;
        break; /* out of loop */
     } /* end if */

     zdebug("search_index() - searching index given\n");
     zdebug1("\tPRID to find %s\n", prid);
     zdebug1("\tCurrent PRID %s\n", get_prid(index));

     /* valid record was found */
     diff=strcmp(prid, get_prid(index)); /* compare the ids */

     if (diff<0) /* index->prid < prid */
    bottom=location-1; /* search upper half of search space */
     else
    if (diff>0) /* index->prid > prid */
       top=location+1; /* search lower half of search space */
        else /* we've found it */
       not_found=FALSE;
  } /* end while */

/* we're out of the loop, if the search was sucessfull return the index entry
 * found, otherwise return NULL to signal an error.
 */
 
 if (not_found)
    return(NULL);
 else
    return(index);
} /* end search_index */

 
problem_record *create_record(char *db_path, index_record *index,
                  problem_record *pro_rec)
{
  reporter_record *rep_rec, /* for ease of reading */
          *rep_old;
  char *log;

  char path[DBPATHLEN],
       prid[MAXPRID],
       assfiles[FNAMELEN];
  index_record new_index;
  long size;
  int i;
  ZFILE *temp;

  zdebug("\nIn create_record()\n");

  if (is_linked(index)) /* problem record does not reside in this leaf */
  { /* so get the original index entry, index->ln_path is the db_path */

     zdebug("create_record() - the problem given is a link\n");

     if ((temp=zopen(get_ln_path(index), ".index", "r", INDEX))==NULL)
     { /* error opening index file */
        return(NULL);
     }
     else /* index was opened, so get original index entry, ln_prid is the ID */
        if (find_problem(temp, get_ln_prid(index), &new_index)==NULL)
        {  /* problem finding index entry */
           zclose(temp);
           return(NULL);
        }
        else /* we found the index entry */
        {
           zdebug("create_record() - found original problem\n");

           zclose(temp);
           nl_strncpy(prid, get_prid(&new_index), MAXPRID);/* prid id used to
                           * access records */
           nl_strncpy(pro_rec->short_description, get_sd(index), MAXSUMLEN);
           nl_strncpy(path, get_ln_path(index), DBPATHLEN); /* path is used to
                              * access records */
        } /* end found index entry */
  } /* end linked problem */
  else /* problem is not linked, so it resides in this leaf */
  {
     nl_strncpy(prid, get_prid(index), MAXPRID);
     nl_strncpy(pro_rec->short_description, get_sd(index), MAXSUMLEN);
     nl_strncpy(path, db_path, DBPATHLEN);
  } /* end else */

  zdebug("create_record() - creating problem record\n");

  nl_strncpy(pro_rec->prid, get_prid(index), MAXPRID);/* user is unaware of
                  * any links */
  pro_rec->status=get_sf(index); /* set the status of the problem being
                  * created SOLVED or UNSOLVED */
  pro_rec->reop_rec=NULL; /* this will point to a reopened record, if the 
               * problem has been reopened, will be set in 
               * view_reopen() */

  pro_rec->priority=get_priority(index); /*Set the priority.  DC 11/21/94*/

  zdebug("create_record() - assigned prid and status\n");
  zdebug3("\tprid - %s\n\tstatus - %d\n\tpriority -%d\n",
         pro_rec->prid, pro_rec->status, pro_rec->priority);

  zdebug1("create_record() - PtsRoot='%s'\n",PtsRoot) ;

  if (exists(PtsRoot, path, prid, ".rep")) /* get the reporters list, if 
                        * it exists */
  {
     zdebug("create_record() - getting reporter records\n");

     sprintf(assfiles, "/%s.rep", prid);
     if ((temp=zopen(path, assfiles, "r", REP))==NULL)
     { /* problem opening reporters file */
        return(NULL);
     }
 /* opened reporters files */
 /* get the number of reporter records in the file */
     size=file_size(temp, sizeof(reporter_record));
     pro_rec->num_rep=size;
     if (pro_rec->reporter_list!=NULL)
    free(pro_rec->reporter_list); /*free any allocated space for the 
                       * last reporters list*/
     if ((rep_rec=(reporter_record *)calloc(size, sizeof(reporter_record)))
                    ==NULL)
     { /* error allocating space for the reporters list */
        db_errorno=errno;
        zclose(temp);
        return(NULL);
     }
     for (i=0; i<size; i++)
     {  /* get the reporters list */
        if (zread(&(rep_rec[i]), sizeof(reporter_record), 1, temp) < 1)
        { /* error reading reporters file */
           db_errorno=errno;
           zclose(temp);
           return(NULL);
        }
     } /* end for */
     zclose(temp);
     pro_rec->reporter_list=rep_rec;
  } /* end if reporters file exists */
  else /* reporters file does not exist */
  {
     zdebug("create_record() - no reporters file\n");

     if (is_unsolved(index, "")) /* unsolved problems must have a reporters */
     {                /* list */
        db_errorno=NOREP;
        return(NULL);
     }
     pro_rec->num_rep=0;
     pro_rec->reporter_list=NULL;
  }
  /* now get the old reporters list, if it exists */

  zdebug1("create_record() - PtsRoot='%s'\n",PtsRoot) ;

  if (exists(PtsRoot, path, prid, ".rep_old"))
  {  /* the file exists */

     zdebug("create_record() - creating old reporters list\n");

     sprintf(assfiles, "/%s.rep_old", prid);
     if ((temp=zopen(path, assfiles, "r", REP))==NULL)
     { /* error opening old reporters file */
        db_errorno=errno;
        return(NULL);
     }
     size=file_size(temp, sizeof(reporter_record));
     pro_rec->num_old_rep=size;
     if (pro_rec->old_reporter_list!=NULL) /* free any space allocated for the
                        * last old rep. list */
    free(pro_rec->old_reporter_list);
     if ((rep_old=(reporter_record *)calloc(size, sizeof(reporter_record)))
      ==NULL)
     {  /* error alloating space for the old reporters list */
        db_errorno=errno;
        zclose(temp);
        return(NULL);
     }
     for (i=0; i<size; i++)
     {  /* get the old reporters list */
    if (zread(&(rep_old[i]), sizeof(reporter_record), 1, temp) < 1)
    {  /* error reading file */
       zclose(temp);
       db_errorno=errno;
       return(NULL);
        }
     } /* end for, got old reporters list */
     zclose(temp);
     pro_rec->old_reporter_list=rep_old;
  } /* end getting old reporters list */
  else /* old reporters list doesn't exist */
  {
     zdebug("create_record() - no old reporter records\n");

     if (is_solved(index, ""))/* solved problems must have an old reporters list */     {
    db_errorno=NOREP;
    return(NULL);
     }
     pro_rec->num_old_rep=0;
     pro_rec->old_reporter_list=NULL;
  }
  /* we've got all the lists and such now for the log file */

  zdebug("create_record() - getting problem log\n");

  sprintf(assfiles, "/%s.log", prid); /* the name of the problem log */
  if ((temp=zopen(path, assfiles, "r", REP))==NULL)
  { /* error opening log file */
     db_errorno=errno;
     return(NULL);
  }
/* we opened the problem log, now find out how much memory to allocate */
 size=file_size(temp, sizeof(char));
 if (pro_rec->log_file!=NULL)
    free(pro_rec->log_file);

/* since the terminating NULL isn't in the log file, we need to allocate space 
 * for it and then append it ourselves.
 */
 if ((log=(char *)malloc(size+1))==NULL)
 {  /* error allocating space for problem log */
    db_errorno=errno;
    zclose(temp);
    return(NULL);
 }
 zdebug("create_record() - allocated space for problem log\n");
 zdebug("\treading problem log into memory\n");

/* we allocated space for it now read it into memory */
 if (zread(log, sizeof(char), size, temp)<size)
 {  /* error reading log */
    zclose(temp);
    db_errorno=errno;
    return(NULL);
 }
 log[size]='\0';
 pro_rec->log_file=log;

 zdebug("create_record() - created problem record\n");

/* we've got everything now so return the problem record */
  zclose(temp);
  return(pro_rec);
} /* end create_record() */

 
problem_record *partial_record(char *path, index_record *index,
                problem_record *pro_rec)
{

 zdebug("\nIn partial_record()\n");
 zdebug("\tcreating partial record\n");

 nl_strncpy(pro_rec->prid, get_prid(index), MAXPRID);
 pro_rec->status=get_sf(index);
 nl_strncpy(pro_rec->short_description, get_sd(index), MAXSUMLEN);
 pro_rec->priority=get_priority(index); /*Set the priority.  DC 11/21/94*/

 zdebug("partial_record() - copied prid, status, priority and short description\n");

 pro_rec->num_rep=0;
 if (pro_rec->reporter_list!=NULL)
 {
    free(pro_rec->reporter_list);
    pro_rec->reporter_list=NULL;
 }

 zdebug("partial_record() - cleared reporters list\n");

 pro_rec->num_old_rep=0;
 if (pro_rec->old_reporter_list!=NULL)
 {
    free(pro_rec->old_reporter_list);
    pro_rec->old_reporter_list=NULL;
 }
 zdebug("partial_record - cleared old reporters list\n");

 if (pro_rec->log_file!=NULL)   /* extra ";" removed DC 3/11/94 */
 {
    free(pro_rec->log_file);
    pro_rec->log_file=NULL;
 }
 zdebug("artial_record() - cleared problem log\n");

 pro_rec->reop_rec=NULL;

 return(pro_rec);
} /* end partial_record() */

 
index_record *check_index(ZFILE *db_pointer, index_record *index)
{
  index_record org_index;
  ZFILE *org_temp;
  int changed; /* the index entry was changed */

  zdebug("\nIn check_index()\n");

  changed=FALSE; /* set to false each time through */

  if ((org_temp=zopen(get_ln_path(index), ".index", "r", INDEX))==NULL)
  {  /* error opening index file */
     return(NULL);
  }

  zdebug("check_index() - Looking for original problem in:\n");
  zdebug1("\tDB Path = %s\n", get_ln_path(index));
  zdebug1("\tPRID = %s\n", get_ln_prid(index));

  /* find the original index entry */
  if (search_index(org_temp, get_ln_prid(index), &org_index, 0L)==NULL)
  {  /* problem finding the index entry */
     zclose(org_temp);
     return(NULL);
  }
  else /* we found the index entry */
     zclose(org_temp);

  zdebug("check_index() - found original problem\n");

  if (is_gone(&org_index))
  {  /* original problem has been deleted */

     zdebug("check_index() - Original problem has been deleted\n");

     set_gone(index);
  }
  else /* the original problem still exists */
  {
     if (is_moved(&org_index))
     {  /* original problem has been moved to another leaf */
    /* so we need to propagate the change back to the index entry */

        zdebug("check_index() - original problem has been moved\n");

        set_ln_path(index, get_ln_path(&org_index), DBPATHLEN);
        set_ln_prid(index, get_ln_prid(&org_index), MAXPRID);
        zclose(org_temp); /* we don't need this anymore */

/* since the recursive call to check_index() will take care of updating the
 * index entry we can just return that return value (that is we don't have
 * to update the entry twice
 */
        return(check_index(db_pointer, index));
     } /* end org_index entry == MOVED */

/* else the problem resides were we think so update the index record passed in,
 * but only if there are any changes made.
 */

     zdebug("chech_index() - updating index entry\n");

     if (get_sf(index)!=get_sf(&org_index))
     {
        changed=TRUE;
        set_sf(index, get_sf(&org_index));
     }
     if (get_edit(index)!=get_edit(&org_index))
     {
        changed=TRUE;
        set_edit(index, get_edit(&org_index));
     }
     if (strcmp(get_acc_name(index), get_acc_name(&org_index))!=0)
     {
        changed=TRUE;
        set_account_name(index, get_acc_name(&org_index), MAXANAMELEN);
     }
     if (strcmp(get_node_name(index), get_node_name(&org_index))!=0)
     {
        changed=TRUE;
        set_node_name(index, get_node_name(&org_index), MAXHOSTNAMELEN);
     }
     if (get_pid(index)!=get_pid(&org_index))
     {
        changed=TRUE;
        set_pid(index, get_pid(&org_index));
     }
         if (get_priority(index)!=get_priority(&org_index)) /*Check priority. */
     {                                                      /*DC 11/21/94*/
        changed=TRUE;
        set_priority(index, get_priority(&org_index));
     }

  } /* end original link has not been moved */

  if (changed) /* the index entry was changed (updated) */
  {
/* now that we've updated the index entry we need to change the file
 * entry to reflect the updated entry.  Since the file pointer is pointing
 * to the beginning of the next index entry, we need to back the pointer
 * up one record to the beginning of the one we need to change.
 */

     zdebug("check_index() - writing out new index entry\n");

     zseek(db_pointer, -(long)sizeof(index_record), SEEK_CUR);
     if (zlock(db_pointer)) /* lock the index before updating */
     { /* lock succeeded, so write the record */
        if (zwrite(index, sizeof(index_record), 1, db_pointer)<1)
        {  /* problem writing the record */
           db_errorno=errno;
       zunlock(db_pointer);
       zseek(db_pointer, 0L, SEEK_CUR); /* since the file is opened for 
                * update there needs to be a seek or EOF 
                * between reads and writes, this takes care 
                * of it. */
       return(NULL);
        }
        else /* the write succeeded */
        {
       zunlock(db_pointer);
           zseek(db_pointer, 0L, SEEK_CUR); /* since the file is opened for 
                * update there needs to be a seek or EOF 
                * between reads and writes, this takes care 
                * of it. */
       return(index);
        } /* end else write suceeded */
     } /* end lock file */
     else /* lock failed */
        return(NULL);
  } /* end index entry was changed (updated) */
} /* end check_index */

 
#ifdef DEBUG /* we only need these functions while debugging the database */

void print_index(index_record *index)
{
 zdebug4("PRID - %s\tMRD - %d\tSF - %d\nEF - %d\n", get_prid(index),
         index->mrd, get_sf(index), get_edit(index));
 zdebug3("Account Name - %s\nNode Name - %s\nPID - %d\n",
         get_acc_name(index), get_node_name(index), get_pid(index));
 zdebug4("LN PRID - %s\nLN PATH - %s\nShort Desc. - %s\nFR - %s\n", 
         get_ln_prid(index), get_ln_path(index), get_sd(index), index->fr);
 zdebug1("PRIORITY - %d\n\n", get_priority(index)) ;
} /* end print_index() */
 
void print_prob(problem_record *pro_rec)
{
 int i;

 zdebug3("PRID - %s\nStatus - %d\nPriority - %d\n",  /*11/21/94 DC*/
         pro_rec->prid, pro_rec->status, pro_rec->priority) ;
 zdebug2("Short Desc. - %s\nReopened Record - %d\n", 
         pro_rec->short_description, pro_rec->reop_rec);

 if (pro_rec->num_rep==0)
    zdebug("No Reporters.\n\n");
 else
    for (i=0; i<pro_rec->num_rep; i++)
    {
       zdebug2("Reporter %d:\n\tAccount Name - %s\n\t", i,
                pro_rec->reporter_list[i].account_name);
       zdebug2("Node Name - %s\n\tTime - %s\n\n",
                pro_rec->reporter_list[i].node_name,
                pro_rec->reporter_list[i].date_time);
    }

 if (pro_rec->num_old_rep==0)
    zdebug("No Old Reporters.\n\n");
 else
    for (i=0; i<pro_rec->num_old_rep; i++)
    {
       zdebug2("Old Reporter %d:\n\tAccount Name - %s\n\t", i,
                pro_rec->old_reporter_list[i].account_name);
       zdebug2("Node Name - %s\n\tTime - %s\n\n",
                pro_rec->old_reporter_list[i].node_name,
                pro_rec->old_reporter_list[i].date_time);
    }
 if (pro_rec->log_file==NULL)
    zdebug("No Log File\n");
 else
    zdebug1("Log File:\n\n%s", pro_rec->log_file);
} /* end print_prob */

#endif
/* end of get_prob.c */
