/*
 * reopen_prob.c -- 
 * Chris Hyde
 * created: 
 *
 * Modified: Dean Collins Sat Feb 12 13:47:37 1994
 * Added unistd.h inclusion for SunOS as requested by
 * magnush@isy.liu.se (Magnus Hammerin).
 *
 * Modified: Dean Collins Sat Feb 12 17:37:09 1994
 * Modifications recommended by Jaap Haalboom (haalboom@prl.philips.nl).
 * Fixed typo in reopened_record().
 * Added code to check the return value of fseek in reopened_record().
 * Added additional locking mechanism for HP (PA) in delete_record().
 * 
 * Modified: Dean Collins Sat Feb 12 18:52:04 1994
 * Fixed several typos.
 *
 * Modified: Dean Collins Wed Feb 23 17:00:42 1994
 * Changed PTSROOT (a constant) to PtsRoot (a variable).
 *
 * Modified: Dean Collins Fri Feb 25 18:23:26 1994
 * Modification recommended by magnush@isy.liu.se (Magnus Hammerin).
 * Changed a NULL to a "" in the extension paramater to a call to create_file.
 *
 * Modifled: Dean Collins Sat Oct 08 19:17:29 1994
 * Changed the way the status of a fread() was checked since it was
 * possible the file being read was empty.  Changed requested by
 * Dan Bechard. (Thanks Dan!)  This was failing on SCO and SunOS.
 *
 * Modified: Chris Hyde 1/10/95
 * Removed include <stdio.h>, since it's now included in zdbm.h.
 * Made changes to I/O to reflect zdbm names (zread(), zwrite(), etc.)
 * Replaced direct references to the index records with functions/macros
 *   defined for that purpose.
 *
 * Modified: Dean Collins Thu Jun  8 13:31:27 PDT 1995
 * Replaced #ifdef DEBUG method of debug output with calls to
 * the zdebug macros.
 *
 */

/*
 * Copyright (c) 1992,1993,1994,1995 Christopher G. Hyde
 * Copyright (c) 1992 University of Idaho, Moscow, Idaho.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation free of charge for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the University of Idaho
 * not be used in advertising or publicity pertaining to distribution of
 * the software without specific, written prior permission.  The University
 * of Idaho makes no representations about the suitability of this software
 * for any purpose.  It is provided "as is" without express or implied warranty.
 *
 * THE UNIVERSITY OF IDAHO DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS.
 * IN NO EVENT SHALL THE UNIVERSITY OF IDAHO BE LIABLE FOR ANY SPECIAL,
 * INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM
 * LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE
 * OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 */

#include <string.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifdef SUNOS
#include <unistd.h>
#endif /*SUNOS*/

#include "zdbm.h"


int reopen_request(char *db_path, char *prid, char *reasons,
           reporter_record *rep_rec)
{
  index_record index,
           prindex;
  long num_rec;
  ZFILE *temp,
        *itemp;
  char path[DBPATHLEN];
  char tprid[MAXPRID];

  zdebug("\nIn reopen_request()\n");
  zdebug2("\tPath of problem to reopen - %s/%s\n", db_path, prid);

  db_errorno=0;

/* open the index of the problem to reopen */
 if ((itemp=zopen(db_path, ".index", "r+", INDEX))==NULL)
    return(FAIL); /* the open failed */

 if (find_problem(itemp, prid, &prindex)==NULL) /* find problem to reopen */
 {  /* error searching for problem */
    zclose(itemp);
    return(FAIL);
 }

 zdebug("reopen_request() - found problem to reopen\n");

 if (is_unsolved(&prindex, "")) /* can only reopen SOLVED problems */
 {
    db_errorno=ISUNSOLVED;
    zclose(itemp);
    return(FAIL);
 }
 if (is_gone(&prindex)) /* can't reopen a deleted problem */
 {
    db_errorno=DELETED;
    zclose(itemp);
    return(FAIL);
 }
 if (is_linked(&prindex)) /* the problem was linked to the leaf given */
 {
    zdebug("reopen_request() - problem to reopen is a link\n");

/* since we are setting the mrd field of records that are requested for
 * reopening we need to have itemp pointing to the original index entry if
 * the problem given is a link
 */
    nl_strncpy(path, get_ln_path(&prindex), DBPATHLEN); /* path is used to
                        * access the records */
    nl_strncpy(tprid, get_ln_prid(&prindex), MAXPRID); /* tprid is used to 
                        * access the records */

    zclose(itemp);
    if ((itemp=zopen(path, ".index", "r+", INDEX))==NULL)
     /* problem opening index file */
       return(FAIL);

    if (find_problem(itemp, tprid, &prindex)==NULL)
    { /* problem getting original index entry */
       zclose(itemp);
       return(FAIL);
    }
 } /* end problem is a link */
 else /* the problem resides in the leaf given */
 {
    nl_strncpy(path, db_path, DBPATHLEN);
    nl_strncpy(tprid, prid, MAXPRID);
 }
 
 if (!exists(PtsRoot, REOPENDIR, "", "")) /* does the reopen directory */
 {                    /* exist ? */
    make_branch(REOPENDIR); /* no so create it */
    if ((temp=zopen(REOPENDIR, ".index", "r+", INDEX))==NULL)
    { /* problem opening new index */
       zclose(itemp);
       return(FAIL);
    }
/* after creating the reopen directory, we need to write out the number of
 * records that are in the directory, 0 in this case. */
    num_rec=0;
    if (zwrite(&num_rec, sizeof(long), 1, temp)<1)
    {   /* the write failed */
       db_errorno=errno;
       zclose(temp);
       zclose(itemp);
       return(FAIL);
    }
    rewind(temp);
    zdebug("reopen_request() - created reopen directory\n");
 } /* end reopen dir. does not exist */
 else /* reopen dir exists */
 {
    zdebug("reopen_request() - found reopen directory\n");

    if ((temp=zopen(REOPENDIR, ".index", "r+", INDEX))==NULL)
    {
       zclose(itemp);
       return(FAIL); /* the open failed */
    }
 } /* end reopen dir. exists */

 if (zlock(temp)==FAIL) /* lock the reopen index */
 {  /* the lock failed */
    zclose(itemp);
    zclose(temp);
    return(FAIL);
 }
 if (zread(&num_rec, sizeof(long), 1, temp)<1) /* get the number of */
 {   /* records in the index, report failure if read fails */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    zclose(itemp);
    return(FAIL);
 }

 zdebug1("reopen_request() - there have been %d reopened requests made\n", num_rec);

/* we use double problem to check for duplicate problems rather than looking
 * at the mrd field, since if the problem is either a link or an original link
 * the field will not be set correctly
 */
 if (double_problem(temp, path, tprid)) /* check for duplicate reports */
 { /* the problem has already been reported */
    zunlock(temp);
    zclose(temp);
    zclose(itemp);
    return(FAIL);
 }

 zdebug("reopen_request() - creating reopened index record\n");

 set_reopen(&index);
 set_sf(&index, UNSOLVED);
 set_priority(&index, get_priority(&prindex));   /* DC 11/21/94 */
 set_edit(&index, FALSE);
 set_account_name(&index, "", 1);
 set_node_name(&index, "", 1);
 set_ln_prid(&index, tprid, MAXPRID);
 set_ln_path(&index, path, DBPATHLEN);
 set_sd(&index, get_sd(&prindex), MAXSUMLEN); 
 set_prid(&index, new_prid(num_rec), MAXPRID);
 if (is_null_string(get_prid(&index)))
 {  /* error creating id */
    zunlock(temp);
    zclose(temp);
    zclose(itemp);
    return(FAIL);
 }

 zdebug("reopen_request() - created record, now appending it\n");

 zseek(temp, 0L, SEEK_END); /* move to the end of the reopen index for 
                 * appending */

 if (zwrite(&index, sizeof(index_record), 1, temp)<1)
 { /* error writing to the index */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    zclose(itemp);
    return(FAIL);
 }

 zdebug("reopen_request() - wrote new record, now updating record count\n");

 /* now go back to the beginning and write out a new record count */
 rewind(temp);
 if (++num_rec>MAXREC) /* we've maxed out the # of possible records */
    num_rec=0; /* hopefully the low numbers have been removed */

 if (zwrite(&num_rec, sizeof(long), 1, temp)<1)
 { /* an error occured while writing */
    zunlock(temp);
    zclose(temp);
    zclose(itemp);
    delete_record(get_prid(&index));
    return(FAIL);
 }
 /* else the write succeeded so close the reopen index and finish up */
 
 zunlock(temp);
 zclose(temp);

 zdebug("reopen_request() - updated record count\n");
 zdebug("reopen_request() - now creating reasons log\n");

 if (write_log(REOPENDIR, get_prid(&index), reasons)==FAIL)
 { /* error creating reasons log */
    delete_record(get_prid(&index));
    zclose(itemp);
    return(FAIL);
 }

 zdebug("reopen_request() - wrote reasons log\n");
 zdebug("reopen_request() - creating reporters file\n");
 
 if ((temp=create_file(REOPENDIR, get_prid(&index), ".rep", FILE_PERMS))==NULL)
 { /* error creating reporters file */
    delete_record(get_prid(&index));
    zclose(itemp);
    return(FAIL);
 }
 if (zwrite(rep_rec, sizeof(reporter_record), 1, temp)<1)
 {  /* error writing reporter record to file */
    db_errorno=errno;
    delete_record(get_prid(&index));
    zclose(temp);
    zclose(itemp);
    return(FAIL);
 }

 zdebug("reopen_request() - created reporter file\n");

 zclose(temp);

/* since we created the reopened record sucessfully we need to set the mrd
 * field of the problem record, but only if the problem is not an original
 * link (since reopen requests are treated like special cases of original
 * links as far as moving and deleting)
 */

 if (is_org_link(&prindex))
 {
    set_reopen(&prindex);

/* since the file pointer is pointing to the end of the index entry we want
 * to overwrite we need to back up one index record.
 */

    zseek(itemp, -(long)(sizeof(index_record)), SEEK_CUR);
    if (zwrite(&prindex, sizeof(index_record), 1, itemp)<1)
    { /* problem writing the new index entry */
       zclose(itemp);
       delete_record(get_prid(&index));
       return(FAIL);
    }
 } /* end !ORGLINK */

/* if we get here everything worked out ok so let the caller know */
 zclose(itemp);
 return(SUCESS);
} /* end reopen_request */

 
int double_problem(ZFILE *db_pointer, char *db_path, char *prid)
{
 index_record index;
 int no_doubles=TRUE;

 zdebug("\nIn double_problem()\n");
 zdebug2("\tProblem to search for - %s/%s\n", db_path, prid);
 zdebug("\tSearching ");

 while ((no_doubles) &&
        (zread(&index, sizeof(index_record), 1, db_pointer) < 1))
 {

    zdebug("?");

    if ((strcmp(get_ln_path(&index), db_path)==0) &&
        (strcmp(get_ln_prid(&index), prid)==0))
       no_doubles=FALSE;
 }
#ifdef DEBUG
 if (no_doubles)
    zdebug("\ndouble_problem() - no doubles found\n");
 else
    zdebug("\ndouble_problem() - found a double\n");
#endif
 if (zerror(db_pointer))
 {  /* there was an i/o error */
    db_errorno=errno;
    return(ERROR);
 }
 else
    if (no_doubles) /* there were no doubles */
       return(FALSE);
    else
    { /* there was a double problem */
       db_errorno=EXISTS;
       return(TRUE);
    } /* end else */
} /* end double_problem */

 
reopened_record *view_reopen(char *reopen_prid, int whole_record)
{
 static reopened_record reop;
 static long location=sizeof(long);
 char db_path[DBPATHLEN];
 index_record index;
 int size;
 char *log;
 ZFILE *temp;

 zdebug("\nIn view_reopen()\n");
 zdebug2("\tPrid = %s\n\tWhole record = %d\n", reopen_prid, whole_record);

 db_errorno=0;

 if ((temp=zopen(REOPENDIR,".index", "r+", INDEX))==NULL)
    return(NULL); /* the open failed */

/* we want to check to see if there any reopen problems before doing anything
 * else, so if file_size() returns a 1 then there are no problems
 */

 if (file_size(temp, sizeof(long))==1)
 { /* no reopen requests */
    db_errorno=NOTFOUND;
    return(NULL);
 }

 if (reopen_prid!=NULL) /* does the caller wants a particular problem */
 {

    zdebug1("view_reopen() - looking for a particular problem - %s\n",
            reopen_prid);

    if (find_reopen(temp, &index, reopen_prid)==NULL)
    { /* problem finding the record */
       zclose(temp);
       return(NULL);
    } /* end if */
    if (bad_problem(&index, temp))
    { /* the original problem has been deleted */

       zdebug1("view_reopen() - problem %s is bad\n", get_prid(&index));

       zclose(temp);
       if (db_errorno==DELETED)
       {
          zdebug2("view_reopen() - problem %s/%s has been deleted\n",
                 get_ln_path(&index), get_ln_prid(&index));

          delete_record(get_prid(&index));
          location=sizeof(long); /* since we don't know where the problem that
                      * was deleted came from (before or after 
                      * location, we just reset it to the beginning
                      */
       }
       return(NULL);
    }
 } /* end user wants a particular problem */
 else
 { /* the user just wants the next one */

    zdebug("view_reopen() - seeking past last record viewed\n");

/* seek to the last location looked at.  If that happens to be the end of 
 * file then rewind the pointer and seek past the record counter at the 
 * begining of the file.
 */

    zseek(temp, location, SEEK_SET);

    zdebug("view_reopen() - reading the next record\n");

/* Now read the index record */

/* the reopen directory is a virtual circular list, when the end is reached
 * you start at the beginning.  The following goto makes the code more
 * "elegant".
 */

    kluge:
    if (zread(&index, sizeof(index_record), 1, temp) < 1)
    {  /* error reading from file */
       if (feof(temp))
       {
          rewind(temp);
          location=sizeof(long); /* seek past the record counter */
          zseek(temp, location, SEEK_SET);
          goto kluge; /* get the first record */
       }
       else /* it's an error */
       {
          db_errorno=errno;
          zclose(temp);
          return(NULL);
       } /* end error */
    } /* end read */

    if (bad_problem(&index, temp))
    { 
       zdebug1("view_reopen() - problem %s is bad\n", get_prid(&index));

       if (db_errorno=DELETED) /* the original problem has been deleted */
       {
          zdebug2("view_reopen() - problem %s/%s has been deleted\n",
                 get_ln_path(&index), get_ln_prid(&index));

/* inorder to remove the offending record from the reopen leaf, we have to 
 * close the file, remove the record and then open it again.  Then since the
 * user just wants the next record we need to get it for them
 */
          zclose(temp);
          delete_record(get_prid(&index));
          if ((temp=zopen(REOPENDIR, ".index", "r+", INDEX))==NULL)
          { /* problem opening index */
         return(NULL);
          }
      if (file_size(temp, sizeof(long))==1) /* there are no more records */
      {
         db_errorno=NOTFOUND;
         zclose(temp);
         return(NULL);
          }
       } /* else it was an error, so skip the record for now */
       else
      location+=sizeof(index_record);
/*       zseek(temp, location, SEEK_SET);*/
                /* This block added 2/12/94, DC */
       if (zseek(temp, location, SEEK_SET) == -1)
       {
#ifdef DEBUG
          zdebug("view_reopen() - error during zseek\n");
                 print_index(&index);
#endif

          db_errorno=errno;
          zclose(temp);
          return(NULL);
       }

       goto kluge;
    } /* end found a bad problem */
#ifdef DEBUG
 zdebug("view_reopen() - index record read\n");
 print_index(&index);
#endif

/* increment the location pointer */
    location+=sizeof(index_record);
 } /* end else the user didn't want a particular problem */

 zdebug("view_reopen() - creating reopened record\n");
 zdebug("\tgetting reasons log\n");

 zclose(temp); /* we don't need the index anymore */
 nl_strncpy(reop.prid, get_prid(&index), MAXPRID); /* the problem id of the 
                       * reopened problem */
 nl_strncpy(reop.db_path, get_ln_path(&index), DBPATHLEN); /* the path to the leaf 
                /* containing the problem to be reopened */

 sprintf(db_path, "%s/%s", REOPENDIR, reop.prid);
 if ((temp=zopen(db_path, ".log", "r", REP))==NULL)
 {  /* error opening reasons log */
    return(NULL);
 }
 size=file_size(temp, sizeof(char));
 if (reop.reasons_log!=NULL) /* free any space allocated for the last log */
    free(reop.reasons_log);

/* since the terminating null character isn't in the log file, we need
 * to append it to the strign we create
 */

 if ((log=(char *)calloc(size+1, sizeof(char)))==NULL)
 {  /* error allocating space for reasons log */
    db_errorno=errno;
    zclose(temp);
    return(NULL);
 }
 zdebug("view_reopen() - allocated space for reasons log\n");


/* The next two lines were modified 10/8/94, DC, as requested
 * by Dan Bechard.  (Thanks Dan!)  This was failing on SCO and SunOS.
 * Use zerror() to check for problems rather than the return of zread().
 * The log could be empty, which can cause zread() to return NULL.
 */
 zread(log, sizeof(char), size, temp) ;

 if (zerror(temp) != 0)  /* zread() failed */
 {  /* error reading reasons log */
    db_errorno=errno;
    zclose(temp);
    return(NULL);
 }
 log[size]='\0'; /* terminate the log with a NULL */
 reop.reasons_log=log;
 zclose(temp);

 zdebug("view_reopen() - read reasons log\n");
 zdebug("\tgetting reporter record\n");

 sprintf(db_path, "%s/%s", REOPENDIR, reop.prid);
 if ((temp=zopen(db_path, ".rep", "r", REP))==NULL)
    return(NULL);  /* the open failed */
 if (zread(&reop.rep_rec, sizeof(reporter_record), 1, temp) < 1)
 {  /* error reading file */
    db_errorno=errno;
    zclose(temp);
    return(NULL);
 }
 zclose(temp);

 zdebug("view_reopen() - got reporter record\n");
 zdebug("\tgetting problem record to reopen\n");

/* Now get the original problem */
 if ((reop.pro_rec=get_problem(get_ln_path(&index), get_ln_prid(&index),
                PARTICULAR, 0, LINKS, whole_record)) == NULL)
    return(NULL); /* there was an error getting the problem record */
 else 
 {
    reop.pro_rec->status=REOPENED; /* the problem is reopened */
    reop.pro_rec->reop_rec=&reop; /* points back to the reopened record */

    zdebug("view_reopen() - changed problem status to REOPENED\n");
    zdebug1("\tstatus - %d\n", reop.pro_rec->status);

    return(&reop);
 }
} /* end view_reopen */

 
char *confirm_reopen(char *reop_prid, reporter_record *reop_rec,
             char *sys_string, char *reasons)
{
 index_record index;
 static index_record org_index;
 ZFILE *temp,
       *org_temp,
       *temp_two;
 char not_found=TRUE;
 char path[COMPLETEPATHLEN];
 long count,
      bottom;

  zdebug("\nIn confirm_reopen()\n");
  zdebug2("\tPRID to confirm - %s\n\tSysop - %s\n", reop_prid,
         sys_string);
  zdebug1("\tReasons - %s\n", reasons);

  db_errorno=0;

 if ((temp=zopen(REOPENDIR, ".index", "r+", INDEX))==NULL)
    return(NULL); /* the open failed */
 else
    if (zlock(temp)==FAIL) /* lock the reopen index file */
    {  /* the lock failed */
       zclose(temp);
       return(NULL);
    }

 if (find_reopen(temp, &index, reop_prid)==NULL)
 {
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }

 zdebug("\nconfirm_reopen() - found desired problem\n");
 zdebug("\tnow find original problem record\n");

/* Open the index for the leaf that holds the problem to be ropened */
 if ((org_temp=zopen(get_ln_path(&index), ".index", "r+", INDEX))==NULL)
 {  /* could not open the index file */
    zunlock(temp);
    zclose(temp);
    return(NULL);
 }

/* Now find the problem to be reopend */
 if (find_problem(org_temp, get_ln_prid(&index), &org_index)==NULL)
 {  /* error looking for the problem */
    zunlock(temp);
    zclose(temp);
    zclose(org_temp);
    return(NULL);
 }

 zdebug("confirm_reopen() - found original problem\n");
 zdebug("\tnow creating the new reporter file\n");

/* create the new reporters list and add the new reporters record */
 if ((temp_two=create_file(get_ln_path(&index), get_prid(&org_index), ".rep",
               FILE_PERMS))==NULL)
 {  /* error creating new file */
    zunlock(temp);
    zclose(temp);
    zclose(org_temp);
    return(NULL);
 }
 if (zwrite(reop_rec, sizeof(reporter_record), 1, temp_two)<1)
 {  /* error writing to the file */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    zclose(org_temp);
    zclose(temp_two);
    remove_file(get_ln_path(&index), get_prid(&org_index), ".rep");
    return(NULL);
 }
 else
    zclose(temp_two);

 zdebug("confirm_reopen() - created new reporter file\n");
 zdebug("\tappending sysop string to log\n");

/* Now add the name of the person confirming the reopen (sys_string) and
 * the reasons for the reopening to the problem log, but first get the position
 * of the current end of file.  This way if an error occurs before completing
 * the reopening the additions can be removed (at least in theory).
 */
 sprintf(path, "%s/%s", get_ln_path(&index), get_prid(&org_index));
 if ((temp_two=zopen(path, ".log", "w", REP))==NULL)
 {  /* error opening log file */
    zunlock(temp);
    zclose(temp);
    zclose(org_temp);
    remove_file(get_ln_path(&index), get_prid(&org_index), ".rep");
    return(NULL);
 }
/* seek to the bottom of the file before appending */
 zseek(temp_two, 0L, SEEK_END);
 bottom=ftell(temp_two); /* get the position of the end of the file */

 if (zwrite(sys_string, sizeof(char), strlen(sys_string), temp_two)
        <strlen(sys_string))
 { /* error writing sys_string to log */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    zclose(org_temp);
    zclose(temp_two);
    remove_file(get_ln_path(&index), get_prid(&org_index), ".rep");
    return(NULL);
 }

 zdebug("confirm_reopen() - wrote sysop string\n");
 zdebug("\tappending reasons\n");

 if (zwrite(reasons, sizeof(char), strlen(reasons), temp_two)
        <strlen(reasons))
 {  /* error writing to the log file */
    db_errorno=errno;
    zunlock(temp);
    zclose(temp);
    zclose(org_temp);
    zclose(temp_two);
    trunc_file(path, ".log", bottom); /* cut off the extra stuff */
    remove_file(get_ln_path(&index), get_prid(&org_index), ".rep");
    return(NULL);
 }

 zdebug("confirm_reopen() - appended reasons\n");
 zdebug("\tnow update the index entry\n");

/* now lock the index file and write out a new index entry for the reopened
 * problem.
 */

 if (zlock(org_temp)==FAIL)
 {  /* the locking failed */
    zunlock(temp);
    zclose(temp);
    zclose(org_temp);
    zclose(temp_two);
    trunc_file(path, ".log", bottom);
    remove_file(get_ln_path(&index), get_prid(&org_index), ".rep");
    return(NULL);
 }

 set_sf(&org_index, UNSOLVED);
 if ((!is_org_link(&org_index)) || (!is_linked(&org_index))
     || (!is_moved(&org_index)))
    set_mrd(&org_index, 0);
 zseek(org_temp, -(long)sizeof(index_record), SEEK_CUR);
 if (zwrite(&org_index, sizeof(index_record), 1, org_temp)<1)
 {  /* error updateing index */
    db_errorno=errno;
    zunlock(temp);
    zunlock(org_temp);
    zclose(temp);
    zclose(org_temp);
    zclose(temp_two);
    trunc_file(path, ".log", bottom);
    remove_file(get_ln_path(&index), get_prid(&org_index), ".rep");
    return(NULL);
 }

 zdebug("confirm_reopen() - updated index entry\n");
 zdebug("\tnow remove the reopen record\n");

 /* now unlock and close everything, then remove the entry from the reopend
  * directory.  We are not overly concerned if the record removal fails, since
  * the update has occured.
  */

  zunlock(org_temp);
  zclose(org_temp);
  zunlock(temp);
  zclose(temp);
  zclose(temp_two);
  if (delete_record(reop_prid)==FAIL)
  {
     zdebug1("deletetion failed,errno=%d\n",errno) ;

    /* deletion failed */
     return(NULL);
  }
  else
     return(get_prid(&org_index));
} /* end confirm_reopen */

 
int deny_reopen(char *reop_prid)
{
 ZFILE *temp;
 index_record index;
 char tprid[MAXPRID],
      tpath[DBPATHLEN];

  zdebug("\nIn deny_reopen()\n");
  zdebug1("\tPRID - %s.\n", reop_prid);

  db_errorno=0;

 if ((temp=zopen(REOPENDIR, ".index", "r", INDEX))==NULL)
 { /* problem opening reopen index */
    return(FAIL);
 }

 if (find_reopen(temp, &index, reop_prid)==NULL)
 { /* problem finding reopen record */
    zclose(temp);
    return(FAIL);
 }

 zdebug("deny_reopen() - found reopen index record desired\n");

 zclose(temp); /* we don't need the reopen index anymore */
 nl_strncpy(tprid, get_ln_prid(&index), MAXPRID);
 nl_strncpy(tpath, get_ln_path(&index), DBPATHLEN);

 zdebug("deny_reopen() - now reset the original problem mrd to 0\n");

/* now open the index the original problem resides in, so we can set the
 * mrd field to 0 (the problem is no longer requested for reopening.
 */

 if ((temp=zopen(tpath, ".index", "r+", INDEX))==NULL)
    return(FAIL);

 if (find_problem(temp, tprid, &index)==NULL)
 { /* error finding problem */
    zclose(temp);
    return(FAIL);
 }

/* we've got the index entry, so reset the mrd field and write the new index
 * entry out, after resetting the file pointer to point to the begining of the
 * entry we want to overwrite.  Also don't reset the mrd field if it's already
 * set ot LINKED, ORGLINK or MOVED
 */

 if (zlock(temp)==FAIL) /* lock the index before updating it */
 {
    zclose(temp);
    return(FAIL);
 }
 if ((!is_org_link(&index))  && (!is_moved(&index)) && (!is_linked(&index)))
 { /* we can reset the mrd field */
    set_mrd(&index, 0);
    zseek(temp, -(long)sizeof(index_record), SEEK_CUR);
    if (zwrite(&index, sizeof(index_record), 1, temp)<1)
    { /* problem writing the index entry */
       zunlock(temp);
       zclose(temp);
       return(FAIL);
    }
    zunlock(temp);
    zclose(temp);
 }
 else
    zclose(temp);

 return(delete_record(reop_prid));
} /* end deny_reopen */

 
int delete_record(char *reop_prid)
{
 index_record index;
 char *iname;
 char tfile[FNAMELEN+1]; /*10/8/94, DC*/
 ZFILE *temp,
       *temp_two;
 long num_rec;
#ifdef __hppa
 int templock = 1; /* HP (PA) seems to need help with locking. 2/12/94 DC */
#endif /*__hppa*/

  zdebug1("\nIn delete_record()\n\tRecord to remove - %s\n", reop_prid);

  nl_strncpy(tfile, TEMPFILE, FNAMELEN);

 if ((temp=zopen(REOPENDIR, ".index", "r", INDEX))==NULL)
    return(FAIL);

#ifdef __hppa
 templock = 0;  /* Reset additional lock for HP (PA) 2/12/94 DC */
#endif

 zdebug2("\tdelete_record() - temp=%p,(file descriptor=%d)\n",temp,fileno(temp)) ;

#ifdef __hppa
 if (templock)  /* added 2/12/94 DC */
#endif
 if (zlock(temp)==FAIL)
 {  /* the lock of the index failed */
    zclose(temp);
    return(FAIL);
 }

 zdebug("delete_record() - creating temporary file\n");

 if ((temp_two=create_file(REOPENDIR, (char *)mktemp(tfile), "", /*2/25/94 DC*/
               FILE_PERMS))==NULL)
 {  /* the creation failed */

#ifdef __hppa
 if (templock)  /* added 2/12/94 DC */
#endif
    zunlock(temp);
    zclose(temp);
    return(FAIL);
 }
 if (zlock(temp_two)==FAIL)
 {  /* the lock the temp file failed */
#ifdef __hppa
    if (templock)   /* added 2/12/94 DC */
#endif
    zunlock(temp);
    zclose(temp);
    zclose(temp_two);
    remove_file(REOPENDIR, tfile, NULL);
    return(FAIL);
 }

/* we need to read the record count from the index file and write it to the
 * new index file before searching for the records. */

 zread(&num_rec, sizeof(long), 1, temp);
 zwrite(&num_rec, sizeof(long), 1, temp_two);

 zdebug("delete_record() - searching for record to remove ");

 while (zread(&index, sizeof(index_record), 1, temp)==1)
 {
    zdebug("?");

    if (strcmp(get_prid(&index), reop_prid)!=0)
       if (zwrite(&index, sizeof(index_record), 1, temp_two)<1)
      break; /* the write failed */
 } /* end while */

 if (zerror(temp) || zerror(temp_two)) /* did we leave the loop because of an 
                    * error reading or writing? */
 {
    db_errorno=errno;
#ifdef __hppa
    if (templock)   /* added 2/12/94 DC */
#endif
    zunlock(temp);
    zclose(temp);
    zunlock(temp_two);
    zclose(temp_two);
    remove_file(REOPENDIR, tfile, NULL);
    return(FAIL);
 }

 zdebug("\ndelete_record() - removed record from index\n");
 zdebug("\trenaming temp file\n");

/* now that we've removed the offending index record, delete the old index
 * and rename the temp file, so it will become the new index.  
 */

 iname=(char *)rindex(REOPENDIR, '/');
 if (iname==NULL)
    iname=REOPENDIR; /* the name of the index file is the name of the dir */
 else
    iname++; /* want char after '/' */
#ifdef __hppa
 if (templock)  /* added 2/12/94 DC */
#endif
 zunlock(temp);
 zunlock(temp_two);
 zclose(temp);
 zclose(temp_two);
 rename_file(REOPENDIR, tfile, "", REOPENDIR, iname, ".index");
 remove_file(REOPENDIR, reop_prid, ".rep");
 remove_file(REOPENDIR, reop_prid, ".log");
 return(SUCESS);
} /* end delete_record */

 
index_record *find_reopen(ZFILE *db_pointer, index_record *index, char *prid)
{

 zdebug("In find_reopen()\n");
 zdebug2("\tsearching for %s/%s\n", REOPENDIR, prid);

/* call search_index() to look for the particular problem */

 return(search_index(db_pointer, prid, index, sizeof(long)));
} /* end find_reopen() */
 
/* now create the reopened record, but before doing that check to make sure
 * the original problem hasn't been deleted or moved.  if it's been deleted
 * then remove the entry from the reopen leaf, if its been moved then update
 * the link fields
 */

int bad_problem(index_record *index, ZFILE *db_pointer)
{
 index_record prindex;
 ZFILE *intemp;

 zdebug("In bad_problem()\n");
 zdebug2("\tProblem to look for %s/%s\n", get_ln_path(index), get_ln_prid(index));

 if ((intemp=zopen(get_ln_path(index), ".index", "r", INDEX))==NULL)
 { /* problem opening the problem index */
    return(ERROR);
 }
 if (find_problem(intemp, get_ln_prid(index), &prindex)==NULL)
 { /* problem finding the index record */
    zclose(intemp);
    if (db_errorno==DELETED)
    { /* the problem has been deleted */

       zdebug("bad_problem() - original problem has been deleted\n");

       return(TRUE);
    }
    else
       return(ERROR);
 } /* end problem with index */
 zclose(intemp); /* we don't need it now */

/* found the index record now update the reopen index record, if needed */

 if (is_moved(&prindex))
 { /* the problem was moved, so update the reopen record */

    zdebug("bad_problem() - original problem has been moved - updating\n");

    set_ln_path(index, get_ln_path(&prindex), DBPATHLEN);
    set_ln_prid(index, get_ln_prid(&prindex), MAXPRID);

    zseek(db_pointer, -(long)sizeof(index_record), SEEK_CUR);
    if (zwrite(index, sizeof(index_record), 1, db_pointer)<1)
    {
       db_errorno=errno;
       return(ERROR);
    }
    else
    {
       zdebug("bad_problem() - reopen index entry has been updated\n");

       return(FALSE);
    }
 } /* end is_moved() */
 else /* the origial proble has no problems */
 {
    zdebug("bad_problem() - problem is good\n");
 
    return(FALSE);
 } /* end else no problems */
} /* end bad_problem() */

int is_reopened(char *path, char *prid)
{
 ZFILE *temp;
 index_record index;
 char tprid[MAXPRID],
      tpath[DBPATHLEN];
 int result;

 zdebug("In is_reopened()\n");
 zdebug2("Checking for problem %s/%s\n", path, prid);

 db_errorno=0;

/* since the index entries of the REOPEN index point back to the original
 * problem in a series of linked problems, we need to get the index of
 * the problem to check and get the path to the origianl problem if it's
 * linked, if it's not then we use the path and prid passed in
 */

 if ((temp=zopen(path, ".index", "r+", INDEX))==NULL)
    return(ERROR); /* problem opening index file */

 if (find_problem(temp, prid, &index)==NULL)
 {
    zclose(temp);
    return(ERROR);
 }
 zclose(temp); /* we don't need to keep the index open after getting the
            * index entry */

 if (is_linked(&index))
 { /* we have different paths and prids for linked problems */
    nl_strncpy(tprid, get_ln_prid(&index), MAXPRID);
    nl_strncpy(tpath, get_ln_path(&index), DBPATHLEN);
 }
 else /* the problem isn't linked */
 {
    nl_strncpy(tprid, prid, MAXPRID);
    nl_strncpy(tpath, path, DBPATHLEN);
 }

 if ((temp=zopen(REOPENDIR, ".index", "r", INDEX))==NULL)
 { /* problem opening the reopen index */
    return(ERROR);
 }
 else
 {
    result=double_problem(temp, tpath, tprid);
    zclose(temp);
    return(result);
 } /* end else */
} /* end is_reopened() */
/* end of reopen_prob.c */
