/*--------------------------------*-C-*---------------------------------*
 * File:	screen.c
 *
 * This module is all new by Robert Nation
 * <nation@rocket.sanders.lockheed.com>
 *
 * Additional modifications by mj olesen <olesen@me.QueensU.CA>
 * No additional restrictions are applied.
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 *
 * Design of this module was heavily influenced by the original xvt
 * design of this module. See info relating to the original xvt elsewhere
 * in this package.
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#include <ctype.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
# include <unistd.h>
#endif
#ifdef HAVE_SYS_IOCTL_H
# include <sys/ioctl.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "screen.h"
#include "command.h"
#include "xsetup.h"
#include "debug.h"
#include "graphics.h"

/* chunk size for retrieving the selection property */
#define PROP_SIZE	4096
#define	TAB_SIZE	8	/* default tab size */

/* Memory copy methods -- rgg 20/11/95 */
#define MEMCOPY         memcpy          /* no overlap -- rgg 04/11/95 */
/* #define MEMCOPY      memmove */      /* allows overlap -- rgg 04/11/95 */

/*----------------------------------------------------------------------*
 * extern variables referenced
 */

/*----------------------------------------------------------------------*
 * extern variables declared here
 */
TermWin_t TermWin;

/* convert pixel dimensions to row/column values */
#define Pixel2Col(x)	(((x) - MARGIN) / TermWin.fwidth)
#define Pixel2Row(y)	(((y) - MARGIN) / TermWin.fheight)

#define Col2Pixel(col)	((col) * TermWin.fwidth  + MARGIN)
#define Row2Pixel(row)	((row) * TermWin.fheight + MARGIN)

#define NEW_WRAPTYPE		/* track which lines have been autowrapped */
#define NEW_SELECTION		/* actually un-mark selection correctly */

/* the `quick' XCopyArea() code destroys wrap information anyhow */
#ifdef USE_XCOPYAREA
#  undef NEW_WRAPTYPE
#endif

/*
 * if last char is not whitespace, selection wraps next line with no '\n'
 * this is preferable 99% of the cases
 */
#ifndef NEW_WRAPTYPE
#  define WRAP_LASTCHAR_HACK
#endif

/*#define OLD_COLOR_MODEL*/	/* use the old color model */

#define SPACE ' '

#ifdef OLD_COLOR_MODEL
#  define CURRENT_RSTYLE RS_NONE
#  define REND_SET(addr,count) \
/**/  memset (addr, RS_NONE, count * sizeof(rend_t))
#else
#  define CURRENT_RSTYLE	(rstyle)
#  define REND_SET(addr,count) do { \
/**/  rend_t *p = addr; int n = count;\
/**/  while (n-- > 0) *p++ = CURRENT_RSTYLE; \
} while (0)
#endif

/*----------------------------------------------------------------------*
 * local variables
 */

#if defined (NO_COLOR) && !defined (KANJI)
typedef	unsigned char rend_t;
#else
typedef	unsigned int rend_t;
#endif
static rend_t rstyle = SET_RSTYLE (RS_NONE, COLORFG, COLORBG);

typedef unsigned char text_t;

/*
 * how the screen accounting works
 *
 * `rend' contains rendition information (font, bold, etc)
 *
 * `text' contains text including the scrollback buffer. Each line is a
 * fixed length (TermWin.ncol+1) the final character of each is '\n' for
 * wrapped lines and `\0' for non-wrapped lines
 *
 * the layout:
 * Rows [0 .. TermWin.hist_size - 1] == scrollback region
 * Rows [TermWin.hist_size .. TermWin.hist_size + TermWin.nrow - 1] ==
 *	screen region [0 .. TermWin.nrow-1]
 *
 * `row', `tscroll', `bscroll' are bounded by (0, TermWin.nrow)
 *
 * `col' is bounded by (0, TermWin.ncol)
 *
 * `TermWin.hist_size'
 *	is the maximum number of lines to save in the scrollback buffer.
 *	This will be a fixed number for any particular rxvt instance.
 *
 * `TermWin.hist_start'
 *	the offset back into the scrollback buffer for our current view
 *
 * `TermWin.hist_count'
 *	how many lines have been scrolled (saved)
 *		0 <= hist_count <= TermWin.hist_size
 *
 * The selection region is defined for [0 .. TermWin.nrow-1], which
 * corresponds to the regular screen and for [-1 .. -TermWin.hist_count]
 * which corresponds to the scrolled region.
 *
 * Actually, the current accounting mechanism for selection is really
 * screwed-up.  Although it is apparently permissible to select from the
 * scrollback buffer, all of the row values for the selection mechanism are
 * in the [0 .. TermWin.nrow-1] range which gives some very strange results
 * and is the likely cause of all the selection problems. -- but I think
 * these selection problems are fixed as of v2.15
 */

typedef struct {
   text_t *text;		/* all the text, including scrollback	*/
   rend_t *rend;		/* rendition, using the `RS_' flags	*/
   int row, col;		/* cursor position			*/
   int tscroll, bscroll;	/* top/bottom of scroll region		*/
   int relative;		/* relative origin mode flag		*/
   int autowrap;		/* auto-wrap flag			*/
   int wrapnext;		/* need to wrap for next char?		*/
   int insert;			/* insert mode (vs. overstrike)		*/
   int charset;			/* character set number [0..3]		*/
} screen_t;

static screen_t screen = { NULL, NULL, 0, 0, 0, 0, 0, 1, 0, 0, 0 };

#ifndef NO_SECONDARY_SCREEN	/* secondary screens */
static screen_t sec_screen = { NULL, NULL, 0, 0, 0, 0, 0, 1, 0, 0, 0 };
# define NSCREENS	1
#else
# define NSCREENS	0
#endif
static int current_screen = PRIMARY;

static int rvideo = 0;
static int focus = 0;

static char *tabs = NULL;	/* a 1 for a location with a tab-stop */
static text_t *linebuf = NULL;

#ifdef KANJI
static int multiByte = 0;
#endif

/* Data for save-screen */
static struct {
   int row, col, charset;
   char charset_char;
   rend_t rstyle;
} save = { 0, 0, 0, 'B', SET_RSTYLE (RS_NONE, COLORFG, COLORBG) };

/* This tells what's actually on the screen */
static text_t *displayed_text = NULL;
static rend_t *displayed_rend = NULL;

static char charsets [4] = {'B','B','B','B'};

static struct {
   text_t *text;		/* selected text */
   int len;			/* length of selected text */
   int op;			/* current operation */
#  define SELECTION_CLEAR	0
#  define SELECTION_BEGIN	1
#  define SELECTION_INIT	2
#  define SELECTION_ONGOING	3
#  define SELECTION_COMPLETE	4
   int screen;			/* which screen is being used */
   struct {
      int row, col;
   } beg, end, mark;
} selection = {NULL, 0, SELECTION_CLEAR, PRIMARY, {0, 0}, {0, 0}, {0, 0} };
/* also could add in these:
 * int firstr, lastr;		-- firstr <= row < lastr
 * if trying to implement X11 mouse highlighting
 */

/*----------------------------------------------------------------------*
 * extern functions referenced
 */

/*----------------------------------------------------------------------*
 * local functions
 */
static int scroll_text (int row1, int row2, int count);
static inline void selection_check (void);

/*----------------------------------------------------------------------*/
/*
 * simple check for integrity of the screen data structures
 */
#ifdef DEBUG_SCREEN
static void
check_text (char *str)
{
   int r, x;
   static char *prev = "?";

   fprintf (stderr, "%s\n", str);
   for (r = 0, x = TermWin.ncol;
	r < (TermWin.nrow + TermWin.hist_size);
	r++, x += (TermWin.ncol+1))
     {
     	if (screen.text [x] != '\0'
#ifdef NEW_WRAPTYPE
	    && screen.text [x] != '\n'
#endif
	    )
	  {
	     fprintf (stderr,"%s: %s Violation on row %d\n", str, prev, x);
	     exit (EXIT_FAILURE);
	  }
     }
   MEM_CHECK ("check_text", str);
   prev = str;
}
#else
#define check_text(a)	((void)0)
#endif

/*
 * Kanji translation units
 */
#ifdef KANJI
static void
eucj2jis (unsigned char *str, int len)
{
   int i;
   for (i = 0; i < len; i++)
     str [i] &= 0x7F;
}

static void
sjis2jis (unsigned char *str, int len)
{
   int i;
   for (i = 0; i < len; i += 2, str += 2)
     {
	unsigned char *high, *low;
	high = str;
	low = str + 1;
	(*high) -= (*high > 0x9F ? 0xB1: 0x71);
	*high = (*high) * 2 + 1;
	if (*low > 0x9E)
	  {
	     *low -= 0x7E;
	     (*high)++;
	  }
	else
	  {
	     if (*low > 0x7E)
	       (*low)--;
	     *low -= 0x1F;
	  }
     }
}
static void (*kanji_decode) (unsigned char *str, int len) = eucj2jis;

void
set_kanji_encoding (const char *str)
{
   if (str == NULL || !*str)
     return;

   if (!strcmp (str, "sjis"))		kanji_decode = sjis2jis;
   else if (!strcmp (str, "eucj"))	kanji_decode = eucj2jis;
}
#endif	/* KANJI */

/*
 * Reset the screen - called whenever the screen needs to be repaired due
 * to re-sizing or initialization
 */
void
scr_reset (void)
{
   /* #define OLD_UGLY_RESIZE */    /* the old, broken resizing code */

   static int prev_nrow = -1, prev_ncol = -1;
   int i;

   if (prev_ncol == TermWin.ncol && prev_nrow == TermWin.nrow) return;
#ifdef DEBUG_SCREEN
   fprintf (stderr, "reset\n");
#endif
   /* In case rows/columns are invalid */
   if (TermWin.ncol <= 0) TermWin.ncol = 80;
   if (TermWin.nrow <= 0) TermWin.nrow = 24;

   if (prev_nrow < 0)
     {
	TermWin.hist_count = TermWin.hist_start = 0;
     }

   for (i = 0; i <= NSCREENS; i++)
     {
	screen_t *scr;
	text_t *text;
	rend_t *rend;
	int r, oldr, oldr_max, start;

	/* allocate space for scrollback (primary screen) + screen */
#if NSCREENS
	if (i)
	  {
	     start = 0;
	     scr = &sec_screen;
	  }
	else
#endif
	  {
	     start = TermWin.hist_size;
	     scr = &screen;
	  }
	text = MALLOC (((TermWin.nrow+start)*(TermWin.ncol+1)) * sizeof(text_t),
		       "text");
	rend = MALLOC (((TermWin.nrow+start)*(TermWin.ncol+1)) * sizeof(rend_t),
		       "rend");

 	/* copy from old buffer to new buffer, as appropriate. */
	oldr = oldr_max = prev_nrow;
	if (prev_nrow > 0)
	  {
	     int n = (TermWin.nrow - prev_nrow);
#ifdef OLD_UGLY_RESIZE
	     scr->row += (n);
	     if (start)
	       TermWin.hist_count -= (n);
#else	/* OLD_UGLY_RESIZE */
	     if (n > 0)		/* window made taller */
	       {
		  oldr = (TermWin.nrow);
		  if (start)
		    {
		       if (n <= TermWin.hist_count)   /* enough */
			 {
			    scr->row += (n);
			    oldr = oldr_max;
			 }
		       else
			 {
			    scr->row += (TermWin.hist_count);
			    oldr     -= (TermWin.hist_count);
			 }
		       TermWin.hist_count -= (n);
		    }
	       }
	     else if (n < 0)	/* window made shorter */
	       {
		  if (scr->row < TermWin.nrow)
		    {
		       oldr_max = TermWin.nrow;
		    }
		  else		/* put extra into scrolled */
		    {
		       oldr_max = (scr->row+1);
		       if (start)
			 TermWin.hist_count += (oldr_max - TermWin.nrow);
		    }
		  oldr = oldr_max;
	       }
	     oldr_max += start;
#endif	/* OLD_UGLY_RESIZE */
	     oldr += start;
	     oldr--;
	  }

	if (scr->row < 0)
	  scr->row = 0;
	else if (scr->row >= TermWin.nrow)
	  scr->row = (TermWin.nrow-1);

	if (scr->col < 0)
	  scr->col = 0;
	else if (scr->col >= TermWin.ncol)
	  scr->col = (TermWin.ncol-1);

	/* reset scroll regions */
	scr->tscroll = 0; scr->bscroll = (TermWin.nrow-1);

	for (r = (TermWin.nrow+start-1); r >= 0; r--, oldr--)
	  {
	     int c, roffset, old_roffset;

	     roffset = r * (TermWin.ncol+1);
	     old_roffset = oldr * (prev_ncol+1);

	     text [roffset + TermWin.ncol] = '\0';
	     rend [roffset + TermWin.ncol] = CURRENT_RSTYLE;
	     for (c = 0; c < TermWin.ncol; c++)
	       {
		  if (c < prev_ncol && oldr >= 0
#ifndef OLD_UGLY_RESIZE
		      && oldr < oldr_max
#endif
		      )
		    {
		       text [roffset + c] = scr->text [old_roffset + c];
		       rend [roffset + c] = scr->rend [old_roffset + c];
		    }
		  else
		    {
		       /* In case there's nothing to copy */
		       text [roffset + c] = SPACE;
		       rend [roffset + c] = CURRENT_RSTYLE;
		    }
	       }
	  }
	FREE (scr->text, "Screen.text", "scr_reset");
	FREE (scr->rend, "Screen.rend", "scr_reset");
	scr->text = text;
	scr->rend = rend;
     }

   /* Make sure the cursor is on the screen */
   if (TermWin.hist_count < 0)
     TermWin.hist_count = 0;
   else if (TermWin.hist_count > TermWin.hist_size)
     TermWin.hist_count = TermWin.hist_size;

   prev_ncol = TermWin.ncol;
   prev_nrow = TermWin.nrow;

   FREE (displayed_text, "displayed_text", "scr_reset");
   FREE (displayed_rend, "displayed_rend", "scr_reset");
   displayed_text = MALLOC ((TermWin.nrow * (TermWin.ncol+1)) * sizeof(text_t),
			    "displayed_text");
   displayed_rend = MALLOC ((TermWin.nrow * (TermWin.ncol+1)) * sizeof(rend_t),
			    "displayed_rend");

   memset (displayed_text, SPACE,
	   (TermWin.nrow * (TermWin.ncol+1)) * sizeof(text_t));
   memset (displayed_rend, RS_NONE,
	   (TermWin.nrow * (TermWin.ncol+1)) * sizeof(rend_t));

   /* ensure the cursor is on the screen */
   if (save.row >= TermWin.nrow) save.row = (TermWin.nrow-1);
   if (save.col >= TermWin.ncol) save.col = (TermWin.ncol-1);

   FREE (tabs, "tabs", "scr_reset");
   tabs = MALLOC (TermWin.ncol * sizeof(char),"tabs");

   for (i = 0; i < TermWin.ncol; i++)
     tabs [i] = !(i % TAB_SIZE);

   FREE (linebuf, "linebuf", "scr_reset");
   linebuf = MALLOC ((TermWin.ncol+1) * sizeof(text_t), "linebuf");
   tty_resize ();
}

/*
 * Restore power-on configuration
 * Clears screen, restores default fonts, etc
 */
void
scr_poweron (void)
{
   screen_t *scr;
   int i, r;

   memset (charsets, 'B', sizeof(charsets));
   rstyle = SET_RSTYLE (RS_NONE, COLORFG, COLORBG);
   rvideo = 0;

#if NSCREENS
   scr_change_screen (SECONDARY);
   scr_erase_screen (DEL_ENTIRE);
#endif
   scr_change_screen (PRIMARY);
   scr_erase_screen (DEL_ENTIRE);

   for (i = 0, scr = &screen; i <= NSCREENS; i++)
     {
	scr->tscroll = 0;
	scr->bscroll = (TermWin.nrow-1);
	scr->row = scr->col = 0;
	scr->relative = 0;
	scr->autowrap = 1;
	scr->insert = 0;
	scr->charset = 0;
#if NSCREENS
	scr = &sec_screen;
#endif
     }

   scr_cursor (SAVE);

   memset (displayed_text, SPACE,
	   (TermWin.nrow * (TermWin.ncol+1)) * sizeof(text_t));
   memset (displayed_rend, RS_NONE,
	   (TermWin.nrow * (TermWin.ncol+1)) * sizeof(rend_t));

   for (r = 0, i = TermWin.ncol; r < TermWin.nrow; r++, i += (TermWin.ncol+1))
     displayed_text [i] = '\0';

   TermWin.hist_start = 0;
   scr_reset ();
   XClearWindow (Xdisplay, TermWin.vt);
   scr_refresh (SLOW_REFRESH);
   Gr_reset ();
}

/*
 * Set the rstyle parameter to reflect the selected font
 */
static inline void
set_font_style (void)
{
   check_text ("font style");

   rstyle &= ~RS_FONTMASK;
   switch (charsets [screen.charset]) {
    case '0': rstyle |= RS_GRFONT;	break;
    case 'A': rstyle |= RS_GBFONT;	break;
   }
}

/*
 * Save/restore the cursor position and rendition style
 */
void
scr_cursor (int mode)
{
   switch (mode) {
    case SAVE:
      save.row = screen.row;
      save.col = screen.col;
      save.rstyle = (rstyle);
      save.charset = screen.charset;
      save.charset_char = charsets [screen.charset];
      break;
    case RESTORE:
      screen.row = save.row;
      screen.col = save.col;
      rstyle = save.rstyle;
      screen.charset = save.charset;
      charsets [screen.charset] = save.charset_char;
      set_font_style ();
      break;
   }
}

/*
 * Handle a backspace
 */
void
scr_backspace (void)
{
   check_text ("backspace");

   if (selection.op) selection_check ();
   if (screen.col == 0 && screen.row > 0)
     {
	screen.row--;
	screen.col = (TermWin.ncol-1);
     }
   else if (screen.wrapnext)
     {
#ifdef NEW_WRAPTYPE
	/* FIXME: anything special needed here? */
	screen.text [screen.row * (TermWin.ncol+1) + screen.col] = '\0';
#endif
	screen.wrapnext = 0;
     }
   else
     scr_gotorc (0, -1, RELATIVE);
}

/*
 * Change between the main and alternate screen
 */
int
scr_change_screen (int scrn)
{
#if NSCREENS
   int x, start;

   check_text ("change screen");

   TermWin.hist_start = 0;
   if (current_screen == scrn)
     return current_screen;

   /* swap screens, but leave scrollback untouched */
   start = (TermWin.hist_size) * (TermWin.ncol+1);
   for (x = 0; x < (TermWin.nrow) * (TermWin.ncol+1); x++)
     {
	text_t t;
	rend_t r;

	t = screen.text [x+start];
	screen.text [x+start] = sec_screen.text [x];
	sec_screen.text [x] = t;

	r = screen.rend [x+start];
	screen.rend [x+start] = sec_screen.rend [x];
	sec_screen.rend [x] = r;
     }

   x = screen.row;
   screen.row = sec_screen.row;
   sec_screen.row = x;

   x = screen.col;
   screen.col = sec_screen.col;
   sec_screen.col = x;

   x = screen.relative;
   screen.relative = sec_screen.relative;
   sec_screen.relative = x;

   x = screen.autowrap;
   screen.autowrap = sec_screen.autowrap;
   sec_screen.autowrap = x;

   x = screen.wrapnext;
   screen.wrapnext = sec_screen.wrapnext;
   sec_screen.wrapnext = x;

   x = screen.insert;
   screen.insert = sec_screen.insert;
   sec_screen.insert = x;

   x = screen.charset;
   screen.charset = sec_screen.charset;
   sec_screen.charset = x;

   if (graphics_up)
     {
	Gr_scroll (0);
	Gr_ChangeScreen ();
     }
   x = current_screen;
   current_screen = scrn;
   return x;
#else
   TermWin.hist_start = 0;
   return current_screen;
#endif
}

/*
 * Change the rendition style
 */
void
scr_rendition (int set, int style)
{
   int color;

   if (set)			/* set the value in question */
     {
	rstyle |= style;
	switch (style) {
#ifndef NO_COLOR
#ifndef USE_FAKE_BOLD
	 case RS_BOLD:
	   color = GET_FG_COLOR (rstyle);
	   if (color >= COLOR0 && color <= COLOR7)
	     scr_fgcolor (color - COLOR0);
	   break;
	 case RS_BLINK:
	   color = GET_BG_COLOR (rstyle);
	   if (color >= COLOR0 && color <= COLOR7)
	     scr_bgcolor (color - COLOR0);
	   break;
#endif
#endif

	 case RS_RVID:
	   if (rvideo)		/* faked reverse video mode */
	     rstyle &= ~RS_RVID;
	   break;
	}
     }
   else				/* unset the value in question */
     {
	rstyle &= ~style;
	switch (style) {
	 case ~RS_NONE:		/* default fg/bg colors */
	   rstyle = SET_RSTYLE (RS_NONE, COLORFG, COLORBG);
	   if (rvideo)		/* faked reverse video mode */
	     rstyle |= RS_RVID;
	   break;

#ifndef NO_COLOR
#ifndef USE_FAKE_BOLD
	 case RS_BOLD:
	   color = GET_FG_COLOR (rstyle);
	   if (color >= BOLD0 && color <= BOLD7)
	     scr_fgcolor (color - BOLD0);
	   break;
	 case RS_BLINK:
	   color = GET_BG_COLOR (rstyle);
	   if (color >= BOLD0 && color <= BOLD7)
	     scr_bgcolor (color - BOLD0);
	   break;
#endif
#endif
	 case RS_RVID:
	   if (rvideo)		/* faked reverse video mode */
	     rstyle |= RS_RVID;
	   break;
	}
     }
   set_font_style ();
}

/*
 * Indicate a change of keyboard focus
 */
void
scr_focus (int in_focus)
{
   focus = in_focus;
}

void
scr_add_lines (text_t *str, int nl_count, int n)
{
   int i, roffset;		/* row offset */
#ifdef KANJI
   enum {SBYTE, WBYTE1, WBYTE2} chstat = SBYTE;
#endif

   check_text ("add lines");
   if (selection.op) selection_check ();

   if (nl_count > 0)
     {
	nl_count += (screen.row - screen.bscroll);

	if ((nl_count > 0) &&
	    (screen.tscroll == 0) && (screen.bscroll == (TermWin.nrow-1)))
	  {
	     scroll_text (screen.tscroll, screen.bscroll, nl_count);
	     screen.row -= nl_count;
	     if (screen.row < -TermWin.hist_size)
	       screen.row = -TermWin.hist_size;
	  }
     }

   if (screen.col >= TermWin.ncol)
     screen.col = (TermWin.ncol-1);

   roffset = (screen.row + TermWin.hist_size) * (TermWin.ncol+1);
   for (i = 0; i < n; i++)
     {
	/*
	 * Adds a single character at the current cursor location
	 * new lines in the string
	 */
	TermWin.hist_start = 0;
	if (graphics_up)
	  Gr_scroll (0);

#ifdef KANJI
	switch (chstat)
#endif
	  {
#ifdef KANJI
	   case WBYTE1:	break;		/* never happens? */

	   case WBYTE2:
	     rstyle |= RL_KANJI2;
	     chstat = SBYTE;
	     break;

	   case SBYTE:
	     if (multiByte || (str[i] & 0x80))
	       {
		  rstyle = (rstyle & ~RL_KANJI2) | RL_KANJI1;
		  chstat = WBYTE2;
	       }
	     else
#endif
	       switch (str [i]) {
		case 127:			continue; break;
		case '\t':	scr_tab (1);	continue; break;
		case '\n':
		  screen.text [roffset + (TermWin.ncol)] = '\0';
		  if (screen.row == screen.bscroll)
		    scroll_text (screen.tscroll, screen.bscroll, 1);
		  else if (screen.row < (TermWin.nrow-1))
		    screen.row++;
		  roffset = ((screen.row + TermWin.hist_size) *
			     (TermWin.ncol+1));
		  screen.wrapnext = 0;
		  continue; break;

		case '\r':
		  screen.text [roffset + (TermWin.ncol)] = '\0';
		  screen.col = 0;
		  screen.wrapnext = 0;
		  continue; break;

		default:
#ifdef KANJI
		  rstyle &= ~RL_KANJI2;
#endif
		  break;
	       }
	  }

	if (screen.wrapnext)
	  {
	     if (screen.row == screen.bscroll)
	       scroll_text (screen.tscroll, screen.bscroll, 1);
	     else if (screen.row < (TermWin.nrow-1))
	       screen.row++;
	     roffset = ((screen.row + TermWin.hist_size) *
			(TermWin.ncol+1));
	     screen.col = 0;
	     screen.wrapnext = 0;
	     screen.text [roffset + (TermWin.ncol)] = '\0';
	  }
	if (screen.insert)
	  scr_insdel_chars (1, INSERT);
	screen.text [roffset + (screen.col)] = str [i];
	screen.rend [roffset + (screen.col)] = rstyle;
	screen.col++;
	if (screen.col == TermWin.ncol)
	  {
	     screen.col--;
	     screen.wrapnext = screen.autowrap;
#ifdef NEW_WRAPTYPE
	     /* for next time */
	     screen.text [roffset + (TermWin.ncol)] = (screen.autowrap ?
						       '\n' : '\0');
#endif
	  }
     }
}

/*
 * Scroll the text on the screen
 * Scroll COUNT lines from ROW1 to ROW2 inclusive (ROW1 <= ROW2)
 * scrolling is up for a +ve COUNT and down for a -ve COUNT
 */
static int
scroll_text (int row1, int row2, int count)
{
   int r;
   text_t *t_dst, *t_src;
   rend_t *r_dst, *r_src;

#ifdef NEW_SELECTION
   if (selection.op)		/* move selected region too */
#endif
     {
#ifdef NEW_SELECTION
	selection.beg.row -= count;
#endif
	selection.end.row -= count;
	selection.mark.row -= count;
	/* could check ranges here and make sure selection is okay
	 * don't scroll into scrollback depending on the region etc,
	 * but leave for now
	 */
     }

   if (count > 0)		/* scroll up */
     {
	int n, x;
	/* if the line scrolls off the top of the screen,
	 * shift the entire scrollback buffer too */
	if ((row1 == 0) && (current_screen == PRIMARY))
	  {
	     row1 = -TermWin.hist_size;
	     TermWin.hist_count += count;
	     if (TermWin.hist_count > TermWin.hist_size)
	       TermWin.hist_count = TermWin.hist_size;
	  }

	x = ((row1 + TermWin.hist_size) * (TermWin.ncol+1));
	t_dst = &screen.text [x];
	r_dst = &screen.rend [x];

	n = (row2 - row1 + 1);
	if (count > n)
	  {
	     count = n;
	     n = 0;
	  }
	else
	  {
	     n -= count;
	  }

	x += count * (TermWin.ncol+1);
	t_src = &screen.text [x];
	r_src = &screen.rend [x];

	/* Forward overlapping memcpy's -- probably OK */
	if (n > 0)
	  {
	     n *= (TermWin.ncol+1);
	     MEMCOPY (t_dst, t_src, n * sizeof(text_t));
	     t_dst += n;
	     MEMCOPY (r_dst, r_src, n * sizeof(rend_t));
	     r_dst += n;
	  }

	/* copy blank lines in at the bottom */
	memset (t_dst, SPACE,
		(count * (TermWin.ncol+1)) * sizeof(text_t));
	REND_SET (r_dst, (count * (TermWin.ncol+1)));
	t_dst--;		/* terminate each line */
	for (r = 0; r < (count+1); r++)
	  {
	     *t_dst = 0;
	     t_dst += (TermWin.ncol+1);
	  }
     }
   else if (count < 0)		/* scroll down */
     {
	int x;

	/* do one line at a time to avoid backward overlapping memcpy's */
	x = (row2 + TermWin.hist_size) * (TermWin.ncol+1);
	t_dst = &screen.text [x];
	r_dst = &screen.rend [x];

	x += (count) * (TermWin.ncol+1);
	t_src = &screen.text [x];
	r_src = &screen.rend [x];
	for (r = row2; r >= (row1 - count); r--)
	  {
	     MEMCOPY (t_dst, t_src, (TermWin.ncol+1) * sizeof(text_t));
	     t_dst -= (TermWin.ncol+1);
	     t_src -= (TermWin.ncol+1);

	     MEMCOPY (r_dst, r_src, (TermWin.ncol+1) * sizeof(rend_t));
	     r_dst -= (TermWin.ncol+1);
	     r_src -= (TermWin.ncol+1);
	  }
	/* copy blank lines in at the top */
	for (; r >= row1; r--)
	  {
	     memset (t_dst, SPACE, TermWin.ncol * sizeof(text_t));
	     t_dst [TermWin.ncol] = '\0';
	     REND_SET (r_dst, (TermWin.ncol));
	     t_dst -= (TermWin.ncol+1);
	     r_dst -= (TermWin.ncol+1);
	  }
     }
   if (graphics_up)
     Gr_scroll (count);
   return count;
}

/*
 * Move the cursor to a new tab position
 *
 * COUNT is +ve, move forward.  COUNT is -ve, move backward
 */
void
scr_tab (int count)
{
   int x = screen.col;

   if (count > 0) 		/* tab forward */
     {
	int i;
	for (i = x+1; i < TermWin.ncol; i++)
	  {
	     if (tabs [i])
	       {
		  x = i;
		  count--;
		  if (!count) break;
	       }
	  }
     }
   else if (count < 0)		/* tab backward */
     {
	int i;
	count = -count;
	for (i = x-1; i >= 0; i--)
	  {
	     if (tabs [i])
	       {
		  x = i;
		  count--;
		  if (!count) break;
	       }
	  }
     }
   else
     return;

   if (x != screen.col)
     scr_gotorc (0, x, R_RELATIVE);
}

/*
 * Move the cursor to a new position.
 * The relative argument is a pair of flags that specify relative
 * rather than absolute motion.
 */
void
scr_gotorc (int row, int col, int relative)
{
   TermWin.hist_start = 0;
   if (graphics_up)
     Gr_scroll (0);
   screen.col = (relative & C_RELATIVE ? (screen.col + col) : col);

   if (screen.col < 0)
     screen.col = 0;
   else if (screen.col >= TermWin.ncol)
     screen.col = (TermWin.ncol-1);

   if (relative & R_RELATIVE)
     {
	if (row > 0)
	  {
	     if ((screen.row <= screen.bscroll) &&
		 (screen.row + row > screen.bscroll))
	       screen.row = screen.bscroll;
	     else
	       screen.row += row;
	  }
	else if (row < 0)
	  {
	     if ((screen.row >= screen.tscroll) &&
		 (screen.row + row < screen.tscroll))
	       screen.row = screen.tscroll;
	     else
	       screen.row += row;
	  }
     }
   else
     {
	if (screen.relative)		/* relative origin mode */
	  {
	     screen.row = row + screen.tscroll;
	     if (screen.row > screen.bscroll)
	       screen.row = screen.bscroll;
	  }
	else
	  screen.row = row;
     }
   if (screen.row < 0)
     screen.row = 0;
   else if (screen.row >= TermWin.nrow)
     screen.row = (TermWin.nrow-1);
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
   /* FIXME: anything special needed here? */
#endif
}

/*
 * Move the cursor down one line and scroll if necessary
 */
void
scr_index (int dirn)
{
   TermWin.hist_start = 0;
   if (graphics_up)
     Gr_scroll (0);

   check_text ("index");

   if ((screen.row == screen.bscroll && dirn == UP) ||
       (screen.row == screen.tscroll && dirn == DOWN))
     scroll_text (screen.tscroll, screen.bscroll, dirn);
   else
     screen.row += dirn;
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
   /* FIXME: anything special needed here? */
#endif
   if (selection.op) selection_check ();
}

/*
 * erase part or the whole of a line
 */
void
scr_erase_line (int mode)
{
   int x, count;
   check_text ("erase line");

   TermWin.hist_start = 0;
   if (graphics_up)
     Gr_scroll (0);

   x = (screen.row + TermWin.hist_size) * (TermWin.ncol+1);
   switch (mode) {
    case DEL_END:		/* 0: erase to end */
      check_text ("erase line End");
      x += (screen.col);
      count = (TermWin.ncol - screen.col);
      break;

    case DEL_START:		/* 1: erase to beginning */
      check_text ("erase line Start");
      count = (screen.col + 1);
      break;

    case DEL_ENTIRE:		/* 2: erase entire */
      check_text ("erase line Entire");
      count = (TermWin.ncol);
      break;

    default:
      check_text ("erase line None");
      return;
   }

   memset (&screen.text [x], SPACE, count * sizeof(text_t));
   REND_SET (&screen.rend [x], count);
   x = (screen.row + TermWin.hist_size) * (TermWin.ncol+1) + (TermWin.ncol);
   screen.text [x] = '\0';

   check_text ("erase line Done");
   if (selection.op) selection_check ();
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
   /* FIXME: anything special needed here? */
#endif
}

/*
 * erase part or the whole of the screen
 */
void
scr_erase_screen (int mode)
{
   int r, startr, startc, endr, endc;

   check_text ("erase screen");

   switch (mode) {
    case DEL_END:		/* 0: erase to end */
      startr = screen.row; endr = (TermWin.nrow-1);
      startc = screen.col; endc = (TermWin.ncol-1);
      break;
    case DEL_START:		/* 1: erase to beginning */
      startr = 0; endr = screen.row;
      startc = 0; endc = screen.col;
      break;
    case DEL_ENTIRE:		/* 2: erase entire */
      startr = 0; endr = (TermWin.nrow-1);
      startc = 0; endc = (TermWin.ncol-1);
      Gr_ClearScreen ();
      break;
    default:
      return;
      break;
   }
   for (r = startr; r <= endr; r++)
     {
	int x, count;
	x = ((r + TermWin.hist_size - TermWin.hist_start) * (TermWin.ncol+1));

	count = (r == endr ? (endc + 1) : (TermWin.ncol));

	if (r == startr)
	  {
	     x += startc;
	     count -= startc;
	  }

	memset (&screen.text [x], SPACE, count * sizeof(text_t));
	REND_SET (&screen.rend [x], count);
     }
}

/*
 * Fill screen with E's
 */
void
scr_E (void)
{
   int r, x;

   check_text ("E");

   x = ((TermWin.hist_size - TermWin.hist_start) * (TermWin.ncol+1));
   for (r = 0; r < (TermWin.nrow); r++)
     {
	memset (&screen.text [x], 'E',
		TermWin.ncol * sizeof(text_t));
	memset (&screen.rend [x], RS_NONE,
		TermWin.ncol * sizeof(rend_t));

	x += (TermWin.ncol + 1);
	screen.text [x-1] = '\0';
     }
}

/*
 * Insert or delete count lines and scroll
 * insdel == +1
 *	delete lines, scroll up the bottom of the screen to fill the gap
 * insdel == -1
 *	insert lines, scroll down the lower lines
 * other values of insdel are undefined
 */
void
scr_insdel_lines (int count, int insdel)
{
   check_text ("insdel lines");

   if (screen.row > screen.bscroll)
     return;

   if (count > screen.bscroll - screen.row + 1)
     {
	if (insdel == DELETE)
	  return;
	else if (insdel == INSERT)
	  count = screen.bscroll - screen.row + 1;
     }

   TermWin.hist_start = 0;
   if (graphics_up)
     Gr_scroll (0);
   scroll_text (screen.row, screen.bscroll, insdel * count);
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
   /* FIXME: anything special needed here? */
#endif
}

/*
 * insert or Delete count characters from the current position
 * insdel == +2, erase  chars
 * insdel == +1, delete chars
 * insdel == -1, insert chars
 */
void
scr_insdel_chars (int count, int insdel)
{
   int roffset;			/* row offset */
   text_t *text, *textend;
   rend_t *rend, *rendend;

   check_text ("insdel chars");

   if (insdel == ERASE)
     {
	if (count > screen.col)
	  count = screen.col;
	if (count <= 0)
	  return;
	screen.col -= count;	/* move backwards */
	insdel = DELETE;	/* delete chars */
     }
   else if (count > (TermWin.ncol - screen.col))
     {
	count = (TermWin.ncol - screen.col);
     }
   if (count <= 0)
     return;

   TermWin.hist_start = 0;
   if (graphics_up)
     Gr_scroll (0);
   if (selection.op) selection_check ();

   roffset = (screen.row + TermWin.hist_size) * (TermWin.ncol+1);

   text = &screen.text [roffset + (screen.col)];
   rend = &screen.rend [roffset + (screen.col)];
   if (insdel == DELETE)
     {
	screen.text [roffset + (TermWin.ncol)] = '\0';	/* not autowrapped */
	/* overlapping copy */
	for (/*nil*/; (*text && text [count]); text++, rend++)
	  {
	     *text = text [count];
	     *rend = rend [count];
	  }

	/* fill in the end of the line */
	for (/*nil*/; *text; text++, rend++)
	  {
	     *text = SPACE;
	     *rend = CURRENT_RSTYLE;
	  }
     }
   else
     {
	/* INSERT count characters */
	textend = &screen.text [roffset + (TermWin.ncol-1)];
	rendend = &screen.rend [roffset + (TermWin.ncol-1)];

	textend [1] = '\0';	/* not autowrapped */
	for (/*nil*/; (textend - count >= text); textend--, rendend--)
	  {
	     *textend = *(textend-count);
	     *rendend = *(rendend-count);
	  }

	/* fill in the gap */
	for (/*nil*/; (textend >= text); textend--, rendend--)
	  {
	     *textend = SPACE;
	     *rendend = CURRENT_RSTYLE;
	  }
     }
   screen.wrapnext = 0;
}

/*
 * set the scroll region
 */
void
scr_scroll_region (int top, int bot)
{
   if (top < 0) top = 0;
   if (bot >= TermWin.nrow) bot = (TermWin.nrow-1);
   if (top > bot) return;

   screen.tscroll = top;
   screen.bscroll = bot;
   scr_gotorc (0, 0, 0);
}

/*
 * set/unset automatic wrapping
 */
void
scr_autowrap (int mode)
{
   screen.autowrap = mode;
}

/*
 * set/unset margin origin mode
 *
 * In absolute origin mode, line numbers are counted relative to top margin
 * of screen, the cursor can be moved outside the scrolling region. In
 * relative mode line numbers are relative to top margin of scrolling
 * region and the cursor cannot be moved outside
 */
void
scr_relative_origin (int mode)
{
   screen.relative = mode;
   scr_gotorc (0, 0, 0);
}

/*
 * set/unset automatic insert mode
 */
void
scr_insert_mode (int mode)
{
   screen.insert = mode;
}

/*
 * Move the display so that line represented by scrollbar value y is at
 * the top of the screen
 */
#ifndef SCROLLBAR_NONE
void
scr_move_to (int y)
{
   check_text ("move to");

   TermWin.hist_start = (((sbar.h-1) - y) * ((TermWin.nrow-1) + TermWin.hist_count)
		     / (sbar.h-1)) - (TermWin.nrow-1);

   if (TermWin.hist_start < 0)
     TermWin.hist_start = 0;
   else if (TermWin.hist_start > TermWin.hist_count)
     TermWin.hist_start = TermWin.hist_count;
   if (graphics_up)
     Gr_scroll (0);
}
#endif

/*
 * page the screen up/down NLINES
 */
void
scr_page (int dirn, int nlines)
{
   check_text ("page");

   if (nlines <= 0)
     nlines = 1;
   else if (nlines > TermWin.nrow)
     nlines = TermWin.nrow;
   TermWin.hist_start += nlines * dirn;

   if (TermWin.hist_start < 0)
     TermWin.hist_start = 0;
   else if (TermWin.hist_start > TermWin.hist_count)
     TermWin.hist_start = TermWin.hist_count;
   if (graphics_up)
     Gr_scroll (0);
}

/*
 * If (row,col) is within a selected region of text, remove the selection
 */
static inline void
selection_check (void)
{
   int c1, c2, r1, r2;
   check_text ("check selection");

   if (current_screen != selection.screen)
     return;

#ifdef NEW_SELECTION
   if ((selection.mark.row < -TermWin.hist_count) ||
       (selection.mark.row >= TermWin.nrow) ||
       (selection.beg.row < -TermWin.hist_count) ||
       (selection.beg.row >= TermWin.nrow) ||
       (selection.end.row < -TermWin.hist_count) ||
       (selection.end.row >= TermWin.nrow))
     {
	scr_selection_clear ();
	return;
     }

   r1 = (screen.row - TermWin.hist_start);
   c1 = ((r1 - selection.mark.row) * (r1 - selection.end.row));
#else
   c1 = ((screen.row - selection.mark.row) *
	 (screen.row - selection.end.row));
#endif
   /* selection.mark.row > screen.row - TermWin.hist_start
    * or
    * selection.end.row > screen.row - TermWin.hist_start
    */
   if (c1 < 0)
     scr_selection_clear ();
   /* selection.mark.row == screen.row || selection.end.row == screen.row */
   else if (c1 == 0)
     {
	/* We're on the same row as the start or end of selection */
	if ((selection.mark.row < selection.end.row) ||
	    ((selection.mark.row == selection.end.row) &&
	     (selection.mark.col < selection.end.col)))
	  {
	     r1 = selection.mark.row;
	     c1 = selection.mark.col;
	     r2 = selection.end.row;
	     c2 = selection.end.col;
	  }
	else
	  {
	     r1 = selection.end.row;
	     c1 = selection.end.col;
	     r2 = selection.mark.row;
	     c2 = selection.mark.col;
	  }
	if ((screen.row == r1) && (screen.row == r2))
	  {
	     if ((screen.col >= c1) && (screen.col <= c2))
	       scr_selection_clear ();
	  }
	else if (((screen.row == r1) && (screen.col >= c1)) ||
		 ((screen.row == r2) && (screen.col <= c2)))
	  scr_selection_clear ();
     }
}

#if 0
void
scr_selection_range (int firstr, int lastr)
{
   if (firstr >= lastr ||
       firstr < 0 || firstr >= TermWin.nrow ||
       lastr <= 0 || lastr > TermWin.nrow)
    return;
   selection.firstr = firstr;
   selection.lastr = lastr;
}
#endif

/*
 * make the selection currently delimited by the selection end markers
 */
void
scr_selection_make (Time time)
{
   text_t *str;
   int r, startr, startc, endr, endc;
   int roffset;			/* row offset */

   switch (selection.op) {
    case SELECTION_ONGOING:
      break;

    case SELECTION_INIT:
      scr_selection_clear ();
      selection.end.row = selection.mark.row = selection.beg.row;
      selection.end.col = selection.mark.col = selection.beg.col;
      /*drop*/
    case SELECTION_BEGIN:
      selection.op = SELECTION_COMPLETE;
    default:
      return;
      break;
   }
   selection.op = SELECTION_COMPLETE;

   FREE (selection.text, "selection.text", "selection make");
   selection.text = NULL;
   selection.len = 0;

   selection.screen = current_screen;
   /* Set start/end row/col to point to the selection endpoints */
   if (selection.end.row < selection.mark.row ||
       (selection.end.row == selection.mark.row &&
	selection.end.col <= selection.mark.col))
     {
	startr = selection.end.row; endr = selection.mark.row;
	startc = selection.end.col; endc = selection.mark.col;
     }
   else
     {
	startr = selection.mark.row; endr = selection.end.row;
	startc = selection.mark.col; endc = selection.end.col;
     }

   if (
#ifdef NEW_SELECTION
       (startr < -TermWin.hist_count || endr >= TermWin.nrow)
#else
       (startr < (TermWin.hist_start - TermWin.hist_size) ||
	endr >= (TermWin.hist_start + TermWin.nrow))
#endif
       )
     {
	scr_selection_clear ();
	return;
     }

   str = MALLOC (((endr - startr + 1)*(TermWin.ncol+1) + 1) * sizeof(text_t),
		 "sel_text");
   selection.text = str;
   *str = '\0';

   /* save all points between start and end with selection flag */
#ifdef NEW_SELECTION
   roffset = ((startr + TermWin.hist_size) * (TermWin.ncol+1));
#else
   roffset = ((startr + TermWin.hist_size - TermWin.hist_start) *
	      (TermWin.ncol+1));
#endif
   for (r = startr; r <= endr; r++, roffset += (TermWin.ncol+1))
     {
	int c1, c2;
	c1 = (r == startr ? startc : 0);
	c2 = (r == endr ? endc : (TermWin.ncol-1));

#ifdef KANJI
	if ((screen.rend [roffset+c1] & RL_KANJI2) == RL_KANJI2) c1--;
	if ((screen.rend [roffset+c2] & RL_KANJI2) == RL_KANJI1) c2++;
#endif
	for (/*nil*/; c1 <= c2; c1++)
	  *str++ = screen.text [roffset + c1];

	if (c2 == (TermWin.ncol-1))
	  {
#ifdef NEW_WRAPTYPE
	     /* not autowrap */
	     if (screen.text [roffset + (TermWin.ncol)] != '\n')
#endif
	       {
#ifdef WRAP_LASTCHAR_HACK
		  int last_is_wspace = isspace (*(--str));
#else
		  str--;
#endif
		  while ((str >= selection.text) && isspace (*str)) str--;
		  str++;
#ifdef WRAP_LASTCHAR_HACK
		  if (last_is_wspace)
#endif
		    *str++ = '\n';
	       }
	  }
     }
   *str = '\0';

   selection.len = strlen (selection.text);
   if (selection.len <= 0)
     return;
   XSetSelectionOwner (Xdisplay, XA_PRIMARY, TermWin.vt, time);
   if (XGetSelectionOwner (Xdisplay, XA_PRIMARY) != TermWin.vt)
     print_error ("can't get primary selection");

   /* Place in CUT_BUFFER0 for backup */
   XChangeProperty (Xdisplay, DefaultRootWindow (Xdisplay), XA_CUT_BUFFER0,
		    XA_STRING, 8, PropModeReplace,
		    selection.text, selection.len);
}

/*
 * respond to a request for our current selection
 */
void
scr_selection_send (XSelectionRequestEvent *rq)
{
   XEvent event;

   event.xselection.type	= SelectionNotify;
   event.xselection.display	= rq->display;
   event.xselection.requestor	= rq->requestor;
   event.xselection.selection	= rq->selection;
   event.xselection.target	= rq->target;
   event.xselection.property	= None;
   event.xselection.time	= rq->time;

   if (rq->target == XA_STRING)
     {
	XChangeProperty (Xdisplay, rq->requestor, rq->property,
			 XA_STRING, 8, PropModeReplace,
			 selection.text, selection.len);
	event.xselection.property = rq->property;
     }
   XSendEvent (Xdisplay, rq->requestor, False, 0, &event);
}

/*
 * Request the current primary selection
 */
void
scr_selection_request (Time time, int x, int y)
{
   Atom prop;

   /* First check that the release is within the window */
   if (x < 0 || y < 0 || x >= TermWin.width || y >= TermWin.height)
     return;
   if (selection.text != NULL)
     {
	/* the selection is internal */
#ifdef SELECTION_REPLACE_NEWLINE
	text_t *p = selection.text;
	text_t *pmax = p + selection.len;
	for (/*nil*/; p < pmax; p++)
	  if (*p == '\n') *p = '\r';
#endif
	tty_write (selection.text, selection.len);
	return;
     }

   if (XGetSelectionOwner (Xdisplay, XA_PRIMARY) == None)
     {
	/* No primary selection so use the cut buffer */
	scr_paste_primary (DefaultRootWindow (Xdisplay),
			   XA_CUT_BUFFER0, False);
	return;
     }
   prop = XInternAtom (Xdisplay, "VT_SELECTION", False);
   XConvertSelection (Xdisplay, XA_PRIMARY, XA_STRING,
		      prop, TermWin.vt, time);
}

/*
 * Respond to a notification that a primary selection has been sent
 */
void
scr_paste_primary (Window win, int prop, int Delete)
{
   Atom actual_type;
   long bytes_after, nread;
   unsigned char * data;

   if (prop == None)
     return;

   nread = 0;
   do
     {
	int i;
	long nitems;

	if ((XGetWindowProperty (Xdisplay, win, prop,
				 nread/4, PROP_SIZE, Delete,
				 AnyPropertyType, &actual_type, &i,
				 &nitems, &bytes_after,
				 &data) != Success) ||
	    (actual_type != XA_STRING))
	  {
	     XFree (data);
	     return;
	  }

	nread += nitems;
#ifdef SELECTION_REPLACE_NEWLINE
	/* make a \n to \r mapping for cut and paste only */
	for (i = 0; i < nitems; i++)
	  if (data [i] == '\n') data [i] = '\r';
#endif
	tty_write (data, nitems);
	XFree (data);
     } while (bytes_after > 0);
}

/*
 * Clear the current selection
 */
void
scr_selection_clear (void)
{
   int x, nrow;

   selection.op = SELECTION_CLEAR;
   selection.end.row = selection.mark.row = 0;
   selection.end.col = selection.mark.col = 0;
   nrow = TermWin.nrow;
   if (current_screen == PRIMARY)
     nrow += TermWin.hist_size;

   for (x = 0; x < nrow * (TermWin.ncol+1); x++)
     screen.rend [x] &= ~RS_SELECTED;
}

void
scr_selection_delete (void)
{
   FREE (selection.text, "sel_text", "clear_sel");
   selection.text = NULL;
   selection.len = 0;

   selection.op = SELECTION_CLEAR;
   scr_selection_clear ();
}

/*
 * mark selected points (used by scr_selection_extend)
 */
static void
selection_setclr (int set, int startr, int startc, int endr, int endc)
{
   int r, roffset;

#ifdef NEW_SELECTION
   /* startr <= endr */
   if ((startr < -TermWin.hist_count) || (endr >= TermWin.nrow))
     {
	scr_selection_clear ();
	return;
     }
#endif

#ifdef NEW_SELECTION
   roffset = ((startr + TermWin.hist_size) * (TermWin.ncol+1));
#else
   roffset = ((startr + TermWin.hist_size - TermWin.hist_start) *
	      (TermWin.ncol+1));
#endif

   for (r = startr; r <= endr; r++)
     {
	int c1, c2;
	c1 = (r == startr ? startc : 0);
	c2 = (r == endr ? endc : (TermWin.ncol-1));

#ifdef KANJI
	if ((screen.rend [roffset+c1] & RL_KANJI2) == RL_KANJI2) c1--;
	if ((screen.rend [roffset+c2] & RL_KANJI2) == RL_KANJI1) c2++;
#endif
	for (/*nil*/; c1 <= c2; c1++)
	  {
	     if (set)
	       screen.rend [roffset + c1] |= RS_SELECTED;
	     else
	       screen.rend [roffset + c1] &= ~RS_SELECTED;
	  }
	roffset += (TermWin.ncol+1);
     }
}

/*
 * start a selection using the specified unit
 */
void
scr_selection_start (int x, int y)
{
#ifdef NEW_SELECTION
   if (selection.op)
     {
	/* startr <= endr */
	if ((selection.end.row < -TermWin.hist_count) ||
	    (selection.mark.row < - TermWin.hist_count))
	  {
	     scr_selection_clear ();
	  }
	else			/* direction of new selection */
	  if (selection.end.row < selection.mark.row ||
	      (selection.end.row == selection.mark.row &&
	       selection.end.col <= selection.mark.col))
	    selection_setclr (0,    /* up */
			      selection.end.row, selection.end.col,
			      selection.mark.row, selection.mark.col);
	else
	  selection_setclr (0,	/* down */
			    selection.mark.row, selection.mark.col,
			    selection.end.row, selection.end.col);
     }
#endif
   selection.op = SELECTION_INIT;

   selection.beg.col = Pixel2Col (x);
   selection.beg.row = Pixel2Row (y);

#ifdef NEW_SELECTION
   selection.beg.row -= TermWin.hist_start;
#endif
}

/*
 * extend the selection
 */
void
scr_selection_extend (int hilite, int x, int y)
{
   int old_row, old_col, old_dirn, dirn;

   switch (selection.op) {
    case SELECTION_INIT:
      scr_selection_clear ();
      selection.end.col = selection.mark.col = selection.beg.col;
      selection.end.row = selection.mark.row = selection.beg.row;
      /*drop*/
    case SELECTION_BEGIN:
      selection.op = SELECTION_BEGIN;
      break;

    case SELECTION_COMPLETE:
    case SELECTION_ONGOING:
      selection.op = SELECTION_ONGOING;
      break;

    case SELECTION_CLEAR:
      scr_selection_start (x, y);
    default:
      return;
      break;
   }

   /* Remember old selection for virtual removal */
   old_row = selection.end.row;
   old_col = selection.end.col;

#ifdef NEW_SELECTION
   if ((old_row < -TermWin.hist_count) ||
       (selection.mark.row < -TermWin.hist_count))
     {
	scr_selection_clear ();
	return;
     }
#endif

   /* Figure out where new selection is */
   selection.end.col = Pixel2Col (x);
   selection.end.row = Pixel2Row (y);

   if (selection.end.col < 0)
     selection.end.col = 0;
   else if (selection.end.col >= TermWin.ncol)
     selection.end.col = (TermWin.ncol-1);

#ifdef NEW_SELECTION
   selection.end.row -= TermWin.hist_start;
   if (selection.end.row < -TermWin.hist_count)
     {
	scr_selection_clear ();
	return;
     }
#else
   if (selection.end.row < 0)
     selection.end.row = 0;
#endif
   else if (selection.end.row >= TermWin.nrow)
     selection.end.row = (TermWin.nrow-1);

   if ((selection.op == SELECTION_BEGIN) &&
       ((selection.end.col != selection.mark.col) ||
	(selection.end.row != selection.mark.row)))
     selection.op = SELECTION_ONGOING;

   /* If new selection is same as old selection just return
    * or if no highlighting was requested
    */
   if (!hilite ||
       (selection.end.row == old_row && selection.end.col == old_col))
     return;

   /* virtual removal -- delete old highlighting and replace with new */

   /* determine direction of old selection */
   old_dirn = ((old_row < selection.mark.row ||
		(old_row == selection.mark.row &&
		 old_col <= selection.mark.col)) ? UP : DOWN);

   /* determine direction of new selection */
   dirn = ((selection.end.row < selection.mark.row ||
	    (selection.end.row == selection.mark.row &&
	     selection.end.col <= selection.mark.col)) ? UP : DOWN);

   /* If old and new direction are different, clear old, set new */
   if (dirn != old_dirn)
     {
	if (old_dirn == UP)
	  {
	     selection_setclr (0,
			       old_row, old_col,
			       selection.mark.row, selection.mark.col);
	     selection_setclr (1,
			       selection.mark.row, selection.mark.col,
			       selection.end.row, selection.end.col);
	  }
	else
	  {
	     selection_setclr (0,
			       selection.mark.row, selection.mark.col,
			       old_row, old_col);
	     selection_setclr (1,
			       selection.end.row, selection.end.col,
			       selection.mark.row, selection.mark.col);
	  }
     }
   else
     {
	if (old_dirn == UP)
	  {
	     if (old_row < selection.end.row ||
		 (old_row == selection.end.row &&
		  old_col < selection.end.col))
	       {
		  selection_setclr (0,
				    old_row, old_col,
				    selection.end.row, selection.end.col);
		  selection_setclr (1,
				    selection.end.row, selection.end.col,
				    selection.end.row, selection.end.col);
	       }
	     else
	       {
		  selection_setclr (1,
				    selection.end.row, selection.end.col,
				    old_row, old_col);
	       }
	  }
	else
	  {
	     if (selection.end.row < old_row ||
		 (selection.end.row == old_row &&
		  selection.end.col < old_col))
	       {
		  selection_setclr (0,
				    selection.end.row, selection.end.col,
				    old_row, old_col);
		  selection_setclr (1,
				    selection.end.row, selection.end.col,
				    selection.end.row, selection.end.col);
	       }
	     else
	       {
		  selection_setclr (1,
				    old_row, old_col,
				    selection.end.row, selection.end.col);
	       }
	  }
     }
}

#ifndef NO_MULTIPLE_CLICK
/*
 * double/triple click selection, by Edward. Der-Hua Liu, Taiwan
 * Added cut char class support: A. Haritsis <ah@doc.ic.ac.uk>
 */
void
scr_selection_click (int clicks, int x, int y)
{
   int c, r, startx, endx;

   switch (clicks) {
    case 1:			/* single click */
      scr_selection_start (x, y);
      return;
      break;

    case 2:			/* double click */
      c = Pixel2Col (x);
      if (c < 0) c = 0; else if (c >= TermWin.ncol) c = (TermWin.ncol-1);
      startx = endx = c;

      r = Pixel2Row (y);
      if (r < 0) r = 0; else if (r >= TermWin.nrow) r = (TermWin.nrow-1);
      x = (r + TermWin.hist_size - TermWin.hist_start)*(TermWin.ncol+1);

      while (startx > 0 && !strchr (rs_cutchars, screen.text [x+startx-1]))
	startx--;
      while (endx < (TermWin.ncol-1) &&
	     !strchr (rs_cutchars, screen.text [x+endx+1]))
	endx++;
      break;

    case 3:			/* triple click */
      startx = 0;
      endx = (TermWin.ncol-1);
      break;

    default:
      return;
      break;
   }
   scr_selection_start (Col2Pixel (startx), y);
   scr_selection_extend (1, Col2Pixel (endx), y);
}
#endif

/*
 * Report the current cursor position
 */
void
scr_report_position (void)
{
   tty_printf ("\033[%d;%dR", screen.row + 1, screen.col + 1);
}

/*
 * Set a font
 */
void
scr_charset_set (int set, text_t a)
{
#ifdef KANJI
   multiByte = (set < 0);
   set = abs (set);
#endif
   charsets [set] = a;
   set_font_style ();
}

/*
 * choose a font
 */
void
scr_charset_choose (int set)
{
   screen.charset = set;
   set_font_style ();
}

/*
 * for the box starting at x, y with size width, height
 * touch the displayed values
 */
void
scr_touch (int x, int y, int width, int height)
{
   int r1, r2, c1, c2;

   check_text ("touch");

   c1 = Pixel2Col (x);
   r1 = Pixel2Row (y);
   c2 = c1 + 1 + width  / TermWin.fwidth;
   r2 = r1 + 1 + height / TermWin.fheight;

   if (r1 < 0) r1 = 0; else if (r1 >= TermWin.nrow) r1 = (TermWin.nrow-1);
   if (r2 < 0) r2 = 0; else if (r2 >= TermWin.nrow) r2 = (TermWin.nrow-1);
   if (c1 < 0) c1 = 0; else if (c1 >= TermWin.ncol) c1 = (TermWin.ncol-1);
   if (c2 < 0) c2 = 0; else if (c2 >= TermWin.ncol) c2 = (TermWin.ncol-1);

   c2 = (c2 - c1 + 1);
   for (r1 = r1; r1 <= r2; r1++)
     {
	int x = c1 + r1 * (TermWin.ncol+1);
	memset (&displayed_text [x], 0, c2 * sizeof(text_t));
	memset (&displayed_rend [x], RS_NONE, c2 * sizeof(rend_t));
     }
}

/*
 * refresh the region defined by rows STARTR and ENDR, inclusively.
 *
 * Actually draws to the X window
 * For X related speed-ups, this is a good place to fiddle.
 * The arrays displayed_text and displayed_rend contain what I
 * believe is currently shown on the screen. The arrays in screen contain
 * what should be displayed. This routine can decide how to refresh the
 * screen. Calls in command.c decide when to refresh.
 */
void
scr_refresh_region (int startr, int endr, int type)
{
   static int d_xcursor = 0;
   int r, xrow, xrow2, y1, xcursor;
#ifdef KANJI
   static int wbyte = 0;
#endif

   /* fix up startr/endr order */
   if (startr > endr)
     {
	r = startr;
	startr = endr;
	endr = r;
     }

   if (type == NO_REFRESH	/* Window not visible, don't update */
       || startr < 0 || endr >= TermWin.nrow)
     return;

   check_text ("refresh region");

   if (d_xcursor < (TermWin.nrow * (TermWin.ncol+1)))
     displayed_rend [d_xcursor] = 0xFF;

   xcursor = ((screen.row + TermWin.hist_size)*(TermWin.ncol+1)
	      + screen.col);
   screen.rend [xcursor] |= RS_CURSOR;
#ifdef KANJI
   if ((screen.rend [xcursor] & RL_KANJI2) == RL_KANJI1)
     screen.rend [xcursor+1] |= RS_CURSOR;
#endif
   d_xcursor = (screen.row + TermWin.hist_start);
   if (d_xcursor >= TermWin.nrow)
     {
	d_xcursor = 0;
	screen.rend [xcursor] &= ~RS_CURSOR;
     }
   else
     {
	d_xcursor *= (TermWin.ncol+1);
     }
   d_xcursor += screen.col;

#ifdef USE_XCOPYAREA
   if (type == FAST_REFRESH && !graphics_up)
     {
	/* scroll using bitblt wherever possible
	 * a dirty approximation will ignore the rendition field here
	 * and fix it up later */
	for (r = startr; r <= endr; r++)
	  displayed_text [r * (TermWin.ncol+1) + TermWin.ncol] = '\0';

	for (r = startr; r <= endr; r++)
	  {
	     int k, xrow, xrow2;

	     xrow = ((r + TermWin.hist_size - TermWin.hist_start) *
		     (TermWin.ncol+1));
	     xrow2 = r * (TermWin.ncol+1);
	     if (!strcmp (&displayed_text[xrow2], &screen.text[xrow]))
	       continue;
	     /* look for a similar line */
	     for (k = startr; k <= endr; k++)
	       {
		  xrow2 = k * (TermWin.ncol+1);
		  if (!strcmp (&displayed_text[xrow2], &screen.text[xrow]))
		    break;
	       }
	     /* found it */
	     if (k <= endr)
	       {
		  int count, j;

		  j = r;
		  xrow  += (TermWin.ncol+1);
		  xrow2 += (TermWin.ncol+1);
		  r++;
		  for (count = 1;
		       ((r <= endr) &&
			(!strcmp (&displayed_text[xrow2],
				  &(screen.text[xrow]))));
		       count++)
		    {
		       xrow  += (TermWin.ncol+1);
		       xrow2 += (TermWin.ncol+1);
		       r++;
		    }
		  r--;
		  XCopyArea (Xdisplay, TermWin.vt, TermWin.vt, Xgc,
			     MARGIN, Row2Pixel (k),
			     TermWin.width,
			     count * TermWin.fheight,
			     MARGIN, Row2Pixel (j));

		  /* Forward overlapping memcpy's are probably OK,
		   * but backwards doesn't work on SunOS 4.1.3 */
		  k *= (TermWin.ncol+1);
		  j *= (TermWin.ncol+1);
		  if (k > j)
		    {
		       while (count-- > 0)
			 {
			    MEMCOPY (&displayed_text [j],
				     &displayed_text [k],
				     (TermWin.ncol+1) * sizeof(text_t));
			    MEMCOPY (&displayed_rend [j],
				     &displayed_rend [k],
				     (TermWin.ncol+1) * sizeof(rend_t));
			    k += (TermWin.ncol+1);
			    j += (TermWin.ncol+1);
			 }
		    }
		  else
		    {
		       k += (count-1) * (TermWin.ncol+1);
		       j += (count-1) * (TermWin.ncol+1);
		       while (count-- > 0)
			 {
			    MEMCOPY (&displayed_text [j],
				     &displayed_text [k],
				     (TermWin.ncol+1) * sizeof(text_t));
			    MEMCOPY (&displayed_rend [j],
				     &displayed_rend [k],
				     (TermWin.ncol+1) * sizeof(rend_t));
			    k -= (TermWin.ncol+1);
			    j -= (TermWin.ncol+1);
			 }
		    }
	       }
	  }
     }
#endif

   xrow  = (startr + TermWin.hist_size - TermWin.hist_start) * (TermWin.ncol+1);
   xrow2 = startr * (TermWin.ncol+1);
   y1    = Xfont->ascent + Row2Pixel (startr);

   /* For a first cut, do it one character at a time */
   for (r = startr; r <= endr; r++)
     {
	int c;
	for (c = 0; c < TermWin.ncol; c++)
	  {
	     int x, x1, count;

	     x = xrow + c;
	     x1 = xrow2 + c;
	     if ((displayed_text [x1] != screen.text [x]) ||
		 (displayed_rend [x1] != screen.rend [x])
#ifdef KANJI
		 || (((screen.rend [x] & RL_KANJI2) == RL_KANJI1) &&
		     (displayed_text [x1+1] != screen.text [x+1]))
#endif
		 )
	       {
		  int fore, back, rend;
#ifndef NO_COLOR
		  unsigned long newgcm = 0;
		  XGCValues newgcv;
#endif
		  int outline = False;
		  GC thisGC = Xgc;

		  displayed_text [x1] = screen.text [x];
		  displayed_rend [x1] = screen.rend [x];
		  rend = screen.rend [x];
		  linebuf [0] = screen.text [x];
		  x1 = Col2Pixel (c);

		  x++; c++;
		  for (count = 1;
		       (c < TermWin.ncol &&
#ifdef KANJI
			((rend & ~RL_KANJI0) ==
			 (screen.rend [x] & ~RL_KANJI0)) &&
#else
			rend == screen.rend [x] &&
#endif
			(displayed_text [xrow2+c] != screen.text [x] ||
			 displayed_rend [xrow2+c] != screen.rend [x] ||
			 (c+1 < TermWin.ncol &&
			  displayed_text [xrow2+c+1] != screen.text [x+1])));
		       count++)
		    {
		       displayed_text [xrow2+c]= screen.text [x];
		       displayed_rend [xrow2+c]= screen.rend [x];
		       linebuf [count] = screen.text [x];
		       x++; c++;
		    }
		  c--;

#ifdef KANJI
		  /* ensure the correct font is used */
		  if (rend & RL_KANJI1)
		    {
		       if (!wbyte)
			 {
			    XSetFont (Xdisplay, Xgc, Kanjifont->fid);
			    XSetFont (Xdisplay, Xrvgc, Kanjifont->fid);
			    wbyte = 1;
			 }
		    }
		  else
		    {
		       if (wbyte)
			 {
			    XSetFont (Xdisplay, Xgc,  Xfont->fid);
			    XSetFont (Xdisplay, Xrvgc, Xfont->fid);
			    wbyte = 0;
			 }
		    }
#endif
		  fore = GET_FG_COLOR (rend);
		  back = GET_BG_COLOR (rend);
		  rend = GET_ATTR (rend);
		  if (rend)
		    {
		       if ((rend & RS_RVID) && (rend & RS_SELECTED))
			 rend &= ~(RS_RVID | RS_SELECTED);

		       if (rend & RS_CURSOR)
			 {
			    if (!focus)
			      {
				 outline = True;
				 rend &= ~RS_CURSOR;
			      }
			    else if (rend & (RS_RVID | RS_SELECTED))
			      rend &= ~(RS_CURSOR | RS_RVID | RS_SELECTED);
			 }

		       if (rend & (RS_CURSOR | RS_RVID | RS_SELECTED))
			 thisGC = Xrvgc;

		       switch (rend & RS_FONTMASK) {
			case RS_GRFONT:
			  for (x = 0; x < count; x++)
			    if (linebuf [x] >= 0x5F && linebuf [x] < 0x7F)
			      linebuf [x] = (linebuf [x] == 0x5F ?
					     0x7F : linebuf [x] - 0x5F);
			  break;
			case RS_GBFONT:
			  for (x = 0; x < count; x++)
			    if (linebuf [x] == '#') linebuf [x] = '\036';
			  break;
		       }
		    }
#ifndef NO_COLOR
		  if (thisGC == Xrvgc)
		    {
		       if (fore)
			 {
			    newgcv.background = pixel_colors [fore];
			    newgcm |= GCBackground;
			 }
		       if (back)
			 {
			    newgcv.foreground = pixel_colors [back];
			    newgcm |= GCForeground;
			 }
		    }
		  else
		    {
		       if (fore)
			 {
			    newgcv.foreground = pixel_colors [fore];
			    newgcm |= GCForeground;
			 }
		       if (back)
			 {
			    newgcv.background = pixel_colors [back];
			    newgcm |= GCBackground;
			 }
		    }

		  if (newgcm)
		    XChangeGC (Xdisplay, thisGC, newgcm, &newgcv);
#endif	/* NO_COLOR */
#ifdef KANJI
		  if (wbyte)
		    {
		       if (linebuf [0] & 0x80)
			 kanji_decode (linebuf, count);
		       XDrawImageString16 (Xdisplay, TermWin.vt, thisGC,
					   x1, y1,
					   (XChar2b *)linebuf, count/2);
		    }
		  else
#endif	/* KANJI */
		    XDrawImageString (Xdisplay, TermWin.vt, thisGC,
				      x1, y1,
				      linebuf, count);

		  if (rend)
		    {
		       if (
#ifdef USE_FAKE_BOLD
			   (rend & (RS_BOLD|RS_BLINK))

#else
			   ((rend & RS_BOLD)  && fore == COLORFG) ||
			   ((rend & RS_BLINK) && back == COLORBG)
#endif
			   )
			 {
#ifdef KANJI
			    if (wbyte)
			      XDrawString16 (Xdisplay, TermWin.vt, thisGC,
					     x1 + 1, y1,
					     (XChar2b *)linebuf, count/2);
			    else
#endif	/* KANJI */
			      XDrawString (Xdisplay, TermWin.vt, thisGC,
					   x1 + 1, y1,
					   linebuf, count);
			 }

		       /*
			* On the smallest font,
			* underline overwrites the next row
			*/
		       if ((rend & RS_ULINE) && (Xfont->descent > 1))
			 XDrawLine (Xdisplay, TermWin.vt, thisGC,
				    x1,
				    y1 + 1,
				    x1 + count * TermWin.fwidth - 1,
				    y1 + 1);
#ifndef NO_COLOR
		       if (newgcm)
			 {
			    /* restore the normal colors */
			    if (thisGC == Xrvgc)
			      {
				 newgcv.foreground = pixel_colors [COLORBG];
				 newgcv.background = pixel_colors [COLORFG];
			      }
			    else
			      {
				 newgcv.foreground = pixel_colors [COLORFG];
				 newgcv.background = pixel_colors [COLORBG];
			      }
			    XChangeGC (Xdisplay, thisGC, newgcm, &newgcv);
			 }
#endif	/* NO_COLOR */
		    }
		  if (outline)
		    XDrawRectangle (Xdisplay, TermWin.vt, thisGC, x1,
				    y1 - Xfont->ascent,
				    TermWin.fwidth - 1,
				    TermWin.fheight - 1);
	       }
	  }
	xrow  += (TermWin.ncol+1);
	xrow2 += (TermWin.ncol+1);
	y1    += (TermWin.fheight);
     }
   screen.rend [xcursor] &= ~RS_CURSOR;
#ifdef KANJI
   screen.rend [xcursor+1] &= ~RS_CURSOR;
#endif
}

void
scr_refresh (int type)
{
   scr_refresh_region (0, (TermWin.nrow-1), type);
}

/*
 *  -1 = clear all tabs
 *  +0 = clear tab stop at current column
 *  +1 = set   tab stop at current column
 */
void
scr_set_tab (int mode)
{
   if (mode < 0)
     memset (tabs, 0, TermWin.ncol);
   else if (screen.col < TermWin.ncol)
     tabs [screen.col] = (mode != 0);
}

/*
 * toggle reverse video settings
 */
void
scr_rvideo_mode (int mode)
{
   if (rvideo != mode)
     {
	register int x, count;

	rvideo = mode;
	rstyle ^= RS_RVID;

	count = (TermWin.nrow + TermWin.hist_size) * (TermWin.ncol+1);
	for (x = 0; x < count; x++) screen.rend [x] ^= RS_RVID;
	scr_refresh (SLOW_REFRESH);
     }
}

/*
 * Handle receipt of a bell character
 */
void
scr_bell (void)
{
#ifdef MAPALERT
   if (map_alert)
     XMapWindow (Xdisplay, TermWin.parent);
#endif
#ifdef VISUAL_BELL
   if (visual_bell)
     {
	scr_rvideo_mode (!rvideo);
	scr_rvideo_mode (!rvideo);
     }
   else
#endif
     XBell (Xdisplay, 0);
}

/*
 * Set the text foreground color
 */
void
scr_fgcolor (int color)
{
#ifndef NO_COLOR
   if (Xdepth >= 3)
     {
	if (color == RESTORE_FG) color = COLORFG;
#ifndef USE_FAKE_BOLD
	else if (rstyle & RS_BOLD) color += BOLD0;
	else
#endif
	  color += COLOR0;
     }
   else
#endif
     color = (color == 7 ? COLORBG : COLORFG);    /* normal/reverse */

   rstyle = SET_FG_COLOR (rstyle, color);
}

/*
 * Set the text foreground color
 */
void
scr_bgcolor (int color)
{
#ifndef NO_COLOR
   if (Xdepth >= 3)
     {
	if (color == RESTORE_BG) color = COLORBG;
#ifndef USE_FAKE_BOLD
	else if (rstyle & RS_BLINK) color += BOLD0;
	else
#endif
	  color += COLOR0;
     }
   else
#endif
     color = (color != 0 ? COLORBG : COLORFG);   /* normal/reverse */

   rstyle = SET_BG_COLOR (rstyle, color);
}

#ifdef RXVT_GRAPHICS
/* return the current foreground/background colors */
int scr_get_fgcolor (void) { return GET_FG_COLOR (rstyle); }
int scr_get_bgcolor (void) { return GET_BG_COLOR (rstyle); }
#endif

#ifdef PRINTPIPE
void
scr_printscreen (int fullhist)
{
   text_t *text;
   int r, nrows;
   FILE *fd;

   fd = popen_printer ();
   if (fd == NULL)
     return;

   nrows = TermWin.nrow;
   if (fullhist)
     nrows += TermWin.hist_start;

   text = &screen.text [((TermWin.hist_size - TermWin.hist_start) *
			 (TermWin.ncol+1))];
   for (r = 0; r < nrows; r++, text += (TermWin.ncol+1))
     {
	int i;
	for (i = (TermWin.ncol-1); i >= 0 && isspace (text[i]); i--) /*nil*/;
	i++;
	fprintf (fd, "%.*s\n", i, text);
     }

   pclose_printer (fd);
}
#endif

/*
 * hidden debugging dump
 */
#ifdef DEBUG_SELECTION
int
scr_debug_selection_state (void)
{
   printf ("\n%dx%d [chars] %dx%d [pixels] (font: %ux%u), scroll = %d/%d\n",
	   TermWin.ncol,   TermWin.nrow,
	   TermWin.width,  TermWin.height,
	   TermWin.fwidth, TermWin.fheight,
	   screen.tscroll, screen.bscroll);

   printf ("%d lines scrollback, %d lines scrolled, offset = %d\n",
	   TermWin.hist_size, TermWin.hist_count, TermWin.hist_start);

   printf ("selection = screen %d, op = ", selection.screen);

   switch (selection.op) {
    case SELECTION_CLEAR:	printf ("CLEAR");	break;
    case SELECTION_BEGIN:	printf ("BEGIN");	break;
    case SELECTION_INIT:	printf ("INIT");	break;
    case SELECTION_ONGOING:	printf ("ONGOING");	break;
    case SELECTION_COMPLETE:	printf ("COMPLETE");	break;
    default:			printf ("Unknown");	break;
   }
   printf ("\n\trow/col\n");
   printf ("beg\t%d %d\nend\t%d %d\nanchor\t%d %d\ncursor\t%d %d\n",
	   selection.beg.row, selection.beg.col,
	   selection.end.row, selection.end.col,
	   selection.mark.row, selection.mark.col,
	   screen.row, screen.col);

   printf ("selection text = %d chars\n", selection.len);
   if (selection.text && selection.len)
     {
	int i;
	printf ("<text>\n\"");
	for (i = 0; i < selection.len; i++)
	  {
	     switch (selection.text[i]) {
	      case '\n': printf ("\\n"); break;
	      case '\r': printf ("\\r"); break;
	     }
	     printf ("%c", selection.text[i]);
	  }
	printf ("\"\n</text>");
     }

   return 0;
}
#endif

/* add the bits: 1 - shift, 2 - meta, 4 - ctrl */
#define KeyState(x) (((x)&(ShiftMask|ControlMask))+(((x) & Mod1Mask)? 2 : 0))

#ifndef NO_MOUSE_REPORT
void
mouse_report (XButtonEvent *ev)
{
   int but, col, row;

   but = (040 + ((ev->button ? (ev->button - Button1) : 3)
		 + (KeyState (ev->state) << 2)));
   col = 041 + Pixel2Col (ev->x);
   row = 041 + Pixel2Row (ev->y);

   tty_printf ("\033[M%c%c%c", but, col, row);
}

#if 0	/* not yet - maybe never! */
void
mouse_tracking (int report, int x, int y, int firstrow, int lastrow)
{
   static int startx, starty, top, bot;

   if (report)
     {
	/* If either coordinate is past the end of the line:
	 * "ESC [ T CxCyCxCyCxCy"
	 * The parameters are startx, starty, endx, endy,
	 * mousex, and mousey */

	if ((selection.beg.row < selection.end.row) ||
	    ((selection.beg.row == selection.end.row) &&
	     (selection.beg.col < selection.end.col)))
	  {
	     if (selection.beg.row >= top && selection.end.row <= bot)
	       tty_printf ("\033[t"); /* start/end are valid locations */
	     else
	       tty_printf ("\033[T%c%c%c%c",
			   selection.beg.col + 1, selection.beg.row + 1,
			   selection.end.col + 1, selection.end.row + 1);
	  }
	else
	  {
	     if (selection.end.row >= top && selection.beg.row <= bot)
	       tty_printf ("\033[t");	/* start/end are valid locations */
	     else
	       tty_printf ("\033[T%c%c%c%c",
			   selection.end.col + 1, selection.end.row + 1,
			   selection.beg.col + 1, selection.beg.row + 1)
	  }
	tty_printf ("%c%c", Pixel2Col (x) + 1, Pixel2Row (y) + 1);
     }
   else
     {
	/* convert column/row to pixels */
	x = Col2Pixel (x - 1);
	y = Row2Pixel (y - 1);

	scr_selection_start (x ,y);
	top = firstrow;
	bot = lastrow;
     }
}
#endif
#endif
/*----------------------- end-of-file (C source) -----------------------*/
