/*--------------------------------*-C-*---------------------------------*
 * File:	xsetup.c
 *
 * Copyright 1992 John Bovey, University of Kent at Canterbury.
 *
 * You can do what you like with this source code as long as
 * you don't try to make money out of it and you include an
 * unaltered copy of this message (including the copyright).
 *
 * This module has heavily modified by R. Nation
 * <nation@rocket.sanders.lockheed.com>
 * No additional restrictions are applied
 *
 * Additional modifications by Garrett D'Amore <garrett@netcom.com>
 * No additional restrictions are applied.
 *
 * Additional modifications by mj olesen <olesen@me.QueensU.CA>
 * No additional restrictions are applied.
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#include <ctype.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
#include <X11/keysym.h>

#include "command.h"
#include "xsetup.h"
#include "screen.h"
#include "debug.h"
#include "graphics.h"
#include "grkelot.h"
#include "defaults.h"

/* change here to suit compiled-in value */
#define HOTKEY_KEYSYM	"Alt-keysym"
/*#define HOTKEY_KEYSYM	"Ctrl-keysym"*/
/*#define HOTKEY_KEYSYM	"Shift-keysym"*/

/*----------------------------------------------------------------------*
 * extern variables referenced
 */

/*----------------------------------------------------------------------*
 * extern variables declared here
 */
Display		*Xdisplay;	/* display */
XFontStruct	*Xfont;		/* main font structure */
#ifdef KANJI
XFontStruct	*Kanjifont;	/* Kanji font structure */
#endif
GC 		Xgc, Xrvgc;	/* GC for drawing text: plain, reverse video */
int		Xdepth;		/* default depth for current screen */

char *rs_color [NCOLORS];
unsigned long pixel_colors [NCOLORS];

#ifndef SCROLLBAR_NONE
sbar_t sbar;
#endif

unsigned char console = 0;
#ifndef META_ESCAPE
/* how to handle meta key events */
unsigned char meta_char = 033;	/* escape prefix */
#endif
unsigned char login_shell = 0;	/* login-shell with `-e rlogin' doesn't work */
#ifdef MAPALERT
unsigned char map_alert = DEF_MAPALERT;
KeySym ks_alert = DEF_KS_ALERT;
#endif
#ifdef VISUAL_BELL
unsigned char visual_bell = DEF_VISUALBELL;
#endif

char *rs_title = NULL;		/* window name for titles etc. */
char *rs_iconName = NULL;	/* name to display in the icon */
char *rs_geometry = NULL;
char *display_name = NULL;

char *rs_font [NFONTS];
#ifdef KANJI
char *rs_kfont [NFONTS];
#endif
#if (NFONTS > 1)
KeySym ks_bigfont = DEF_KS_BIGFONT;
KeySym ks_smallfont = DEF_KS_SMALLFONT;
#endif

#ifdef PRINTPIPE
char *rs_print_pipe = NULL;
KeySym ks_printscreen = DEF_KS_PRINTSCREEN;
#endif

KeySym ks_pageup   = DEF_KS_PAGEUP;
KeySym ks_pagedown = DEF_KS_PAGEDOWN;
#ifdef GREEK_SUPPORT
KeySym ks_greektoggle = DEF_KS_GREEKTOGGLE;
#endif

#ifndef NO_MULTIPLE_CLICK
char *rs_cutchars = NULL;
#endif

/*----------------------------------------------------------------------*
 * local variables
 */
static int Xscreen;		/* the X screen number */

static XSizeHints sizehint = {
   PMinSize | PResizeInc | PBaseSize | PWinGravity,
   0, 0, 80, 24,		/* x, y, width and height */
   1, 1,			/* Min width and height */
   0, 0,			/* Max width and height */
   1, 1,			/* Width and height increments */
   {1, 1},			/* x, y increments */
   {0, 0},			/* Aspect ratio - not used */
   2 * MARGIN, 2 * MARGIN,	/* base size */
   NorthWestGravity		/* gravity */
};

static char *rs_name = NULL;	    /* the name the program is run under */
static unsigned char iconic = 0;    /* boolean */

static char *Def_color_names [] = { DEF_COLORFG, DEF_COLORBG,
#ifndef NO_COLOR
     DEF_COLOR0, DEF_COLOR1, DEF_COLOR2, DEF_COLOR3,
     DEF_COLOR4, DEF_COLOR5, DEF_COLOR6, DEF_COLOR7,
#ifndef USE_FAKE_BOLD
     DEF_COLOR10, DEF_COLOR11, DEF_COLOR12, DEF_COLOR13,
     DEF_COLOR14, DEF_COLOR15, DEF_COLOR16, DEF_COLOR17,
#endif
#endif
     NULL
};

static char *Def_font_names [] = { DEF_FONT0,
#if (NFONTS > 1)
     DEF_FONT1, DEF_FONT2, DEF_FONT3, DEF_FONT4,
#endif
     NULL
};

#ifdef KANJI
static char *Def_kfont_names [] = { DEF_KFONT0,
#if (NFONTS > 1)
     DEF_KFONT1, DEF_KFONT2, DEF_KFONT3, DEF_KFONT4,
#endif
     NULL
};
#endif

/*----------------------------------------------------------------------*
 * extern functions referenced
 */
extern void extract_resources (char *name);
#ifdef DISPLAY_ENV_AND_ANSWER_IS_IP
extern char * network_display (char *display);
#endif

/*----------------------------------------------------------------------*
 * local functions
 */
static void create_window (int argc, char *argv[]);

/*----------------------------------------------------------------------*/
static void
usage (void)
{
   fprintf(stderr, "Usage v"VERSION":\n\t"
	   APL_NAME" [options] [-e command args]\n\n"
	   "where options include:\n"
	   "  -display displayname  X server to contact\n"
	   "  -geometry geom        size (in characters) and position\n"
	   "  -bg color             background color\n"
	   "  -fg color             foreground color\n"
#ifndef NO_COLOR
	   "  -color<n> color       color <n>; <n> = 0-7, 10-17\n"
#endif
	   "  -fn fontname          normal font\n"
#if (NFONTS > 1)
	   "  -font<n> fontname     alternative font <n>; <n> = 1-%d\n",
	   (NFONTS-1)
#endif
	   );

#ifdef KANJI
   fprintf(stderr,
	   "  -km mode              kanji encoding; mode = sjis | eucj\n"
	   "  -fk fontname          kanji font\n"
#if (NFONTS > 1)
	   "  -kfont<n> fontname    kanji alternative font <n>; <n> = 1-%d\n",
	   (NFONTS-1)
#endif
	   );
#endif	/* KANJI */

   fprintf(stderr,
	   "  -name string          client instance, icon, and title strings\n"
	   "  -title string         title name for window\n"
	   "  -n string             icon and title name for window\n"
	   "  -/+ls                 turn on/off login shell\n"
#ifdef UTMP_SUPPORT
	   "  -/+ut                 turn on/off utmp inhibit\n"
#endif
#ifdef MAPALERT
	   "  -/+ma                 turn on/off deiconify (map) on audio alert\n"
#endif
#ifdef VISUAL_BELL
	   "  -/+vb                 turn on/off visual bell\n"
#endif
#ifdef SCROLLBAR_ANY
	   "  -/+sb                 turn on/off scrollbar\n"
	   "  -/+arrows             turn on/off scrollbar arrows\n"
#endif	/* ANY */
	   "  -ic                   start iconic\n"
#ifndef META_ESCAPE
	   "  -/+meta               Meta key as Escape prefix on/off\n"
	   "  -meta8                Meta key as 8th bit set\n"
#endif
	   "  -sl number            number of scrolled lines to save\n"
#ifdef REFRESH_RESOURCE
	   "  -refresh number       maximum number of buffered screenfuls\n"
#endif
#ifdef GREEK_SUPPORT
	   "  -grk9                 greek kbd - ELOT 928 (default)\n"
	   "  -grk4                 greek kbd - IBM 437\n"
#endif
#ifdef PRINTPIPE
	   "  -print-pipe name      specify pipe for vt100 printer\n"
#endif
#ifdef WITH_7BIT_MODE
	   "  -/+7                  turn on/off 7 bit mode\n"
#endif
	   );

#ifdef NO_RESOURCES	/* options only if there are no resources */
   fprintf(stderr,
	   "  -pageup keysym        "HOTKEY_KEYSYM" to scroll up\n"
	   "  -pagedown keysym      "HOTKEY_KEYSYM" to scroll down\n"
#if (NFONTS > 1)
	   "  -bigfont keysym       "HOTKEY_KEYSYM" to switch to a bigger font\n"
	   "  -smallfont keysym     "HOTKEY_KEYSYM" to switch to a smaller font\n"
#endif
#ifdef PRINTPIPE
	   "  -prkey keysym         keysym to print screen\n"
#endif
	   );
#endif	/* NO_RESOURCES */

   fprintf(stderr,
	   "  -C                    intercept console messages\n"
	   "  -e command arg ...    command to execute\n"
	   );
}

char *
my_basename (const char *string)
{
   char *base;
   return ((base = strrchr (string, '/')) == NULL) ? string : base + 1;
}

void
String2Keysym (KeySym *keysym, char *string)
{
   KeySym ks;
   if (string != NULL && ((ks = XStringToKeysym (string)) != 0))
     *keysym = ks;
}

static XErrorHandler
RxvtErrorHandler (Display *dpy, XErrorEvent *event)
{
   exit (EXIT_FAILURE);
   return 0;
}

#ifndef NO_COLOR
static void
color_aliases (int index)
{
   char *name = rs_color [index];

   if (*name == '#' && strlen (name) <= 3)
     {
	int n = isdigit (*++name) ? atoi (name) : -1;
	if (n >= 0 && n <= 7)		/* normal colors */
	  rs_color [index] = rs_color [n + COLOR0];
#ifndef USE_FAKE_BOLD
	else if (n >= 10 && n <= 17)	/* bold colors */
	  rs_color [index] = rs_color [n-10 + BOLD0];
#endif
     }
}
#endif

/*
 * Open display, get options, resources and create the window
 */
void
init_display (int argc, char *argv[], char *cmd)
{
   static char id_string [29], *display_string;
   char *opt, *val;
   XGCValues gcv;
   int i, n, bad_option = 0;

   rs_name = my_basename (argv [0]);

   if (cmd != NULL)
     rs_title = my_basename (cmd);

   if ((display_name = getenv ("DISPLAY")) == NULL)
     display_name = ":0";

   /* first pass (before getting resources),
    * MUST find -display & -name (if they exist)
    * Also, find options with string arguments to avoid allocation
    */
   for (i = 1; i < argc; i++)
     {
	opt = argv [i];
	val = argv [i+1];

	if (*opt != '-' || val == NULL) continue;
	opt++;

	if (!strcmp(opt, "display") || !strcmp(opt, "d"))
	  { display_name = val; }
	else if (!strcmp(opt, "name"))	{ rs_name = val; }
	else if (!strcmp(opt, "title") || !strcmp(opt, "T"))
	  { rs_title = val; }
	else if (!strcmp(opt, "n"))	{
	   rs_iconName = val;
	   if (rs_title == NULL)
	     rs_title = val;
	}
	else if (!strcmp(opt, "geometry") || !strcmp(opt, "g"))
	  { rs_geometry = val; }
	else if (!strcmp(opt, "fn") && val)	{ rs_font [0] = val; }
	else if (!strncmp(opt, "font", 4))	{
	   /* also filters out & ignores `-font_list' */
	   opt += 4;
	   n = isdigit (*opt) ? atoi (opt) : -1;
	   if (n >= 0 && n < NFONTS)
	     rs_font [n] = val;
	}
	else if (!strcmp(opt, "km") && val)	{
#ifdef KANJI
	   set_kanji_encoding (val);
#endif
	}
	else if (!strcmp(opt, "fk") && val)	{
#ifdef KANJI
	   rs_kfont [0] = val;
#endif
	}
	else if (!strncmp(opt, "kfont", 5))	{
#ifdef KANJI
	   opt += 5;
	   n = isdigit (*opt) ? atoi (opt) : -1;
	   if (n >= 0 && n < NFONTS)
	     rs_kfont [n] = val;
#endif
	}
	else if (!strcmp(opt, "fg"))	{ rs_color [COLORFG] = val; }
	else if (!strcmp(opt, "bg"))	{ rs_color [COLORBG] = val; }
	else if (!strncmp(opt, "color", 5))	{
#ifndef NO_COLOR
	   opt += 5;
	   n = isdigit (*opt) ? atoi (opt) : -1;
	   if (n >= 0 && n <= 7)		/* normal colors */
	     rs_color [n + COLOR0] = val;
#ifndef USE_FAKE_BOLD
	   else if (n >= 10 && n <= 17)	/* bold colors */
	     rs_color [n-10 + BOLD0] = val;
#endif
#endif
	}
	else if (!strcmp(opt, "print-pipe"))	{
#ifdef PRINTPIPE
	   rs_print_pipe = val;
#endif
	}

	/* only bother with these options if there is no resource available */
#ifdef NO_RESOURCES
	else if (!strcmp(opt, "prkey"))		{
#ifdef PRINTPIPE
	   String2Keysym (&ks_printscreen, val);
#endif
	}
	else if (!strcmp(opt, "bigfont"))	{
#if (NFONTS > 1)
	   String2Keysym (&ks_bigfont, val);
#endif
	}
	else if (!strcmp(opt, "smallfont"))	{
#if (NFONTS > 1)
	   String2Keysym (&ks_smallfont, val);
#endif
	}
	else if (!strcmp(opt, "pageup"))	{
	   String2Keysym (&ks_pageup, val);
	}
	else if (!strcmp(opt, "pagedown"))	{
	   String2Keysym (&ks_pagedown, val);
	}
#endif	/* NO_RESOURCES */

	else continue;		/* not a string option, skip */

	argv [i] = NULL;
	i++;
	argv [i] = NULL;
     }

   Xdisplay = XOpenDisplay (display_name);
   if (!Xdisplay)
     {
	print_error ("can't open display %s", display_name);
	exit (EXIT_FAILURE);
     }

   Xscreen = DefaultScreen (Xdisplay);
   Xdepth  = DefaultDepth (Xdisplay, Xscreen);

   /* set some fallback values */
   TermWin.hist_size = DEF_SCROLLBACK;
#ifdef SCROLLBAR_ANY
   sbar.type = DEF_SCROLLBAR;
#endif

   extract_resources (rs_name);

   /* now get all the options */
   for (i = 1; i < argc; i++)
     {
	int flag_on;
	opt = argv [i];
	if (opt == NULL) continue;	/* already processed */
	val = argv [i+1];

	flag_on = *opt++;		/* -/+ sign */
	if (flag_on != '-' && flag_on != '+')
	  {
	     bad_option = 1;
	     print_error ("bad option %s", (opt-1));
	     continue;
	  }
	flag_on = (flag_on == '-');

	if (!strcmp(opt, "C"))	{ console = flag_on; }
	else if (!strcmp(opt, "7"))	{
#ifdef WITH_7BIT_MODE
	   hibit_mask = (flag_on ? 0x7F : 0xFF);
#endif
	}
	else if (!strcmp(opt, "ls"))	{ login_shell = flag_on; }
	else if (!strcmp(opt, "ut"))	{
#ifdef UTMP_SUPPORT
	   utmp_inhibit = flag_on;
#endif
	}
	else if (!strcmp(opt, "sb"))	{
#ifdef SCROLLBAR_ANY
	   if (!flag_on)		/* '+' */
	     sbar.type = SBAR_NONE;
	   else if (sbar.type == SBAR_NONE)
	     sbar.type = SBAR_XTERM;
#endif
	}
	else if (!strncmp(opt, "arrow", 5))	{
#ifdef SCROLLBAR_ANY
	   if (flag_on)
	     sbar.type = SBAR_ARROWS;
	   else if (sbar.type == SBAR_ARROWS)
	     sbar.type = SBAR_XTERM;
#endif
	}
	else if (!strcmp(opt, "ma"))	{
#ifdef MAPALERT
	   map_alert = flag_on;
#endif
	}
	else if (!strcmp(opt, "vb"))	{
#ifdef VISUAL_BELL
	   visual_bell = flag_on;
#endif
	}
	else if (!strncmp(opt, "ic", 3))	{ iconic = flag_on; }
	else if (!strcmp(opt, "meta8"))	{
#ifndef META_ESCAPE
	   meta_char = 0x80;
#endif
	}
	else if (!strcmp(opt, "meta"))		{
#ifndef META_ESCAPE
	   meta_char = (flag_on ? 033 : 0);
#endif
	}
	else if (!strncmp(opt, "grk", 4))	{
#ifdef GREEK_SUPPORT
	   switch (opt [4]) {
	    case '4':	greek_setmode (GREEK_IBM437);	break;
	    case '9':	greek_setmode (GREEK_ELOT928);	break;
	   }
#endif
	}
	else if (!strcmp(opt, "sl") && val)	{
	   i++;
	   TermWin.hist_size = atoi (val);
	   if (TermWin.hist_size < 0)
	     TermWin.hist_size = 0;
	}
#ifdef REFRESH_RESOURCE
	else if (!strcmp(opt, "refresh") && val)
	  { i++; refresh_period = atoi(val); }
#endif
	else if (!strcmp(opt, "help"))
	  {
	     usage ();
	     exit (EXIT_FAILURE);
	  }
	else
	  /* various old-style options, just ignore
	   * Obsolete since about Jan 96, so they can probably eventually
	   * be removed.
	   */
	  if (!strcmp(opt, "8") ||
	      !strcmp(opt, "fat") ||
	      !strcmp(opt, "thin") ||
	      !strncmp(opt, "meta", 4))
	  {
	     print_error ("obsolete option %c%s",
			  (flag_on ? '-' : '+'), opt);
	  }
	else
	  {
	     bad_option = 1;
	     print_error ("bad option %c%s", (flag_on ? '-' : '+'), opt);
	  }
     }

   if (bad_option)
     {
	print_error ("for usage use -help");
	exit (EXIT_FAILURE);
     }

   /*
    * set any defaults not already set
    */
   if (rs_title == NULL) rs_title = rs_name;
   if (rs_iconName == NULL) rs_iconName = rs_name;

   if (rs_geometry == NULL) rs_geometry = DEF_GEOMETRY;

#ifdef PRINTPIPE
   if (rs_print_pipe == NULL) rs_print_pipe = PRINTPIPE;
#endif
#ifndef NO_MULTIPLE_CLICK
   if (rs_cutchars == NULL) rs_cutchars = DEF_CUTCHARS;
#endif

   for (i = 0; i < NFONTS; i++)
     {
	if (rs_font [i] == NULL) rs_font [i] = Def_font_names [i];
#ifdef KANJI
	if (rs_kfont [i] == NULL) rs_kfont [i] = Def_kfont_names [i];
#endif
     }

   for (i = 0; i < NCOLORS; i++)
     {
	if (rs_color [i] == NULL) rs_color [i] = Def_color_names [i];
     }

#ifndef NO_COLOR
   /* convenient aliases for setting fg/bg to colors */
   color_aliases (COLORFG);
   color_aliases (COLORBG);
#endif

   if (argc)
     {
	argc = 1;
	argv[1] = NULL;
     }
   create_window (argc, argv);

   /* Add a DISPLAY entry to the environment, in case we started with
    * rxvt -display name. */
#ifdef DISPLAY_ENV_AND_ANSWER_IS_IP
   /* Fixup display_name for export over pty to any interested terminal
    * clients via "ESC[7n" (e.g. shells).  Note we use the pure IP number
    * (for the first non-loopback interface) that we get from
    * network_display().  This is more "name-resolution-portable", if you
    * will, and probably allows for faster x-client startup if your name
    * server is beyond a slow link or overloaded at client startup.  Of
    * course that only helps the shell's child processes, not us.
    */
   val = display_name = network_display (display_name);
   if (val == NULL)
#endif
     val = XDisplayString (Xdisplay);
   i = strlen (val);
   display_string = MALLOC (i+10,"display_string");
   sprintf (display_string, "DISPLAY=%s", val);
   sprintf (id_string,"WINDOWID=%u",(unsigned int)TermWin.parent);

   /* add entries to the environment */

   putenv (display_string);
   putenv (id_string);

   putenv ("TERM=" TERM_NAME);
#ifdef NO_COLOR
   putenv ("COLORTERM=" APL_NAME "-mono");
#else
   putenv ("COLORTERM=" APL_NAME);
#endif

   if (display_name == NULL)
     display_name = val;	/* use broken `:0' value */

   scr_reset ();		/* initialize screen */
   Gr_reset ();

   /* Create the graphics contexts */
   gcv.font = Xfont->fid;
   gcv.foreground = pixel_colors [COLORBG];
   gcv.background = pixel_colors [COLORFG];
   Xrvgc = XCreateGC (Xdisplay, TermWin.parent,
		      GCForeground|GCBackground|GCFont, &gcv);
   gcv.foreground = pixel_colors [COLORFG];
   gcv.background = pixel_colors [COLORBG];
   Xgc = XCreateGC (Xdisplay, TermWin.parent,
		    GCForeground|GCBackground|GCFont, &gcv);

   sbar_init (TermWin.parent, &gcv);	/* initialize the scrollbar */

#ifdef DEBUG_X
   XSynchronize (Xdisplay, True);
   XSetErrorHandler ((XErrorHandler)abort);
#else
   XSetErrorHandler ((XErrorHandler)RxvtErrorHandler);
#endif
}

/*
 * Open and map the window
 */
static void
create_window (int argc, char *argv[])
{
   const char *font_msg = "can't load font \"%s\"";

   Colormap cmap;		/* colormap */
   XColor xcolor_bg, xcolor_fg;
   XClassHint xcls;
   XWMHints wmhint;
   Cursor cursor;
   int i, flags, x, y;
   /*unsigned*/ int width, height;

   /* First get the font since we need it to set the size */
   Xfont = XLoadQueryFont (Xdisplay, rs_font [0]);
   if (!Xfont)
     {
	print_error (font_msg, rs_font [0]);
	rs_font [0] = "fixed";
	Xfont = XLoadQueryFont (Xdisplay, rs_font [0]);
	if (!Xfont)
	  {
	     print_error (font_msg, rs_font [0]);
	     goto Abort;
	  }
     }
#ifdef KANJI
   Kanjifont = XLoadQueryFont (Xdisplay, rs_kfont [0]);
   if (!Kanjifont)
     {
	print_error (font_msg, rs_kfont [0]);
	goto Abort;
     }
#endif

   /*
    * allocate colors now, before netscape gets them :)
    */
   cmap = DefaultColormap (Xdisplay, Xscreen);
   for (i = 0; i < NCOLORS; i++)
     {
	const char *color_msg = "can't load color \"%s\"";
	XColor  *pxcol, xcol;

	switch (i) {
	 case COLORFG:	pxcol = &xcolor_fg;	break;
	 case COLORBG:	pxcol = &xcolor_bg;	break;
	 default:	pxcol = &xcol;		break;
	}

	if (!XParseColor (Xdisplay, cmap, rs_color [i], pxcol) ||
	    !XAllocColor (Xdisplay, cmap, pxcol))
	  {
	     print_error (color_msg, rs_color [i]);
	     rs_color [i] = Def_color_names [i];
	     if (!XParseColor (Xdisplay, cmap, rs_color [i], pxcol) ||
		 !XAllocColor (Xdisplay, cmap, pxcol))
	       {
		  print_error (color_msg, rs_color [i]);

		  if (i == COLORFG || i == COLORBG)
		    goto Abort;
		  pxcol->pixel = 0;	/* none */
	       }
	  }
	pixel_colors [i] = pxcol->pixel;
     }

   TermWin.fheight = Xfont->ascent + Xfont->descent;
   TermWin.fwidth  = XTextWidth (Xfont, "MMMMMMMMMM",10) / 10;
   sizehint.height_inc = TermWin.fheight;
   sizehint.width_inc  = TermWin.fwidth;

   flags = XParseGeometry (rs_geometry, &x, &y, &width, &height);
   if (flags & WidthValue)
     {
	sizehint.width = width;
	sizehint.flags |= USSize;
     }
   if (flags & HeightValue)
     {
	sizehint.height = height;
	sizehint.flags |= USSize;
     }

   TermWin.ncol = sizehint.width;
   TermWin.nrow = sizehint.height;
   TermWin.width  = TermWin.ncol * TermWin.fwidth;
   TermWin.height = TermWin.nrow * TermWin.fheight;

#ifndef SCROLLBAR_NONE
#ifdef SCROLLBAR_ANY
   if (sbar.type != SBAR_NONE)
#endif
     sizehint.base_width += SBAR_WIDTH;
#endif	/* SCROLLBAR_NONE */

   sizehint.width  = (sizehint.width * sizehint.width_inc
		      + sizehint.base_width);
   sizehint.height = (sizehint.height * sizehint.height_inc
		      + sizehint.base_height);
   sizehint.min_width  = sizehint.width_inc  + sizehint.base_width;
   sizehint.min_height = sizehint.height_inc + sizehint.base_height;
   if (flags & XValue)
     {
	if (flags & XNegative)
	  {
	     x += (DisplayWidth (Xdisplay, Xscreen)
		   - (sizehint.width + MARGIN));
	     sizehint.win_gravity = NorthEastGravity;
	  }
	sizehint.x = x;
	sizehint.flags |= USPosition;
     }
   if (flags & YValue)
     {
	if (flags & YNegative)
	  {
	     y += (DisplayHeight (Xdisplay, Xscreen)
		   - (sizehint.height + MARGIN));
	     sizehint.win_gravity = (sizehint.win_gravity == NorthEastGravity ?
				     SouthEastGravity : SouthWestGravity);
	  }
	sizehint.y = y;
	sizehint.flags |= USPosition;
     }

   /* create the window */
   TermWin.parent = XCreateSimpleWindow (Xdisplay,
					 DefaultRootWindow (Xdisplay),
					 sizehint.x, sizehint.y,
					 sizehint.width, sizehint.height,
					 1,
					 pixel_colors [COLORFG],
					 pixel_colors [COLORBG]);

   xterm_seq (NEW_TITLE_NAME, rs_title);
   xterm_seq (NEW_ICON_NAME, rs_iconName);
   xcls.res_name = rs_name;
   xcls.res_class = APL_CLASS;
   wmhint.input = True;
   wmhint.initial_state = iconic ? IconicState : NormalState;
   wmhint.flags = InputHint | StateHint;

   XSetWMProperties (Xdisplay, TermWin.parent, NULL, NULL, argv, argc,
		     &sizehint, &wmhint, &xcls);

   XSelectInput (Xdisplay, TermWin.parent,
		 (KeyPressMask|FocusChangeMask|
		  StructureNotifyMask|VisibilityChangeMask)
		 );

#ifndef SCROLLBAR_NONE
   sbar.w = 0;
#ifdef SCROLLBAR_ANY
   if (sbar.type != SBAR_NONE)
#endif
     {
	sbar.w = (SBAR_WIDTH - 1);
	sbar.h = sizehint.height;

	sbar.sb.win = XCreateSimpleWindow (Xdisplay, TermWin.parent,
					   -1, -1,
					   sbar.w, sbar.h,
					   1,
					   pixel_colors [COLORFG],
					   pixel_colors [COLORBG]);

	/* cursor: Black-on-White */
	cursor = XCreateFontCursor (Xdisplay, XC_sb_v_double_arrow);
	XDefineCursor (Xdisplay, sbar.sb.win, cursor);

	XSelectInput (Xdisplay, sbar.sb.win,
		      (ExposureMask|ButtonPressMask|ButtonReleaseMask|
		       Button1MotionMask|Button2MotionMask|Button3MotionMask)
		      );
#ifndef SCROLLBAR_XTERM
#ifdef SCROLLBAR_ANY
	if (sbar.type == SBAR_ARROWS)
#endif	/* ANY */
	  {
	     sbar.up.win = XCreateSimpleWindow (Xdisplay, sbar.sb.win,
						0, 0,
						sbar.w, SBAR_HEIGHT,
						0,
						pixel_colors [COLORFG],
						pixel_colors [COLORBG]);

	     /* do placement in resize_window() */
	     sbar.dn.win = XCreateSimpleWindow (Xdisplay, sbar.sb.win,
						0, 0,
						sbar.w, SBAR_HEIGHT,
						0,
						pixel_colors [COLORFG],
						pixel_colors [COLORBG]);

	     /* cursor: Black-on-White */
	     cursor = XCreateFontCursor (Xdisplay, XC_sb_left_arrow);
	     XDefineCursor (Xdisplay, sbar.up.win, cursor);
	     XDefineCursor (Xdisplay, sbar.dn.win, cursor);

	     XSelectInput (Xdisplay, sbar.up.win,
			   (ExposureMask|EnterWindowMask|
			    LeaveWindowMask|ButtonPressMask));
	     XSelectInput (Xdisplay, sbar.dn.win,
			   (ExposureMask|EnterWindowMask |
			    LeaveWindowMask|ButtonPressMask));

	     XMapWindow (Xdisplay, sbar.up.win);
	     XMapWindow (Xdisplay, sbar.dn.win);

	     sbar.h -= 2 * SBAR_HEIGHT;	/* account for arrows */
	  }
#endif	/* ! SCROLLBAR_XTERM */
	XMapWindow (Xdisplay, sbar.sb.win);
     }
#endif	/* ! SCROLLBAR_NONE */

   /* correct placement now, do size in resize_window() */
   TermWin.vt = XCreateSimpleWindow (Xdisplay, TermWin.parent,
#if !defined (SCROLLBAR_NONE) && !defined (SBAR_RIGHT)
				     (sbar.w ? sbar.w + 1 : 0),
#else
				     0,
#endif
				     0, sizehint.width, sizehint.height,
				     0,
				     pixel_colors [COLORFG],
				     pixel_colors [COLORBG]);

   /* cursor: Black-on-White is normal, but this is more popular */
   cursor = XCreateFontCursor (Xdisplay, XC_xterm);
   XRecolorCursor (Xdisplay, cursor, &xcolor_fg, &xcolor_bg);
   XDefineCursor (Xdisplay, TermWin.vt, cursor);

   XSelectInput (Xdisplay, TermWin.vt,
		 (ExposureMask|ButtonPressMask|ButtonReleaseMask|
		  Button1MotionMask|Button3MotionMask));

   XMapWindow (Xdisplay, TermWin.vt);
   XMapWindow (Xdisplay, TermWin.parent);

   return;
   Abort:
   print_error ("aborting");	/* fatal problem */
   exit (EXIT_FAILURE);
}

/*
 * Redraw window
 */
void
redraw_window (unsigned int width, unsigned int height)
{
   int curr_width, curr_height;

   curr_width  = TermWin.width;
   curr_height = TermWin.height;

#ifndef SCROLLBAR_NONE
#ifdef SCROLLBAR_ANY
   if (sbar.type != SBAR_NONE)
#endif
     {
	XResizeWindow (Xdisplay, sbar.sb.win, sbar.w, height);
	width -= sbar.w;
#ifdef SBAR_RIGHT
	width--;
	XMoveWindow (Xdisplay, sbar.sb.win, width, -1);
#endif
     }
   sbar.h = height;
#ifndef SCROLLBAR_XTERM
#ifdef SCROLLBAR_ANY
   if (sbar.type == SBAR_ARROWS)
#endif
     {
	sbar.h -= 2 * (SBAR_HEIGHT - 1);
	XMoveWindow (Xdisplay, sbar.dn.win, 0, height - SBAR_HEIGHT);
     }
#endif	/* ! SCROLLBAR_XTERM */
#endif	/* SCROLLBAR_NONE */

   XResizeWindow (Xdisplay, TermWin.vt, width, height+1);
   XClearWindow (Xdisplay, TermWin.vt);
   XSync (Xdisplay, 0);
}

static void
resize_all_windows (int width, int height)
{
   int old_width  = TermWin.width;
   int old_height = TermWin.height;

   TermWin.width  = TermWin.ncol * TermWin.fwidth;
   TermWin.height = TermWin.nrow * TermWin.fheight;

#ifndef SCROLLBAR_NONE
#ifdef SCROLLBAR_ANY
   if (sbar.type != SBAR_NONE)
#endif
     {
	XResizeWindow (Xdisplay, sbar.sb.win, sbar.w, height);
	width -= sbar.w;
#ifdef SBAR_RIGHT
	width--;
	XMoveWindow (Xdisplay, sbar.sb.win, width, -1);
#endif
     }
   sbar.h = height;
#ifndef SCROLLBAR_XTERM
#ifdef SCROLLBAR_ANY
   if (sbar.type == SBAR_ARROWS)
#endif
     {
	sbar.h -= 2 * (SBAR_HEIGHT - 1);
	XMoveWindow (Xdisplay, sbar.dn.win, 0, height - SBAR_HEIGHT);
     }
#endif	/* ! SCROLLBAR_XTERM */
#endif	/* SCROLLBAR_NONE */

   XResizeWindow (Xdisplay, TermWin.vt, width, height+1);
   if (old_width)
     Gr_Resize (old_width, old_height);
   XClearWindow (Xdisplay, TermWin.vt);
   XSync (Xdisplay, 0);
}

/*
 * Redraw window after exposure or size change
 */
void
resize_window (void)
{
   static int old_ncol = -1, old_nrow = -1;
   int new_ncol, new_nrow;
   int x, y;
   unsigned int border, depth, width, height;

   Window root;
   XEvent dummy;

   while (XCheckTypedWindowEvent (Xdisplay, TermWin.parent,
				  ConfigureNotify, &dummy));
   XGetGeometry (Xdisplay, TermWin.parent,
		 &root, &x, &y, &width, &height, &border, &depth);

   new_ncol = (width  - sizehint.base_width)  / TermWin.fwidth;
   new_nrow = (height - sizehint.base_height) / TermWin.fheight;

   if ((new_ncol != old_ncol) || (new_ncol != TermWin.ncol) ||
       (new_nrow != old_nrow) || (new_nrow != TermWin.nrow) )
     {
	int curr_screen  = -1;

	/* scr_reset only works on the primary screen */
	if (old_ncol >= 0)	/* not first time thru */
	  {
	     scr_selection_clear ();
	     curr_screen = scr_change_screen (PRIMARY);
	  }

	old_ncol = new_ncol;	/* save for next time */
	old_nrow = new_nrow;

	TermWin.ncol = new_ncol;
	TermWin.nrow = new_nrow;

	resize_all_windows (width, height);
	scr_reset ();

	if (curr_screen >= 0)	/* not first time thru */
	  scr_change_screen (curr_screen);
     }
}

/*
 * good for toggling 80/132 columns
 */
void
set_width (unsigned int width)
{
   unsigned int height = TermWin.nrow;

   if (width == TermWin.ncol)
     return;

   width  = width  * TermWin.fwidth  + sizehint.base_width;
   height = height * TermWin.fheight + sizehint.base_height;

   XResizeWindow (Xdisplay, TermWin.parent, width, height);
   resize_window ();
}

/*
 * Print an error message
 */
void
print_error (const char *fmt, ...)
{
   va_list arg_ptr;

   va_start (arg_ptr, fmt);
   fprintf (stderr, "%s: ", rs_name);
   vfprintf (stderr, fmt, arg_ptr);
   fprintf (stderr, "\n");
   va_end (arg_ptr);
}

/*
 * change title/icon name
 */
void
xterm_seq (int op, char *string)
{
   XTextProperty name;

   if (!XStringListToTextProperty (&string, 1, &name))
     {
	print_error ("can't allocate title/icon name");
	return;
     }

   switch (op) {
    case NEW_NAMES:
      XSetWMName (Xdisplay, TermWin.parent, &name);	/* drop */
    case NEW_ICON_NAME:
      XSetWMIconName (Xdisplay, TermWin.parent, &name);	break;
    case NEW_TITLE_NAME:
      XSetWMName (Xdisplay, TermWin.parent, &name);	break;
   }
   XFree (name.value);
}

/*
 * Switch to a new font
 */
#if (NFONTS > 1)
void
new_font (int dirn)
{
   static int font_num = NORM_FONT;
   int num;

   switch (dirn) {
    case UP:
      font_num++;
      if (font_num >= NFONTS)
	{
	   font_num = (NFONTS - 1);
	   return;
	}
      break;

    case DOWN:
      font_num--;
      if (font_num < 0)
	{
	   font_num = 0;
	   return;
	}
      break;

    default:
      return;
      break;
   }

   /* take case of shuffling font numbers */

   num = font_num;
   if (num == NORM_FONT)	/* position of the normal font */
     num = 0;
   else if (num < NORM_FONT)
     num++;

   XFreeFont (Xdisplay, Xfont);
     {
	char *name = rs_font [num];
	Xfont = XLoadQueryFont (Xdisplay, name);
	if (!Xfont)
	  {
	     print_error ("can't load font \"%s\"", name);
	     Xfont = XLoadQueryFont (Xdisplay, rs_font [0]);
	     rs_font [num] = rs_font [0];
	  }
     }
   XSetFont (Xdisplay, Xgc, Xfont->fid);
   XSetFont (Xdisplay, Xrvgc, Xfont->fid);

#ifdef KANJI
   XFreeFont (Xdisplay, Kanjifont);
     {
	char *name = rs_kfont [num];
	Kanjifont = XLoadQueryFont (Xdisplay, name);
	if (!Kanjifont)
	  {
	     print_error ("can't load kfont \"%s\"", name);
	     Kanjifont = XLoadQueryFont (Xdisplay, rs_kfont [0]);
	     rs_kfont [num] = rs_kfont [0];
	  }
     }
#endif
   sizehint.width_inc  = XTextWidth (Xfont, "MMMMMMMMMM", 10) / 10;
   sizehint.height_inc = Xfont->ascent + Xfont->descent;
   if ((TermWin.fwidth == sizehint.width_inc) &&
       (TermWin.fheight == sizehint.height_inc))
     return;

   TermWin.fwidth  = sizehint.width_inc;
   TermWin.fheight = sizehint.height_inc;

   TermWin.width  = TermWin.fwidth  * TermWin.ncol;
   TermWin.height = TermWin.fheight * TermWin.nrow;

   sizehint.min_width  = sizehint.width_inc  + sizehint.base_width;
   sizehint.min_height = sizehint.height_inc + sizehint.base_height;

   sizehint.width  = TermWin.width  + sizehint.base_width;
   sizehint.height = TermWin.height + sizehint.base_height;

   sizehint.flags = PMinSize|PResizeInc|PBaseSize|PWinGravity;

   XSetWMNormalHints (Xdisplay, TermWin.parent, &sizehint);
   XResizeWindow (Xdisplay, TermWin.parent, sizehint.width, sizehint.height);

   resize_all_windows (sizehint.width, sizehint.height);
}
#endif
/*----------------------- end-of-file (C source) -----------------------*/
