/* 
 * MapProject.c
 */

#include "MapProject.h"
#include "MapP.h"

#include <limits.h>
#include <math.h>

#define d_asin(x) (fabs (x) >= 1.0 ? copysign(M_PI_2,(x)) : asin(x)) 
#define SMALL 1.0e-4

static double
DegToRad(double deg)
{
  return deg * M_PI / 180.0;
}

static double
RadToDeg(double rad)
{
  return rad * 180.0 / M_PI;
}

static void
ProjectCylEqdist(float lat, float lon, double* x, double* y)
{
  *x = DegToRad(lon);
  *y = DegToRad(lat);
}

static Boolean
InvProjectCylEqdist(double x, double y, float* lat, float* lon)
{
  *lat = RadToDeg(y);
  *lon = RadToDeg(x);

  return True;
}

static void
ProjectEckertVI(float lat, float lon, double* x, double* y)
{
  double delta;
  double phi = lat = DegToRad(lat);
  int n_iter = 0;

  do
    {
      delta = -(phi + sin(phi) - (1.0 + M_PI_2) * sin(lat)) / (1.0 + cos(phi));
      phi += delta;
    }
  while (delta > SMALL && n_iter++ < 100);

  *x = DegToRad(lon) * (1.0 + cos(phi)) / 2.0;
  *y = phi;
}

static Boolean
InvProjectEckertVI(double x, double y, float* lat, float* lon)
{
  double phi = y;

  if (fabs(x *= 2.0 / (1.0 + cos(phi))) > M_PI)
    return FALSE;
  else
  {
    *lat = RadToDeg(d_asin((phi + sin(phi)) / (1.0 + M_PI_2)));
    *lon = RadToDeg(x);

    return True;
  }
}

static void
ProjectHammer(float lat, float lon, double* x, double* y)
{
  double clat;
  double D;

  lat = DegToRad(lat);
  clat = cos(lat);

  lon = DegToRad(lon) / 2.0;

  D = sqrt(2.0 / (1.0 + clat * cos(lon)));
  *x = 2.0 * D * clat * sin(lon);
  *y = D * sin(lat);
}

static Boolean
InvProjectHammer(double x, double y, float* lat, float* lon)
{
  double rho = hypot(x *= 0.5, y);
  double c = 2.0 * d_asin(rho / 2.0);

  if (fabs(c) > M_PI_2)
    return FALSE;
  else if (rho == 0.0)
  {
    *lat = 0.0;
    *lon = 0.0;
  }
  else
  {
    *lat = RadToDeg(d_asin(y * sin(c) / rho));

    if (fabs(c - M_PI_2) < SMALL)
      *lon = (x == 0.0) ? 0.0 : copysign(180.0, x);
    else
      *lon = RadToDeg(2.0 * atan(x * tan(c) / rho));
  }

  return True;
}

#if 0
static Boolean
ProjectMercator(float *y, float *x, float lat, float lon,
	      float a, float lambda0)
{
  float h, k;

  *x = a * (lon - lambda0);
  *y = a * atanh(sin(lat));
  h = k = 1 / cos(lat);

  return 1;
}

static Boolean
InvProjectMercator(float *lat, float *lon, float y, float x,
		 float a, float lambda0)
{
  const pio2 = 1.5707963267949;

  *lat = pio2 - 2*atan(exp(y / a));
  *lon = x/a + lambda0;

  return 1;
}

static Boolean
ProjectMiller(float *y, float *x, float lat, float lon,
	      float a, float lambda0)
{
  float h, k;

  *x = a * (lon - lambda0);
  *y = a * atanh(sin(0.8*lat)) / 0.8;
  h = 1 / cos(0.8*lat);
  k = 1 / cos(lat);

  return 1;
}

static Boolean
InvProjectMiller(float *lat, float *lon, float y, float x,
		 float a, float lambda0)
{
  *lat = 2.5 * atan(exp(0.8 * y/ a)) - 1.9634954; /* 5pi/8 */
  *lon = x/a - lambda0;

  return 1;
}
#endif

static void
ProjectMercator(float lat, float lon, double *x, double *y)
{

}

static Boolean
InvProjectMercator(double x, double y, float* lat, float* lon)
{
  return True;
}

static void
ProjectMiller(float lat, float lon, double* x, double* y)
{
  *x = DegToRad(lon);
  *y = log(tan(M_PI_4 + DegToRad(lat) * 0.4)) * 0.85242889474565752828;
}

static void
NewProjectMiller(float lat, float lon, float center_lon, double *x, double *y)
{
  *x = DegToRad(lon) - DegToRad(center_lon);
  *y = log(tan(M_PI_4 + DegToRad(lat) * 0.4)) * 0.85242889474565752828;
}

static Boolean
InvProjectMiller(double x, double y, float* lat, float* lon)
{
  /* 1.17... = log(tan(M_PI_4 + M_PI_2 * 0.4)) / M_PI_2 */
  *lat = RadToDeg(2.5 * (atan(exp(1.17311837522632766984 * y)) - M_PI_4));
  *lon = RadToDeg(x);

  return True;
}

static Boolean
NewInvProjectMiller(double x, double y, float center_lon,
		    float *lat, float *lon)
{
  /* 1.17... = log(tan(M_PI_4 + M_PI_2 * 0.4)) / M_PI_2 */
  *lat = RadToDeg(2.5 * (atan(exp(1.17311837522632766984 * y)) - M_PI_4));
  *lon = RadToDeg(x) - DegToRad(center_lon);

  return True;
}

static void
ProjectMollweide(float lat, float lon, double* x, double* y)
{
  if (fabs(lat) == 90.0)
  {
    *x = 0.0;
    *y = lat / 90.0;
  }
  else
  {
    double delta;
    double phi = lat = DegToRad(lat);
    int i = 0;

    do
      {
	delta = -(phi + sin(phi) - M_PI * sin(lat)) / (1.0 + cos(phi));
	phi += delta;
      }
    while (fabs(delta) > 1.0e-7 && i++ < 100);

    phi *= 0.5;
    *x = DegToRad(lon * cos(phi));
    *y = sin(phi);
  }

  *y *= M_PI_2;
}

static Boolean
InvProjectMollweide(double x, double y, float* lat, float* lon)
{
  double phi = 2.0 * d_asin(y / M_PI_2);

  if (fabs(x /= cos(phi / 2.0)) > M_PI)
    return FALSE;
  else
  {
    *lat = RadToDeg(d_asin((phi + sin(phi)) / M_PI));
    *lon = RadToDeg(x);

    return True;
  }
}

static void
ProjectSinusoidal(float lat, float lon, double* x, double* y)
{
  lat = DegToRad(lat);
  lon = DegToRad(lon);

  *x = lon * cos(lat);
  *y = lat;
}

static Boolean
InvProjectSinusoidal(double x, double y, float* lat, float* lon)
{
  if (fabs(y) == M_PI_2)
  {
    if (x != 0.0)
      return FALSE;

    *lat = 90.0;
    *lon = 0.0;
  }
  else
  {
    x /= cos(y);
    if (fabs(x) > M_PI)
      return FALSE;

    *lat = RadToDeg(y);
    *lon = RadToDeg(x);
  }

  return True;
}

Boolean
MapGPointToXPoint(MapWidget mw, const XmMapGPoint* gpt, XPoint* xp)
{
  double deltaLat = M_PI / mw->map.zoom_factor;
  double deltaLon = 2.0 * M_PI / mw->map.zoom_factor;
  double latB = M_PI_2 / mw->map.zoom_factor + mw->map.zoom_center_y;
  double lonB = -M_PI / mw->map.zoom_factor + mw->map.zoom_center_x;
  double latP;
  double lonP;

  switch(mw->map.projection)
  {
    case XmCYL_EQDIST_PROJECTION:
	ProjectCylEqdist(gpt->lat, gpt->lon, &lonP, &latP);
	break;

    case XmECKERT_VI_PROJECTION:
	ProjectEckertVI(gpt->lat, gpt->lon, &lonP, &latP);
	break;

    case XmHAMMER_PROJECTION:
	ProjectHammer(gpt->lat, gpt->lon, &lonP, &latP);
	break;

    case XmMERCATOR_PROJECTION:
	ProjectMiller(gpt->lat, gpt->lon, &lonP, &latP);
	break;

    case XmMILLER_PROJECTION:
      /* ProjectMiller(gpt->lat, gpt->lon, &lonP, &latP); */
	NewProjectMiller(gpt->lat, gpt->lon, mw->map.center_lon, &lonP, &latP);
	break;

    case XmMOLLWEIDE_PROJECTION:
	ProjectMollweide(gpt->lat, gpt->lon, &lonP, &latP);
	break;

    case XmSINUSOIDAL_PROJECTION:
	ProjectSinusoidal(gpt->lat, gpt->lon, &lonP, &latP);
	break;

    default:
	latP = 0.0;
	lonP = 0.0;
  }

  /* Calculate the XPoint with respect to the zoom */
  latP = (latB - latP) / deltaLat * mw->core.height;
  lonP = (lonP - lonB) / deltaLon * mw->core.width;

  /* Make sure that the value is within the XPoint range */
  if (latP < SHRT_MIN)
    xp->y = SHRT_MIN;
  else if (latP > SHRT_MAX)
    xp->y = SHRT_MAX;
  else
    xp->y = latP;

  if (lonP < SHRT_MIN)
    xp->x = SHRT_MIN;
  else if (lonP > SHRT_MAX)
    xp->x = SHRT_MAX;
  else
    xp->x = lonP;

  return (xp->x >= 0 && xp->x <= (short) mw->core.width
	  && xp->y >= 0 && xp->y <= (short) mw->core.height);
}

Boolean
MapXPointToGPoint(MapWidget mw, const XPoint* xp, XmMapGPoint* gpt)
{
  Boolean status;
  double delta_lat = M_PI / mw->map.zoom_factor;
  double delta_lon = 2.0 * M_PI / mw->map.zoom_factor;
  double lat_b = M_PI_2 / mw->map.zoom_factor + mw->map.zoom_center_y;
  double lon_b = -M_PI / mw->map.zoom_factor + mw->map.zoom_center_x;
  double lat_p = lat_b - xp->y * delta_lat / (double) mw->core.height;
  double lon_p = lon_b + xp->x * delta_lon / (double) mw->core.width;;

  switch (mw->map.projection)
  {
    case XmCYL_EQDIST_PROJECTION:
	status = InvProjectCylEqdist(lon_p, lat_p, &gpt->lat, &gpt->lon);
	break;

    case XmECKERT_VI_PROJECTION:
	status = InvProjectEckertVI(lon_p, lat_p, &gpt->lat, &gpt->lon);
	break;

    case XmHAMMER_PROJECTION:
	status = InvProjectHammer(lon_p, lat_p, &gpt->lat, &gpt->lon);
	break;

    case XmMERCATOR_PROJECTION:
	status = InvProjectMercator(lon_p, lat_p, &gpt->lat, &gpt->lon);
	break;

    case XmMILLER_PROJECTION:
	status = NewInvProjectMiller(lon_p, lat_p, mw->map.center_lon,
				     &gpt->lat, &gpt->lon);
	break;

    case XmMOLLWEIDE_PROJECTION:
	status = InvProjectMollweide(lon_p, lat_p, &gpt->lat, &gpt->lon);
	break;

    case XmSINUSOIDAL_PROJECTION:
	status = InvProjectSinusoidal(lon_p, lat_p, &gpt->lat, &gpt->lon);
	break;

    default:
        status = False;
	break;
  }

  return status;
}
