/* worldmap - a world map display program
   Copyright (C) 1998 Andrew Skypeck
  
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
  
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
  
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * rpc-server.c
 */

#include <stdio.h>
#include <rpc/rpc.h>
#include <X11/Intrinsic.h>

#include "xtutil.h"
#include "Map.h"
#include "rmap.h"
#include "rmapxdr.h"

static SVCXPRT *xprt;
static XtInputId input_id;
static Widget map;

static void
SocketInputCallback(caddr_t client_data, int *source, XtInputId *id)
{
  fd_set readfds;

  if (*id != input_id || *source != xprt->xp_sock)
    {
      fprintf(stderr,
	      "SocketInputCallback(): unexpected input from who knows where");
      exit(1);
    }

  readfds = svc_fdset;
  svc_getreqset(&readfds);	/* calls the dispatcher */
}

/* convert a high level RMap network gc to a real x gc on the map server */
static void
ConvertGC(RMapGCValues rmgcv, unsigned long mask, XGCValues *xgcv)
{
  /* gcd.values is only a high level representation of
     xgcvalues.  we must first do some converting before we
     can use them*/
  memset((char *) xgcv, 0, sizeof(XGCValues));

  /* this little ditty was taken crom X11R6.3:CrGC.c.  It sets up the
     mask for the nice comparison expressions */
  mask &= (1L << (GCLastBit + 1)) - 1;

  if (mask & GCFunction)
    xgcv->function = rmgcv.function;
	
  if (mask & GCPlaneMask)
    xgcv->plane_mask = rmgcv.plane_mask;

  if (mask & GCForeground)
    xgcv->foreground = \
      (unsigned long) ColorNameToPixel((Widget) map, rmgcv.foreground);

  if (mask & GCBackground)
    xgcv->background = \
      (unsigned long) ColorNameToPixel((Widget) map, rmgcv.background);

  if (mask & GCLineWidth)
    xgcv->line_width = rmgcv.line_width;

  if (mask & GCLineStyle)
    xgcv->line_style = rmgcv.line_style;

  if (mask & GCCapStyle)
    xgcv->cap_style  = rmgcv.cap_style;
    
  if (mask & GCJoinStyle)
    xgcv->join_style = rmgcv.join_style;

  if (mask & GCFillStyle)
    xgcv->fill_style = rmgcv.fill_style;

  if (mask & GCFillRule)
    xgcv->fill_rule  = rmgcv.fill_rule;

  /*
  if (mask & GCArcMode)
    if (mask & GCTile)
      if (mask & GCStipple) {
	if (mask & GCTileStipXOrigin)
	  if (mask & GCTileStipYOrigin)
  */
  if (mask & GCFont)
    xgcv->font = FontNameToFont((Widget) map, rmgcv.font);
  /*
    if (mask & GCSubwindowMode)
    if (mask & GCGraphicsExposures)
    if (mask & GCClipXOrigin)
    if (mask & GCClipYOrigin)
    if (mask & GCClipMask)
    if (mask & GCDashOffset)
    if (mask & GCDashList)
  */
}

/* Decode the requested service and provide it. This is a synchronous
   dispatcher, but there's no reason why you couldn't use the async.
   rls_svc.c dispatcher to avoid prolonged RPC client blocking. */
static void
Dispatch(struct svc_req *request, SVCXPRT *transport)
{
  RMapGPointArray gpa;
  RMapStringArray sa;
  RMapMarkerArray ma;
  RMapGCDesc gcd;

  switch (request->rq_proc)
    {
    case NULLPROC:
      (void) svc_sendreply(transport, xdr_void, 0);
      break;

    case CLEAR:
      XmMapClear(map);
      if (!svc_sendreply(transport, xdr_void, 0))
	{
	  svcerr_systemerr(transport);
	  return;
	}
      break;

    case FREEZE:
      XmMapFreeze(map);
      if (!svc_sendreply(transport, xdr_void, 0))
	{
	  svcerr_systemerr(transport);
	  return;
	}
      break;

    case DRAW_POINTS:
      (void) memset((char *) &gpa, 0, sizeof(gpa));
      if (!svc_getargs(transport, (xdrproc_t) xdr_RMapGPointArray,
		       (char *) &gpa))
	svcerr_decode(transport);
      else
	{
	  XmMapDrawPoints(map, (XmMapGPoint *) gpa.pts, (Cardinal) gpa.npts);

	  if (!svc_sendreply(transport, xdr_void, 0))
	    {
	      svcerr_systemerr(transport);
	      return;
	    }
	}
      break;

    case DRAW_MARKERS:
      (void) memset((char *) &ma, 0, sizeof(ma));
      if (!svc_getargs(transport, (xdrproc_t) xdr_RMapMarkerArray,
		       (char *) &ma))
	svcerr_decode(transport);
      else
	{
	  XmMapDrawMarkers(map, (XmMapGPoint *) ma.pts, (Cardinal) ma.npts,
			   (XmMarkerType) ma.marker_type,
			   (Cardinal) ma.marker_width);

	  if (!svc_sendreply(transport, xdr_void, 0))
	    {
	      svcerr_systemerr(transport);
	      return;
	    }
	}
      break;

    case DRAW_LINES:
      (void) memset((char *) &gpa, 0, sizeof(gpa));
      if (!svc_getargs(transport, (xdrproc_t) xdr_RMapGPointArray,
		       (char *) &gpa))
	svcerr_decode(transport);
      else
	{
	  XmMapDrawLines(map, (XmMapGPoint *) gpa.pts, (Cardinal) gpa.npts);

	  if (!svc_sendreply(transport, xdr_void, 0))
	    {
	      svcerr_systemerr(transport);
	      return;
	    }
	}
      break;

    case DRAW_STRINGS:
      (void) memset((char *) &sa, 0, sizeof(sa));
      if (!svc_getargs(transport, (xdrproc_t) xdr_RMapStringArray,
		       (char *) &sa))
	svcerr_decode(transport);
      else
	{
	  int i;
	  String *s = (String *) XtMalloc(sa.npts * sizeof(String *));

          /* APS - I would like to just pass &(sa.str) but this points
             to char *[40] not char **.  Unless I can figure out how
             to send ragged arrays through xdr we are stuck with
             duplicating. yuk! */
	  for (i = 0; i < sa.npts; i++)
	    s[i] = XtNewString(sa.str[i]);

	  XmMapDrawStrings(map, (XmMapGPoint *) sa.pts, s, (Cardinal) sa.npts);

	  for (i = 0; i < sa.npts; i++)
	    XtFree((char *) s[i]);
	  XtFree((char *) s);

	  if (!svc_sendreply(transport, xdr_void, 0))
	    {
	      svcerr_systemerr(transport);
	      return;
	    }
	}
      break;

    case FILL_POLYGON:
      (void) memset((char *) &gpa, 0, sizeof(gpa));
      if (!svc_getargs(transport, (xdrproc_t) xdr_RMapGPointArray,
		       (char *) &gpa))
	svcerr_decode(transport);
      else
	{
	  XmMapFillPolygon(map, (XmMapGPoint *) gpa.pts, (Cardinal) gpa.npts);

	  if (!svc_sendreply(transport, xdr_void, 0))
	    {
	      svcerr_systemerr(transport);
	      return;
	    }
	}
      break;

    case CHANGE_GC:
      (void) memset((char *) &sa, 0, sizeof(sa));
      if (!svc_getargs(transport, (xdrproc_t) xdr_RMapGCDesc,
		       (char *) &gcd))
	svcerr_decode(transport);
      else
	{
	  XGCValues xgcv;

	  /* we must translate the high level gc values to actual gc
             values */
	  ConvertGC(gcd.values, gcd.mask, &xgcv);
	  XChangeGC(XtDisplay(map), XmMapGC(map), gcd.mask, &xgcv);

	  if (!svc_sendreply(transport, xdr_void, 0))
	    {
	      svcerr_systemerr(transport);
	      return;
	    }
	}
      break;

    default:
      svcerr_noproc(transport);
      break;
    }

  return;
}

void
RPCRegisterServer()
{
  /* Establish the server daemon */
  (void) pmap_unset(PROGNUM, VERSION);
  xprt = (SVCXPRT *) svcudp_create(RPC_ANYSOCK);
  if (xprt == NULL)
    {
      fprintf(stderr, "RPCRegisterServer(): cannot create udp service\n");
      exit(1);
    }
  if (!svc_register(xprt, PROGNUM, VERSION, Dispatch, IPPROTO_UDP))
    {
      fprintf(stderr,
	      "RPCRegisterServer(): unable to register (PROGNUM, VERSION, udp).\n");
      exit(1);
    }
}

void
RPCAddSocketInput(XtAppContext app_context, Widget w)
{
  /* Before starting the dispatch loop, tell it to look at the server
     socket information in fd_set to notify us of RPC requests. */
  input_id = \
    XtAppAddInput(app_context, xprt->xp_sock,
		  (XtPointer) XtInputReadMask,
		  (XtInputCallbackProc) SocketInputCallback,
		  (XtPointer) NULL);

  /* record a global reference to the map widget the server will issue
     requests to */
  map = w;
}
