/*
 * rmapguile.c - guile interpreter that is aware of rmap extensions
 */

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>

/* the C to scheme primatives */
#include <guile/gh.h>

/* remote map drawing functions */
#include <rpc/rpc.h>
#include "rmap.h"

#define SCM2FLOAT(_s) ((float) gh_scm2double(_s))
#define SCM2INT(_s) ((int) gh_scm2long(_s))

static void
warning(char *msg)
{
  fprintf(stderr, "rmapguile: warning: %s\n", msg);
}

static void
error(char *msg)
{
  fprintf(stderr, "rmapguile: error: %s\n", msg);
  exit(1);
}

char *
xmalloc(size_t size)
{
  register char *value = malloc (size);
  if (value == 0)
    error("virtual memory exhausted");
  return value;
}

void
xfree(char *val)
{
  if (val != NULL)
    free(val);
}

/* convert a scheme object (either string or integer) to an XFont */
static SCM
c_RMapClear(SCM shost)
{
  char *host;
  int len;

  gh_defer_ints();
  host = gh_scm2newstr(shost, &len);
  RMapClear(host);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_RMapFreeze(SCM shost)
{
  char *host;
  int len;

  gh_defer_ints();
  host = gh_scm2newstr(shost, &len);
  RMapFreeze(host);
  gh_allow_ints();

  return (SCM) 0;
}

/* draw a vector of lists of length 2 */
static SCM
c_RMapDrawPoints(SCM shost, SCM pv)
{
  char *host;
  RMapGPoint *gp;
  int i, n, m;
  enum clnt_stat clnt_stat;

  n = (int) gh_vector_length(pv);

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  host = gh_scm2newstr(shost, &m);  
  gp = (RMapGPoint *) xmalloc(n * sizeof(RMapGPoint));
  for (i = 0; i < n; i++)
    {
      SCM ls;

      ls = gh_vref(pv, gh_long2scm((long) i));
      gp[i].lat = SCM2FLOAT(gh_car(ls));
      gp[i].lon = SCM2FLOAT(gh_cadr(ls));
    }

  clnt_stat = RMapDrawPoints(host, gp, n);
  if (clnt_stat != RPC_SUCCESS)
    error(clnt_sperrno(clnt_stat));

  xfree((char *) gp);
  xfree((char *) host);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_RMapDrawMarkers(SCM shost, SCM stype, SCM swidth, SCM ls)
{
  char *host;
  RMapMarkerType marker_type = (RMapMarkerType) SCM2INT(stype);
  int marker_size = SCM2INT(swidth);
  RMapGPoint *gp;
  unsigned long n = gh_list_length(ls);
  int i, m;
  enum clnt_stat clnt_stat;

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  host = gh_scm2newstr(shost, &m);  
  gp = (RMapGPoint *) xmalloc(n * sizeof(RMapGPoint));

  for (i = 0; i < n; i++)
    {
      SCM lls;

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      ls = gh_cdr(ls);
    }

  clnt_stat = RMapDrawMarkers(host, gp, n, marker_type, marker_size);
  if (clnt_stat != RPC_SUCCESS)
    error(clnt_sperrno(clnt_stat));

  xfree((char *) gp);
  xfree((char *) host);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_RMapDrawStrings(SCM shost, SCM ls)
{
  char *host;
  RMapGPoint *gp;
  RMapString *s;
  unsigned long n = gh_list_length(ls);
  int i, m;
  enum clnt_stat clnt_stat;

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  host = gh_scm2newstr(shost, &m);  
  gp = (RMapGPoint *) xmalloc(n * sizeof(RMapGPoint));
  s = (RMapString *) xmalloc(n * sizeof(RMapString));
  memset((char *) s, 0, n * sizeof(RMapString));
  
  /* loop through the list of points and strings converting to c as we
     go */
  for (i = 0; i < n; i++)
    {
      SCM lls;			/* this is the inner list, ls is outer */
      char *t;			/* temporary string */

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      t = gh_scm2newstr(gh_caddr(lls), &m);
      strcpy(s[i], t);
      xfree(t);
      ls = gh_cdr(ls);
    }

  /*  the call should be all set up */
  clnt_stat = RMapDrawStrings(host, gp, s, n);
  if (clnt_stat != RPC_SUCCESS)
    error(clnt_sperrno(clnt_stat));

  /* now free everything that was malloced */
  xfree((char *) gp);
  for (i = 0; i < n; i++)
    xfree((char *) s[i]);
  xfree((char *) s);
  xfree((char *) host);
  gh_allow_ints();

  return (SCM) 0;
}

/* draw a vector of lists of length 2 */
static SCM
c_RMapDrawLines(SCM shost, SCM ls)
{
  char *host;
  RMapGPoint *gp;
  unsigned long n = gh_list_length(ls);
  int i, m;
  enum clnt_stat clnt_stat;

  if (n == 0) return (SCM) 0;

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  host = gh_scm2newstr(shost, &m);
  gp = (RMapGPoint *) xmalloc(n * sizeof(RMapGPoint));
  for (i = 0; i < n; i++)
    {
      SCM lls;

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      ls = gh_cdr(ls);
    }

  clnt_stat = RMapDrawLines(host, gp, n);
  if (clnt_stat != RPC_SUCCESS)
    error(clnt_sperrno(clnt_stat));

  xfree((char *) gp);
  xfree((char *) host);
  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_RMapFillPolygon(SCM shost, SCM ls)
{
  char *host;
  RMapGPoint *gp;
  unsigned long n = gh_list_length(ls);
  int i, m;
  enum clnt_stat clnt_stat;

  if (n == 0) return (SCM) 0;

  /* prevent GC while we use dynamic vars */
  gh_defer_ints();
  host = gh_scm2newstr(shost, &m);
  gp = (RMapGPoint *) xmalloc(n * sizeof(RMapGPoint));
  for (i = 0; i < n; i++)
    {
      SCM lls;

      lls = gh_car(ls);
      gp[i].lat = SCM2FLOAT(gh_car(lls));
      gp[i].lon = SCM2FLOAT(gh_cadr(lls));
      ls = gh_cdr(ls);
    }

  clnt_stat = RMapFillPolygon(host, gp, n);
  if (clnt_stat != RPC_SUCCESS)
    error(clnt_sperrno(clnt_stat));

  xfree((char *) gp);
  xfree((char *) host);
  gh_allow_ints();

  return (SCM) 0;
}

static void
GetGCValuesFromSCMArgs(SCM args, RMapGCValues *gcv, unsigned long *mask)
{
  SCM sbit, svalue;
  unsigned long bit, list_len;

  *mask = 0;
  memset((char *) gcv, 0, sizeof(RMapGCValues));

  if ((list_len = gh_list_length(args)) == 0) return;

  while (list_len)
    {
      char *str;
      int str_len;		/* string argument length */
      sbit = gh_car(args);
      svalue = gh_cadr(args);
      args = gh_cddr(args);

      bit = gh_scm2ulong(sbit);

      *mask |= bit;
      switch (bit)
	{
	case RMapGCFunction:  /* int */
	  gcv->function = SCM2INT(svalue);
	  break;

	case RMapGCPlaneMask: /* unsigned long */
	  gcv->plane_mask = gh_scm2ulong(svalue);
	  break;

	case RMapGCForeground: /* string */
	  str = gh_scm2newstr(svalue, &str_len);
	  strncpy(gcv->foreground, str, str_len);
	  xfree(str);
	  break;

	case RMapGCBackground: /* long */
	  str = gh_scm2newstr(svalue, &str_len);
	  strncpy(gcv->background, str, str_len);
	  xfree(str);
	  break;

	case RMapGCLineWidth: /* int */
	  gcv->line_width = SCM2INT(svalue);
	  break;

	case RMapGCLineStyle: /* int */
	  gcv->line_style = SCM2INT(svalue);
	  break;

	case RMapGCFillStyle: /* int */
	  gcv->fill_style = SCM2INT(svalue);
	  break;

	case RMapGCStipple:
	  str = gh_scm2newstr(svalue, &str_len);
	  strncpy(gcv->stipple, str, str_len);
	  xfree(str);
	  break;

	case RMapGCFont:
	  str = gh_scm2newstr(svalue, &str_len);
	  strncpy(gcv->font, str, str_len);
	  xfree(str);
	  break;

	default:
	  {
	    char s[80];
	    sprintf(s, "invalid GC flag `%d' would crash map server", bit);
	    error(s);
	  }
	  break;
	}

      list_len -= 2;
    }
}

static SCM
c_RMapChangeGC(SCM shost, SCM args)
{
  char *host;
  RMapGCValues gcvalues;
  unsigned long mask = 0; 
  int m;
  enum clnt_stat clnt_stat;

  gh_defer_ints();
  host = gh_scm2newstr(shost, &m);
  GetGCValuesFromSCMArgs(args, &gcvalues, &mask);

  clnt_stat = RMapChangeGC(host, gcvalues, mask);  
  if (clnt_stat != RPC_SUCCESS)
    error(clnt_sperrno(clnt_stat));

  gh_allow_ints();

  return (SCM) 0;
}

static SCM
c_sleep(SCM s_sec)
{
  if (gh_exact_p(s_sec))
    sleep((unsigned int) gh_scm2long(s_sec));

  return (SCM) 0;
}

int
load_rmap()
{
#include "rmap-defs.h"

  gh_eval_str(gc_defs);

  /* install the new scheme functions and set the target widget */
  gh_new_procedure("rmap-clear", c_RMapClear, 1, 0, 0);
  gh_new_procedure("rmap-freeze", c_RMapFreeze, 1, 0, 0);
  gh_new_procedure("rmap-change-gc", c_RMapChangeGC, 2, 0, 0);
  gh_new_procedure("rmap-draw-points", c_RMapDrawPoints, 2, 0, 0);
  gh_new_procedure("rmap-draw-markers", c_RMapDrawMarkers, 4, 0, 0);
  gh_new_procedure("rmap-draw-strings", c_RMapDrawStrings, 2, 0, 0);
  gh_new_procedure("rmap-draw-lines", c_RMapDrawLines, 2, 0, 0);
  gh_new_procedure("rmap-fill-polygon", c_RMapFillPolygon, 2, 0, 0);
  /* gh_new_procedure("rmap-sync", c_RMapSync, 0, 0, 0); */
  gh_new_procedure("sleep", c_sleep, 1, 0, 0);
}

void
usage()
{
  fprintf(stderr, "usage: rmapguile file1 [file2] ...");
  exit(1);
}

void
main_func(int argc, char *argv[])
{
  int i;
  /* define the new rmap procedures to the guile interpreter */
  load_rmap();
  if (argc < 2)
    usage();

  for (i = 1; i < argc; i++)
    gh_eval_file(argv[i]);
}

int
main(int argc, char *argv[])
{
  /* yes, this is funky.  guile needs to take over the main function
     for the purposes of garbage collection */
  gh_enter(argc, argv, (void (*)(int, char *[])) main_func);

  exit(0);
}
