/********************************************************************\
 * Account.h -- the Account data structure                          *
 * Copyright (C) 1997 Robin D. Clark                                *
 *                                                                  *
 * This program is free software; you can redistribute it and/or    *
 * modify it under the terms of the GNU General Public License as   *
 * published by the Free Software Foundation; either version 2 of   *
 * the License, or (at your option) any later version.              *
 *                                                                  *
 * This program is distributed in the hope that it will be useful,  *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of   *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the    *
 * GNU General Public License for more details.                     *
 *                                                                  *
 * You should have received a copy of the GNU General Public License*
 * along with this program; if not, write to the Free Software      *
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.        *
 *                                                                  *
 *   Author: Rob Clark                                              *
 * Internet: rclark@cs.hmc.edu                                      *
 *  Address: 609 8th Street                                         *
 *           Huntington Beach, CA 92648-4632                        *
\********************************************************************/

#ifndef __ACCOUNT_H__
#define __ACCOUNT_H__

#include "config.h"

#include "QuickFill.h"
#include "Transaction.h"

/* The account types */
/* Note: the actual values of these are *very* important, 
 * as it is the values, not the enums, that are stored 
 * in the file format! */
enum {
  BANK = 0,
  CASH = 1,
  ASSET = 2,
  CREDIT = 3,
  LIABILITY = 4,
  STOCK = 5,
  MUTUAL= 6, 
  INCOME = 7,
  EXPENSE = 8,
  EQUITY = 9,
  NUM_ACCOUNT_TYPES = 10
};

/* the english-language names here should match 
 * the enumerated types above */
extern char *account_type_name[];

/** STRUCTS *********************************************************/
typedef struct _Account {
  /* public data, describes account */

  struct _account_group *parent;     /* back-pointer to parent */
  struct _account_group *children;   /* pointer to sub-accounts */
  int       id;                      /* unique account id, internally assigned */
  char      flags;
  short     type;
  String    accountName;
  String    description;
  String    notes;


  /* protected data, cached parameters */
  double balance;
  double cleared_balance;
  double reconciled_balance;

  double running_balance;
  double running_cleared_balance;
  double running_reconciled_balance;

  int         numTrans;
  Transaction **transaction;    /* pointer to array of pointers to transactions */

  /* private data, controls main window display */
  Widget arrowb;  /* arrow button in the main window */
  short expand;   /* expand display of subaccounts in main window */
  int PreviousArrowReason; /* arrow workaround */

  /* private data, controls register window display and editing */
  struct _RegWindow       *regData;       /* the register window if open, or NULL    */
  struct _RegWindow       *regLedger;     /* the ledger window if open, or NULL    */
  struct _RegWindow       **ledgerList;   /* zero or more ledger windows             */
  struct _RecnWindow      *recnData;      /* the reconcile window, if open, or NULL  */
  struct _AdjBWindow      *adjBData;      /* the adjust-balance window, or NULL      */
  struct _editaccwindow   *editAccData;   /* the edit account-name window, or NULL   */
  struct _editnoteswindow *editNotesData; /* the edit-account-notes window, or NULL  */
  QuickFill               *qfRoot;        /* the quickfill tree for this Account     */
} Account;

/** PROTOTYPES ******************************************************/

Account     *mallocAccount( void );
void         freeAccount( Account * );
Transaction *getTransaction( Account *, int );
void         xaccRemoveTransaction( Account *, Transaction * );
int          insertTransaction( Account *, Transaction * );
int          getNumOfTransaction( Account *, Transaction *);

int          xaccGetAccountID (Account *);


/* The following two functions set and get the amount on the transaction 
 * The account argument is needed to determine if this is a credit
 * or a debit, since this is a double-entry system */
double       xaccGetAmount (Account *, Transaction *);
double       xaccGetShareAmount (Account *, Transaction *);
void         xaccSetAmount (Account*, Transaction *, double);
void         xaccSetShareAmount (Account*, Transaction *, double);

/* Get the other account to which a transaction belongs to */
Account      * xaccGetOtherAccount( Account *acc, Transaction *trans );

/* the following recompute the partial balances (stored with the transaction)
 * and the total balance, for this account */
void         xaccRecomputeBalance (Account *);
void         xaccRecomputeBalances (Account **);

/* The following four subroutines return the running balance up 
 * to & including the indicated transaction. 
 *
 * The balance is the currency-denominated balance.  For accounts
 * with non-unit share prices, it is correctly adjusted for
 * share prices. 
 *
 * The share-balance is the number of shares.
 * Price fluctuations do not change the share balance.
 *
 * The cleared-balance is the currency-denominated balance
 * of all transactions that have been marked as cleared or reconciled.
 * It is correctly adjusted for price fluctuations.
 *
 * The reconciled-balance is the currency-denominated balance
 * of all transactions that have been marked as reconciled.
 */
double       xaccGetBalance (Account *, Transaction *);
double       xaccGetClearedBalance (Account *, Transaction *);
double       xaccGetReconciledBalance (Account *, Transaction *);
double       xaccGetShareBalance (Account *, Transaction *);

/* the following methods check the date order of the transacton */
int          xaccCheckDateOrder   (Account *, Transaction *);
int          xaccCheckDateOrderDE (Transaction *);

/* The xaccIsAccountInList() subroutine returns the number of times
 * that an account appears in the account list. */
int          xaccIsAccountInList (Account * acc, Account **list);
void         xaccZeroRunningBalances (Account **list);

/* The xaccConsolidateTransactions() subroutine scans through
 * all of the transactions in an account, and compares them.
 * if any of them are exact duplicates, the duplicates are removed.
 * duplicates may occur when accounts from multiple sources are 
 * merged.  Note that this can be a dangerous operation to perform */

void        xaccConsolidateTransactions (Account *);

/** GLOBALS *********************************************************/

extern int next_free_unique_account_id;


#endif
