/****************************************************************************

xapplaunch - X application launcher

Copyright 1995 Craig Knudsen

Permission to use, copy, modify and distribute this software and
its documentation for any purpose is hereby granted without fee, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting documentation,
and that the name of Craig Knudsen not be used in advertising or
publicity pertaining to distribution of the software without
specific, written prior permission.  Craig Knudsen makes no representations
about the suitability of this software for any purpose. It is
provided "as is" without express or implied warranty.


Command line args:
	-f <file>	Read in specified menu file rather than
			$HOME/.xapplaunchrc


reads in $HOME/.xapplaunchrc and preprocesses through cpp

bitmaps/pixmaps should be 75x75 or smaller

define HAVE_XPM if you have the xpm library (available at ftp.x.org).

.xapplaunchrc is in the following format:

<T>Title
<B>bitmap file
<E>exec

   --- OR ---

<T>Title
<P>pixmap file
<E>exec

   --- OR ---

<T>Title
<E>exec

where <B> indicates an X bitmap file and <P> indicates an XPM file
created by either pixed or something.  <E> specifies what to execute
and should include a "&" unless you want to wait for it.  If <P> and
<B> are not specified for an entry, then <T> will be used and "\n" will
be translated into new lines.  There are the following predefined
functions that may be specified for <E>:

<E>f.quit	Ask the user if they want to quit
<E>f.quit_now	Quit now (now confirmation) (Motif only)
<E>f.restart	Restart xapplaunch
<E>f.info	Display text info to user (Motif only)

History:
	22-Jul-93  Craig Knudsen	Created
	01-Jan-94  Craig Knudsen	Added f.quit, f.restart and f.quit_now
	09-Aug-94  Craig Knudsen	ANSI-C changes
	10-Jan-94  Craig Knudsen	Version 1.1:
					  Xpm support added
					  Xaw support added
					  Removed bottom label

***************************************************************************
*/


#define DEFAULT_SIZE	75


#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>

#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/RowColumn.h>
#include <Xm/PushB.h>
#include <Xm/DrawnB.h>
#include <Xm/Label.h>
#include <Xm/MessageB.h>
#else
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Box.h>
#endif

#ifdef HAVE_XPM
#include <X11/xpm.h>
#endif



typedef struct {
  Pixmap pixmap;
  Pixmap shape_mask;
#ifdef HAVE_XPM
  XpmAttributes attr;
#endif
  int is_pixmap;
  unsigned int width, height;
  char *file;
} MyPixmap;

static MyPixmap **pixmaps;
static int num_pixmaps = 0;


static void button_pressed (
#ifndef _NO_PROTO
  Widget widget,
  XtPointer client,
  XtPointer call
#endif
);

static Widget create_text_button (
#ifndef _NO_PROTO
  char *name,
  char *label
#endif
);

static Widget create_graphic_button (
#ifndef _NO_PROTO
  char *name,
  MyPixmap *my_pixmap;
#endif
);

static void expose_handler (
#ifndef _NO_PROTO
  Widget widget,
  XtPointer client,
  XEvent *event,
  Boolean *continue_to_dispatch
#endif
);

#ifdef HAVE_MOTIF
static void confirm (
#ifndef _NO_PROTO
  char *msg,
  void (*yes_cb)(),       /* or NULL */
  void (*no_cb)(),        /* or NULL */
  XtPointer data
#endif
);
#endif

#ifdef HAVE_MOTIF
static void info (
#ifndef _NO_PROTO
  char *msg,
#endif
);
#endif

static void destroy_self (
#ifndef _NO_PROTO
  Widget widget,
  XtPointer client,
  XtPointer call
#endif
);

static void restart (
#ifndef _NO_PROTO
  Widget widget,
  XtPointer client,
  XtPointer call
#endif
);

static void my_exit (
#ifndef _NO_PROTO
  Widget widget,
  XtPointer client,
  XtPointer call
#endif
);

static void remove_arg (
#ifndef _NO_PROTO
  int *argc,
  char **argv
#endif
);


/*
** Global variables
*/
static int verbose = 0;
static Widget toplevel;
static Widget ButtonForm;
static Pixel fg, bg;
static Cursor cursor;
char *argv_save[100];
#ifndef HAVE_MOTIF
static Widget last_button = NULL;
#endif


int main ( argc, argv )
int argc;
char *argv[];
{
  FILE *fp;
  char path[300], text[1000], title[300], bitmap[300], ex[300];
  char cmd[300], hostname[30];
  Widget form, button, frame;
  char *env;
  int len, loop;
  int is_pixmap, i;
  struct stat buf;
  char *temp, *file, *otherfile = NULL, *ptr;

  /*
  ** Save copy of argument so we can restart with same command line args
  */
  loop = -1;
  do {
    loop++;
    if ( argv[loop] ) {
      argv_save[loop] = (char *) malloc ( strlen ( argv[loop] ) + 1 );
      strcpy ( argv_save[loop], argv[loop] );
    }
    else {
      argv_save[loop] = argv[loop];	/* NULL */
    }
  } while ( argv[loop] != NULL );

  for ( i = 1; i < argc; ) {
    if ( strcmp ( argv[i], "-v" ) == 0 ) {
      remove_arg ( &argc, argv + i - 1 );
      verbose = 1;
    }
    else if ( strcmp ( argv[i], "-f" ) == 0 ) {
      otherfile = argv[i+1];
      remove_arg ( &argc, argv + i - 1 );
      remove_arg ( &argc, argv + i - 1 );
    }
    else
      i++;
  }

  if ( ! otherfile ) {
    env = (char *) getenv ( "HOME" );
    sprintf ( path, "%s/.xapplaunchrc", env );
  }
  else {
    strcpy ( path, otherfile );
  }

  if ( stat ( path, &buf ) != 0 ) {
    fprintf ( stderr, "Cannot stat %s\n", path );
    exit ( 1 );
  }

#ifndef NO_CPP_SUPPORT
  temp = tmpnam ( NULL );
  gethostname ( hostname, 30 );
  ptr = strtok ( hostname, "." );	/* don't define the domain name */
  sprintf ( cmd, "%s -D_%s_ -DHOSTNAME=%s -P %s > %s", CPP, ptr,
    hostname, path, temp );
  if ( verbose ) {
    printf ( "%s\n", cmd );
    fflush ( stdout );
  }
  if ( system ( cmd ) != 0 ) { 
    fprintf ( stderr, "Error in exec: %s\n", cmd );
    exit ( 1 );
  }
  file = temp;
#else
  file = path;
#endif

  fp = fopen ( file, "r" );
  if ( fp == NULL ) {
    fprintf ( stderr, "Cannot open %s\n", file );
    exit ( 1 );
  }


  toplevel = XtInitialize ( argv[0], "Xapplaunch", NULL, 0, &argc, argv );

#ifdef HAVE_MOTIF
  frame = XtVaCreateManagedWidget ( "frame",
    xmFrameWidgetClass, toplevel,
    XmNleftAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM,
    XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM, NULL );

  form = XtVaCreateManagedWidget ( "form",
    xmFormWidgetClass, frame,
    /*XmNmappedWhenManaged, False,*/
    XmNleftAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM,
    XmNtopAttachment, XmATTACH_FORM, XmNbottomAttachment, XmATTACH_FORM,
    XmNleftOffset, 3, XmNrightOffset, 3, XmNtopOffset, 3,
    XmNbottomOffset, 3, NULL );

  ButtonForm = XtVaCreateManagedWidget ( "ButtonForm",
    xmRowColumnWidgetClass, form,
    XmNleftAttachment, XmATTACH_FORM, XmNrightAttachment, XmATTACH_FORM,
    XmNtopAttachment, XmATTACH_FORM, XmNorientation, XmHORIZONTAL, NULL );
#else
  frame = form = ButtonForm = XtVaCreateManagedWidget ( "ButtonForm",
    formWidgetClass, toplevel,
    XtNheight, DEFAULT_SIZE, XtNwidth, DEFAULT_SIZE,
    NULL );
#endif

#ifdef HAVE_MOTIF
  XtVaGetValues ( ButtonForm, XmNbackground, &bg, XmNforeground, &fg, NULL );
#else
  XtVaGetValues ( ButtonForm, XtNbackground, &bg, XtNforeground, &fg, NULL );
#endif

#ifdef HAVE_MOTIF
  button = XtVaCreateManagedWidget ( "__TEMP__",
    xmDrawnButtonWidgetClass, ButtonForm,
    XmNlabelType, XmPIXMAP,
    XmNwidth, DEFAULT_SIZE, XmNheight, DEFAULT_SIZE,
    XmNpushButtonEnabled, TRUE,
    XmNshadowType, XmSHADOW_OUT, NULL );
#else
  button = XtVaCreateManagedWidget ( "__TEMP__",
    commandWidgetClass, ButtonForm,
    XtNwidth, DEFAULT_SIZE, XtNheight, DEFAULT_SIZE,
    NULL );
#endif

  XtRealizeWidget ( toplevel );
  ex[0] = bitmap[0] = title[0] = '\0';

  pixmaps = (MyPixmap **) malloc ( 4 );
  while ( fgets ( text, 1000, fp ) ) {
    if ( strncmp ( text, "<T>", 3 ) == 0 ) {
      strcpy ( title, text + 3 );
      len = strlen ( title );
      title[len-1] = '\0';
    }
    else if ( strncmp ( text, "<B>", 3 ) == 0 ) {
      strcpy ( bitmap, text + 3 );
      len = strlen ( bitmap );
      bitmap[len-1] = '\0';
      is_pixmap = FALSE;
    }
#ifdef HAVE_XPM
    else if ( strncmp ( text, "<P>", 3 ) == 0 ) {
      strcpy ( bitmap, text + 3 );
      len = strlen ( bitmap );
      bitmap[len-1] = '\0';
      is_pixmap = TRUE;
    }
#endif
    else if ( strncmp ( text, "<E>", 3 ) == 0 ) {
      strcpy ( ex, text + 3 );
      len = strlen ( ex );
      ex[len-1] = '\0';
      process_item ( title, bitmap, is_pixmap, ex );
      ex[0] = bitmap[0] = title[0] = '\0';
    }
  }
  fclose ( fp );

#ifndef NO_CPP_SUPPORT
  unlink ( temp );
#endif

  XtDestroyWidget ( button );

  cursor = XCreateFontCursor ( XtDisplay ( toplevel ), XC_watch );

  XtSetMappedWhenManaged ( form, TRUE );
/*
  XtMapWidget ( form );
*/

  XtMainLoop ();
}


/* remove an argument from argv[] */
static void remove_arg ( argc, argv )
int *argc;
char **argv;
{
  int loop;

  for ( loop = 2; loop < *argc; loop++ )
    argv[loop -1] = argv[loop];

  (*argc)--;
}


  


/*
** Function to create a button for one entry.
*/
static int process_item ( title, bitmap_file, is_pixmap, ex )
char *title;
char *bitmap_file;
int is_pixmap;
char *ex;
{
  char *ptr;
  Widget button;
  MyPixmap *my_pixmap;
  struct stat buf;
  int width, height;
  int status;
  int x_hot_return, y_hot_return;
  char *ptr1, *ptr2, temp[200];
  int read_error = 0;
#ifdef HAVE_MOTIF
  XmString xmstr;
#endif

  my_pixmap = (MyPixmap *) malloc ( sizeof ( MyPixmap ) );
  memset ( my_pixmap, '\0', sizeof ( MyPixmap ) );

  my_pixmap->pixmap = 0;
  my_pixmap->is_pixmap = is_pixmap;
  my_pixmap->file = (char *) malloc ( strlen ( bitmap_file ) + 1 );
  strcpy ( my_pixmap->file, bitmap_file );

  strcpy ( temp, title );
  for ( ptr1 = title, ptr2 = temp; *ptr1 != '\0'; ptr1++ ) {
    if ( *ptr1 == '\134' && *( ptr1 + 1 ) == 'n' ) {
      *ptr2 = '\012';
      ptr1++;
    }
    else {
      *ptr2 = *ptr1;
    }
    ptr2++;
  }
  *ptr2 = '\0';
  if ( ! strlen ( bitmap_file ) ) {
    button = create_text_button ( temp, temp );
    free ( my_pixmap );
  }
  else if ( stat ( bitmap_file, &buf ) != 0 ) {
    fprintf ( stderr, "xapplaunch: bitmap file not found %s\n", bitmap_file );
    button = create_text_button ( temp, temp );
  }
  else {
    if ( is_pixmap ) {
#ifdef HAVE_XPM
      status = XpmReadFileToPixmap ( XtDisplay ( toplevel ),
        XtWindow ( toplevel ), my_pixmap->file,
        &my_pixmap->pixmap, &my_pixmap->shape_mask, &my_pixmap->attr );
      read_error = ( status != XpmSuccess );
      my_pixmap->width = my_pixmap->attr.width;
      my_pixmap->height = my_pixmap->attr.height;
#endif
    }
    else {
      status = XReadBitmapFile ( XtDisplay ( toplevel ), XtWindow ( toplevel ),
        my_pixmap->file, &my_pixmap->width, &my_pixmap->height,
        &my_pixmap->pixmap, &x_hot_return, &y_hot_return );
      read_error = ( status != BitmapSuccess );
      my_pixmap->width += 2;
      my_pixmap->height += 2;
    }
    if ( read_error ) {
      fprintf ( stderr, "Unable to read: %s\n", my_pixmap->file );
      button = create_text_button ( temp, temp );
      free ( my_pixmap );
    }
    else {
      pixmaps = (MyPixmap **) realloc ( pixmaps,
        ( ( num_pixmaps + 1 ) * sizeof ( MyPixmap * ) ) );
      pixmaps[num_pixmaps++] = my_pixmap;
      button = create_graphic_button ( temp, my_pixmap );
    }
  }

  ptr = (char *) malloc ( strlen ( ex ) + 5 );
  if ( strncmp ( ex, "f.", 2 ) == 0 )
    strcpy ( ptr, ex );
  else
    sprintf ( ptr, "(%s)&", ex );

#ifdef HAVE_MOTIF
  XtAddCallback ( button, XmNactivateCallback, button_pressed, ptr );
#else
  XtAddCallback ( button, XtNcallback, button_pressed, ptr );
#endif
}






/*
** Create a text push button
*/
static Widget create_text_button ( name, label )
char *name;
char *label;
{
  Widget button;
#ifdef HAVE_MOTIF
  XmString xmstr;
#endif

#ifdef HAVE_MOTIF
  xmstr = XmStringCreateLtoR ( name, XmSTRING_DEFAULT_CHARSET );
  button = XtVaCreateManagedWidget ( name,
    xmPushButtonWidgetClass, ButtonForm,
    XmNheight, DEFAULT_SIZE,
    XmNlabelType, XmSTRING,
    XmNlabelString, xmstr,
    XmNalignment, XmALIGNMENT_CENTER,
    NULL );
  XmStringFree ( xmstr );
#else
  if ( last_button )
    button = XtVaCreateManagedWidget ( name,
      commandWidgetClass, ButtonForm,
      XtNheight, DEFAULT_SIZE,
      XtNstring, name,
      XtNhorizDistance, 1,
      XtNfromHoriz, last_button,
      NULL );
  else
    button = XtVaCreateManagedWidget ( name,
      commandWidgetClass, ButtonForm,
      XtNheight, DEFAULT_SIZE,
      XtNstring, name,
      XtNhorizDistance, 1,
      NULL );
  last_button = button;
#endif

  return ( button );
}




/*
** Create a pushbutton with a graphic icon
*/
static Widget create_graphic_button ( name, my_pixmap )
char *name;
MyPixmap *my_pixmap;
{
  Widget button;

#ifdef HAVE_MOTIF
  button = XtVaCreateManagedWidget ( name,
    xmDrawnButtonWidgetClass, ButtonForm,
    XmNlabelType, XmPIXMAP,
#if XmVersion >= 1002
    /*XmNwidth, DEFAULT_SIZE, XmNheight, DEFAULT_SIZE,*/
    XmNwidth, (my_pixmap->width + 10), XmNheight, DEFAULT_SIZE,
#else
    XmNwidth, (my_pixmap->width + 6), XmNheight, DEFAULT_SIZE,
#endif
    XmNpushButtonEnabled, TRUE,
    XmNshadowType, XmSHADOW_OUT,
    NULL );
  XtAddEventHandler ( button, ExposureMask, False,
    (XtEventHandler) expose_handler, (XtPointer) my_pixmap );
#else
  /* For Athena widgets, give no name so there is no text... */
  if ( last_button )
    button = XtVaCreateManagedWidget ( " ",
      commandWidgetClass, ButtonForm,
      XtNheight, DEFAULT_SIZE,
      XtNwidth, (my_pixmap->width + 6),
      XtNhorizDistance, 1,
      XtNfromHoriz, last_button,
      NULL );
  else
    button = XtVaCreateManagedWidget ( " ",
      commandWidgetClass, ButtonForm,
      XtNheight, DEFAULT_SIZE,
      XtNwidth, (my_pixmap->width + 6),
      NULL );
  XtAddEventHandler ( button, ExposureMask, False,
    (XtEventHandler) expose_handler, (XtPointer) my_pixmap );
  last_button = button;
#endif
  return ( button );
}


#ifdef HAVE_MOTIF
/*
** Ask the user if they really want to quit.
*/
static void confirm ( msg, yes_cb, no_cb, data )
char *msg;
void (*yes_cb)();       /* or NULL */
void (*no_cb)();        /* or NULL */
XtPointer data;
{
  Widget top, widget;
  Arg args[10];
  XmString xmstr, title;
  Dimension w, h;
  int screen_w, screen_h, x, y;

  xmstr = XmStringCreateLtoR ( msg, XmSTRING_DEFAULT_CHARSET );
  XtSetArg ( args[0], XmNmessageString, xmstr );
  title = XmStringCreateLtoR ( "Confirm", XmSTRING_DEFAULT_CHARSET );
  XtSetArg ( args[1], XmNdialogTitle, title );
  XtSetArg ( args[2], XmNdialogStyle, XmDIALOG_FULL_APPLICATION_MODAL );
  top = XmCreateQuestionDialog ( toplevel, "Confirm", args, 3 );
  /* get rid of the help button */
  widget = XmMessageBoxGetChild ( top, XmDIALOG_HELP_BUTTON );
  XtUnmanageChild ( widget );
  /* add callbacks */
  if ( yes_cb != NULL )
    XtAddCallback ( top, XmNokCallback, yes_cb, data );
  XtAddCallback ( top, XmNokCallback, destroy_self, (XtPointer)top );
  if ( no_cb != NULL )
    XtAddCallback ( top, XmNcancelCallback, no_cb, data );
  XtAddCallback ( top, XmNcancelCallback, destroy_self, top );
  XtManageChild ( top );
  if ( yes_cb == NULL && no_cb == NULL ) {
    /* get rid of the cancel button */
    widget = XmMessageBoxGetChild ( top, XmDIALOG_CANCEL_BUTTON );
    XtUnmanageChild ( widget );
  }

  /*
  ** Place it in the center of the screen.
  */
  XtManageChild ( top );
  XtVaGetValues ( top, XmNheight, &h, XmNwidth, &w, NULL );
  screen_w = DisplayWidth ( XtDisplay ( top ),
      DefaultScreen ( XtDisplay ( top ) ) );
  screen_h = DisplayHeight ( XtDisplay ( top ),
      DefaultScreen ( XtDisplay ( top ) ) );
  x = ( screen_w - w - 10 ) / 2;
  y = ( screen_h - h - 10 ) / 2;
  XtMoveWidget ( XtParent ( top ), x, y );
}



/*
** Display info to user
*/
static void info ( msg )
char *msg;
{
  Widget top, widget;
  Arg args[10];
  XmString xmstr, title;
  Dimension w, h;
  int screen_w, screen_h, x, y;
  char *newmsg, *ptr1, *ptr2;

  /*
  ** Translate \n to newline.
  */
  newmsg = (char *) malloc ( strlen ( msg ) + 1 );
  for ( ptr1 = msg, ptr2 = newmsg; *ptr1 != '\0'; ptr1++, ptr2++ ) {
    if ( *ptr1 == '\134' && *(ptr1+1) == 'n' ) {
      /* change '\''n' into '\n' */
      *ptr2 = '\012';
      ptr1++;
    }
    else {
      *ptr2 = *ptr1;
    }
  }
  ptr2++;
  *ptr2 = '\0';

  xmstr = XmStringLtoRCreate ( newmsg, XmSTRING_DEFAULT_CHARSET );
  free ( newmsg );
  XtSetArg ( args[0], XmNmessageString, xmstr );
  title = XmStringCreateLtoR ( "Info", XmSTRING_DEFAULT_CHARSET );
  XtSetArg ( args[1], XmNdialogTitle, title );
  top = XmCreateInformationDialog ( toplevel, "Info", args, 2 );
  /* get rid of the help button */
  widget = XmMessageBoxGetChild ( top, XmDIALOG_HELP_BUTTON );
  XtUnmanageChild ( widget );
  /* add callbacks */
  XtAddCallback ( top, XmNokCallback, destroy_self, (XtPointer)top );
  XtManageChild ( top );
  /* get rid of the cancel button */
  widget = XmMessageBoxGetChild ( top, XmDIALOG_CANCEL_BUTTON );
  XtUnmanageChild ( widget );

  /*
  ** Place it in the center of the screen.
  */
  XtManageChild ( top );
  XtVaGetValues ( top, XmNheight, &h, XmNwidth, &w, NULL );
  screen_w = DisplayWidth ( XtDisplay ( top ),
      DefaultScreen ( XtDisplay ( top ) ) );
  screen_h = DisplayHeight ( XtDisplay ( top ),
      DefaultScreen ( XtDisplay ( top ) ) );
  x = ( screen_w - w - 10 ) / 2;
  y = ( screen_h - h - 10 ) / 2;
  XtMoveWidget ( XtParent ( top ), x, y );
}
#endif


static void destroy_self ( widget, client, call )
Widget widget;
XtPointer client;
XtPointer call;
{
  XtDestroyWidget ( XtParent ( client ) );
}




/*
** Expose event handler for buttons with pixmaps on them.
** Redraw the pixmap.
*/
static void expose_handler ( widget, client, event, continue_to_dispatch )
Widget widget;
XtPointer client;	/* Contiains system call */
XEvent *event;
Boolean *continue_to_dispatch;
{
  MyPixmap *my_pixmap = (MyPixmap *) client;
  static GC gc = NULL;
  int status, x, y;
  int x_hot_return, y_hot_return;
  int width, height;

  if ( gc == NULL )
    gc = XCreateGC ( XtDisplay ( widget ), XtWindow ( widget ), NULL, NULL );

  if ( my_pixmap->is_pixmap ) {
#ifdef HAVE_XPM
    /* Handle a pixmap */
    if ( my_pixmap->pixmap == 0 ) {
      status = XpmReadFileToPixmap ( XtDisplay ( widget ),
        XtWindow ( widget ), my_pixmap->file,
        &my_pixmap->pixmap, &my_pixmap->shape_mask, &my_pixmap->attr );
      my_pixmap->width = my_pixmap->attr.width;
      my_pixmap->height = my_pixmap->attr.height;
    }
    if ( my_pixmap->width > DEFAULT_SIZE ) {
      width = DEFAULT_SIZE;
      x = 3;
    }
    else {
      x = 3;
      /*x = ( DEFAULT_SIZE - my_pixmap->width ) / 2; */
      width = my_pixmap->width;
    }
    if ( my_pixmap->height > DEFAULT_SIZE ) {
      height = DEFAULT_SIZE;
      y = 0;
    }
    else {
      y = ( DEFAULT_SIZE - my_pixmap->height ) / 2;
      height = my_pixmap->height;
    }
    XSetForeground ( XtDisplay ( widget ), gc, fg );
    XSetClipMask ( XtDisplay ( widget ), gc, my_pixmap->shape_mask );
#if XmVersion >= 1002
    XSetClipOrigin ( XtDisplay ( widget ), gc, x + 2, y );
    XCopyArea ( XtDisplay ( widget ), my_pixmap->pixmap,
      XtWindow ( widget ), gc, 0, 0, width, height, x + 2, y );
#else
    XSetClipOrigin ( XtDisplay ( widget ), gc, x, y );
    XCopyArea ( XtDisplay ( widget ), my_pixmap->pixmap,
      XtWindow ( widget ), gc, 0, 0, width, height, x, y );
#endif
    XSetClipMask ( XtDisplay ( widget ), gc, None );
#endif
  }
  else {
    /* Handle a bitmap */
    if ( my_pixmap->pixmap == 0 ) {
      status = XReadBitmapFile ( XtDisplay ( toplevel ), XtWindow ( toplevel ),
        my_pixmap->file, &my_pixmap->width, &my_pixmap->height,
        &my_pixmap->pixmap, &x_hot_return, &y_hot_return );
    }
    if ( my_pixmap->width > DEFAULT_SIZE ) {
      width = DEFAULT_SIZE;
      x = 3;
    }
    else {
      /*x = ( DEFAULT_SIZE - my_pixmap->width ) / 2;*/
      x = 3;
      width = my_pixmap->width;
    }
    if ( my_pixmap->height > DEFAULT_SIZE ) {
      height = DEFAULT_SIZE;
      y = 1;
    }
    else {
      y = ( DEFAULT_SIZE - my_pixmap->height ) / 2;
      height = my_pixmap->height;
    }
    XSetForeground ( XtDisplay ( widget ), gc, bg );
    XSetFillStyle ( XtDisplay ( widget ), gc, FillSolid );
    XFillRectangle ( XtDisplay ( widget ), XtWindow ( widget ), gc,
      0, 0, DEFAULT_SIZE, DEFAULT_SIZE );
    XSetFillStyle ( XtDisplay ( widget ), gc, FillOpaqueStippled );
    XSetForeground ( XtDisplay ( widget ), gc, fg );
    XSetBackground ( XtDisplay ( widget ), gc, bg );
    XSetStipple ( XtDisplay ( widget ), gc, my_pixmap->pixmap );
    XSetTSOrigin ( XtDisplay ( widget ), gc, x, y );
    XFillRectangle ( XtDisplay ( widget ), XtWindow ( widget ), gc,
      x, y, width, height );
    XSetFillStyle ( XtDisplay ( widget ), gc, FillSolid );
  }
}



/*
** Callback for when user presses one of the buttons
*/
static void button_pressed ( widget, client, call )
Widget widget;
XtPointer client;	/* Contiains system call */
XtPointer call;
{
  char *sys_call;
  int ret;
  char *ptr;
  char temp[200];
#ifdef HAVE_MOTIF
  XmString xmstr;
#endif

  sys_call = (char *) client;

#ifdef HAVE_MOTIF
  if ( strcmp ( sys_call, "f.quit" ) == 0 ) {
    confirm ( "Are you sure\nyou want to exit?", my_exit, NULL, NULL );
  }
#else
  if ( strcmp ( sys_call, "f.quit" ) == 0 ) {
    exit ( 0 );
  }
#endif
  else if ( strcmp ( sys_call, "f.quit_now" ) == 0 ) {
    exit ( 0 );
  }
  else if ( strcmp ( sys_call, "f.restart" ) == 0 ) {
#ifdef HAVE_MOTIF
    confirm ( "Are you sure\nyou want to restart?", restart, NULL, NULL );
#else
    restart ( NULL, NULL, NULL );
#endif
  }
#ifdef HAVE_MOTIF
  else if ( strncmp ( sys_call, "f.info", 6 ) == 0 ) {
    info ( sys_call + 7 );
  }
#endif
  else {
    XDefineCursor ( XtDisplay ( toplevel ), XtWindow ( toplevel ), cursor );
    if ( verbose ) {
      printf ( "%s\n", sys_call );
      fflush ( stdout );
    }
    ret = system ( sys_call );
    if ( ret != 0 ) {
      sprintf ( temp, "Error starting : %s", sys_call );
    }
    XUndefineCursor ( XtDisplay ( toplevel ), XtWindow ( toplevel ) );
  }
}


static void restart ( widget, client, call )
Widget widget;
XtPointer client, call;
{
  execv ( argv_save[0], argv_save );
}

static void my_exit ( widget, client, call )
Widget widget;
XtPointer client, call;
{
  int loop;

  for ( loop = 0; loop < num_pixmaps; loop++ ) {
    XFreePixmap ( XtDisplay ( toplevel ), pixmaps[loop]->pixmap );
    if ( pixmaps[loop]->shape_mask )
      XFreePixmap ( XtDisplay ( toplevel ), pixmaps[loop]->shape_mask );
  }
  exit ( 0 );
}

