/* asc2bin -- convert ASCII files to ACCBIN
 *
 * asc2bin [-wi] [-a n] [-s scale] [file ...] [-o outfile]
 *
 * asc2bin reads lines of ASCII input from the named files, logically
 * concatenated, or from stdin if no files are named.  An ACCBIN file
 * is written on stdout.
 *
 * The input file consists of data lines and parameter lines.  A data
 * line is just a series of numbers separated by whitespace or commas.
 * Since these numbers will be converted to 16-bit integers they
 * should be in the range [-32768, 32767].  Numbers outside that range
 * will be converted to the appropriate extreme, and a warning will be
 * written to stdout.
 *
 * Parameter lines have the format:
 *
 *	<parameter> = <value>
 *
 * The value type depends on the parameter.  asc2bin understands the
 * following parameters:
 *
 *	parameter	value type	meaning
 *
 *	Channels	string		channels recorded, e.g.
 *					1,2:5,8
 *	TZero		number		Time at point 0
 *	ULimit0		number		Upper voltage limit, channel 0
 *	LLimit0		number		Lower voltage limit, channel 0
 *	VMul0		number		Scale factor, channel 0
 *	VOffset0	number		Offset, channel 0
 *	Ulimit1		number		Upper voltage limit, channel 1
 *	...		...
 *	and so on, up to channel 8
 *
 *	ULimit		number		Synonym for ULimit0
 *	LLimit		number		Synonym for LLimit0
 *	VMul		number		Synonym for VMul0
 *	VOffset		number		Synonym for VOffset0
 *	Clock		number		Clock frequency
 *	InterchannelDelay number	Interchannel delay
 *	Comment		string		text comment, up to 355 chars
 *
 * Parameter lines that asc2bin doesn't recognize are skipped without
 * warning unless the -w option is used, in which case a warning is
 * written to stderr when an unrecognized parameter is used.
 *
 * Anything including or after a '#' character in a line is ignored,
 * allowing comments.  There is currently no way to include leading
 * blanks, newline, or # in a string parameter.
 *
 * Options:
 *	-w	print warning for unknown parameter
 *	-i	assume input numbers are integers.  Speeds up
 *		conversion.
 *	-a n	abort after <n> warnings
 *	-s scale multiply data by scale 
 *	-o outfile specify output file
 *
 * Note for programmers: this code assumes it is running with IEEE
 * floating-point arithmetic.  If not, you're sunk.
 */
#include <stdio.h>
#include <ctype.h>
#include <math.h>

#include "config.h"
#include "readline.h"

#define	Nfree(p)	(((p) == NULL) || (free(p), (p) = NULL))

#ifdef	__STDC__
void	options(int, char**);
void	usage(void);
void	warning(void);
void	files(int, char**);
char	*estrdup(char *);
void	exit_cleanup(void);
void	open_temp(void);
void	print_hdr(void);
void	print_temp(void);
void	doit(void);
char	*hdr_init(void);
char	*uncomment(FILE **, FNL **);
int	isparam(char *);
void	parse_param(char *, char *);
void	parse_numbers(char *, FILE *);
int	ab_string(char *, VOIDST, int);
int	ab_i16(char *, VOIDST, int);
int	ab_i32(char *, VOIDST, int);
int	ab_sgl(char *, VOIDST, int);
long	ab_strtol(char *, char **, int);
#else	/* __STDC__ */
void	options();
void	usage();
void	warning();
void	files();
char	*estrdup();
void	exit_cleanup();
void	open_temp();
void	print_hdr();
void	print_temp();
void	doit();
char	*hdr_init();
char	*uncomment();
int	isparam();
void	parse_param();
void	parse_numbers();
int	ab_string();
int	ab_i16();
int	ab_i32();
int	ab_sgl();
long	ab_strtol();
#endif	/* __STDC__ */

/*
 * ACCBIN definitions
 */
#define	AB_HDRSZ	(1000)
#define	AB_MIN		(-32767)
#define	AB_MAX		(32767)
#define	I16_MIN		(-32767)
#define	I16_MAX		(32767)
#define	I32_MIN		(-0xffffffff)
#define	I32_MAX		(0xffffffff)

/*
 * offsets
 */
#define	ABO_MAGIC	(0)
#define	AB_MAGLEN	(27)
#define	AB_MAGIC	"accbin format #2(header=1k)"
#define	ABO_CHAN	(27)
#define	AB_CHAN		"0"
#define	AB_CHLEN	(30)
#define	ABO_T0		(57)
#define	ABO_VHI0	(61 + 0*16)
#define	ABO_VLO0	(65 + 0*16)
#define	ABO_VMUL0	(69 + 0*16)
#define	ABO_VOFF0	(73 + 0*16)
#define	ABO_VHI1	(61 + 1*16)
#define	ABO_VLO1	(65 + 1*16)
#define	ABO_VMUL1	(69 + 1*16)
#define	ABO_VOFF1	(73 + 1*16)
#define	ABO_VHI2	(61 + 2*16)
#define	ABO_VLO2	(65 + 2*16)
#define	ABO_VMUL2	(69 + 2*16)
#define	ABO_VOFF2	(73 + 2*16)
#define	ABO_VHI3	(61 + 3*16)
#define	ABO_VLO3	(65 + 3*16)
#define	ABO_VMUL3	(69 + 3*16)
#define	ABO_VOFF3	(73 + 3*16)
#define	ABO_VHI4	(61 + 4*16)
#define	ABO_VLO4	(65 + 4*16)
#define	ABO_VMUL4	(69 + 4*16)
#define	ABO_VOFF4	(73 + 4*16)
#define	ABO_VHI5	(61 + 5*16)
#define	ABO_VLO5	(65 + 5*16)
#define	ABO_VMUL5	(69 + 5*16)
#define	ABO_VOFF5	(73 + 5*16)
#define	ABO_VHI6	(61 + 6*16)
#define	ABO_VLO6	(65 + 6*16)
#define	ABO_VMUL6	(69 + 6*16)
#define	ABO_VOFF6	(73 + 6*16)
#define	ABO_VHI7	(61 + 7*16)
#define	ABO_VLO7	(65 + 7*16)
#define	ABO_VMUL7	(69 + 7*16)
#define	ABO_VOFF7	(73 + 7*16)
#define	ABO_VHI8	(61 + 8*16)
#define	ABO_VLO8	(65 + 8*16)
#define	ABO_VMUL8	(69 + 8*16)
#define	ABO_VOFF8	(73 + 8*16)
#define	ABO_VHI		ABO_VHI0
#define	ABO_VLO		ABO_VLO0
#define	ABO_VMUL	ABO_VMUL0
#define	ABO_VOFF	ABO_VOFF0
#define	ABO_CLK		(637)
#define	ABO_ICD		(641)
#define	ABO_COM		(645)
#define	ABO_HLEN	(1000)
#define	AB_COMLEN	(ABO_HLEN - ABO_COM)

#define	COM_CHAR	'#'

/*
 * parameter definitions
 */
typedef	struct AB_PARAM {
    char	*ab_name;		/* Name in asc file		*/
    int		ab_off;			/* offset in header		*/
    int		ab_siz;			/* size in bytes		*/
    char	*ab_init;		/* initial value		*/
    int		(*ab_dcd)();		/* ASCII decode			*/
}		AB_PARAM;

AB_PARAM	ab_params[] = {
    {"", ABO_MAGIC, AB_MAGLEN, AB_MAGIC, ab_string},
    {"Channels", ABO_CHAN, AB_CHLEN, AB_CHAN, ab_string},
    {"TZero", ABO_T0, 4, "0", ab_sgl},
    {"ULimit", ABO_VHI, 4, "2047.0", ab_sgl},
    {"LLimit", ABO_VLO, 4, "-2047.0", ab_sgl},
    {"VMul", ABO_VMUL, 4, "1.0", ab_sgl},
    {"VOffset", ABO_VOFF, 4, "0.0", ab_sgl},
    {"ULimit0", ABO_VHI0, 4, "2047.0", ab_sgl},
    {"LLimit0", ABO_VLO0, 4, "-2047.0", ab_sgl},
    {"VMul0", ABO_VMUL0, 4, "1.0", ab_sgl},
    {"VOffset0", ABO_VOFF0, 4, "0.0", ab_sgl},
    {"ULimit1", ABO_VHI1, 4, "2047.0", ab_sgl},
    {"LLimit1", ABO_VLO1, 4, "-2047.0", ab_sgl},
    {"VMul1", ABO_VMUL1, 4, "1.0", ab_sgl},
    {"VOffset1", ABO_VOFF1, 4, "0.0", ab_sgl},
    {"ULimit2", ABO_VHI2, 4, "2047.0", ab_sgl},
    {"LLimit2", ABO_VLO2, 4, "-2047.0", ab_sgl},
    {"VMul2", ABO_VMUL2, 4, "1.0", ab_sgl},
    {"VOffset2", ABO_VOFF2, 4, "0.0", ab_sgl},
    {"ULimit3", ABO_VHI3, 4, "2047.0", ab_sgl},
    {"LLimit3", ABO_VLO3, 4, "-2047.0", ab_sgl},
    {"VMul3", ABO_VMUL3, 4, "1.0", ab_sgl},
    {"VOffset3", ABO_VOFF3, 4, "0.0", ab_sgl},
    {"ULimit4", ABO_VHI4, 4, "2047.0", ab_sgl},
    {"LLimit4", ABO_VLO4, 4, "-2047.0", ab_sgl},
    {"VMul4", ABO_VMUL4, 4, "1.0", ab_sgl},
    {"VOffset4", ABO_VOFF4, 4, "0.0", ab_sgl},
    {"ULimit5", ABO_VHI5, 4, "2047.0", ab_sgl},
    {"LLimit5", ABO_VLO5, 4, "-2047.0", ab_sgl},
    {"VMul5", ABO_VMUL5, 4, "1.0", ab_sgl},
    {"VOffset5", ABO_VOFF5, 4, "0.0", ab_sgl},
    {"ULimit6", ABO_VHI6, 4, "2047.0", ab_sgl},
    {"LLimit6", ABO_VLO6, 4, "-2047.0", ab_sgl},
    {"VMul6", ABO_VMUL6, 4, "1.0", ab_sgl},
    {"VOffset6", ABO_VOFF6, 4, "0.0", ab_sgl},
    {"ULimit7", ABO_VHI7, 4, "2047.0", ab_sgl},
    {"LLimit7", ABO_VLO7, 4, "-2047.0", ab_sgl},
    {"VMul7", ABO_VMUL7, 4, "1.0", ab_sgl},
    {"VOffset7", ABO_VOFF7, 4, "0.0", ab_sgl},
    {"ULimit8", ABO_VHI8, 4, "2047.0", ab_sgl},
    {"LLimit8", ABO_VLO8, 4, "-2047.0", ab_sgl},
    {"VMul8", ABO_VMUL8, 4, "1.0", ab_sgl},
    {"VOffset8", ABO_VOFF8, 4, "0.0", ab_sgl},
    {"Clock", ABO_CLK, 4, "1.0", ab_sgl},
    {"InterchannelDelay", ABO_ICD, 4, "0.0", ab_sgl},
    {"Comment", ABO_COM, AB_COMLEN, "", ab_string}
};
#define	AB_NPARAMS	(sizeof(ab_params)/sizeof(AB_PARAM))

struct {
    char	*cval;
    long	lval;
}	ab_vals[] = {
    {"true", 1L},
    {"True", 1L},
    {"TRUE", 1L},
    {"yes", 1L},
    {"Yes", 1L},
    {"YES", 1L},
    {"false", 0L},
    {"False", 0L},
    {"FALSE", 0L},
    {"no", 0L},
    {"No", 0L},
    {"NO", 0L}
};
#define	AB_NVALS	(sizeof(ab_vals)/sizeof(ab_vals[0]))

char	*PROGNAME;
int	debug=FALSE;			/* on for debugging messages	*/
int	indent=0;			/* debug msg indent		*/
int	wopt = FALSE;			/* extra warnings option	*/
int	iopt = FALSE;			/* integer arithmetic option	*/
double	scale = 1.0;			/* data scale factor		*/
long	maxwarn = MAXLONG;		/* max wornings			*/
char	*ofile = NULL;			/* output file name		*/
FILE	*cif = NULL;			/* current input file		*/
FNL	*inf = NULL;			/* input file list		*/
char	*tfn = NULL;			/* temp file name		*/
FILE	*tfs = NULL;			/* temp file stream		*/
char	*header = NULL;			/* file header			*/
long	warns = 0L;			/* number warnings so far	*/

int
main(argc, argv)
int	argc;
char	*argv[];
{
    int			err = NOERR;

    PROGNAME = *argv++; argc--;
    options(argc, argv);
    files(argc, argv);
    ON_EXIT(exit_cleanup, NULL);
    open_temp();			/* open temp file		*/
    doit();				/* write ACCBIN in it		*/
    print_hdr();			/* copy header			*/
    print_temp();			/* copy temp file to stdout	*/
    exit_cleanup();
    exit(err);
}

void
exit_cleanup()
{
    if (NULL != tfs) {
	fclose(tfs);
	tfs = NULL;
    }
    if (NULL != tfn) {
	unlink(tfn);
	free(tfn);
	tfn = NULL;
    }
    Nfree(header);
}

void
open_temp()
{
    Nfree(tfn);
    if (NULL != ofile) {
	if (NULL == (tfs = fopen(ofile, "w+")))
	    error1("unable to open output file %s", ofile);
	if (EOF == fseek(tfs, (long) AB_HDRSZ, 0))
	    error1("unable to position output file %s", ofile);
    }
    else {
	if (NULL == (tfn = tempnam(NULL, "asc2bin.")))
	    error("no memory");
	if (NULL == (tfs = fopen(tfn, "w+"))) {
	    exit_cleanup();
	    error1("unable to open temporary file %s", tfn);
	}
    }
}

void
print_temp()
{
    register int	c;

    if (NULL != ofile) return;
    rewind(tfs);
    while(EOF != (c = getc(tfs))) putchar(c);
}

void
print_hdr()
{
    if (NULL != ofile) {
	rewind(tfs);
	if (1 != fwrite(header, AB_HDRSZ, 1, tfs))
	    error("unable to write ACCBIN header");
    }
    else {
	if (1 != fwrite(header, AB_HDRSZ, 1, stdout))
	    error("unable to write ACCBIN header");
    }
}

void
options(argc, argv)
int	argc;
char	*argv[];
{
    register int	c;
    register char	*s;
    register char	*t;
    char		*end;

    while(argc-- > 0) {
	if (
	    (NULL == (s = *argv++)) ||
	    ('-' != *s++)
	) continue;
	argv[-1] = NULL;
	while('\0' != (c = *s++)) {
	    switch(c) {
	    case 'w':			/* extra warnings		*/
		wopt = TRUE;
		break;

	    case 'i':			/* integer arithmetic		*/
		iopt = TRUE;
		break;

	    case 's':			/* scale data			*/
		if (NULL == (t = *argv)) usage();
		*argv++ = NULL;
		scale = strtod(t, &end);
		if ((end == t) || ('\0' != *end)) usage();
		break;

	    case 'a':			/* abort after n warnings	*/
		if (NULL == (t = *argv)) usage();
		*argv++ = NULL;
		maxwarn = strtol(t, &end, 0);
		if ((end == t) || ('\0' != *end)) usage();
		break;

	    case 'o':			/* write direct to output file	*/
		if (NULL == (ofile = *argv)) usage ();
		*argv++ = NULL;
		break;

	    default:
		usage();
	    }
	}
    }
}

void
usage()
{
    error("usage: %s [-wi] [-a n] [-s scale] [files...] [-o outfile]", PROGNAME);
}

/* files -- get list of input files					*/
void
files(argc, argv)
int	argc;
char	*argv[];
{
    register FNL	**fp = &inf;

    for(;argc-- > 0; argv++) {
	if (NULL == *argv) continue;
	*fp = (FNL *) ealloc(sizeof(FNL));
	(*fp)->f_n = estrdup(*argv);
	(*fp)->f_nxt = NULL;
	fp = &(*fp)->f_nxt;
    }
    if (NULL == inf) {			/* if no files, use stdin	*/
	cif = stdin;
	inf = (FNL *) ealloc(sizeof(FNL));
	inf->f_nxt = NULL;
	inf->f_n = estrdup("Standard Input");
    }
}

void
doit()
{
    char		*line;

    errno = 0;
    header = hdr_init();
    warns = 0L;
    while(NULL != (line = uncomment(&cif, &inf))) {
	if (isparam(line)) {
	    parse_param(line, header);
	}
	else {
	    parse_numbers(line, tfs);
	}
	free(line);
    }
}

char *
hdr_init()
{
    register int	i;
    register char	*hdr;

    if (NULL == (hdr = calloc(AB_HDRSZ, 1))) {
	exit_cleanup();
	error("no memory");
    }
    for(i=0; i<AB_NPARAMS; i++) {
	(ab_params[i].ab_dcd)(
	     ab_params[i].ab_init,
	     hdr+ab_params[i].ab_off,
	     ab_params[i].ab_siz
	);
    }
    return(hdr);
}

char *
uncomment(cp, ip)
FILE		**cp;
FNL		**ip;
{
    register char	*s;
    char		*line;

    while(EOF != readline(&line, cp, ip)) {
	if (NULL == (s = STRCHR(line, COM_CHAR)))
	    s = line + strlen(line);	/* strip comment...		*/
	while((s > line) && isspace(s[-1])) s--;
	*s = '\0';			/* ...trailing white		*/
	for(s=line; (('\0' != *s) && isspace(*s)); s++);
	bcopy(s, line, strlen(s)+1);	/* strip leading white		*/
	if ('\0' != *line) return(line);
	free(line);
    }
    return(NULL);
}

void
parse_numbers(line, tfs)
char	*line;
FILE	*tfs;
{
    register char	*s;
    double		dval = 0.0;
    long		lval = 0L;
    short		ival;

    for(s=line;; s=line) {
	if (iopt) {
	    lval = strtol(s, &line, 0);
	    if (1.0 != scale) lval *= scale;
	}
	else {
	    dval = scale * strtod(s, &line);
	}
	if (line == s) {		/* error or end of line		*/
	    while(isspace(*line)) line++;
	    if ('\0' == *line) break;	/* end of line			*/
	    fprintf(stderr, "%s: bad number \"", PROGNAME);
	    while(('\0' != *line) && !isspace(*line)) putc(*line++, stderr);
	    fprintf(stderr, "\"\n");
	    warning();
	    continue;
	}
	if (iopt) {
	    if (lval > AB_MAX) {
		fprintf(stderr, "%s: number too large \"", PROGNAME);
		while(s < line) putc(*s++, stderr);
		fprintf(stderr, "\"\n");
		warning();
		ival = AB_MAX;
	    }
	    else if (lval < AB_MIN) {
		fprintf(stderr, "%s: number too small \"", PROGNAME);
		while(s < line) putc(*s++, stderr);
		fprintf(stderr, "\"\n");
		warning();
		ival = AB_MIN;
	    }
	    else {
		ival = lval;
	    }
	}
	else {
	    if (dval > AB_MAX) {
		fprintf(stderr, "%s: number too large \"", PROGNAME);
		while(s < line) putc(*s++, stderr);
		fprintf(stderr, "\"\n");
		warning();
		ival = AB_MAX;
	    }
	    else if (dval < AB_MIN) {
		fprintf(stderr, "%s: number too small \"", PROGNAME);
		while(s < line) putc(*s++, stderr);
		fprintf(stderr, "\"\n");
		warning();
		ival = AB_MIN;
	    }
	    else {
		ival = floor(dval + 0.5);
	    }
	}
	putc(ival >> 8, tfs);
	putc(ival & 0xff, tfs);
    }
}

int
isparam(line)
char	*line;
{
    if (!isalpha(*line) && ('_' != *line)) return(FALSE);
    while(isalnum(*line) || (NULL != STRCHR("$_.", *line))) line++;
    while(isspace(*line)) line++;
    return('=' == *line);
}

void
parse_param(line, header)
char	*line;
char	*header;
{
    register char	*s = line;
    register int	i;

    while(isalnum(*s) || (NULL != STRCHR("$_.", *s))) s++;
    for(i=0; i<AB_NPARAMS; i++) {	/* look up  param		*/
	if (
	    (strlen(ab_params[i].ab_name) == s - line) &&
	    (0 == strncmp(ab_params[i].ab_name, line, s - line))
	) break;
    }
    if (i >= AB_NPARAMS) {		/* not found			*/
	if (wopt) {			/* warn if desired		*/
	    fprintf(stderr, "%s: unknown parameter \"", PROGNAME);
	    while(line < s) putc(*line++, stderr);
	    fprintf(stderr, "\"\n");
	    warning();
	}
	return;				/* ignore it			*/
    }
    while(isspace(*s)) s++;		/* skip to argument		*/
    s++;				/* (skip '=')			*/
    while(isspace(*s)) s++;
    (ab_params[i].ab_dcd)(		/* decode arg into header	*/
	s,
	header+ab_params[i].ab_off,
	ab_params[i].ab_siz
    );
}

int
ab_sgl(arg, where, size)
char		*arg;
VOIDST		where;
int		size;
{
    register char	*s = arg;
    register unsigned char *w = (unsigned char *) where;
    union {
	float		f;
	char		b[4];
    }			val;

    val.f = strtod(s, &arg);
    if (arg == s) {
	fprintf(stderr, "%s: bad floating-point number \"%s\"\n",
		PROGNAME, s);
	warning();
	return(TRUE);
    }
#ifdef	WORDS_BIGENDIAN
    w[0] = val.b[0];
    w[1] = val.b[1];
    w[2] = val.b[2];
    w[3] = val.b[3];
#else	/* WORDS_BIGENDIAN */
    w[0] = val.b[1];			/* ****************************	*/
    w[1] = val.b[0];			/*           UNTESTED		*/
    w[2] = val.b[3];			/*	       CODE		*/
    w[3] = val.b[2];			/* ****************************	*/
#endif	/* WORDS_BIGENDIAN */
    return(FALSE);
}

int
ab_i16(arg, where, size)
char		*arg;
VOIDST		where;
int		size;
{
    register char	*s = arg;
    register unsigned char *w = (unsigned char *) where;
    int			ival;

    ival = ab_strtol(s, &arg, 0);
    if (arg == s) {
	fprintf(stderr, "%s: bad 16-bit number \"%s\"\n", PROGNAME, s);
	warning();
	return(TRUE);
    }
    if ((ival < I16_MIN) || (ival > I16_MAX)) {
	fprintf(stderr, "%s: 16-bit number out of range \"%s\"\n",
		PROGNAME, s);
	warning();
	if (ival < I16_MIN) ival = I16_MIN;
	else ival = I16_MAX;
    }
    w[0] = ival >> 8;
    w[1] = ival & 0xff;
    return(FALSE);
}

int
ab_i32(arg, where, size)
char		*arg;
VOIDST		where;
int		size;
{
    register char	*s = arg;
    register unsigned char *w = (unsigned char *) where;
    long		ival;

    ival = ab_strtol(s, &arg, 0);
    if (arg == s) {
	fprintf(stderr, "%s: bad 32-bit number \"%s\"\n", PROGNAME, s);
	warning();
	return(TRUE);
    }
    if ((ival < I32_MIN) || (ival > I32_MAX)) {
	fprintf(stderr, "%s: 32-bit number out of range \"%s\"\n",
		PROGNAME, s);
	warning();
	if (ival < I32_MIN) ival = I32_MIN;
	else ival = I32_MAX;
    }
    w[0] = ival >> 24;
    w[1] = (ival >> 16) & 0xff;
    w[2] = (ival >> 8) & 0xff;
    w[3] = ival & 0xff;
    return(FALSE);
}

int
ab_string(arg, where, size)
char		*arg;
VOIDST		where;
int		size;
{
    strncpy((char *) where, arg, size);
    return(FALSE);
}

long
ab_strtol(s, ptr, base)
char	*s;
char	**ptr;
int	base;
{
    register int	i;

    for(i=0; i<AB_NVALS; i++) {
	if (0 == strcmp(s, ab_vals[i].cval)) {
	    *ptr = s + strlen(s);
	    return(ab_vals[i].lval);
	}
    }
    return(strtol(s, ptr, base));
}

void
warning()
{
    warns++;
    if (warns >= maxwarn) error("too many warnings");
}
