/*
  
Copyright (C) 10/1995 Klaus Hartenstein

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted, provided
that the above copyright notice appear in all copies and that both that
copyright notice and this permission notice appear in supporting
documentation.  This software is provided "as is" without express or
implied warranty.

*/


#define adr_file
#include "adr.h"

#ifdef __hpux
#define _INCLUDE_POSIX_SOURCE
#endif
#if (defined(__sgi) || defined(sgi))
#define _POSIX_SOURCE
#endif

//#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>

static char lower(char c);
     
/********************************************************************************
 * constructor
 ********************************************************************************/
dir_t::dir_t()
{
  num_addr_ = 0;
  max_addr_ = 0;
  addresses_ = (address_t **)0;
}

/********************************************************************************
 * destructor: delete all addresses
 ********************************************************************************/
dir_t::~dir_t()
{
  int i;
  for(i=0 ; i < (int)num_addr_ ; i++)
    delete addresses_[i];
}

/********************************************************************************
 * add address at the end of records
 ********************************************************************************/
void dir_t::add_addr(address_t *addr)
{
  if(num_addr_ == max_addr_){
    max_addr_ += 100;
    address_t **buffer =
      (address_t **) calloc(  sizeof(address_t *) , max_addr_);
    if(addresses_ != NULL){
      memcpy((char *)buffer, (char *)addresses_, num_addr_ * sizeof(address_t *));
      (void)free((char *)addresses_);
    }
    addresses_ = buffer;
  }
  addresses_[num_addr_++] = addr;
}

/********************************************************************************
 * add address at index
 ********************************************************************************/
void dir_t::add_addr(unsigned int i, address_t *addr)
{
  int j;
  if(num_addr_ == max_addr_){
    max_addr_ += 100;
    address_t **buffer =
      (address_t **) calloc( sizeof(address_t *) , max_addr_);
    if(addresses_ != NULL){
      memcpy((char *)buffer, (char *)addresses_, num_addr_ * sizeof(address_t *));
      (void)free((char *)addresses_);
    }
    addresses_ = buffer;
  }
  for( j=num_addr_ ; j > (int)i ; j--)
    addresses_[j] = addresses_[j-1];
  addresses_[i] = addr;
  num_addr_++;
}

/********************************************************************************
 * add address with string-data
 ********************************************************************************/
void dir_t::add_addr(char *cp)
{
  if(cp){
    address_t *addr;
    addr = new address_t();
    addr->addr(cp);
    add_addr(addr);
  }
}

/********************************************************************************
 * add address with string-data at index
 ********************************************************************************/
void dir_t::add_addr(unsigned int i, char *cp)
{
  if(cp){
    address_t *addr;
    addr = new address_t();
    addr->addr(cp);
    add_addr(i, addr);
  }
}

/********************************************************************************
 * return indexed address
 ********************************************************************************/
address_t *dir_t::addr(int i)const
{
  if((i < (int)num_addr_) && ( i >= 0))
    return addresses_[i];
  else
    return (address_t *)0;
}

/********************************************************************************
 * kill specified address
 ********************************************************************************/
void dir_t::kill_addr(int *i)
{
  unsigned int j;
  
  if((*i > (int)num_addr_) || (num_addr_ == 0))
    return;
  delete addr(*i);
  for( j = *i ; j<num_addr_ ; j++)
    addresses_[j] = addresses_[j+1];
  
  --num_addr_;
  
  if(*i != 0)
    *i = *i-1;

}

/********************************************************************************
 * kill specified address
 ********************************************************************************/
void dir_t::kill_addr(address_t *addr)
{
  int i;
  for( i=0 ; i < (int)num_addr_ ; i++)
    if(addresses_[i] == addr){
      kill_addr(&i);
      return;
    }
}

/********************************************************************************
 * return number of stored addresses
 ********************************************************************************/
unsigned int dir_t::num_addr(void)const
{
  return num_addr_;
}

/********************************************************************************
 * load addresses from file
 ********************************************************************************/
void dir_t::load_addr(char *filename)
{
  FILE *fp;
  if((fp = fopen(filename, "r")) == NULL)
    fprintf(stderr, "couldn't open file %s for reading\n", filename);
  else{
    char *buf;
    struct stat file_stat;
    
    if((fstat((int)fileno(fp), &file_stat)) < 0){
      perror("fstat");
      return;
    }
    buf = (char *)calloc((unsigned)sizeof(char) , (unsigned)file_stat.st_size*2);
    size_t r = fread((char *)buf, sizeof(char), (size_t)file_stat.st_size, fp);
    
    if(r != (size_t)file_stat.st_size){
      fprintf(stderr, "read error on %s\n", filename);
      fclose(fp);
      free(buf);
      return;
    }
    else{                                                       /* read success */
      char *beg = buf;                              /* beginning of one address */
      int i;
      for( i=0 ; i<file_stat.st_size ; i++){
	if(buf[i] == ''){
	  if(i>0){
	    if(buf[i-1]=='\n')
	      buf[i-1]='\0';
	  }
	  buf[i] = '\0';
	  add_addr(beg);
	  beg = &buf[i+1];
	}
      }
    }
    fclose(fp);
    free(buf);
  }
}

/********************************************************************************
 * save all addresses in file 
 ********************************************************************************/
void dir_t::save_addr(char *filename)const
{
  int i;
  FILE *fp;
  if((fp = fopen(filename, "w")) == NULL)
    fprintf(stderr, "couldn't open file %s for writing\n", filename);
  else{
    for( i=0 ; i < (int)num_addr_ ; i++){
      fprintf(fp, "%s", addr(i)->addr());
      fprintf(fp, "\n");
    }
    fclose(fp);
  }
}

/********************************************************************************
 * search for str in all addresses
 ********************************************************************************/

int dir_t::search(char *str, int index)const
{
  int i, j;
  short flag=0;
  char *cp = 0;
  char *lstr = 0;
  
  if(index==-1)                                 /* index parameter specified ? */
    index=0;
  else
    index++;

  lstr = (char *)calloc(sizeof(char), strlen(str)+1);
  for(j=0 ; j <= (int)strlen(str) ; j++)
    lstr[j]=str[j];

  for( j=0 ; j < (int)strlen(lstr) ; j++)
    lstr[j] = lower(lstr[j]);
  
  for( i=index ; i < (int)num_addr_ ; i++){
    cp = (char *)calloc(sizeof(char), strlen(addr(i)->addr())+1);
    strcpy(cp, addr(i)->addr());
    for( j=0 ; j < (int)strlen(cp); j++)
      cp[j] = lower(cp[j]);
    if(strstr(cp, lstr)){
      free(cp);
      flag = 1;
      break;
    }
    free(cp);
  }
  free(lstr);

  if(flag)
    return i;
  else
      return -1;
}

static char lower(char c)                   /* returns lowercase of given char */
{                                           /* also works with 8-bit chars     */
  if((((int)c) >= 65) && ((int)c <= 90))
    return (char)c+32;
  return c;
}
/********************************************************************************
 * returns 1 of one of addresses has changed; 0 otherwise
 ********************************************************************************/
int dir_t::changed(void)const
{
  int i,j=0;
  for(i=0 ; i < (int)num_addr_ ; i++)
    if(addresses_[i]->changed()){
      return j;
//      j=1;
    }
  return j;
}

/********************************************************************************
 * return next record
 ********************************************************************************/
address_t *dir_t::next(int *i)const
{
  if(num_addr_ <= 0)
    return(address_t *)0;
  if( *i < (int)num_addr_ -1 )
    *i = *i+1;
  return(addr(*i));
}

/********************************************************************************
 * return prev record
 ********************************************************************************/
address_t *dir_t::prev(int *i)const
{
  if(num_addr_ <= 0)
    return(address_t *)0;
  if( *i > 0 )
    *i = *i-1;
  return(addr(*i));
}

/********************************************************************************/
/******** class address_t *******************************************************/

/********************************************************************************
 * constructor
 ********************************************************************************/
address_t::address_t()
{
  data_ = (char *)0;
  key_start_ = 0;
  key_end_ = 0;
  deleted_ = 0;
  data_length_ = 0;
  changed_ = 0;
}

/********************************************************************************
 * destructor
 ********************************************************************************/
address_t::~address_t()
{
  if(data_) free(data_);
}

/********************************************************************************
 * set or update address data
 ********************************************************************************/
void address_t::addr(char *cp)
{
  if(cp){
    if(data_ != (char *)0)                                       /* just update */
      free(data_);
    data_ = (char *)calloc(strlen(cp)+1, sizeof(char));
    sprintf(data_, "%s", cp);
    data_length_ = strlen(cp);
    changed_ = 0;
  }
}

/********************************************************************************
 * get address data
 ********************************************************************************/
char *address_t::addr(void)const
{
  return data_;
}

/********************************************************************************
 * set deleted flag of address
 ********************************************************************************/
void address_t::deleted(unsigned short val)
{
  deleted_ = val;
}

/********************************************************************************
 * get deleted status of address
 ********************************************************************************/
unsigned short address_t::deleted(void)
{
  return deleted_;
}

/********************************************************************************
 * set changed field 
 ********************************************************************************/
void address_t::changed(unsigned short i)
{
  changed_ = i;
}

/********************************************************************************
 * get changed status
 ********************************************************************************/
unsigned short address_t::changed(void)
{
  return(changed_);
}
