/*
  Xquote Copyright (c)1997 Mark Buser,
  All Rights Reserved.

  Permission is hereby granted to copy
  and freely distribute copies of this
  program for non-commercial purposes
  without fee, provided that this notice
  appears in all copies.
  
  All redistributions must be in their
  entirety as originally distributed.
  Feel free to modify Xquote, but
  modified versions may not be distributed
  without prior consent of the author.
  
  This software is provided 'as-is'
  without any express or implied warranty.
  In no event will the author be held
  liable for any damages resulting from
  the use of this software.

  $Revision: 1.12 $ $Date: 1997/10/31 23:08:04 $
*/

#include <sys/types.h>
#include <stdlib.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <sys/socket.h>
#include <netdb.h>
#include <errno.h>
#include <fcntl.h>
#include <string.h>
#include <time.h>

#include <X11/Xos.h>
#include <Xm/XmAll.h>

#include "about.h"
#include "optdetail.h"
#include "optnet.h"
#include "opttick.h"
#include "optupdate.h"
#include "parse.h"
#include "remote.h"
#include "server.h"
#include "status.h"
#include "tape.h"
#include "xutil.h"
#include "xquote.h"

/* Extern */
extern Widget Toplevel;
extern char time_msg[80];

/* Globals */
int netFinished = True;
XtWorkProcId work;

/* Local Globals */
Widget optmenu;

/* Pending vars */
static int pServer;

/*
** Net option support
*/

static void netTellXinvest ()
{
  Display *dpy = XtDisplay (Toplevel);
  int i;
  char *price;
  char date[11];

  double nav;
  time_t t = time ( (time_t *)NULL ); 
  struct tm *now = localtime(&t);

  char **commands = NULL;
  int num_commands = 0;

#if !defined (NO_XMU)
  write_status_line ("Contacting Xinvest...");
  XmUpdateDisplay(Toplevel);

  /* 
  ** Send "NAV ticker mm/dd/yyyy fffff.fff" for each ticker.
  */ 
  for (i=0; i < tickGetNum(); i++) {
    QUERY_STRUCT *data = tickGetDetail(i);
    if (data && data->values[PRICE]) {

      price = data->values[PRICE];
      if (strcmp (price, "N/A") == 0) {     /* no price */
        continue;
      } else if (strchr (price, '/')) {     /* fraction or decimal */
	int whole, num, den;
	sscanf( price, "%d %d / %d", &whole, &num, &den);
	nav = (double)whole + ((double)num/(double)den);
      } else
	nav = strtod (price, NULL);

      /* Dates from the net could be anything, punt and use today */
      strftime ( date, 11, "%m/%d/%Y", now);

      /* Add new command plus NULL command */
      commands = (char **)XtRealloc ( (char *)commands, 
		                      (num_commands+2) * sizeof(char *));
      commands[num_commands] = XtCalloc ( 40, sizeof(char) );
      sprintf ( commands[num_commands++], "NAV %s %s %.3f", 
		tickGetName(i), date, nav);
      commands[num_commands] = NULL;
    }
  }

  if (num_commands)
    xinvest_remote_commands ( dpy, commands);

  for (i=0; i < num_commands; i++) 
    XtFree (commands[i]);
  XtFree ((char *)commands);
#endif
}

static char *makeQuery (int proxy, int tick)
{
  char *url, *name;
  char *insert, *nohost;
  static char buf[BUFSIZ];

  if (tick != WORK_NEWS) {
    setServerType (tickGetType(tick));
    url = getServer(URL);
    name = tickGetName (tick);
  } else {
    url = "sunsite.auc.dk/SYMBOL/";
    name = "xinvest";
  }
  insert = strstr (url, "SYMBOL");
  nohost = strchr (url, '/');
  if (insert && nohost) {

    if (proxy)
      sprintf ( buf, "GET http://%.*s%s%s HTTP/1.0\r\n"
                     "User-Agent: Xquote/1.0\r\n"
                     "Accept: */*\r\n"
                     "\r\n",
  	        insert-url, url, name, insert + strlen("SYMBOL") );
    else
      sprintf ( buf, "GET %.*s%s%s HTTP/1.0\r\n"
                     "User-Agent: Xquote/1.0\r\n"
                     "Accept: */*\r\n"
                     "\r\n",
  	        insert-nohost, nohost, name, insert + strlen("SYMBOL") );

  } else 
    strcpy (buf, "");

#if 0
  fprintf (stderr, buf);
#endif

  return (buf);
}


static int readQuery(int s, char **html) 
{
  int num;
  int size = 0;
  char buf[BUFSIZ];
	  
  if (*html)
    size = strlen(*html);
  else
    size = 0;

  /* Read the page */
  if ( (num = read (s, buf, BUFSIZ-1)) > 0) {
    buf[num] = '\0';
    *html = XtRealloc ( *html, (size + num +1)*sizeof(char) );
    if (*html) {
      if (size == 0)
        strcpy ( *html, buf);
      else
        strcat ( *html, buf);
    } else {
      write_status ("Could not allocate memory for HTML.", ERR);
      return (0);
    }
  }

#if 0
  fprintf (stderr, buf);
#endif

  return (num);
}


#define IDLE    0
#define SOCKET  1
#define CONNECT 2
#define READ    3
#define WRITE   4
#define PARSE   5
#define TEST    6
#define UPDATE  7
#define END     8

int workNet (int command)
{
  static int state = IDLE;
  static int i = 0;

  static int s;
  int status;
  static struct sockaddr_in serv_addr;
  struct servent *sp;
  struct hostent *hp;
  char **name;
  int size;

  static int proxy;
  static char *serv_machine;
  static int port;

  static char *html;

  if (command == WORK_CLEAN)
    state = END;

  AllowShellResize (Toplevel, DISALLOW);

  switch (state) {

    case IDLE:
      cylonStart (True);

      /* This assumes the host is the same regardless of the server type */
      i = 0;
      setServerType (tickGetType(i));

      /* 
      ** Determine server and port
      */
      proxy = True;
      getProxy (&serv_machine, &port);
      if (serv_machine == NULL) {                    /* No proxy */
        if (command == WORK_NEWS) {
          serv_machine = XtNewString ("sunsite.auc.dk");
        } else {
          size = strcspn ( getServer(URL), "/");     /* Get machine from URL */
          serv_machine = XtCalloc ( size+1, sizeof(char));
          sprintf ( serv_machine, "%.*s", size, getServer(URL));
        }

        if ( (sp = getservbyname ("http", "tcp")) )  /* Get port from lookup */
          port = ntohs(sp->s_port);
        else
          port = 80;                                 /* or default */
        proxy = False;
      }
      sprintf (errmsg, "Connecting to %s...", serv_machine);
      write_status_line (errmsg);

      state = SOCKET;
      netFinished = False;
      return (False);
      /* break; */

    case SOCKET:
      /* 
      ** Get server address struct.
      */
      memset ( (char *)&serv_addr, 0, sizeof(serv_addr) );
      serv_addr.sin_port = htons(port);

      if ((serv_addr.sin_addr.s_addr = inet_addr(serv_machine)) != -1) {
        /* Dotted number address */
        serv_addr.sin_family = AF_INET;   

        s = socket (serv_addr.sin_family, SOCK_STREAM, 0);
        if ( s < 0 ) {
	  sprintf (errmsg, "Socket create failed:\n%.*s", 
		   (int)XtNumber(errmsg)-23, strerror(errno));
          write_status (errmsg, ERR);
          state = END;
          return (False);
        }
      } else {
        /* Dotted name address */
        if ((hp = gethostbyname(serv_machine)) == NULL) {
          sprintf(errmsg, "Unknown host: '%s'.\n", serv_machine);
          write_status ( errmsg, ERR);
          state = END;
          return (False);
        }
        serv_addr.sin_family = hp->h_addrtype;
 
#ifdef h_addr
        for (s = 0, name = hp->h_addr_list; *name; name++) {
          memcpy((char *) &serv_addr.sin_addr, *name, hp->h_length);

          if ((s = socket (serv_addr.sin_family, SOCK_STREAM, 0)) >= 0) 
            break;     /* Got one */
        }
#else
        memcpy((char *) &serv_addr.sin_addr, hp->h_addr, hp->h_length);
        s = socket (serv_addr.sin_family, SOCK_STREAM, 0);
        if ( s < 0 ) {
	  sprintf (errmsg, "Socket create failed:\n%.*s", 
		   XtNumber(errmsg)-23, strerror(errno));
          write_status (errmsg, ERR);
          state = END;
          return (False);
        }
#endif
      }
 
      /* Set non blocking, not fatal if we can't */
      if ( (status = fcntl (s, F_GETFL)) != -1 ) {
	status |= O_NONBLOCK;
        fcntl (s, F_SETFL, status);
      }

      state = CONNECT;
      return (False);
      /* break; */

    case CONNECT:
      /*
      ** Connect to http server 
      */
      if (connect(s, (struct sockaddr *)&serv_addr, sizeof (serv_addr)) < 0) {
	if (errno == EISCONN)
          state = WRITE;
        else if (errno == EALREADY || errno == EINPROGRESS || errno == EAGAIN)
          state = CONNECT;
	else {
          state = END;
	  sprintf (errmsg, "Socket connect failed:\n%.*s", 
		   (int)XtNumber(errmsg)-24, strerror(errno));
          write_status (errmsg, ERR);
          close(s);
        }
      } else
        state = WRITE;

      if (state == WRITE) {
        if (command == WORK_NEWS)
	  html = makeQuery ( proxy, WORK_NEWS);
        else
	  html = makeQuery ( proxy, i);
      }

      return (False);
      /* break; */

    case WRITE:

      status = write (s, html, strlen(html));
      if (status < 0) {                     /* ERROR */
	if (errno == EAGAIN) {    /* no, just can't write yet */
	  state = WRITE;
        } else {
	  sprintf (errmsg, "Write failed:\n%.*s", 
		 (int)XtNumber(errmsg)-15, strerror(errno));
	  write_status (errmsg, ERR);
	  html = NULL;
	  state = END;
	}
      } else if (status == strlen(html)) {  /* DONE */
        sprintf ( errmsg, "Reading %s...", 
		  (command!=WORK_NEWS) ? tickGetName(i):"Xquote News");
        write_status_line (errmsg);
        html = NULL;
        state = READ;
      } else {                              /* NOT DONE */
	html += status;
	state = WRITE;
      }

      return (False);
      /* break; */

    case READ:

      /* Read the query response */
      status = readQuery (s, &html);
      if (status < 0) {                  /* ERROR */
	if (errno == EAGAIN) {    /* no, just no data yet */
	  state = READ;
        } else {
	  sprintf (errmsg, "Read failed:\n%.*s", 
		 (int)XtNumber(errmsg)-14, strerror(errno));
	  write_status (errmsg, ERR);
	  XtFree (html);
	  html = NULL;
	  state = END;
	}
      } else if (status == 0) {  /* DONE */
	char *start, *end;
	int code, num;

        /* Check for error codes 'HTTP/1.0 200 OK\r\n' */
	if (html) {
	  sscanf ( html, "HTTP/%*f %d %n", &code, &num);
	  if (code != 200) {
	    start = html + num;
	    if ( (end = strchr ( start, '\r')) == NULL)
	      end = start + 20;
	    sprintf ( errmsg, "%s: %d %.*s", 
		     tickGetName(i), code, end-start, start);
	    write_status (errmsg, ERR);
	  }
	}
        state = PARSE;
      } else {                  /* MAYBE MORE */
        state = READ;
      }
      return (False);
      /* break; */

    case PARSE:
      close (s);
      if (command != WORK_NEWS) {
        if (html)
          tickSetDetail ( i, findPattern(html) );
        state = TEST;
      } else {
        aboutText( findNews(html) );
        state = END;
      }
      XtFree(html);
      html = NULL;

      return (False);
      /* break; */

    case TEST:
      if ( ++i < tickGetNum() ) {
        state = SOCKET;
      } else {
        detailUpdateMainWindow ();
        makeTextTape();
        state = UPDATE;
      }
      return (False);
      /* break; */

    case UPDATE:
      netTellXinvest();
      state = END;
      return (False);
      /* break; */

    case END:
    default:
      if (!proxy)
        XtFree (serv_machine);
      sprintf (errmsg, "%s%.*s", version, 
               80-(int)strlen(version), time_msg);
      write_status_line (errmsg);

      cylonStart (False);

      state = IDLE;
      netFinished = True;
      work = (XtWorkProcId) NULL;
      return (True);
      /* break; */
  }
}
#undef IDLE
#undef SOCKET
#undef CONNECT
#undef READ
#undef PARSE
#undef TEST
#undef UPDATE
#undef END

/* ARGSUSED */
static void serverNet (Widget w, XtPointer client_data, XtPointer call_data)
{
  pServer = (int) client_data;
}


/* ARGSUSED */
void procNet (Widget w, XtPointer which, XtPointer call_data)
{
  static Widget proxy = (Widget) NULL, port;
  static char *server, *portno;

  if (proxy == (Widget)NULL) {
    proxy = XtNameToWidget (GetTopShell(w),
                            "NetPane.NetRow.NetProxy.form_1.text_0");
    port = XtNameToWidget  (GetTopShell(w),
                            "NetPane.NetRow.NetProxy.form_1.text_1");
  }

  switch ((int) which) {

    case 0: /* Ok */
	    { 
              XtPopdown (GetTopShell(w));
              /* 
              ** PROXY
              */
	      if (server) XtFree (server);
	      if (portno) XtFree (portno);
	      server = XmTextFieldGetString (proxy);
	      portno = XmTextFieldGetString (port);
	      if (strlen(server)) {
                if (strlen(portno))
                  setProxy (server, portno);
	      } else {
                setProxy (NULL, "80");
              }

	      /*
	      ** Current Server
	      */
              serverCB ((Widget)NULL, (XtPointer)pServer, (XtPointer)NULL);

	    }
            break;

    case 1: /* Cancel */
            XtPopdown (GetTopShell(w));

	    /* Restore previous values */
	    if (server)
	      XmTextFieldSetString (proxy, server);
	    if (portno)
	      XmTextFieldSetString (port, portno);

            { 
              Widget menu, cascade;
              WidgetList buttons;
              XmString label;
              int value = getCurServer();

              /* Change menu history */
              XtVaGetValues (optmenu, XmNsubMenuId, &menu, NULL);
              XtVaGetValues (menu, XmNchildren, &buttons, NULL);
              XtVaSetValues (optmenu, XmNmenuHistory, buttons[value], NULL);

              /* Change cascade button label */
              cascade = XmOptionButtonGadget (optmenu);
              XtVaGetValues (buttons[value], XmNlabelString, &label, NULL);
              XtVaSetValues (cascade, XmNlabelString, label, NULL);

              XmUpdateDisplay (cascade);
            }
            break;

    default: break;
  }
}

Widget createNetDialog ()
{
  char *frames[] =  {"NetSource", "NetProxy"};
  char *forms[] =   {"form_0", "form_1"};
  char *buttons[] = {"button_0", "button_1"};

  Widget dialog, form, pane, row, frame, button, label, text;
  int num;

  Dimension width, height, border;

  dialog = XtVaCreatePopupShell ("OptionNet", 
                           xmDialogShellWidgetClass,
                           GetTopShell(Toplevel),
                           NULL);

  pane = XtVaCreateWidget ("NetPane", xmPanedWindowWidgetClass, 
                           dialog,
                           XmNsashWidth, 1,
                           XmNsashHeight, 1,
                           NULL);

  row = XtVaCreateWidget ("NetRow", xmRowColumnWidgetClass, pane,
                          XmNnavigationType, XmNONE,
                          XmNtopAttachment, XmATTACH_FORM,
                          XmNleftAttachment, XmATTACH_FORM,
                          NULL);

  for (num = 0; num < XtNumber(frames); num ++) {
    
     frame =   XtVaCreateManagedWidget( frames[num],
                          xmFrameWidgetClass, row,
                          NULL);
     XtVaCreateManagedWidget ( frames[num],
                          xmLabelGadgetClass, frame,
                          XmNchildType, XmFRAME_TITLE_CHILD,
                          XmNchildVerticalAlignment, XmALIGNMENT_CENTER,
                          NULL);
     form = XtVaCreateWidget ( forms[num], 
                            xmFormWidgetClass, frame, 
                            NULL );

     switch (num) {

        case 0: {
		  Widget menu, cascade;
                  Widget visible = (Widget)NULL;
		  XmString optlabel;
		  int servers;

		  menu = XmCreatePulldownMenu ( form, "NetOptionMenu_pulldown", 
			     NULL, 0);
		  optmenu = XmCreateOptionMenu (form, "NetOptionMenu", NULL, 0);

                  /* Add menu items */
		  for (servers = 0; servers < numServer(); servers++) {
		    button = XtVaCreateManagedWidget ( getServerTitle(servers),
		                       xmPushButtonGadgetClass, menu, NULL);
		    XtAddCallback (button, XmNactivateCallback, serverNet,
		  	           (XtPointer)servers);
                    if (servers == getCurServer())
                      visible = button;
		  }

                  /* Make server label the one shown */
		  optlabel = XmStringCreateLocalized (
				        getServerTitle (getCurServer()));
		  cascade = XmOptionButtonGadget (optmenu);
		  XtVaSetValues (cascade, XmNlabelString, optlabel, NULL);
		  XmStringFree (optlabel);

                  /* Make the active menu item the one shown */
		  XtVaSetValues (optmenu, 
		                 XmNsubMenuId, menu,
                                 XmNmenuHistory, visible, 
                                 NULL);

		  setServerType (0);
                  XtManageChild (optmenu);
		}
                break;

        case 1: {
	          char *host, port[12];
		  int portno;

		  getProxy (&host, &portno);
		  if (host)
		    sprintf ( port, "%d", portno );

                  label = XtVaCreateManagedWidget ( "label_0",
                            xmLabelGadgetClass, form,
                            XmNleftAttachment, XmATTACH_FORM,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
                  text = XtVaCreateManagedWidget ( "text_0",
                            xmTextFieldWidgetClass, form,
                            XmNleftAttachment, XmATTACH_WIDGET,
                            XmNleftWidget, label,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
		  if (host)
		    XmTextFieldSetString (text, host);

                  label = XtVaCreateManagedWidget ( "label_1",
                            xmLabelGadgetClass, form,
                            XmNleftAttachment, XmATTACH_WIDGET,
                            XmNleftWidget, text,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
                  text = XtVaCreateManagedWidget ( "text_1",
                            xmTextFieldWidgetClass, form,
                            XmNleftAttachment, XmATTACH_WIDGET,
                            XmNleftWidget, label,
                            XmNtopAttachment, XmATTACH_FORM,
                            XmNbottomAttachment, XmATTACH_FORM,
                            NULL );
		  if (host)
		    XmTextFieldSetString (text, port);
		}
                break;

     }
     XtManageChild (form);

  }

  XtManageChild (row);

  /* Buttons to add delete cancel exit */
  form = XtVaCreateWidget ( "ButForm", xmFormWidgetClass, pane,
                            XmNfractionBase, 5,
                            NULL );

  for (num=0; num < XtNumber(buttons); num++) {
    button = XtVaCreateManagedWidget ( buttons[num], 
                                       xmPushButtonWidgetClass, form,
                                       XmNtopAttachment, XmATTACH_FORM,
                                       XmNbottomAttachment, XmATTACH_FORM,
                                       XmNleftAttachment, XmATTACH_POSITION,
                                       XmNleftPosition, 2*num+1,
                                       XmNrightAttachment, XmATTACH_POSITION,
                                       XmNrightPosition, 2*num+2,
                                       XmNshowAsDefault, (num==0)?True:False,
                                       XmNdefaultButtonShadowThickness, 1,
                                       NULL );
    XtAddCallback ( button, XmNactivateCallback,
                            (XtCallbackProc) procNet, (XtPointer)num );
  }
  XtManageChild (form);
  XtManageChild (pane);

  /* Prevent pane from changing size */
  XtVaGetValues ( dialog, 
                  XmNwidth, &width,
                  XmNheight, &height,
                  XmNborderWidth, &border,
                  NULL );

  XtVaSetValues ( dialog,
                  XmNminWidth,  width +  border,
                  XmNmaxWidth,  width +  border,
                  XmNminHeight, height + border,
                  XmNmaxHeight, height + border,
                  NULL );
  return (dialog);
}
