/*
  Xquote is copyright 1997 by Mark Buser.

  Permission is hereby granted to copy
  and freely distribute copies of this
  program for non-commercial purposes
  without fee, provided that this notice
  appears in all copies.
  
  All redistributions must be in their
  entirety as originally distributed.
  Feel free to modify Xquote, but
  modified versions may not be distributed
  without prior consent of the author.
  
  This software is provided 'as-is'
  without any express or implied warranty.
  In no event will the author be held
  liable for any damages resulting from
  the use of this software.

  $Revision: 1.4 $ $Date: 1997/05/30 21:38:46 $

  Modified from the original Netscape source.

*/
/* -*- Mode:C; tab-width: 8 -*-
 * remote.c --- remote control of Netscape Navigator for Unix.
 * version 1.1.3, for Netscape Navigator 1.1 and newer.
 *
 * Copyright  1996 Netscape Communications Corporation, all rights reserved.
 * Created: Jamie Zawinski <jwz@netscape.com>, 24-Dec-94.
 *
 * Permission to use, copy, modify, distribute, and sell this software and its
 * documentation for any purpose is hereby granted without fee, provided that
 * the above copyright notice appear in all copies and that both that
 * copyright notice and this permission notice appear in supporting
 * documentation.  No representations are made about the suitability of this
 * software for any purpose.  It is provided "as is" without express or 
 * implied warranty.
 *
 * To compile:
 *
 *    cc -o netscape-remote remote.c -DSTANDALONE -lXmu -lX11
 *
 * To use:
 *
 *    netscape-remote -help
 *
 * Documentation for the protocol which this code implements may be found at:
 *
 *    http://home.netscape.com/newsref/std/x-remote.html
 *
 * Bugs and commentary to x_cbug@netscape.com.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>

#include <X11/Xlib.h>
#include <X11/Xatom.h>

#if !defined (NO_XMU)
#include <X11/Xmu/WinUtil.h>	/* for XmuClientWindow() */
#endif


/* vroot.h is a header file which lets a client get along with `virtual root'
   window managers like swm, tvtwm, olvwm, etc.  If you don't have this header
   file, you can find it at "http://home.netscape.com/newsref/std/vroot.h".
   If you don't care about supporting virtual root window managers, you can
   comment this line out.
 */
#include "vroot.h"
#include "status.h"


#ifdef STANDALONE
 static const char *progname = 0;
 static const char *expected_xinvest_version = "1.0";
#else  /* !STANDALONE */
 const char *progname = "Xinvest";
 const char *expected_xinvest_version = "2.4";
#endif /* !STANDALONE */

#define XINVEST_VERSION_PROP   "_XINVEST_VERSION"
#define XINVEST_LOCK_PROP      "_XINVEST_LOCK"
#define XINVEST_COMMAND_PROP   "_XINVEST_COMMAND"
#define XINVEST_RESPONSE_PROP  "_XINVEST_RESPONSE"
static Atom XA_XINVEST_VERSION  = 0;
static Atom XA_XINVEST_LOCK     = 0;
static Atom XA_XINVEST_COMMAND  = 0;
static Atom XA_XINVEST_RESPONSE = 0;

#if !defined (NO_XMU)
static void
xinvest_remote_init_atoms (Display *dpy)
{
  if (! XA_XINVEST_VERSION)
    XA_XINVEST_VERSION = XInternAtom (dpy, XINVEST_VERSION_PROP, False);
  if (! XA_XINVEST_LOCK)
    XA_XINVEST_LOCK = XInternAtom (dpy, XINVEST_LOCK_PROP, False);
  if (! XA_XINVEST_COMMAND)
    XA_XINVEST_COMMAND = XInternAtom (dpy, XINVEST_COMMAND_PROP, False);
  if (! XA_XINVEST_RESPONSE)
    XA_XINVEST_RESPONSE = XInternAtom (dpy, XINVEST_RESPONSE_PROP, False);
}

static Window
xinvest_remote_find_window (Display *dpy)
{
  int i;
  Window root = RootWindowOfScreen (DefaultScreenOfDisplay (dpy));
  Window root2, parent, *kids;
  unsigned int nkids;
  Window result = 0;
  Window tenative = 0;
  unsigned char *tenative_version = 0;
  static int found = False;

  if (! XQueryTree (dpy, root, &root2, &parent, &kids, &nkids))
    {
      sprintf (errmsg, "XQueryTree failed on display %s",
	       DisplayString (dpy));
      write_status (errmsg, ERR);;
      return ((Window)0);
    }

  /* root != root2 is possible with virtual root WMs. */

  if (! (kids && nkids))
    {
      sprintf (errmsg, "Root window has no children on display %s",
	       DisplayString (dpy));
      write_status (errmsg, ERR);;
      return ((Window)0);
    }

  for (i = nkids-1; i >= 0; i--)
    {
      Atom type;
      int format;
      unsigned long nitems, bytesafter;
      unsigned char *version = 0;
      Window w = XmuClientWindow (dpy, kids[i]);
      int status = XGetWindowProperty (dpy, w, XA_XINVEST_VERSION,
				       0, (65536 / sizeof (long)),
				       False, XA_STRING,
				       &type, &format, &nitems, &bytesafter,
				       &version);
      if (! version)
	continue;
      if (strcmp ((char *) version, expected_xinvest_version) &&
	  !tenative)
	{
	  tenative = w;
	  tenative_version = version;
	  continue;
	}
      XFree (version);
      if (status == Success && type != None)
	{
	  result = w;
	  break;
	}
    }

  if (result && tenative)
    {
      sprintf (errmsg,
	       "Warning: both version %s (0x%x) and version\n"
	       "\t%s (0x%x) are running.  Using version %s.",
	       tenative_version, (unsigned int) tenative,
	       expected_xinvest_version, (unsigned int) result,
	       expected_xinvest_version);
      write_status (errmsg, WARN);
      XFree (tenative_version);
      return result;
    }
  else if (tenative)
    {
      sprintf (errmsg,
	       "Warning: expected version %s but found version\n"
	       "\t%s (0x%x) instead.",
	       expected_xinvest_version,
	       tenative_version, (unsigned int) tenative);
      write_status (errmsg, WARN);
      XFree (tenative_version);
      return tenative;
    }
  else if (result)
    {
      found = True;
      return result;
    }
  else
    {
      sprintf (errmsg, "%s not found on display %s.", progname,
	       DisplayString (dpy));
      write_status_line (errmsg);
      if (found == True) {
        write_status (errmsg, INFO);
        found = False;
      }
      return ((Window)0);
    }
}

/* ARGSUSED */
static void bad_window_handler(Display *disp, XErrorEvent *err)
{
  /* The window id we saved is probably no good anymore */
  return;
}

static int
xinvest_remote_check_window (Display *dpy, Window window)
{
  Atom type;
  int format;
  unsigned long nitems, bytesafter;
  unsigned char *version = 0;
  int status;
  int (*old_handler)();

  old_handler = XSetErrorHandler( (XErrorHandler)bad_window_handler);
  status = XGetWindowProperty (dpy, window, XA_XINVEST_VERSION,
				   0, (65536 / sizeof (long)),
				   False, XA_STRING,
				   &type, &format, &nitems, &bytesafter,
				   &version);
  (void) XSetErrorHandler(old_handler);

  if (status != Success || !version)
    {
      sprintf (errmsg, "Window 0x%x is not an Xinvest window.",
	       (unsigned int) window);
      write_status (errmsg, ERR);
      return (False);
    }
  else if (strcmp ((char *) version, expected_xinvest_version))
    {
      sprintf (errmsg,
	       "Warning: window 0x%x is Xinvest version %s;\n"
	       "expected version %s.",
	       (unsigned int) window,
	       version, expected_xinvest_version);
      write_status (errmsg, WARN);
      return (False);
    }
  XFree (version);
  return (True);
}


static char *lock_data = 0;

static int
xinvest_remote_obtain_lock (Display *dpy, Window window)
{
  Bool locked = False;
  Bool waited = False;

  if (! lock_data)
    {
      lock_data = (char *) malloc (255);
      sprintf (lock_data, "pid %d at ", getpid ());
      if (gethostname (lock_data + strlen (lock_data), 100))
	{
	  sprintf ( errmsg, "gethostname: %s", strerror(errno) );
	  write_status (errmsg, ERR);
	  return False;
	}
    }

  do
    {
      int result;
      Atom actual_type;
      int actual_format;
      unsigned long nitems, bytes_after;
      unsigned char *data = 0;

      XGrabServer (dpy);   /* ################################# DANGER! */

      result = XGetWindowProperty (dpy, window, XA_XINVEST_LOCK,
				   0, (65536 / sizeof (long)),
				   False, /* don't delete */
				   XA_STRING,
				   &actual_type, &actual_format,
				   &nitems, &bytes_after,
				   &data);
      if (result != Success || actual_type == None)
	{
	  /* It's not now locked - lock it. */
#ifdef DEBUG_PROPS
	  fprintf (stderr, "%s: (writing " XINVEST_LOCK_PROP
		   " \"%s\" to 0x%x)\n",
		   progname, lock_data, (unsigned int) window);
#endif
	  XChangeProperty (dpy, window, XA_XINVEST_LOCK, XA_STRING, 8,
			   PropModeReplace, (unsigned char *) lock_data,
			   strlen (lock_data));
	  locked = True;
	}

      XUngrabServer (dpy); /* ################################# danger over */
      XSync (dpy, False);

      if (! locked)
	{
	  /* We tried to grab the lock this time, and failed because someone
	     else is holding it already.  So, wait for a PropertyDelete event
	     to come in, and try again. */

	  sprintf (errmsg, 
                   XINVEST_LOCK_PROP" is held by %s.\n"
                   "If this process is no longer running you may break\n"
                   "the lock manually: 'xprop -remove " XINVEST_LOCK_PROP "'\n"
                   "and click on the Xinvest window.",
                   data );
	  write_status (errmsg, INFO);
          /* 
          ** Relax the protocol no need to get into a snit and
          ** spin on the lock. 
          */
          return False;   

	  waited = True;

	  while (1)
	    {
	      XEvent event;
	      XNextEvent (dpy, &event);
	      if (event.xany.type == DestroyNotify &&
		  event.xdestroywindow.window == window)
		{
		  sprintf (errmsg, "Window 0x%x unexpectedly destroyed.",
			   (unsigned int) window);
		  write_status (errmsg, ERR);
		  return False;
		}
	      else if (event.xany.type == PropertyNotify &&
		       event.xproperty.state == PropertyDelete &&
		       event.xproperty.window == window &&
		       event.xproperty.atom == XA_XINVEST_LOCK)
		{
		  /* Ok!  Someone deleted their lock, so now we can try
		     again. */
#ifdef DEBUG_PROPS
		  fprintf (stderr, "Window 0x%x unlocked, trying again...\n",
			   (unsigned int) window);
#endif
		  break;
		}
	    }
	}
      if (data)
	XFree (data);
    }
  while (! locked);

  if (waited) {
    sprintf (errmsg, "Obtained lock.");
    write_status (errmsg, INFO);
  }
  return True;
}


static void
xinvest_remote_free_lock (Display *dpy, Window window)
{
  int result;
  Atom actual_type;
  int actual_format;
  unsigned long nitems, bytes_after;
  unsigned char *data = 0;

#ifdef DEBUG_PROPS
	  fprintf (stderr, "%s: (deleting " XINVEST_LOCK_PROP
		   " \"%s\" from 0x%x)\n",
		   progname, lock_data, (unsigned int) window);
#endif

  result = XGetWindowProperty (dpy, window, XA_XINVEST_LOCK,
			       0, (65536 / sizeof (long)),
			       True, /* atomic delete after */
			       XA_STRING,
			       &actual_type, &actual_format,
			       &nitems, &bytes_after,
			       &data);
  if (result != Success)
    {
      sprintf (errmsg, "Unable to read and delete " XINVEST_LOCK_PROP
	       " property");
      write_status (errmsg, ERR);
      return;
    }
  else if (!data || !*data)
    {
      sprintf (errmsg, "Invalid data on " XINVEST_LOCK_PROP
	       " of window 0x%x.",
	       (unsigned int) window);
      write_status (errmsg, ERR);
      return;
    }
  else if (strcmp ((char *) data, lock_data))
    {
      sprintf (errmsg, 
               "Unexpected data on "XINVEST_LOCK_PROP".\n"
	       "Expected \"%s\", found \"%s\"!",
	       lock_data, data);
      write_status (errmsg, ERR);
      return;
    }

  if (data)
    XFree (data);
}


static int
xinvest_remote_command (Display *dpy, Window window, const char *command)
{
  int result;
  Bool done = False;
  char *new_command = 0;

#ifdef DEBUG_PROPS
  fprintf (stderr, "%s: (writing " XINVEST_COMMAND_PROP " \"%s\" to 0x%x)\n",
	   progname, command, (unsigned int) window);
#endif

  XChangeProperty (dpy, window, XA_XINVEST_COMMAND, XA_STRING, 8,
		   PropModeReplace, (unsigned char *) command,
		   strlen (command));

  while (!done)
    {
      XEvent event;
      XNextEvent (dpy, &event);
      if (event.xany.type == DestroyNotify &&
	  event.xdestroywindow.window == window)
	{
	  /* Print to warn user...*/
	  sprintf (errmsg, "Window 0x%x was destroyed.",
		   (unsigned int) window);
	  write_status (errmsg, ERR);
	  result = 6;
	  goto DONE;
	}
      else if (event.xany.type == PropertyNotify &&
	       event.xproperty.state == PropertyNewValue &&
	       event.xproperty.window == window &&
	       event.xproperty.atom == XA_XINVEST_RESPONSE)
	{
	  Atom actual_type;
	  int actual_format;
	  unsigned long nitems, bytes_after;
	  unsigned char *data = 0;

	  result = XGetWindowProperty (dpy, window, XA_XINVEST_RESPONSE,
				       0, (65536 / sizeof (long)),
				       True, /* atomic delete after */
				       XA_STRING,
				       &actual_type, &actual_format,
				       &nitems, &bytes_after,
				       &data);
#ifdef DEBUG_PROPS
	  if (result == Success && data && *data)
	    {
	      fprintf (stderr, "%s: (server sent " XINVEST_RESPONSE_PROP
		       " \"%s\" to 0x%x.)\n",
		       progname, data, (unsigned int) window);
	    }
#endif

	  if (result != Success)
	    {
	      sprintf (errmsg, "Failed reading " XINVEST_RESPONSE_PROP
		       " from window 0x%0x.",
		       (unsigned int) window);
	      write_status (errmsg, ERR);
	      result = 6;
	      done = True;
	    }
	  else if (!data || strlen((char *) data) < 5)
	    {
	      sprintf (errmsg, "Invalid data on " XINVEST_RESPONSE_PROP
		       " property of window 0x%0x.",
		       (unsigned int) window);
	      write_status (errmsg, ERR);
	      result = 6;
	      done = True;
	    }
	  else if (*data == '1')	/* positive preliminary reply */
	    {
#if 0
	      fprintf (stderr, "%s: %s\n", progname, data + 4);
#endif
	      /* keep going */
	      done = False;
	    }
#if 1
	  else if (!strncmp ((char *)data, "200", 3)) /* positive completion */
	    {
	      result = 0;
	      done = True;
	    }
#endif
	  else if (*data == '2')		/* positive completion */
	    {
#if 0
	      fprintf (stderr, "%s: %s\n", progname, data + 4);
#endif
	      result = 0;
	      done = True;
	    }
	  else if (*data == '3')	/* positive intermediate reply */
	    {
#if 0
	      fprintf (stderr, "%s: internal error: "
		       "server wants more information?  (%s)\n",
		       progname, data);
#endif
	      result = 3;
	      done = True;
	    }
	  else if (*data == '4' ||	/* transient negative completion */
		   *data == '5')	/* permanent negative completion */
	    {
#if 0
	      fprintf (stderr, "%s: %s\n", progname, data + 4);
#endif
	      result = (*data - '0');
	      done = True;
	    }
	  else
	    {
	      sprintf (errmsg,
		       "Unrecognised " XINVEST_RESPONSE_PROP
		       " from window 0x%x: %s",
		       (unsigned int) window, data);
	      write_status (errmsg, ERR);
	      result = 6;
	      done = True;
	    }

	  if (data)
	    XFree (data);
	}
#ifdef DEBUG_PROPS
      else if (event.xany.type == PropertyNotify &&
	       event.xproperty.window == window &&
	       event.xproperty.state == PropertyDelete &&
	       event.xproperty.atom == XA_XINVEST_COMMAND)
	{
	  fprintf (stderr, "%s: (server 0x%x has accepted "
		   XINVEST_COMMAND_PROP ".)\n",
		   progname, (unsigned int) window);
	}
#endif /* DEBUG_PROPS */
    }

 DONE:

  if (new_command)
    free (new_command);

  return result;
}

int
xinvest_remote_commands (Display *dpy, char **commands)
{
  static Window window = 0;   
  int status = 0;

  xinvest_remote_init_atoms (dpy);

  if (window == 0)
    window = xinvest_remote_find_window (dpy);
  else if (!xinvest_remote_check_window (dpy, window) )
    window = xinvest_remote_find_window (dpy);

  if (window == 0)
    return (5);

  XSelectInput (dpy, window, (PropertyChangeMask|StructureNotifyMask));

  if (xinvest_remote_obtain_lock (dpy, window) ) {

    while (*commands)
      {
        status = xinvest_remote_command (dpy, window, *commands);
        if (status != 0)
	  break;
        commands++;
      }

    /* When status = 6, it means the window has been destroyed */
    /* It is invalid to free the lock when window is destroyed. */

    if ( status != 6 )
      xinvest_remote_free_lock (dpy, window);
  }

  return status;
}


#ifdef STANDALONE

static void
usage (void)
{
  fprintf (stderr, "usage: %s [ options ... ]\n\
       where options include:\n\
\n\
       -help                     to show this message.\n\
       -display <dpy>            to specify the X server to use.\n\
       -remote <remote-command>  to execute a command in an already-running\n\
                                 Netscape process.  See the manual for a\n\
                                 list of valid commands.\n\
       -id <window-id>           the id of an X window to which the -remote\n\
                                 commands should be sent; if unspecified,\n\
                                 the first window found will be used.\n\
       -raise                    whether following -remote commands should\n\
                                 cause the window to raise itself to the top\n\
                                 (this is the default.)\n\
       -noraise                  the opposite of -raise: following -remote\n\
                                 commands will not auto-raise the window.\n\
",
	   progname);
}


void
main (int argc, char **argv)
{
  Display *dpy;
  char *dpy_string = 0;
  char **remote_commands = 0;
  int remote_command_count = 0;
  int remote_command_size = 0;
  unsigned long remote_window = 0;
  Bool sync_p = False;
  int i;

  progname = strrchr (argv[0], '/');
  if (progname)
    progname++;
  else
    progname = argv[0];

  /* Hack the -help and -version arguments before opening the display. */
  for (i = 1; i < argc; i++)
    {
      if (!strcasecmp (argv [i], "-h") ||
	  !strcasecmp (argv [i], "-help"))
	{
	  usage ();
	  exit (0);
	}
      else if (!strcmp (argv [i], "-d") ||
	       !strcmp (argv [i], "-dpy") ||
	       !strcmp (argv [i], "-disp") ||
	       !strcmp (argv [i], "-display"))
	{
	  i++;
	  dpy_string = argv [i];
	}
      else if (!strcmp (argv [i], "-sync") ||
	       !strcmp (argv [i], "-synchronize"))
	{
	  sync_p = True;
	}
      else if (!strcmp (argv [i], "-remote"))
	{
	  if (remote_command_count == remote_command_size)
	    {
	      remote_command_size += 20;
	      remote_commands =
		(remote_commands
		 ? realloc (remote_commands,
			    remote_command_size * sizeof (char *))
		 : calloc (remote_command_size, sizeof (char *)));
	    }
	  i++;
	  if (!argv[i] || *argv[i] == '-' || *argv[i] == 0)
	    {
	      fprintf (stderr, "%s: invalid `-remote' option \"%s\"\n",
		       progname, argv[i] ? argv[i] : "");
	      usage ();
	      exit (-1);
	    }
	  remote_commands [remote_command_count++] = argv[i];
	}
      else if (!strcmp (argv [i], "-raise") ||
	       !strcmp (argv [i], "-noraise"))
	{
	  char *r = argv [i];
	  if (remote_command_count == remote_command_size)
	    {
	      remote_command_size += 20;
	      remote_commands =
		(remote_commands
		 ? realloc (remote_commands,
			    remote_command_size * sizeof (char *))
		 : calloc (remote_command_size, sizeof (char *)));
	    }
	  remote_commands [remote_command_count++] = r;
	}
      else if (!strcmp (argv [i], "-id"))
	{
	  char c;
	  if (remote_command_count > 0)
	    {
	      fprintf (stderr,
		"%s: the `-id' option must preceed all `-remote' options.\n",
		       progname);
	      usage ();
	      exit (-1);
	    }
	  else if (remote_window != 0)
	    {
	      fprintf (stderr, "%s: only one `-id' option may be used.\n",
		       progname);
	      usage ();
	      exit (-1);
	    }
	  i++;
	  if (argv[i] &&
	      1 == sscanf (argv[i], " %ld %c", &remote_window, &c))
	    ;
	  else if (argv[i] &&
		   1 == sscanf (argv[i], " 0x%lx %c", &remote_window, &c))
	    ;
	  else
	    {
	      fprintf (stderr, "%s: invalid `-id' option \"%s\"\n",
		       progname, argv[i] ? argv[i] : "");
	      usage ();
	      exit (-1);
	    }
	}
    }

  dpy = XOpenDisplay (dpy_string);
  if (! dpy)
    exit (-1);

  if (sync_p)
    XSynchronize (dpy, True);

  exit (xinvest_remote_commands (dpy, (Window) remote_window,
				 remote_commands));
}

#endif /* STANDALONE */
#endif /* NO_XMU */
