/*
  Xquote Copyright (c)1997 Mark Buser,
  All Rights Reserved.

  Permission is hereby granted to copy
  and freely distribute copies of this
  program for non-commercial purposes
  without fee, provided that this notice
  appears in all copies.
  
  All redistributions must be in their
  entirety as originally distributed.
  Feel free to modify Xquote, but
  modified versions may not be distributed
  without prior consent of the author.
  
  This software is provided 'as-is'
  without any express or implied warranty.
  In no event will the author be held
  liable for any damages resulting from
  the use of this software.

  $Revision: 1.9 $ $Date: 1997/10/17 01:00:50 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <Xm/Xm.h>
#include <Xm/Protocols.h>
#if XtSpecificationRelease > 5
#include <X11/SM/SMlib.h>
#endif

#include "optdetail.h"
#include "optnet.h"
#include "opttick.h"
#include "optupdate.h"
#include "server.h"
#include "status.h"
#include "tape.h"
#include "util.h"
#include "view.h"
#include "xutil.h"

/* Application defined options that we don't want to save in state file */
static XrmOptionDescRec kill_options[] = {
  {
     "-restore", "restore", XrmoptionSepArg, NULL
  }
};

int restoreState( char *from )
{
  extern Widget Toplevel;

  FILE *savefile;
  char line[80];
  short major, minor; /* Xquote state file rev */
  int xrev;           /* X11Rx revision */
  int ticks = 0;
  int autonet = 0;
  int view = 0;

  char *param;

  /* Open session file */
  savefile = fopen ( from, "r");

  if (savefile == NULL) {
    sprintf ( errmsg, 
              "Could not open state file '%s',\nstate not restored.",
              from );
    write_status ( errmsg, ERR);
    return (False);
  }

  /* Read save area version number */
  if (fgets (line, 80, savefile))
    sscanf (line, "VERSION %hd.%hd (X11R%d)\n", &major, &minor, &xrev);
  else {
    sprintf (errmsg, 
             "Could not read state file '%s',\nstate not restored.",
             from);
    write_status (errmsg, ERR);
    return (False);
  }

  if (major != 1) {
    sprintf (errmsg, 
             "Expected state version 1.1, not '%hd.%hd' from\n"
             "'%s', state not restored.",
             major, minor, from);
    write_status (errmsg, ERR);
    return (False);
  }
  if (minor < 1) {
    sprintf (errmsg, 
             "Expected version 1.1, not %hd.%hd in state\n"
             "file '%s', only\n"
             "restoring compatible state data.",
             major, minor, from);
    write_status (errmsg, WARN);
  }

  /* Odds are we have a good state file, take note */
  sprintf (errmsg, "Restoring state saved under X11R%d protocol.", xrev);
  write_status (errmsg, LOG);

  /* Lock window managers concept of app size */
  AllowShellResize (Toplevel, LOCK|DISALLOW);

  while (fgets(line, 80, savefile)) {
    param = strrchr (line, '\n');  /* strip newline */
    if (param)
      *param = '\0';
    param = strchr (line, ' ');   /* param is after first space */

    switch (line[0]) {

            case 'T': /* "TICK symbol" */
                      if (param) {
                        tickAddItem (++param);
			ticks++;
		      }
                      break;

            case 'D': /* "DETAIL which" */
                      if (param) {
			int state = 0;
			sscanf ( ++param, "%d", &state);
                        detailSetState ( state );
			detailAddMainWindow();
                      }
                      break;

            case 'P': /* "PROXY host port" */
                      if (param) {
			char *host, *port; 
			host = XtCalloc ( 1, strlen(++param)+1 * sizeof(char));
			port = XtCalloc ( 1, strlen(param)+1 * sizeof(char));

			if (host && port) {
			  sscanf ( param, "%s %s", host, port );
                          setProxy (host, port);
			}
			XtFree (host);
			XtFree (port);
                      }
                      break;

            case 'A': /* "AUTO on minutes" */
                      if (param) {
			int min;
			sscanf (++param, "%d %d", &autonet, &min);
			updateSetAuto (autonet, min);
                      }
                      break;

            case 'R': /* "RANGE day start_hour end_hour" */
		      if (param) {
			int d, s, e;
			sscanf (++param, "%d %d %d", &d, &s, &e);
			updateSetTime (d, s, e);
                      }
                      break;

	    case 'S': /* SCROLL smooth delay */
		      if (param) {
			int smooth, delay;
			sscanf ( ++param, "%d %d", &smooth, &delay);
			tapeSetScroll ( smooth, delay );
	              }
		      break;

	    case 'V': /* VIEW which */
		      if (param)
			sscanf ( ++param, "%d", &view);
		      break;

	    case 'C': /* CURSERVER which */
		      if (param) {
			int server;
			sscanf ( ++param, "%d", &server);
                        if (server <= numServer()-1)
			  setCurServer (server);
                        else {
                          sprintf (errmsg, 
                                   "Saved quote server does not\n" 
                                   "exist, state not restored.");
                          write_status (errmsg, ERR);
                          return (False);
                        }
		      }
		      break;

            default:  break;
    }
  }
  if (ticks)
    tickAddMainWindow();

  triggerUpdate ((autonet)?TRIGGER_START:TRIGGER_STOP, (XtIntervalId) NULL);

  if (view)
    viewSet (view);

  AllowShellResize (Toplevel, UNLOCK|ALLOW);

  fclose (savefile);
  return (True);
}


int saveState (char *to)
{
  FILE *savefile;
  static int log = True;

  if (log) {
    sprintf ( errmsg, "Saving state under X11R%d protocol.", 
              XtSpecificationRelease);
    write_status ( errmsg, LOG);
    log = False;
  }

  /* 
  ** Create session file 
  */
  if ( (savefile = fopen (to, "w")) == NULL) {
    sprintf (errmsg, "Could not open state file '%s',\nstate not saved.", to);
    write_status (errmsg, ERR);
    return (False);
  }

  /* 
  ** Save save area version number 
  */
  fprintf (savefile, "VERSION 1.1 (X11R%d)\n", XtSpecificationRelease);

  /* Save which server used FIRST */
  fprintf (savefile, "CURSERVER %d\n", getCurServer());

  /* Save Ticker Symbols */
  {
    char *item;
    int num;
    for (num=0; num < tickGetNum(); num++) {
      if ( (item = tickGetItem (num)) ) {
        fprintf ( savefile, "TICK %s\n", item );
        XtFree (item);
      }
    }
  }

  /* Save detail setting */
  fprintf ( savefile, "DETAIL %d\n", detailGetState(-1) );

  /* Save proxy info */
  { 
    char *host;
    int port;
    getProxy ( &host, &port);
    if (host)
      fprintf ( savefile, "PROXY %s %d\n", host, port);
  }

  /* Save auto update. Must be saved _after_ proxy save */
  {
    int autonet, min;
    updateGetAuto (&autonet, &min);
    fprintf ( savefile, "AUTO %d %d\n", autonet, min );
  }

  /* Save time range. */
  {
    int d, s, e;
    updateGetTime (&d, &s, &e);
    fprintf (savefile, "RANGE %d %d %d\n", d, s, e);
  }

  /* Save tape scroll. */
  {
    int smooth, delay;
    tapeGetScroll ( &smooth, &delay );
    fprintf ( savefile, "SCROLL %d %d\n", smooth, delay );
  }

  /* Save tape or table view */
  {
    int tape = viewGet();
    fprintf ( savefile, "VIEW %d\n", tape );
  }

  fclose (savefile);
  return (True);
}

char *sessSaveFile (Boolean manual)
{
  char *savepathname;
  char *savefilename;
  char sessfilename[] = "xqteXXXXXX";
  char manfilename[] = ".xquoterc";
  char *fullpathname;

  /*
  ** Determine session save area and filename.
  */
  if ( (savepathname = getenv ("SM_SAVE_DIR")) == NULL)
    savepathname = getenv ("HOME");

  if (manual) {
    savefilename = manfilename;
    fullpathname = XtMalloc (strlen(savepathname) + strlen(savefilename) +2);
    if (fullpathname == NULL) {
      write_status ("Could not allocate memory for path.\n", ERR);
      return (NULL);
    }
    sprintf (fullpathname, "%s/%s", savepathname, savefilename);

  } else {
    savefilename = sessfilename;

#ifndef X_NOT_POSIX
    fullpathname = XtMalloc (strlen(savepathname) + strlen(savefilename) +2);
    if (fullpathname == NULL) {
      write_status ("Could not allocate memory for path.\n", ERR);
      return (NULL);
    }
    sprintf (fullpathname, "%s/%s", savepathname, savefilename);
    fullpathname = mktemp (fullpathname);
#else
    fullpathname = tempnam (savepathname, "xqte");
#endif
  }

  return (fullpathname);
}

#if XtSpecificationRelease > 5

/* ARGSUSED */
void sessSave ( Widget shell, XtPointer client_data, XtPointer call_data)
{
  XtCheckpointToken token = (XtCheckpointToken) call_data;
  String saveFile = sessSaveFile(False);

  char **argv;
  int  argc;

  String *discard = (String *)NULL;
  String *restart = (String *)NULL;
  int i;

  if (token->shutdown || token->interact_style != SmInteractStyleNone)
    XtSetSensitive ( shell, False);

  if ( (token->save_success = saveState(saveFile)) ) {
    /* 
    ** Update restore command 
    */

    XtVaGetValues ( shell, XtNrestartCommand, &argv, NULL);
    argc = 0;     /* How many arguments */
    if (argv) {
      XrmDatabase database = (XrmDatabase) NULL;

      while ( argv[argc++] )
        ;
      if (argc) argc--;       /* don't count NULL last arg */

      /* Remove arguments we don't want to save */
      XrmParseCommand ( &database, kill_options, XtNumber(kill_options),
                        "Xquote", &argc, argv );
      XrmDestroyDatabase (database);

      restart = (char **) XtMalloc ((argc+3) * sizeof(String));
      if (restart) {
        for (i=0; i < argc; i++)
          restart[i] = XtNewString ( argv[i] );
        restart[argc++] = XtNewString("-restore");
        restart[argc++] = XtNewString(saveFile);
        restart[argc]   = NULL;
  
        XtVaSetValues ( shell, XtNrestartCommand, restart, NULL);
      }
    }

    /* 
    ** Update discard command 
    */
    i = 0;
    discard = (String *) XtMalloc ( 2 * sizeof(String) );
    if (discard) {
      discard[i] = XtMalloc ( strlen(saveFile) + strlen ("rm ") +1 );
      if (discard[i])
        sprintf ( discard[i++], "rm %s", saveFile );
      discard[i] = NULL;

      XtVaSetValues ( shell, XtNdiscardCommand, discard, NULL);
    }
  }

  XtFree (saveFile);
}
#else

/* ARGSUSED */
void sessSave ( Widget shell, XtPointer client_data, XtPointer call_data)
{
  char *saveFile = sessSaveFile(False);

  char **argv;
  int argc;
  char restore[] = "-restore";

  XGetCommand ( XtDisplay (shell), XtWindow (shell), &argv, &argc );

  if ( saveState (saveFile) ) {
    XrmDatabase database = (XrmDatabase) NULL;

    /* Remove arguments we don't want to save (-restore) */
    XrmParseCommand ( &database, kill_options, XtNumber(kill_options),
                      "Xquote", &argc, argv );
    XrmDestroyDatabase (database);

    /* Add new -restore to read state file */
    argv = (char **) XtRealloc ( (char *)argv, (argc+2) * sizeof(char *));
    argv[argc++] = restore;
    argv[argc++] = saveFile;
    XSetCommand ( XtDisplay (shell), XtWindow (shell), argv, argc );

  } else
    XSetCommand ( XtDisplay (shell), XtWindow (shell), argv, argc );

  XtFree (saveFile);
  XFreeStringList ( argv );
}
#endif

/* ARGSUSED */
void sessResume ( Widget shell, XtPointer client_data, XtPointer call_data)
{
  XtSetSensitive ( shell, True);
}

/* ARGSUSED */
void sessDie ( Widget shell, XtPointer client_data, XtPointer call_data)
{
  XtDestroyApplicationContext (XtWidgetToApplicationContext(shell));
  exit(0);
}
