/******************************************************************************
 * $RCSfile: tree.h,v $   $Revision: 1.2 $
 * Copyright (C) 1994  A.Michael.Leliveld@Informatik.TU-Muenchen.De
 *
 * data structure tree
 * ===================
 *
 * Definition:
 *	Ein Bin"arbaum ist genau dann ausgeglichen, wenn f"ur alle Knoten
 *	die H"ohendifferenz der beiden Teilb"aume maximal 1 betr"agt.
 *	Ein ausgeglichener Baum wird auch h"ohenbalancierter, ausgeglichener
 *	oder AVL-Baum genannt.
 *	Der Wert der Knoten des linken Unterbaums ist kleiner als der Wert
 *	der Wurzel; der Wert der Knoten des rechten ist gr"o\3er oder gleich.
 *
 *	Mit h"ohenbalancierten B"aumen k"onnen die Operationen Einf"ugen,
 *	Suchen und L"oschen garantiert in O( log n ) Schritten ausgef"uhrt
 *	werden.  Die Funktion InOrder_AVLtree() liest einen Bin"arbaum in
 *	aufsteigend sortierter Reihenfolge aus.
 *
 *	Um die Wertigkeit von Knoten zu vergleichen ben"otigt man eine
 *	Ordnungsrelation cmp:
 *		int	< 0		e1 < e2
 *			== 0		e1 == e2
 *			> 0		e1 > e2
 *		cmp(
 *			void*,	        element e1
 *			void*	        element e2
 *		);
 ******************************************************************************/

#ifndef _MISC_TREE_H
#define _MISC_TREE_H


#include <stdio.h>
#include <stdlib.h>

#include <misc/mem.h>



struct tree {
	struct tree * left;
	struct tree * right;
	void* data;
	signed char balance;
};
typedef struct tree *  tree;


/*----------------------------------------------------------------------------*
 * extern prototypes
 *----------------------------------------------------------------------------*/

/* Returns an empty tree */
extern tree
    New_tree();


/* Is the tree empty? */
extern int
    Empty_tree(tree);


/* Add an element to a tree */
extern void
    AddTrue_tree(tree*,  /* pointer to the root of a tree */
		 void*,  /* data of the element to be added to the tree */
		 int (*)(void*, void*)  /* compare function */
		 );


/* Add an element ne to a tree, only if there is no other element e
 * in the tree with cmp(ne,e) == 0 */
extern void*  /* data of the rejected element  or  NULL */
    AddTrueNew_tree(tree*,  /* pointer to the root of the tree */
		    void*,  /* data of the element to be added to the tree */
		    int (*)(void*, void*)  /* compare function */
		    );


/* Replace the element oe from the tree with an element ne, if cmp(oe,ne) == 0
 * if oe doesn't exist, ne is added to the tree */
extern void*  /* data of the replaced element oe  or  NULL */
    ReplaceTrue_tree(tree*,  /* pointer to the root of the tree */
		     void*,  /* data of the element ne */
		     int (*)(void*, void*)  /* compare function */
		     );


/* Seeks the data in the tree.
 * Returns the pointer to the data from the first node of the tree with
 * cmp(tree->data,data) == 0, or NULL.
 * The rest of the tree will be set to the right tree of that node or NULL,
 * if that node doesn't exist. */
extern void*  /* pointer to the seeked data from the tree  or  NULL */
    Seek_tree(tree,   /* tree to be searched */
	      void*,  /* data to be seeked */
	      tree*,  /* pointer to the rest of the tree, if the pointer
		       * is NULL this parameter will be ignored */
	      int (*)(void*, void*)  /* compare function */
	      );


/* Deletes an element from the tree.
 * The rest of the tree will be set to the right tree of that node or NULL,
 * if that node doesn't exist */
extern void*  /* data of the deleted element  or  NULL */
    Del_tree(tree*,  /* pointer to the root of a tree */
	     void*,  /* data to be deleted from the tree */
	     tree*,  /* pointer to the rest of the tree, if the pointer
		      * is NULL this parameter will be ignored */
	     int (*)(void*, void*)  /* compare function */
	     );


/* Uses the function in the sorted order on every element of the list */
extern void
    InOrder_tree(tree,  /* root of the tree */
		 void (*)(void*)  /* function to apply to the element */
		 );


/* Frees the whole tree */
extern void
    Free_tree(tree*,  /* pointer to the root of the tree */
	      void (*)(void**)  /* function to free an element */
	      );


#endif /* _MISC_TREE_H */
