///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////
//         This example code is from the book:
//
//           Object-Oriented Programming with C++ and OSF/Motif, 2nd Edition
//         by
//           Douglas Young
//           Prentice Hall, 1995
//           ISBN 0-13-20925507
//
//         Copyright 1995 by Prentice Hall
//         All Rights Reserved
//
//  Permission to use, copy, modify, and distribute this software for 
//  any purpose except publication and without fee is hereby granted, provided 
//  that the above copyright notice appear in all copies of the software.
///////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////



/////////////////////////////////////////////////////
// Cmd.h: A base class for all command objects
/////////////////////////////////////////////////////
#ifndef CMD_H
#define CMD_H

#include "SimpleList.h"
#include "CmdInterface.h"

class CmdList;

class Cmd {
    
friend CmdInterface;
friend CmdList;
    
public:
    
    virtual ~Cmd ();           // Destructor
    
    enum CmdType { NoOp, Action, Toggle, Separator, List };
    
    // Public interface for executing and undoing commands
    
    virtual void execute();  
    virtual void undo();
    
    virtual void activate();   // Activate this object
    virtual void deactivate(); // Deactivate this object
    
    // Functions to register dependent commands
    
    void addToActivationList ( Cmd * );
    void addToDeactivationList ( Cmd * );
    
    // Register a UIComponent used to execute this command
    
    void registerInterface ( CmdInterface * );
    
    // Access functions 
    
    int active () { return ( _active ); }
    int hasUndo() { return ( _hasUndo ); }
    const char *const name () { return ( _name ); }
    virtual const char *const className () { return ( "Cmd" ); }
    virtual CmdType type() { return ( Action ); }
    
protected:
    
    int           _hasUndo;     // True if this object supports undo
    static Cmd   *_lastCmd;     // Pointer to last Cmd executed
    
    virtual void doit()   = 0;  // Specific actions must be defined
    virtual void undoit() = 0;  // by derived classes
    
    Cmd ( const char *,  int active = TRUE);  // Protected constructor
    virtual void  revert();   // Reverts object to previous state
    
private:
    
    // Lists of other commands to be activated or deactivated
    // when this command is executed or "undone"
    
    CmdList        *_activationList;
    CmdList        *_deactivationList;
    int            _active;     // Is this command currently active?
    int            _previouslyActive; // Previous value of _active
    char          *_name;             // Name of this command
    SimpleList<CmdInterface *>  _ci;            
};
#endif





































