/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */


#include "Rt.h"
#ifdef MOTIF
#include <Xm/Xm.h>
#include "Widget.h"
#endif
#include "Error.h"

static RtResource *locate_resource ();
static RtResource *find_resource ();
static void initialize_object ();
static int validate_object ();

/*
 * Standard message object
 */
 
MessageObj stdmsg = NULL;

#ifdef OLD
Obj RtCreateObject (parent, class)
Obj parent;
ObjClass class;
#endif
Obj RtCreateObject (object_name, class)
char *object_name;
ObjClass class;

{

Obj object;
static int initted = 0;

   /*
    * Create stdmsg
    */

   if (!initted) {
	initted++;
	stdmsg = RtCreateObject (NULL, messageObjClass);
   }
   	 
   if (!class)
   	return (NULL);
   
   if (!class->core_class.class_inited)
   	initialize_class (class);
   	
   object = (Obj) malloc (class->core_class.object_size);
   
   
   if (!object)
   	return (NULL);

   object->core.object_class = class;

   RtInitializeObject (object);
   
   if (object_name)
   	object->core.object_name = strdup (object_name);
   	
   return (object);	
   
}

/*
 * initialize_object: Initialize object
 */
 
static void initialize_object (obj, class)
Obj obj;
ObjClass class;
{

   if (!class)
   	return;

   /* do the top level class (core) first */
   
   if (class->core_class.superclass) {
   	initialize_object (obj, class->core_class.superclass);
   }

   /* In your way back, do the rest */
   
   if (class->core_class.initialize)
        class->core_class.initialize (obj);
    
}

/*
 * RtInitializeObject: Invoke the initialize method for each class
 *		       in the hierarchy. Start with Core.
 */
 

void RtInitializeObject (obj)
Obj obj;
{
   ObjClass class = obj->core.object_class;
   
   if (!obj || !class)
   	return;

   memset (obj, '\0', class->core_class.object_size);
   obj->core.object_class = class;
   obj->core.self = obj;
   
   initialize_object (obj, class);

}

/*
 * RtSetObjectValues: change resource values
 * bug: it should call the set values method of the each superclass
 *      wait until machines get faster
 */
 
RtSetObjectValues (subclass, current, new)
ObjClass subclass;
Obj current, new;
{

   ObjClass class = current->core.object_class;

   /* 
    * Invoke the set values method of the object class
    */
    
   if (class->core_class.set_values)
        class->core_class.set_values (current, new);	
   
   /*
    * Invoke the set values method of the object superclass
    * that owns the resource
    */
    
   if (subclass != class)
   	subclass->core_class.set_values (current, new);
   	
   /* copy new object onto the original object */
   memcpy (current, new, class->core_class.object_size);
   free (new);       
}

/*
 * RtActivateObject: activate object
 */

RtActivateObject (object)
Obj object;
{
ObjClass class;

   if (!object)
   	return;

#ifdef DEBUG
   if (!validate_object (object)) {
	fprintf (stderr, ">>>>> RtActivateObject: invalid object %d\n",
		object);	      
   }  
#endif

   class = object->core.object_class;
   	
   if (class->core_class.activate)
        class->core_class.activate (object);	
   
}

/*
 * RtRealizeObject: realize object
 */
 
int RtRealizeObject (object)
Obj object;
{
ObjClass class;

   if (!object)
   	return;

#ifdef DEBUG
   if (!validate_object (object)) {
	fprintf (stderr, ">>>>> RtRealizeObject: invalid object %d\n",
		object);	      
   }  
#endif
   class = object->core.object_class;
   	
   if (class->core_class.realize)
        return (class->core_class.realize (object));	
   
}

/*
 * RtValidateObject: validate the object
 */

int RtValidateObject (obj)
Obj obj;
{
int status;

   status = validate_object (obj);

#ifdef  DEBUG
   if (!status) {
	fprintf (stderr, ">>>>>> RtValidateObject: invalid object %d\n",
		obj);	   
   }
#endif
   
   return (status);
}

 
/*
 * RtSendMessage: send a message to an object
 */
 
RtSendMessage (object, data, client)
Obj object;
void *data;
void *client;
{
ObjClass class;
int id;
#ifdef DEBUG
MessageObj msg;
int type;
char *content;
#endif

   if (!object || !data)
   	return;

#ifdef DEBUG
   if (!validate_object (object)) {
	fprintf (stderr, ">>>>> RtRealizeObject: invalid object %d\n",
		object);	      
   }  
#endif
   class = object->core.object_class;

#ifdef DEBUG
	msg = (MessageObj) data;
	RtGetValue (msg, RtNmsgId, &id);
	RtGetValue (msg, RtNmsgType, &type);
	RtGetValue (msg, RtNmsgContent, &content);
	
	switch (type) {
	   case RtMSG_INT:
	   	fprintf (stderr, "RtSendMessage (%d) %d to %s ",
	   	    id, (int) content, object->core.object_name);
	   	break;	
	   case RtMSG_STRING:
	   	fprintf (stderr, "RtSendMessage (%d) %s to %s ",
	   	    id, content, object->core.object_name);
	   	break;	
	   default:	
		fprintf (stderr, "RtSendMessage %d to %s ", id,
	  	    object->core.object_name);
	}   
	fprintf (stderr, "(%s)\n", class->core_class.class_name);
   
#endif   	
   RtGetValue ((MessageObj) data, RtNmsgId, &id);
   if (id == RtDISABLE) {
#ifdef DEBUG
	fprintf (stderr, "RtSendMessage: disable object\n");
#endif
	object->core.object_flags |= RtOBJ_DISABLED;
	return;   
   }
   
   if (id == RtENABLE) {
   	object->core.object_flags &= ~RtOBJ_DISABLED;
   }   

   if (object->core.object_flags & RtOBJ_DISABLED) {
#ifdef DEBUG
	fprintf (stderr, "RtSendMessage: disabled object\n");
#endif
   	return;
   }
   
   	
   if (class->core_class.process_message)
        class->core_class.process_message (object, data,
        	client);

#ifdef DEBUG
   if (object->core.status)
   	fprintf (stderr, ">>>>> RtSendMessage: error status: %d\n",
   		object->core.status);  
#endif
        	
   /* Check if this object is chained in which case */
   /* Messages should be broadcasted to the next    */
   /* object in the sequence			    */
   /* Chech this portion of the code                */
    
   if ((object->core.object_flags & RtOBJ_CHAINED) 
   	&& (id >= RtRESERVED_IDS)) {
   	/* Invoke the superclass method to broadcast the message */
   	(*broadCasterObjClass->core_class.process_message) (object, 
                        (char *) data, NULL);   
   }
}

/*
 * bad_object: function called when a bad object is detected.
 *	       This function is used for debugging purposes.
 */
bad_object (obj)
Obj obj;
{


}
 
/*
 * validate_object: verify that this a valid object
 */

static int validate_object (obj) 
Obj obj;
{
#ifdef DEBUG
ObjClass class;
#endif

  if (!obj) {
#ifdef DEBUG
	bad_object (obj);
#endif
   	return (0);
   }
   
   if (obj->core.self != obj) {
#ifdef DEBUG
	bad_object (obj);
#endif
   	return (0);
   }
   	
   if (obj->core.object_flags & RtOBJ_DESTROYED) {
#ifdef DEBUG
        class = obj->core.object_class;
	fprintf (stderr, ">>>>validate_object: destroyed object %d,",
		obj);
	fprintf (stderr, "class = %s, name = %s\n", 
		class->core_class.class_name, 
		obj->core.object_name);
	bad_object (obj);
#endif
   	return (0);   
   }
   return (1);
}

 
/*
 * destroy_object: destroy the object (one subclass at a time)
 */
 
static void destroy_object (obj, class)
Obj obj;
ObjClass class;
{

   if (!class)
   	return;
   	
   if (class->core_class.destroy)
        class->core_class.destroy (obj);
    
   if (class->core_class.superclass) {
   	destroy_object (obj, class->core_class.superclass);
   }

   if (class == objClass) {
	obj->core.object_flags |= RtOBJ_DESTROYED;  
#ifndef DEBUG
   	free (obj);	/* destroy the object pointer */
#endif   	
   }
}

/*
 * RtDestroyObject: destroy the object
 */

RtDestroyObject (object)
Obj object;
{
ObjClass class;

   if (!object)
   	return;

#ifdef DEBUG
   if (!validate_object (object)) {
	fprintf (stderr, ">>>>> RtDestroyObject: invalid object %d\n",
		object);	      
   }  
#endif

   destroy_object (object, object->core.object_class);
   	
}

/*
 * RtPipeObject: Create a pipe between two objects
 *		  Returns a pipe object
 */
 
Obj RtPipeObject (src, dest)
Obj src, dest;
{ 
   Obj pipe;
   
   if (!dest || !src)
        return (NULL);
   
#ifdef DEBUG
   if (!validate_object (src)) {
	fprintf (stderr, 
		">>>>> RtPipeObject: invalid object(src) %d\n",
		src);	      
   }  
   if (!validate_object (dest)) {
	fprintf (stderr, 
		">>>>> RtPipeObject: invalid object(dest) %d\n",
		dest);	      
   }  
#endif

   /*
    * Create a Plug object to establish the connection
    */
        
   pipe = RtCreateObject (NULL, broadCasterObjClass);
   if (pipe) {
   
   	/* Plug pipe into src */ 
        RtSetValue(pipe, RtNmsgId, RtADD_RECEPTOR);
        RtSetValue(pipe, RtNmsgContent, (char *) pipe);
        RtSendMessage (src, pipe, NULL);
        
        /* Plug dest into pipe */
        RtSetValue(pipe, RtNmsgContent, (char *) dest);        
        RtSendMessage (pipe, pipe, NULL); /* check */
   }
   
   return (pipe);   
}



/*
 * RtCreateResourceLink: Create a resource link 
 */
 
ResourceLinkObj RtCreateResourceLink (obj, obj1, res, res1)
Obj obj;
Obj obj1;
char *res;
char *res1;
{

RtLinksTbl links_tbl;
ResourceLinkObj *tmp;

   if (!obj || !res || !obj1 || !res)
   	return (NULL);
   	
   links_tbl = obj->core.rlinks;
   
   if (!links_tbl) {
        links_tbl = obj->core.rlinks = (RtLinksTbl)
        	malloc (sizeof (struct _RtLinksTbl));
        	
	if (!links_tbl)
	   return (NULL);
	           
        memset (links_tbl, '\0', sizeof (struct _RtLinksTbl));
        		
	links_tbl->rlink_list = (Obj *) 
           	malloc (sizeof (ResourceLinkObj) 
                * LTBL_CHUNK_SIZE);
                
        if (!links_tbl->rlink_list)
              return (NULL);
        links_tbl->max_cnt = LTBL_CHUNK_SIZE;
   } else if (links_tbl->index >= links_tbl->max_cnt) {
           /* Allocate an additional chunk of memory */
           tmp = (Obj *) realloc (links_tbl->rlink_list, 
           		sizeof (ResourceLinkObj) 
                        * (links_tbl->max_cnt + LTBL_CHUNK_SIZE));
           if (tmp) {
                links_tbl->rlink_list = tmp;
                links_tbl->max_cnt += LTBL_CHUNK_SIZE;                    
           }
                           
   }
        
   if (links_tbl->index >= links_tbl->max_cnt)
           return (NULL);
       
   tmp = links_tbl->rlink_list + links_tbl->index;
   
   *tmp = (ResourceLinkObj) RtCreateObject (NULL, resourceLinkObjClass);
   
   if (!*tmp)
   	return (NULL);
   	    
   (*tmp)->rlink.res = strdup (res);
   (*tmp)->rlink.obj1  = obj1;
   (*tmp)->rlink.res1 = strdup (res1);
   links_tbl->index++;
   return (*tmp);  
}
  
/*
 * check_res_links: look for a resource in the links table 
 */
 
int check_res_links (obj, resource, value, bool)
Obj obj;
RtResource *resource;
void *value;
int bool;
{
RtLinksTbl links_tbl;
register ResourceLinkObj *rlink;
register int i = 0;
char **cp;

   links_tbl = obj->core.rlinks;
   if (!links_tbl)
      	return (0);
      
   rlink = (ResourceLinkObj *) links_tbl->rlink_list;
   
   while (i++ < links_tbl->index) {
	if (!resource->name ||
		strcmp ((*rlink)->rlink.res, resource->name)) {
		rlink++;
		continue;
	}
	
	/* this is a link to another resource */		   
    	switch (resource->type) {
    	   case RtSTRING_TYPE:
    	   	if (bool) {
		   RtSetValue ((*rlink)->rlink.obj1, (*rlink)->rlink.res1, 
			(char *) value);
		} else
		   RtGetValue ((*rlink)->rlink.obj1, (*rlink)->rlink.res1, 
			(char **) value);

		if ((*rlink)->rlink.flags & RtRLINK_CASCADE)
		   return (0); /* propagate changes */
		else
		   return (1);
    	   case RtBIT_TYPE:
    	   	if (bool) {
		   RtSetValue ((*rlink)->rlink.obj1, (*rlink)->rlink.res1, 
			(char *) value);
		} else
		   RtGetValue ((*rlink)->rlink.obj1, (*rlink)->rlink.res1, 
			(char **) value);

		if ((*rlink)->rlink.flags & RtRLINK_CASCADE)
		   return (0); /* propagate changes */
		else
		   return (1);
#ifdef OLD			
		/* this avoids memory leaks	*/ 
    	   	cp = (char **) (((char *) obj) + resource->offset);
    	   	*cp = *value_pointer;
		return (1);
		break; 
#endif		  	
    	}
    	rlink++;
   }
   return (0);   	   
}

/*
 * RtPlugObjectInto: Plug destination into source
 */
 
RtPlugObjectInto(dest,src,msg)
Obj dest, src;
MessageObj msg; /* New message Id */
{ 

MessageObj aux;
 
   aux = (MessageObj) RtCreateObject (NULL,  messageObjClass);
   
   if (!dest || !src)
        return (NULL);

#ifdef DEBUG   
   if (!validate_object (src)) {
	fprintf (stderr, 
		">>>>> RtPlugObjectInto: invalid object(src) %d\n",
		src);	      
   }  
   if (!validate_object (dest)) {
	fprintf (stderr, 
		">>>>> RtPlugObjectInto: invalid object(dest) %d\n",
		dest);	      
   }
#endif
 
   /*
    * Create a Plug object to establish the connection
    */
        
   /* Plug dest into src */
    
   RtSetValue(aux, RtNmsgId, RtADD_RECEPTOR);
   RtSetValue(aux, RtNmsgContent, (char *) dest);
   RtSetValue(aux, RtNmsgClient, (char *) msg);
   RtSendMessage (src, aux, NULL);
   
   RtDestroyObject (aux);
        
}

/*
 * set_resource: set a specific resource
 */
 
ObjClass xx_rt_set_resource (obj, res, value)
Obj obj;
char *res;
char *value;
{
   ObjClass class, subclass = NULL;
   register RtResource *resources;
   register int i;
   char **cp;
   int *ip;
   MessageObj aux;
   
   if (!res)
   	return (NULL);
   
   class = obj->core.object_class;

   	resources = find_resource (obj, res, &subclass);
        
   	if (!resources)
	   return (NULL);
    
   	if (obj->core.rlinks) {
	   if (check_res_links (obj, resources, value, 1))
	   	return (subclass); 
   	}

	if (resources->flags & RtWidgetXtResource) {
   	   aux = (MessageObj) RtCreateObject (NULL,  messageObjClass);	   
	   RtSetValue (aux, RtNmsgId, RtWIDGET_SET_XT_RESOURCE);
	   RtSetValue (aux, RtNmsgContent, resources->client);
	   RtSetValue (aux, RtNmsgClient, value);
	   RtSendMessage (obj, aux);
	   RtDestroyObject (aux);
	   return (subclass); /* bug */
	}
	
    	switch (resources->type) {
    	   case RtOBJ_TYPE: {
    	    Obj *objp;
   	   	objp = (Obj *) (((char *)  obj) + resources->offset);
    	   	*objp = (Obj) value;
    	   	return (subclass);
    	   	break;
    	   }
	   case RtSUBOBJ_TYPE: {
	    Obj *objp;
	   	objp = (Obj *) (((char *)  obj) + resources->offset);
	   	RtSetValue ((*objp), resources->client, value);	   		
	   	return (subclass);
	   }
    	   case RtSTRING_TYPE:
    	   case RtPOINTER_TYPE:
    	   	cp = (char **) (((char *) obj) + resources->offset);
    	   	*cp = value;
    	   	return (subclass);
    	   	break;
    	   case RtINT_TYPE:
    	   	ip = (int *) (((char *)  obj) + resources->offset);
    	   	*ip = (int) value;
    	   	return (subclass);
    	   	break;
    	   case RtBIT_TYPE:
    	   	ip = (int *) (((char *)  obj) + resources->offset);
    	   	if (value)
    	   	   *ip |= (int) resources->client;
    	   	else
    	   	   *ip &= ~((int) resources->client);
#ifdef DEBUG
		fprintf (stderr, "xx_rt_set_resource(%s): %s %d\n",
			obj->core.object_name,
			res, *ip);
#endif			
    	   	return (subclass);
    	   	break;
    	   case RtUID_TYPE: {
    	    uid_t *uidp;
   	   	uidp = (uid_t *) (((char *)  obj) + resources->offset);
    	   	*uidp = (uid_t) value;
    	   	return (subclass);
    	   	break;
    	   }
    	   case RtTIME_TYPE: {
    	    time_t *timp;
   	   	timp = (int *) (((char *)  obj) + resources->offset);
    	   	*timp = (time_t) value;
    	   	return (subclass);
    	   	break;
    	   }
	   default:
#ifdef DEBUG
		fprintf (stderr, "set_resource: invalid resource type\n");
#endif
		obj->core.status = RtINVALID_RESOURCE_TYPE;	   
	   	break;	   
	}
#ifdef OLD				   
   class = class->core_class.superclass;
   }
#endif 
#ifdef DEBUG
   fprintf (stderr, 
	"xx_rt_set_resource:%s:%s, no such resource\n",
		obj->core.object_name, res);
#endif
   return (NULL); 
}


/*
 * RtCloneObject: clone the object
 */

Obj RtCloneObject (obj)
Obj obj;
{
Obj new_obj;

   if (!obj->core.object_class)
   	return (NULL);

   new_obj = RtCreateObject (NULL, 
    		obj->core.object_class);
   
   RtCopyObject (obj, new_obj);
   return (new_obj);
}

/*
 * RtCopyObject: copies the content of obj1 onto obj2
 */

void RtCopyObject (obj1, obj2)
Obj obj1, obj2;
{
   ObjClass class1, class2;
   RtResource *resources;
   register int i;
   char *dummy;	/* check */

   if (!obj1 || !obj2)
   	return;

   class1 = obj1->core.object_class;
   class2 = obj2->core.object_class;
   
   if (class1 != class2)
   	return;

   resources = class2->core_class.resources;

   if (!resources) 
   	return;
   	

   for (i = 0; i < class2->core_class.num_resources; i++) {
   	if (!resources->name) {
   	   resources++;
   	   continue;
   	}
   	RtGetValue (obj1, resources->name, &dummy);
   	RtSetValue (obj2, resources->name, dummy);
	resources++;
   }

}

 
/*
 * locate_resource: locate a resource in the resource table
 */
 
static RtResource *locate_resource (class, res)
ObjClass class;
char *res;
{
 int mid, lo, hi, cnt;
 int aux;
 RtResource *resources;
    
   if (!class)
   	return (NULL);

   cnt = class->core_class.num_resources;	
   resources = class->core_class.resources; 
   
   if (!resources || (cnt <= 0) || !res)
	return (NULL);
   
   lo = 0;
   hi = cnt - 1;

#ifdef OLD
#ifdef DEBUG
   fprintf (stderr, "locate_resource: %d\n", cnt);
#endif
#endif

   while (lo <= hi) {
   	mid = (lo+hi)/2;

#ifdef OLD
#ifdef DEBUG
   	fprintf (stderr, "locate_resource: try %s\n", 
   		resources[mid].name);
#endif
#endif
   
	aux = strcmp (res, resources[mid].name); /* check */
   
	if (!aux)
   	   return (&resources[mid]);
   	
   	if (aux > 0) {
	   lo = mid+1; /* upper portion ? */	   
   	} else
   	   hi = mid-1;

   }
   return (NULL);

}

/*
 * RtLocateClassResource: locate a resource in the specified class
 */
 
RtResource *RtLocateClassResource (class, res)
ObjClass class;
char *res;
{
   return (locate_resource (class, res));
}

/*
 * resource_prefix: return the resource name prefix
 */

static char *resource_prefix (res)
char *res;
{

   register char *cp = res;
   register int cnt = 0;
   register char *prefix;
   
   if (!cp)
   	return (NULL);
   
   while (*cp) {   	
   	if (!islower (*cp++))
   	   break;
   	cnt++;
   }
   
   if (!cnt)
      return (NULL);
      
   prefix = malloc ((cnt+1) * sizeof (char));
   
   if (!prefix)
      return (NULL);
      
   memcpy (prefix, res, cnt);
   
   prefix[cnt] = '\0';
   
   return (prefix); 
}

 
/*
 * fast_find: find a specific resource by trying a faster algorithm 
 */

static RtResource *fast_find (obj, res, subclass)
Obj obj;
char *res;
ObjClass *subclass;
{
   ObjClass class;
   RtResource *resources;
   char *prefix;

   *subclass = NULL;
   if (!res)
   	return (NULL);
   
   prefix = resource_prefix (res);
   
   if (!prefix)
   	return (NULL);
   
   for (class = obj->core.object_class; class != NULL; 
   	class = class->core_class.superclass) {
	
        if (!class->core_class.prefix)
   	    continue;

	/* check the resource prefix */
	if (strcmp (prefix, class->core_class.prefix))
	   continue;
#ifdef DEBUG
#ifdef OLD
	fprintf (stderr, "prefix %s matched\n", prefix);
#endif	 
#endif	

   	resources = locate_resource (class, res); 
   		
   	if (resources) {
#ifdef DEBUG
#ifdef OLD
	   fprintf (stderr, "prefix/resource %s matched\n", prefix);
#endif	    
#endif
	   *subclass = class;	
   	   free (prefix);
   	   return (resources);
   	}

   }
   
   free (prefix);
   return (NULL);
}

/*
 * find_resource: traverse the class hierarchy looking
 *                for a specific resource
 */


static RtResource *find_resource (obj, res, subclass)
Obj obj;
char *res;
ObjClass *subclass;
{
   ObjClass class;
   RtResource *resources;
   register int i;

   *subclass = NULL;
   if (!res)
   	return (NULL);
   
   /* try a faster algorithm first */
   resources = fast_find (obj, res, subclass);
   
   class = obj->core.object_class;

   if (resources) 
   	return (resources);
   	
   /* traverse the whole class hierarchy */
   while (class) {
   	resources = class->core_class.resources;

   	if (!resources) {
   	   class = class->core_class.superclass;
    	   continue;
   	}
    
   	resources = locate_resource (class, res); 
   		
  	if (!resources) {
   		class = class->core_class.superclass;
    		continue;
  	}
	*subclass = class;
	return (resources); /* resource found */
   }
#ifdef DEBUG
   fprintf (stderr, 
	">>>>>>>> find_resource:%s:%s, no such resource\n",
		obj->core.object_name, res);
#endif
   return (NULL);
} 

static void get_value (class, obj, res, value_pointer)
ObjClass class;
Obj obj;
char *res;
void **value_pointer;
{

   if (!class || !obj || !res || !value_pointer)
   	return;

   while (class) {
        
        if (!locate_resource (class, res)) {
           class = class->core_class.superclass;
	   continue;                
        }

        if (class->core_class.get_values) {
   	   (*class->core_class.get_values) (obj,
        	res, value_pointer);
        }
        return;
   }
}

static void remove_trailing_spaces(value)
char *value;
{
   register char *cp = value;

   if (!value)
   	return;
   	
   while (*cp)
   	cp++;
   
   while (cp != value) {
	if (*cp && !isspace (*cp)) {
	   break;
	}
	*cp = '\0';   
   	cp--;			   
   }
   if (isspace (*cp))
   	*cp = '\0';
   	
}

static void remove_leading_spaces (value)
char *value;
{
   register char *cp = value;
   register char *cp1 = value;
   
   if (!value)
   	return;
      
   while (*cp) {
	if (!isspace (*cp)) {
	   break;
	}	   
	cp++;	   	
   }
   
   if (cp == value)
   	return;
   
   while (*cp1++ = *cp++);
   
}


massage_value (resources, value_pointer)
RtResource *resources;
char **value_pointer;
{

   if (!resources)
   	return;
   	
   if (resources->type == RtSTRING_TYPE) {   
	if (resources->flags & RtRES_DEL_LEADING_SPACES)
	   remove_leading_spaces (*value_pointer);
	if (resources->flags & RtRES_DEL_TRAILING_SPACES)
	   remove_trailing_spaces (*value_pointer);
	 	
   }

}


/*
 * xx_rt_get_resource: get the value of a specific resource
 */
 
int xx_rt_get_resource (obj, res, value_pointer)
#ifdef MOTIF
RtWidgetObj obj;
#else
Obj obj;
#endif
char *res;
void **value_pointer;
{
   ObjClass class, subclass = NULL;
   RtResource *resources;
   register int i;
   char **cp;
   int *ip;
   char **str_ptr;
   int *int_ptr;
#ifdef MOTIF
   XmString Str;
#endif
   
   if (!res)
   	return;

   resources = find_resource (obj, res, &subclass);
        
   if (!resources)
	return;
    
   if (obj->core.rlinks) {
	if (check_res_links (obj, resources, value_pointer, 0)) {
	   massage_value (resources, value_pointer);
	   return; 
	}
   }

   if (resources->flags & RtInvokeClassGetValue) {
	get_value (obj->core.object_class,
		obj, res, value_pointer);
	massage_value (resources, value_pointer);
	
	return;	   	
   }
   switch (resources->type) {
#ifdef MOTIF
    	   case RtXMSTRING_TYPE: {
    	     char *str;
    	     
    	        if (!obj->widget.widget && !resources->client)
    	           return;
		XtVaGetValues (obj->widget.widget,
			(char *) resources->client, &Str, NULL);
    	   	str_ptr = (char **) value_pointer;
    	   	if (!XmStringGetLtoR (Str, XmSTRING_DEFAULT_CHARSET,
    	   		&str))
    	   	   return; /* check leak */
    	   	*str_ptr = str;
    	   	XmStringFree (Str);
    	   	return; 				    	    	    	   
    	   }
#endif    	   
	case RtSUBOBJ_TYPE: {
	   Obj *objp;
	   objp = (Obj *) (((char *)  obj) + resources->offset);
	   RtGetValue (*objp,
		(char *) resources->client, value_pointer);	   		
	   return;
	}
    	case RtOBJ_TYPE: {
    	   Obj *objp, *obj_ptr;
    	   objp = (Obj *) (((char *)  obj) + resources->offset);
    	   obj_ptr = (Obj *) value_pointer;
    	   *obj_ptr = *objp;
    	   return;
    	}
    	case RtSTRING_TYPE:
    	case RtPOINTER_TYPE:
    	   cp = (char **) (((char *) obj) + resources->offset);
    	   str_ptr = (char **) value_pointer;
    	   *str_ptr = *cp;
	   massage_value (resources, value_pointer);
    	   return;
    	case RtINT_TYPE:
    	   ip = (int *) (((char *)  obj) + resources->offset);
    	   int_ptr = (int *) value_pointer;
    	   *int_ptr = *ip;
    	   return;
    	case RtBIT_TYPE:
    	   ip = (int *) (((char *)  obj) + resources->offset);
    	   int_ptr = (int *) value_pointer;
    	   *int_ptr = (*ip & (int) resources->client);
#ifdef DEBUG
	   fprintf (stderr, "xx_rt_get_resource(%s): %s %d\n",
	   		obj->core.object_name,
			res, *int_ptr);
#endif			
    	   return;
    	case RtUID_TYPE: {
    	uid_t *uidp, *uid_ptr;
    	   uidp = (uid_t *) (((char *)  obj) + resources->offset);
    	   uid_ptr = (uid_t *) value_pointer;
    	   *uid_ptr = *uidp;
    	   return;
    	}
    	case RtTIME_TYPE: {
    	time_t *timp, *tim_ptr;
    	   timp = (time_t *) (((char *)  obj) + resources->offset);
    	   tim_ptr = (time_t *) value_pointer;
    	   *tim_ptr = *timp;
    	   return;
    	}
	default:
#ifdef DEBUG
	   fprintf (stderr, 
			"xx_rt_get_resource: invalid resource type\n");
#endif
	   obj->core.status = RtINVALID_RESOURCE_TYPE;	   
	   break;	   
	}

}
#ifdef OLD 
RtResource *RtGetPResource (obj, pos)
Obj obj;
int pos;
{
   ObjClass class;
   RtResource *resources;
   int i = 0;

   if (!obj)
   	return (NULL);
   	
   class = obj->core.object_class;
   
   if (!class)
   	return (NULL);

   resources = class->core_class.resources;
   if (!resources) 
   	return (NULL);
   	

   for (i = 0; i < class->core_class.num_resources; i++) {
	if ((int) resources->client == pos)
	   return (resources);
	resources++;
   }
}
#endif

/*
 * RtSuperclassProcessMessage: invoke the superclass to process
 *			       the message
 */
 
RtSuperclassProcessMessage (class, obj, data, client)
ObjClass class;
Obj obj;
void *data;
void *client;
{
   int flags;
   
   if (!class)
	return;
   
   while (class = class->core_class.superclass) {
   	if (class->core_class.process_message) {
   	     /* save the object flags */
             flags = obj->core.object_flags;
             /* invoke the process message in the superclass */
             obj->core.object_flags |= RtOBJ_SUPERCLASS_PROCESS_MESSAGE;
  	     (*class->core_class.process_message) (obj,
        	(char *) data, client);
             obj->core.object_flags = flags;
             return;
        }
   }
   	
}

/*
 * Send an error message to the stderr object
 */
 
static void send_err_msg (obj, id, client)
Obj obj;
int id;
char *client;
{

   RtSetValue (obj->core.obj_stderr, RtNmsgId, 
	RtERR_ID);
   RtSetValue (obj->core.obj_stderr, RtNmsgContent, 
        id);
   RtSetValue (obj->core.obj_stderr, RtNmsgClient, 
        client);

   RtSendMessage (obj->core.obj_stderr,
        	   obj->core.obj_stderr);

}

/*
 * single_word: single word 
 */
  
static int single_word (s)
char *s;
{
   register char *cp = s;
   
   if (!s)
   	return (1);
   
   /* consume leading spaces */	
   while (*cp) {
	if (!isspace (*cp))
	   break;   
	cp++;      
   }

   /* first character after last space */
   while (*cp) {
	if (isspace (*cp))
	   break;   
	cp++;      
   }
   
   if (!cp)
   	return (1); /* one word */
   	
   /* first space after the first word */
   
   while (*cp) {
	if (!isspace (*cp))
	   return (0); /* there is another word */
	cp++;  	   
   }
   return (1);   /* only spaces after the first word */
}

/*
 * validate_resource: validate resource
 */

static void validate_resource (obj, resource)
Obj obj;
RtResource *resource;
{
char *res, *aux;
char *cp;
   res = NULL;
   
   switch (resource->type) { 
	case RtSTRING_TYPE: 
	   if (!resource->flags)
	   	return;
	   xx_rt_get_resource (obj, resource->name, &res); /* check */
    	   if ((resource->flags & RtRES_REQUIRED) && 
    	   	(!res || !*res)) {
    	        if (resource->default_address)
    	   	   aux = _strconcat(resource->default_address,
    	        		": value required.");
    	        else
    	   	   aux = _strconcat(resource->name,
    	        		": value required.");
    	        	
   	   	obj->core.status = RtREQUIRED_RESOURCE;
#ifdef OLD
        	RtSetValue (obj->core.obj_stderr, RtNmsgId, 
        		RtERR_ID);
        	RtSetValue (obj->core.obj_stderr, RtNmsgContent, 
        		RtREQUIRED_RESOURCE);
        	RtSetValue (obj->core.obj_stderr, RtNmsgClient, 
        		aux);
#endif		
		send_err_msg (obj, RtREQUIRED_RESOURCE,
				aux);
        	if (aux)
        	   free (aux);
#ifdef OLD
        	RtSendMessage (obj->core.obj_stderr,
        	   obj->core.obj_stderr);
#endif
        	return;
   	   }
	   if (res && (resource->flags & RtRES_PRINTING)) {
		cp = res;		
		while (*cp) {
		   if (!isprint (*cp)) {
   	   		obj->core.status = RtINVALID_VALUE;
    	        	if (resource->default_address)
    	   	   	   aux = _strconcat(resource->default_address,
    	        		": Invalid value");
    	        	else
    	   	   	   aux = _strconcat(resource->name,
    	        		": Invalid value");
			send_err_msg (obj, RtINVALID_VALUE,
				aux);
			if (aux)
			   free (aux);
				
		   }			
		   cp++;		
		}			   
	   }
	   if (res && (resource->flags & RtRES_SINGLE_WORD)) {
		if (!single_word (res)) {
   	   	   obj->core.status = RtINVALID_VALUE;
    	           if (resource->default_address)
    	   	   	aux = _strconcat(resource->default_address,
    	        		": Invalid value");
    	           else
    	   	   	aux = _strconcat(resource->name,
    	        		": Invalid value");
		   send_err_msg (obj, RtINVALID_VALUE,
				aux);
		   if (aux)
		   	free (aux);				
		}
						   
	   }
   	   break;
   	default:
   	   break;
   }
}


/*
 * RtValidateResources: validate the value of each one of the object
 *			resources
 */

void RtValidateResources (obj)
Obj obj;
{
   ObjClass class;
   RtResource *resources;
   register int i;
   int status;

   class = obj->core.object_class;
   
   if (!obj)
        return;
        
   if (!class)
   	return;


   resources = class->core_class.resources;

   if (!resources) 
   	return;
   	

   for (i = 0; i < class->core_class.num_resources; i++) {
#ifdef OLD
	resources = RtGetPResource (obj, i);
#endif
        
	validate_resource (obj, resources++);
	
	if (obj->core.status)
		return;
   }

   return;

}
