/*
 * Freedom Desktop
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

/*
 * Directory Object: object used to manipulate directories
 */
 
void DIRInitialize ();
void DIRActivate ();
void DIRRealize ();
void DIRDestroy ();
void DIRSetValues ();
void DIRProcessMessage ();

#include "Directory.h"


static RtResource resources[] = {
   {RtNdirExcludeParent, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(DirObj,dir.exclude_parent), 0, NULL, 0, NULL },   
   {RtNdirExcludeCurrent, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(DirObj,dir.exclude_current), 0, NULL, 0, NULL },   
   {RtNdirSorted, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(DirObj,dir.sorted), 0, NULL, 0, NULL },   
};

DirClassRec dirClassRec = {
  {
    (ObjClass) &fileClassRec,	/* superclass		*/
    "Directory",		/* class name		*/
    NULL,	
    0,
    resources,			/* resource table	*/
    RtNumber(resources),	/* number of resources  */
    /* object size              */  sizeof(DirRec),
    DIRInitialize,
    DIRRealize,
    NULL,
    DIRSetValues,
    DIRActivate,
    DIRDestroy,
    DIRProcessMessage,
    "dir",
    "Rt1.00",
    NULL
  },
  {
  0,
  }
  
};

ObjClass dirObjClass = (ObjClass) &dirClassRec; 

/*
 * string_cmp: compare two directory entries
 */
 
static int string_cmp (str1, str2)
register char **str1, **str2;
{
char c1, c2;
register char *s1, *s2;

   /* Check if one of the strings is NULL */
   
   if (!*str1) {
   	if (*str2)
   	   return (-1);
   	else
   	   return (0);
   }
    
   if (!*str2) {
   	if (*str1)
   	   return (1);
   	else
   	   return (0);
   }

   s1 = *str1;
   s2 = *str2;
   
   while (1) {
	if (*s1 != *s2) {
   	   c1 = *s1;
   	   c2 = *s2;
   	   
   	   /* Convert everything to lower case before making the */
   	   /* comparison					 */
   	   
   	   if (isupper (c1))
        	c1 = tolower (c1);
   	   if (isupper (c2))
        	c2 = tolower (c2);
	   if (c1 != c2)
	   	return (c1 - c2);
	   s1++;
	   s2++;
	   continue;
	}

	if (*s1 == '\0')
	  return (0);
	s1++;
	s2++;
   }
}


/*
 * list_sorted_dir: sort and broadcast directory entries
 */
 
static void list_sorted_dir (obj)
DirObj obj;
{
DIR		*dirp;
struct dirent	*dp;
char		*path = obj->file.path;
char		**tmp;
register char	**cp;
register int	i;

   if (!path)
  	return;


   if ((dirp = opendir(path)) == NULL) {
#ifdef DEBUG
	perror (path);
#endif	
	obj->core.status = errno;
	return;
   }

   obj->dir.index = 0;
   obj->dir.max_cnt = 0;
   obj->dir.dir_list = NULL;
   
   while ((dp = readdir(dirp)) != NULL) {   	

	if (!strcmp (dp->d_name, ".") && 
		obj->dir.exclude_current)
	   continue;
	      
	if (!strcmp (dp->d_name, "..") && 
		obj->dir.exclude_parent)
	   continue;

	if (!obj->dir.dir_list) {
	   obj->dir.dir_list = (char **) malloc 
           	(sizeof (char *) * DIR_LIST_CHUNK_SIZE);
           if (!obj->dir.dir_list) {
   		closedir(dirp);
           	return;
           }
	   obj->dir.max_cnt = DIR_LIST_CHUNK_SIZE;
   	} else if (obj->dir.index >= obj->dir.max_cnt) {
	   tmp = realloc (obj->dir.dir_list, sizeof (Obj) 
                        * (obj->dir.max_cnt + 
                        	DIR_LIST_CHUNK_SIZE));
   	   if (tmp) {
                obj->dir.dir_list = tmp;
                obj->dir.max_cnt += DIR_LIST_CHUNK_SIZE;                    
   	   }
                           
   	}
        
   	if (obj->dir.index >= obj->dir.max_cnt)
		break;
        *(obj->dir.dir_list + obj->dir.index) =
                        strdup (dp->d_name);

        obj->dir.index++;
		   
   }
   closedir(dirp);

   if (!obj->dir.index) {
   	RtSetValue (obj, RtNmsgId, RtMESSAGE);
   	/* Broadcast end of sequence */
   	RtSetValue (obj, RtNmsgContent, NULL);
   	(*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);
    	return;
   }
   
   /* sort the directory entries */
   
   qsort ((char *) obj->dir.dir_list, obj->dir.index, sizeof (char *),
   		string_cmp);
   
   cp = obj->dir.dir_list;
   
   RtSetValue (obj, RtNmsgId, RtMESSAGE); /* check */
   RtSetValue (obj, RtNmsgType, RtMSG_STRING);

   /* Broadcast sorted entries */
   i = 0;
   while (i++ < obj->dir.index) {   
   
	/* Invoke the superclass method to broadcast the message */
	
	RtSetValue (obj, RtNmsgContent, *cp);
	(*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL); 
        free (*cp++);  
   }		 
   

   /* Broadcast end of sequence */
   RtSetValue (obj, RtNmsgContent, NULL);
   (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);

   /* free the memory allocated for the directory list */
   if (obj->dir.dir_list) {
   	free (obj->dir.dir_list);
  	obj->dir.dir_list = NULL;
   }
   	
   obj->dir.index = 0;
   obj->dir.max_cnt = 0;
}

/*
 * list_dir: broadcast the content of a directory
 */
 
static void list_dir (obj)
DirObj obj;
{
DIR		*dirp;
struct dirent	*dp;
char		*path = obj->file.path;

   if (!path)
  	return;


   if ((dirp = opendir(path)) == NULL) {
#ifdef DEBUG
	perror (path);
#endif	
	obj->core.status = errno;
	return;
   }

   RtSetValue (obj, RtNmsgId, RtMESSAGE); /* check */
   /* Broadcast file names */
   while ((dp = readdir(dirp)) != NULL) {   	
   
	if (!strcmp (dp->d_name, ".") && 
		obj->dir.exclude_current) /* Exclude . */
	   continue;
	      
	if (!strcmp (dp->d_name, "..") && 
		obj->dir.exclude_parent)  /* Exclude .. */
	   continue;
	   
	/* Invoke the superclass method to broadcast the message.*/
	/* The directory remains open during what can be a long  */
	/* period of time					 */
	
   	RtSetValue (obj, RtNmsgContent, dp->d_name);
        (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);
   }
   
   /* Broadcast end of sequence */
   RtSetValue (obj, RtNmsgContent, NULL);
   (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);
  
   closedir(dirp);

}

/*
 * DIRSetValues: set object resources
 */
 
void DIRSetValues (current, new)
DirObj current, new;
{
   /* Check - The superclass should do this */
   if (current->core.object_name != new->core.object_name)
   	RtUpdateString (current->core.object_name, 
   			new->core.object_name);
}

/*
 * DIRInitialize: initialize the object
 */
 
void DIRInitialize (obj)
DirObj obj;
{
   /* Exclude . and .. */
   obj->dir.exclude_current = 1;
   obj->dir.exclude_parent = 1;   
#ifdef DEBUG
   obj->core.object_name = strdup ("Dir Object");
#endif
}

/*
 * DIRActivate: activate the object
 */
 
void DIRActivate (obj)
DirObj obj;
{
}


/*
 * DIRRealize: realize the object
 */
 
void DIRRealize (obj)
DirObj obj;
{
}


/*
 * DIRDestroy: destroy the object
 */
 
void DIRDestroy (obj)
DirObj obj;
{
#ifdef OLD
    /* destroy the file part */
    (*fileObjClass->core_class.destroy) (obj);

    /* Invoke the superclass method to destroy the broadcast part */
    (*broadCasterObjClass->core_class.destroy) (obj);
    
    /* free the object memory */
    if (obj->core.object_class == dirObjClass)
    	free (obj);
#endif
    if (obj->dir.dir_list) {
    	free (obj->dir.dir_list);
    	obj->dir.dir_list = NULL;
    }
}

/*
 * DIRProcessMessage: process object messages
 */
 
void DIRProcessMessage (obj, data, client)
DirObj obj;
void *data;
char *client;
{
MessageObj 	msg;
int message_id;

   msg = (MessageObj) data;
   RtGetValue (msg, RtNmsgId, &message_id);   

   switch (message_id) {
      case RtRELEASE_DATA:
        if (obj->dir.sorted)
           list_sorted_dir (obj); /* sort the directory entries */
        else
           list_dir (obj);
        break;
      case RtADD_RECEPTOR:
      case RtADD_CONVERSION:
      case RtADD_ID_CONVERSION:
        /* Invoke the superclass method to add a new receptor */
        (*broadCasterObjClass->core_class.process_message) (obj, 
                        (char *) data, NULL);
        break;
      default:
#ifdef DEBUG
   	fprintf (stderr, 
   	"DIRProcessMessage: I don't know how to process this message\n");
#endif
        break;
   }
}

