/*
 * Freedom Plug & Play reusable toolkit (Rt)
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

/*
 * File Object: object used to manipulate files/directories
 */
 
#include "File.h"

void FILInitialize ();
void FILActivate ();
void FILRealize ();
void FILDestroy ();
void FILSetValues ();
void FILProcessMessages ();
static int write_n (); 

static void read_lines ();
static void read_line ();
extern char *_vstrconcat ();
static void open_file ();
static void change_mode ();
static void change_times ();
static char *dirname ();
static char *path_append ();
static int _iswritableby ();
static char *basename ();

static RtResource resources[] = {
   {RtNfileBasename, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(FileObj,file.basename), 0, NULL, 0, NULL},
   {RtNfileDirectory, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(FileObj,file.dir), 0, NULL, 0, NULL},
   {RtNfilePath, NULL, RtSTRING_TYPE, sizeof(char *), 
   RtOffset(FileObj,file.path), 0, NULL, 0, NULL},
   {RtNfileUid, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.st.st_uid), 0, NULL, 0, NULL},
   {RtNfileGid, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.st.st_gid), 0, NULL, 0, NULL},
   {RtNfileMode, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.st.st_mode), 0, NULL, 0, NULL},
   {RtNfileAccessTime, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.st.st_atime), 0, NULL, 0, NULL},
   {RtNfileModificationTime, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.st.st_mtime), 0, NULL, 0, NULL},
   {RtNfileChangeTime, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.st.st_ctime), 0, NULL, 0, NULL},
   {RtNfileOpenFlags, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.open_flags), 0, NULL, 0, NULL},
   {RtNfileDbFile, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(FileObj,file.db_file), 0, NULL, 0, NULL},
};


FileClassRec fileClassRec = {
  {
    (ObjClass) &broadCasterClassRec,	/* superclass		*/
    "File",			/* class name		*/
    NULL,	
    0,
    resources,			/* resource table	*/
    RtNumber(resources),	/* number of resources  */
    /* object size              */  sizeof(FileRec),
    FILInitialize,
    FILRealize,
    NULL,
    FILSetValues,
    FILActivate,
    FILDestroy,
    FILProcessMessages,
    "file",
    "Rt1.00",
    NULL
  },
  {
  0,
  }
  
};

ObjClass fileObjClass = (ObjClass) &fileClassRec; 

/*
 * FILSetValues: set object resources
 */
 
void FILSetValues (current, new)
FileObj current, new;
{
   RtUpdateString (current->core.object_name, new->core.object_name);
   RtUpdateString (current->file.path, new->file.path);
   RtUpdateString (current->file.dir, new->file.dir);
   RtUpdateString (current->file.basename, new->file.basename);
}

/*
 * FILInitialize: initialize the object
 */
 
void FILInitialize (obj)
FileObj obj;
{
#ifdef DEBUG
   obj->core.object_name = strdup ("File Object");
#endif
   obj->file.fd = -1;
   
   /* default mode for the open operation */
   obj->file.st.st_mode = S_IRUSR | S_IWUSR | S_IRGRP | S_IROTH;
}

/*
 * FILRealize: realize the object
 */ 

void FILRealize (new)
FileObj new;
{
}

/*
 * FILDestroy: destroy the object
 */
 
void FILDestroy (obj)
FileObj obj;
{
   if (obj->file.fd != 1) {
        close (obj->file.fd);   /* Close the file descriptor */
        obj->file.fd = -1;
   }
   
   if (obj->file.fp) {
        fclose (obj->file.fp);   /* Close the file stream */
        obj->file.fp = NULL;
   }

   /* free the memory allocated for the strings */

   RtDestroyString (obj->file.path);
   RtDestroyString (obj->file.dir);
   RtDestroyString (obj->file.basename);

   /* destroy the broadcaster portion of the object */
   (*broadCasterObjClass->core_class.destroy) (obj);

   /* destroy the core portion of the object */
   RtDestroyString (obj->core.object_name);   

   /* free the object itself */
   if (obj->core.object_class == fileObjClass)
      free (obj);   
}

/*
 * FILActivate: Activate the object
 */
 
void FILActivate (obj)
FileObj obj;
{
}


#ifdef STANDALONE
static char *_dircat ();
static int _isdir ();
#else
char *_dircat ();
int _isdir ();
#endif



/*
 * FILProcessMessages: process object messages
 */

void FILProcessMessages (obj, data, client)
FileObj obj;
void *data;
char *client;
{
int len, tmp;
char *newpath;
MessageObj msg;
char *content;
char *aux;
int flags;
int message_id;
char *msg_client;

   obj->core.status = 0;
   
   msg = (MessageObj) data;
   RtGetValue (msg, RtNmsgId, &message_id);   
   RtGetValue (msg, RtNmsgContent, &content);   
   RtGetValue (msg, RtNmsgFlags, &flags);   
   
   /* ignore the function paramenter  		*/
   /* get the client info from the message 	*/
   RtGetValue (msg, RtNmsgClient, &msg_client); 

   switch (message_id) {
#ifdef OLD /* new */
      case RtADD_RECEPTOR:
      case RtADD_CONVERSION:
      case RtADD_ID_CONVERSION:
        /* Invoke the superclass method 	*/
        (*broadCasterObjClass->core_class.process_message) (obj, 
                        (char *) data, NULL);
        break;
#endif
      case RtMESSAGE:
      case RtFILE_WRITE_LINE:

	if (flags & RtFILE_STRIP_NEW_LINES)
	   obj->file.strip_newlines = 1;
	   	
#ifdef DEBUG      
        fprintf (stderr, "data received %s\n", content);
#endif 
	/* open the file if needed */  
	if (obj->file.fd < 0) {
	   open_file (obj);
	}
	   
	if (obj->core.status)
	   return;

	/* client contains the length of the record    */
	/* or 0 in the case of NULL terminated strings */	         

        if (msg_client)
          len = (int) msg_client;
        else
          len = (content?strlen (content):0);

	/* close the file when a NULL		*/ 
	/* or a zero-length record is received  */
	
        if (!content || !len) {
	   if (obj->file.fd != 1) { /* check stdin & stdout */
              close (obj->file.fd);
              obj->file.fd = -1;
           }
           return;
        }
                
        /* write content to output file */ 
        write_n (obj->file.fd, (char *) content, len);

	/* write a newline after each record (RtFILE_WRITE_LINE */
	/* message)						*/
	
        if (message_id == RtFILE_WRITE_LINE)
           write_n (obj->file.fd, (char *) "\n", 1); 
        
        /* Close the file if needed */
        if (flags & RtFILE_CLOSE) 
           if (obj->file.fd != 1) { /* check stdin & stdout */
              close (obj->file.fd);
              obj->file.fd = -1;
           }
        break;
      case RtFILE_READ_LINES:      
      case RtFILE_READ_LINE:

	if (flags & RtFILE_STRIP_NEW_LINES)
	   obj->file.strip_newlines = 1;
	   	
	/* open the file if needed */  
	if (obj->file.fd < 0) {
	   tmp = obj->file.open_flags;
	   obj->file.open_flags = O_RDONLY;
	   open_file (obj);
	   obj->file.open_flags = tmp;
	}
	
	/* check the status of the object after the open */
	/* operation					 */
	  
	if (obj->core.status)
	   return;	/* failed to open the file */
	
	/* Associate a stream with the file descriptor */
	   
	obj->file.fp = fdopen (obj->file.fd, "r");
	
	if (!obj->file.fp) {
	   obj->core.status = errno;
	   return;
	}

        if (message_id == RtFILE_READ_LINES)
        /* read the file line by line */
           read_lines (obj);
        else
           read_line (obj);
        
        if (message_id == RtFILE_READ_LINES ||
                (flags & RtFILE_CLOSE)) {
                /* close the stream and the file descriptor */
                fclose (obj->file.fp);
                close (obj->file.fd);
                obj->file.fp = NULL;
                obj->file.fd = -1;
        }
	break;        
      case RtFILE_STAT: /* get the status information */
        obj->core.status = stat (obj->file.path, &obj->file.st);
        if (obj->core.status < 0)
          obj->core.status = errno; 
        break;
      case RtFILE_OPEN: /* open the file	      */
        open_file (obj);
        break;      
      case RtFILE_CHMOD: /* change the protections of the file */
        change_mode (obj, (mode_t) data);
        break;
      case RtFILE_UTIME: /* Change the access and modification time of a file*/
        change_times (obj, (time_t) data, (time_t) msg_client);
        break;
      case RtFILE_CHOWN: /* Change owner and group */
        if ((obj->core.status = chown 
                (obj->file.path, (uid_t) data, (gid_t) msg_client)) < 0) {
#ifdef DEBUG
           perror (obj->file.path);
#endif           
        }
        if (obj->core.status < 0)
           obj->core.status = errno;
        break;
      case RtFILE_DIRNAME:
        if (obj->file.dir)
           free (obj->file.dir);
        obj->file.dir = dirname (obj->file.path);
        break;
      case RtFILE_BASENAME:
        if (obj->file.basename)
           free (obj->file.basename);
        obj->file.basename = basename (obj->file.path);
        break;
      case RtFILE_UNLINK:/* unlink file         */
        if (!obj->file.path)
           return;
        obj->core.status =  unlink(obj->file.path);
        if (obj->core.status < 0)
           obj->core.status = errno;
        break;
      case RtFILE_APPEND_NAME: /* Add a new component to the path */
        if (!data)
           return;
        newpath = _dircat (obj->file.path, (char *) data);
        if (!newpath) {
           obj->core.status = RtFILE_INVALID_PATH;
           return;
        }
        RtSetValue (obj, RtNfilePath, newpath);
        free (newpath);
        break;
      case RtFILE_PATH_APPEND: /* Add a new component to the path */
      			       /* and broadcast the result        */
        newpath = _dircat (obj->file.path, (char *) content);
        if (!newpath) {
           obj->core.status = RtFILE_INVALID_PATH;
           return;
        }

        RtSetValue (obj, RtNmsgId, RtMESSAGE);
   	RtSetValue (obj, RtNmsgContent, newpath);
        RtSetValue (obj, RtNmsgClient, NULL);
#ifdef DEBUG
        RtSetValue (obj, RtNmsgType, RtMSG_STRING);
#endif        
   	
   	/* Broadcast result */
   	
        (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);

        free (newpath);
        break;
      case RtFILE_EXISTS:
        obj->core.status = _exists (obj->file.path);
        break;
      case RtFILE_IS_DIRECTORY:
        obj->core.status = _isdir (obj->file.path);
        break; 
      case RtFILE_VALVE_IS_DIRECTORY:
        /* Act as a valve. Allow the flow of directories */
        if (!content)
           return;
           
        if (_isdir (content))
           (*broadCasterObjClass->core_class.process_message) (obj, 
                        (char *) msg, NULL);
        else {           
           if (!content)
           	return;
           	
           /* Send a message to the stderr object */
           /* check */
           aux = _vstrconcat (content, 
      		": ", "this is not a directory", NULL);
           
      	   if (!aux)
          	return;
          
           RtSetValue (msg, RtNmsgId, RtMESSAGE);
           RtSetValue (msg, RtNmsgContent, aux);
           RtSetValue (msg, RtNmsgClient, NULL);
	   RtSendMessage (obj->core.obj_stderr, (char *) msg, NULL);        
      	   free (aux);
        }
        break;
      case RtFILE_WRITABLE_BY:
        obj->core.status = _iswritableby (obj->file.path, (int) data);
        break; 
      case RtFILE_RENAME: /* Rename a file */
        if (!content)
           return;
        obj->core.status = rename (obj->file.path, (char *) content);
        if (!obj->core.status) {
          RtSetValue (obj, RtNfilePath, (char *) content);
        } else
          obj->core.status = errno;
        break;
      default:
#ifdef DEBUG
        fprintf (stderr,
           "File Object: unknown message ID, I don't know how to process this message\n");
#endif
	RtSuperclassProcessMessage (fileObjClass,
		obj, data, client); /* new */
	break;
   }
}

/*
 * open_file: open the file
 */
 
static void open_file (new)
FileObj new;
{
   new->core.status = 0;
   
   if (new->file.fd == 1)	/* check - avoid closing stdout */
   	return;
   	
   if (!new->file.path) {
   	new->core.status = RtINVALID_RESOURCES;
        return;
   }
   
   if ((new->file.fd = open (new->file.path, 
        	new->file.open_flags, 
        	new->file.st.st_mode)) < 0) {
#ifdef DEBUG
        perror (new->file.path);
#endif
   }

   if (new->file.fd < 0)
   	new->core.status = errno;
}

/*
 * change_mode: change the mode of the file or directory
 */
 
static void change_mode (obj, mode)
FileObj obj;
mode_t mode;
{
   if (!obj->file.path) {
   	obj->core.status = RtFILE_INVALID_PATH;
        return;
   }
   
   if ((obj->core.status = chmod (obj->file.path, mode)) < 0) {
#ifdef DEBUG
        perror (obj->file.path);
#endif
	obj->core.status = errno;       
   }
}

/*
 * change_times: change the access and the modificatio dates
 */
 
static void change_times (obj, access_time, modification_time)
FileObj obj;
time_t access_time;
time_t modification_time;
{

   struct utimbuf timebuf;

   if (!obj->file.path) {
	obj->core.status = RtFILE_INVALID_PATH;   	
        return;
   }
   
   if (!access_time || !modification_time) {
	obj->core.status = RtINVALID_PARAMETERS;   	
   }

   timebuf.actime = access_time;
   timebuf.modtime = modification_time;
   
   if ((obj->core.status = utime (obj->file.path, &timebuf)) < 0) {
#ifdef DEBUG
        perror (obj->file.path);
#endif
	obj->core.status = errno;
        return;
   }
}

/*
 * basename: returns the last component of the path. It allocates
 *	     memory before returning the string
 *	     
 */
 
static char *basename (path)
char *path;
{
char *cp;
   
   if (!path)
        return (NULL);
   
   if (!(cp = rindex (path, '/'))) {
        return (strdup (path));
   }

   return (strdup (++cp));              
}

/*
 * dirname: returns all but the last level of the path name. This
 *	    function allocates memory before returning the string	       
 */
 
static char *dirname (path)
char *path;
{
char *cp, *aux;
int len;

   if (!path)
        return (NULL);
   
   if (!(cp = rindex (path, '/'))) {
        return (strdup ("."));
   }

   len = cp - path;
   
   if (!len)
        return (strdup ("/"));
   
   aux = (char *) malloc ((len+1) * sizeof (char)); 
   memcpy (aux, path, len);
   aux[len] = '\0';
     
   return (aux);                
}

#ifdef STANDALONE
/*
 * _dircat: Append a filename to a directory path
 * 
 * Limitation: It does not consider NULL parameters
 * Taken from FD. Reviewed 8/95
 */

static char *_dircat (path, subpath)
register char *path, *subpath;
{

   size_t length;
   char *newpath;

   if (!path) {
        fprintf (stderr, "_dircat: Invalid arguments\n");
        return (NULL);
   }

   length = _strlen (path) + _strlen (subpath) + 2;/* One byte might be wasted */


   if ((newpath = malloc (length)) == NULL) {
        fprintf (stderr, "dircat: not enough memory\n");
        return (NULL);
   }

   newpath[0] = '\0';
   if (path[strlen(path) - 1] == '/')
        strcpy (newpath, path);
   else
        sprintf (newpath, "%s/", path);

   strcat (newpath, subpath);


   return (newpath);
}

/*
 * _exists: check if fname or if the file pointed by fname (symbolic 
 *          links) exists
 */

int _exists (fname)
char *fname;
{
struct stat fstat;

   if(stat(fname, &fstat) < 0) { 
        return (0);
   } else
        return (1);
}

/*
 * _isdir: Determines if fname is a directory or if fname points
 *          to a directory (symbolic links)
 */

static int _isdir (fname)
char *fname;
{
struct stat fstat;

   if(stat(fname, &fstat) < 0) { 
        return (0);
   } else
        return (S_ISDIR(fstat.st_mode));
}
#endif

/*
 * _iswritableby: check if fname is writable by the user with id uid
 * check:	  This function needs to handle groups             
 */
 
static int _iswritableby (fname, uid)
char *fname;
uid_t uid;
{
struct stat fstat;
mode_t    mode;

    if (!fname)
        return (-1);
       
    if (stat (fname, &fstat) < 0) {
        return (-1);    
    }
    
   mode = fstat.st_mode;
   
   if (uid == fstat.st_uid)
        return (mode & S_IWUSR);

   return (mode & S_IWOTH); /* check */

}

/*
 * _stripnl: remove the ending newline from s
 */

static char *_stripnl (s)
register char *s;
{
   int l;

   if (!s || !*s)
	return (s);

   l = strlen (s);

   if (s[l - 1] == '\n')
	s[l - 1] = '\0';

   return (s);
	
}

/*
 * read_line: read one line from a file  
 */

static void read_line (obj)
FileObj obj;
{
  char line[MAXFLINE];
  
   if (!obj->file.fp) {
        obj->core.status = RtINVALID_RESOURCES;
        return;
   }    

   RtSetValue (obj, RtNmsgId, RtMESSAGE);
   RtSetValue (obj, RtNmsgClient, NULL);
   if (fgets (line, MAXFLINE, obj->file.fp)) { /* check */
        /* strip newline  */
        _stripnl (line);
        /* broadcast line */
        RtSetValue (obj, RtNmsgContent, line);
        
        /* Invoke the superclass method to broadcast the message */
        (*broadCasterObjClass->core_class.process_message) (obj, 
                        (char *) obj, NULL);
           
   }
}

/*
 * read_lines: read lines from a file  
 */

static void read_lines (obj)
FileObj obj;
{
  char line[MAXFLINE];
  static char *_stripnl ();
  
   if (!obj->file.fp) {
   	obj->core.status = RtINVALID_RESOURCES;
	return;
   }	

   RtSetValue (obj, RtNmsgClient, NULL);
   while (fgets (line, MAXFLINE, obj->file.fp)) {
        /* strip newline  */
        if (obj->file.strip_newlines)
   	   _stripnl (line);
	/* broadcast line */
   	RtSetValue (obj, RtNmsgContent, line);
   	
    	if (obj->file.db_file) {
    	   /* load into the database (database class) */
   	   RtSetValue (obj, RtNmsgId, obj->file.db_file);
	   RtSendMessage (obj, obj, NULL);	    	   
    	} else {
  	   /* Invoke the superclass method to broadcast the message */
   	   RtSetValue (obj, RtNmsgId, RtMESSAGE);
           (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);
        }
   }
   /* broadcast a NULL */
   RtSetValue (obj, RtNmsgContent, NULL);
   	
   /* Invoke the superclass method to broadcast the message */
   (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);
}

/* 
 * write_n: Write exactly n bytes 
 */

static int
write_n (fd, vptr, n)    /* Write exactly n bytes */
int fd;
void *vptr;
size_t n;
{
   size_t nleft, nwritten;
   char *ptr;
   
   ptr = vptr;
   nleft = n;
   
   while (nleft > 0) {
        if ((nwritten = write(fd, ptr, nleft)) < 0)
           if (errno == EINTR)
              continue;         /* interrupted system call */
           else
              return (nwritten); /* Error */
        
        if (!nwritten)
           break;       /* EOF */
           
         nleft -= nwritten;
         ptr += nwritten;     

   }
   return (n);

}
