/*
 * Freedom Desktop 
 * Copyright 1994 by Freedom Software
 *
 * Freedom Software retains all rights to Freedom Desktop (hereafter Software)
 * in binary and in source code form.
 *
 * The commercial use of this Software shall be governed by a separate License
 * agreement. Any individual or institution wishing to make commercial use of
 * the Software must sign a license agreement with Freedom Software. In such
 * cases, the Licensee agrees to abide by the terms contained in the License
 * Agreement and not those contained in this document. Examples of commercial
 * use include (without limitation): (i) integration of the Software (source
 * code form), in whole or in part, into a commercial product sold by or on
 * on behalf of the Licensee; (ii) distribution of the Software (binary form or
 * source code form) in combination with a commercial product sold by or on
 * behalf of the Licensee.
 *
 * Freedom Software (Licensor) grants you (Licensee) a license: (i) to use,
 * copy and make changes and improvements to this Software for licensee's
 * internal business purposes; (ii) to use, copy, and distribute this Software
 * or the derivative works provided that the copyright notice and this
 * permission notice appear on all copies and that NO CHARGE is associated
 * with such copies. However, if Licensee distributes any derivative work
 * based on the Software, then Licensee shall (i) notify Licensor in writing
 * (ii) clearly state that such derivative work is a modified and not the
 * original Freedom Desktop distributed by Freedom Software (iii) publish
 * the corresponding machine-readable source code or information as to
 * where it may be obtained. Each time Licensee redistribute the Software
 * or any derivative work, the recipient automatically agrees to abide
 * by the same terms as the Licensee. Licensee may not impose terms
 * more restrictive than the terms granted herein.
 *
 * By using, copying, modifying or distributing this Software (or any
 * derivative work based on this Software) Licensee indicates acceptance
 * of the terms and conditions set forth in this License.
 *
 * Licensor reserves the right to terminate this License immediately on written
 * notice, for material breach by the Licensee.
 *
 * FREEDOM SOFTWARE DISCLAIMS ALL WARRANTIES EXPRESS OR IMPLIED WITH REGARD
 * TO THIS SOFTWARE INCLUDING BUT NOT LIMITED TO ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND  FITNESS,  IN  NO  EVENT  SHALL LICENSOR BE LIABLE
 * FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE
 */

/*
 * XmList wrapper Object - Object used to handle list objects
 */

#include "ListWrapper.h"

void XMLWInitialize ();
void XMLWActivate ();
int XMLWRealize ();
void XMLWDestroy ();
void XMLWSetValues ();
void XMLWProcessMessage ();
char *get_selected ();
static void list_items ();
static void release_selected_items ();
static void extended_selection ();
static void delete_selected_item ();
static void add_item ();
static void insert_oitem ();
static char *find_by_key ();
extern char *strgetw ();
void replace_item ();

static RtResource resources[] = {
   {RtNxmlwOneInstance, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(XmListWrapperObj,list_wrapper.one_instance), 0, NULL, 0, 
   NULL},
   {RtNxmlwOrdered , NULL, RtBIT_TYPE, 1, 
   RtOffset(XmListWrapperObj,list_wrapper.flags), 0, NULL, 0, 
   (char *) RtXMLW_ORDERED},   
   {RtNxmlwImmediateMode, NULL, RtINT_TYPE, sizeof(int), 
   RtOffset(XmListWrapperObj,list_wrapper.immediate_mode), 0, NULL, 0, 
   NULL},
};

XmListWrapperClassRec xmListWrapperClassRec = {
  {
    (ObjClass) &rtWidgetClassRec,
    "List Wrapper",
    NULL,
    0,
    resources,
    RtNumber (resources),
    /* object size              */  sizeof(XmListWrapperRec),
    XMLWInitialize,
    XMLWRealize,
    NULL,
    XMLWSetValues,
    XMLWActivate,
    XMLWDestroy,
    XMLWProcessMessage,
    "xmlw",
    "Rt1.00",
    NULL
  },
  {
  0,
  }
  
};

ObjClass xmListWrapperObjClass = (ObjClass) &xmListWrapperClassRec; 

/*
 * XMLWSetValues: set object resources
 */
 
void XMLWSetValues (current, new)
XmListWrapperObj current, new;
{
   if (current->core.object_name != new->core.object_name)
        RtUpdateString (current->core.object_name, 
        	new->core.object_name);
}


/*
 * XMLWInitialize: initialize the object
 */

void XMLWInitialize (obj)
XmListWrapperObj obj;
{
#ifdef DEBUG
   obj->core.object_name = strdup ("XmList Wrapper");
#endif
}

/*
 * XMLWActivate: activate the object
 */

void XMLWActivate (obj)
XmListWrapperObj obj;
{
}


/*
 * XMLWRealize: realize the object
 */

int XMLWRealize (obj)
XmListWrapperObj obj;
{
   Widget widget = obj->widget.widget;
   unsigned char selection_policy;

   if (!widget)
        return;
        
   XtVaGetValues (widget, XmNselectionPolicy, 
                &selection_policy, NULL);
                        
   if (obj->list_wrapper.immediate_mode
        && (selection_policy == XmEXTENDED_SELECT)) {
        XtAddCallback (widget, XmNextendedSelectionCallback,
                extended_selection, obj);          
   }
}

/*
 * XMLWDestroy: destroy the object
 */

void XMLWDestroy (obj)
XmListWrapperObj obj;
{
   RtDestroyString (obj->core.object_name);   

   /* destroy the broadcaster portion of the object */
   (*broadCasterObjClass->core_class.destroy) (obj);

   /* free the object itself */
   if (obj->core.object_class == xmListWrapperObjClass)
      free (obj);   
}


/*
 * XMLWProcessMessage: process object messages
 */

void XMLWProcessMessage (obj, data, client)
XmListWrapperObj obj;
void *data;
void *client;
{
MessageObj msg;
char *content;
int message_id;
char *msg_client;

   msg = (MessageObj) data;
   RtGetValue (msg, RtNmsgId, &message_id);   
   RtGetValue (msg, RtNmsgContent, &content);   
      
   switch (message_id) {

      case RtXMLW_ADD_ITEM:    /* add an item to the list      */
        if (obj->list_wrapper.flags & RtXMLW_ORDERED)
           insert_oitem (obj, content);
        else
           add_item (obj, content);
        break;
      case RtXMLW_REPLACE_ITEM:
   	RtGetValue (msg, RtNmsgClient, &msg_client);   
	if (!msg_client || !content)
	   return;
	replace_item (obj, content, msg_client); 
	break;        
      case RtXMLW_DELETE_ITEM: /* delete an item from the list */
        delete_item (obj, content);
        break;
      case RtLIST_DELETE_ALL_ITEMS:
        if (obj->widget.widget)
           XmListDeleteAllItems (obj->widget.widget);
        break;
      case RtXMLW_DELETE_SELECTED_ITEM:
	delete_selected_item (obj);      
        break;
      case RtXMLW_LIST_ITEMS:	    /* list items		    */
        list_items (obj);
        break;
      case RtRELEASE_DATA:
#ifdef DEBUG
   	RtSetValue (obj, RtNmsgType, RtMSG_STRING);
#endif
	release_selected_items (obj);
        break;
      default:
#ifdef DEBUG
   	fprintf (stderr, 
   	"XMLWProcessMessage: I don't know how to process this message\n");
#endif
	/* Invoke the superclass to process the message */
	RtSuperclassProcessMessage (rtWidgetObjClass,
		obj, data, client); /* check */
        return;
   }
}

/*
 * delete_selected_item: delete the first selected item 
 */
 
static void delete_selected_item (obj)
XmListWrapperObj obj;
{

int *pos_list;
int cnt;

   if (!obj->widget.widget)
   	return;
   if (!XmListGetSelectedPos (obj->widget.widget,
   		&pos_list,
   		&cnt))
   	return;
   XmListDeletePos (obj->widget.widget, *pos_list);
   XtFree (pos_list);
}


/*
 * delete_item: delete an item from the list
 */
 
delete_item (obj, item)
XmListWrapperObj obj;
char *item;
{
   int pos;   
   XmString cs;
   Widget widget = obj->widget.widget;
   
   if (!item || !widget)
   	return;
   
   cs = XmStringCreateSimple (item);
   if (!cs)
	return;

   if (pos = XmListItemPos (widget, cs)) 	
   	XmListDeletePos (widget, pos);
   XmStringFree (cs);
}

/*
 * replace_item: replace a list item
 */
 
void replace_item (obj, item, new_item)
XmListWrapperObj obj;
char *item;
char *new_item;
{
   XmString cs[1], cs1[1];
   char *orig_item;
   
   if (!item || !obj->widget.widget || !new_item)
   	return;

   orig_item = find_by_key (obj, item);
   
   if (!orig_item)
   	return;
   
   cs[0] = XmStringCreateSimple (orig_item);
   free (orig_item);
   cs1[0] = XmStringCreateSimple (new_item);
   if (!(cs[0]) || !(cs1[0]))
	return;

   XmListReplaceItems (obj->widget.widget, cs, 1, cs1);
   /* check notify parameter */
   XmListSelectItem (obj->widget.widget, cs1[0], False); 
   XmStringFree (cs[0]);
   XmStringFree (cs1[0]);
}

/*
 * find_insertion_pos: find the position where
 *	the new item is to be inserted 
 */
 
static int find_insertion_pos (widget, item)
Widget widget;
char *item;
{
 int mid, lo, hi, cnt;
 int aux;
 XmString current;
 XmString *Items;
 char *value;
    
   if (!item)
   	return (0);

   XtVaGetValues (widget, XmNitems, &Items, 
   		XmNitemCount, &cnt, 
   		NULL);
   
   if (cnt == 0)
   	return (1);
   	
   lo = 0;
   hi = cnt - 1;

   while (lo <= hi) {
   	mid = (lo+hi)/2;
   	
   	current = Items[mid];

        if (!XmStringGetLtoR(current, XmSTRING_DEFAULT_CHARSET, 
        	&value))
	   return (0);

	if (!value)
	   value = "";
	      
	aux = strcmp (item, value);
   	XtFree (value);
   	
	if (!aux)
   	   return (mid+1);
   	
   	if (aux > 0) {
	   lo = mid+1; /* upper portion ? */	   
   	} else
   	   hi = mid-1;

   }
   return (lo+1);

}

/*
 * insert_oitem: insert an item in the list
 */
 
static void insert_oitem (obj, item)
XmListWrapperObj obj;
char *item;
{
   int i, pos;   
   XmString cs;
   Widget widget = obj->widget.widget;
   
   if (!item || !*item || !widget)
      return;
   
   i = obj->list_wrapper.nitems;
   
   cs = XmStringCreateSimple (item);
   if (!cs)
	return;
	
   if (obj->list_wrapper.one_instance) {
   	if (pos = XmListItemPos (widget, cs)) {
   	   XmStringFree (cs);
   	   return;
   	} 	   
   }	

   pos = find_insertion_pos (widget, item);
   XmListAddItem (widget, cs, pos);
   XmListSelectPos (widget, pos, False);
   XmListSetBottomPos (widget, pos);
   
   obj->list_wrapper.nitems++;
   XmStringFree (cs);
} 

/*
 * find_by_key: find item by key
 */
 
static char *find_by_key (obj, item)
XmListWrapperObj obj;
char *item;
{
   Widget widget = obj->widget.widget;
   XmString *Items;
   int nsel = 0;
   char *text;
   char *key, *key1;

   if (!widget)
      return (NULL);
      
   key = strgetw (item, ' ');
   if (!key)
      return (NULL);
         
   XtVaGetValues (widget, XmNitems, &Items, 
			XmNitemCount, &nsel, NULL);

   while (nsel--) {
   	if (!XmStringGetLtoR(*Items++, 
   		XmSTRING_DEFAULT_CHARSET, &text))
		text = NULL;
	key1 = 	strgetw (text, ' ');
	if (!text || !key1)
	   continue;
	   
	if (!strcmp (key, key1)) {
	   free (key1);
	   free (key);
	   return (text);
	}
	free (key1);	
   	if (text)
   	   XtFree (text);
   }
   free (key);
   return (NULL);
 }

/*
 * add_item: add an item to the list
 */
 
static void add_item (obj, item)
XmListWrapperObj obj;
char *item;
{
   int i, pos;   
   XmString cs;
   Widget widget = obj->widget.widget;
   
   if (!item || !*item || !widget)
      return;
   
   i = obj->list_wrapper.nitems;
   
   cs = XmStringCreateSimple (item);
   if (!cs)
	return;
	
   if (obj->list_wrapper.one_instance) {
   	if (pos = XmListItemPos (widget, cs)) {
   	   XmStringFree (cs);
   	   return;
   	} 	   
   }	
   XmListAddItem (widget, cs, 0);
   XmListSelectPos (widget, 0, False);
   XmListSetBottomPos (widget, 0);
   
   obj->list_wrapper.nitems++;
   XmStringFree (cs);
} 

/*
 * get_selected: get the selected item. It needs to be enhanced
 *               to handle multiple items
 */
 
char *get_selected (obj)
XmListWrapperObj obj;
{
   XmString *selItems;
   int nsel;
   char *text;
   Widget widget = obj->widget.widget;

   XtVaGetValues (widget, XmNselectedItems, &selItems, 
			XmNselectedItemCount, &nsel, NULL);

   if (nsel == 0)
	return (NULL);

   if (!XmStringGetLtoR(*selItems, XmSTRING_DEFAULT_CHARSET, &text))
	return (NULL);
   return (text);
}

/*
 * list_items: list all of the items of the list
 */
 
static void list_items (obj)
XmListWrapperObj obj;
{
   Widget widget = obj->widget.widget;
   XmString *Items;
   int nsel = 0;
   char *text;

   if (!widget)
      return;
      

   XtVaGetValues (widget, XmNitems, &Items, 
			XmNitemCount, &nsel, NULL);

   RtSetValue (obj, RtNmsgId, RtMESSAGE);
   RtSetValue (obj, RtNmsgClient, NULL);
   while (nsel--) {
   	if (!XmStringGetLtoR(*Items++, 
   		XmSTRING_DEFAULT_CHARSET, &text))
		text = NULL;
		
   	RtSetValue (obj, RtNmsgContent, text);
   		
   	/* Invoke the superclass method to broadcast the message */
        (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);
   	if (text)
   		XtFree (text);
   }
   
   RtSetValue (obj, RtNmsgContent, NULL);
   (*broadCasterObjClass->core_class.process_message) (obj, 
        		(char *) obj, NULL);

}

/*
 * release_selected_items: broadcast the selected items.
 */
 
static void release_selected_items (obj)
XmListWrapperObj obj;
{
   XmString *selItems;
   int nsel;
   char *text;
   Widget widget = obj->widget.widget;
   unsigned char selection_policy;

   XtVaGetValues (widget, XmNselectedItems, &selItems, 
                        XmNselectedItemCount, &nsel, 
                        XmNselectionPolicy, &selection_policy, NULL);

   if (nsel == 0) {
	if (!obj->core.obj_stderr)
           return;
        obj->core.status = RtXMLW_NONE_SELECTED_ITEMS;
        RtSetValue (obj->core.obj_stderr, RtNmsgId, 
        		RtERR_ID);
        RtSetValue (obj->core.obj_stderr, RtNmsgContent, 
        		RtXMLW_NONE_SELECTED_ITEMS);
        RtSetValue (obj->core.obj_stderr, RtNmsgClient, 
        		NULL);
        RtSendMessage (obj->core.obj_stderr,
        		obj->core.obj_stderr);
   }

   RtSetValue (obj, RtNmsgId, RtMESSAGE); /* For now */
   while (nsel--) {
        if (!XmStringGetLtoR(*selItems, XmSTRING_DEFAULT_CHARSET, &text))
           text = NULL;
           
        RtSetValue (obj, RtNmsgContent, text);
        
        /* Invoke the superclass method to broadcast the message */
        (*broadCasterObjClass->core_class.process_message) (obj, 
                        (char *) obj, NULL);
        if (!text)
           break;
        else
           XtFree (text);
        selItems++;
   }
   
   if (selection_policy == XmSINGLE_SELECT || 
        selection_policy == XmBROWSE_SELECT)
        return;
   
   /* Send a NULL message to indicate that all the list items have */
   /* been sent                                                    */
   RtSetValue (obj, RtNmsgContent, NULL);
   (*broadCasterObjClass->core_class.process_message) (obj, 
                        (char *) obj, NULL);
     
}

static void
extended_selection (widget, client_data, cbs)
Widget widget;
XtPointer client_data;
XmListCallbackStruct *cbs;
{
   XmListWrapperObj obj = (XmListWrapperObj) client_data;
   
   if (obj)
        release_selected_items (obj);  

}


