/* Hash Library */

#include <stdio.h>
#include <stdlib.h>
#include "hash.h"

static size_t _n_buckets = 0;

/*
 * DefaultHash: Calculates a simple hash function
 */

int DefaultHash(key)
char *key;
{
    char c;
    int index=0;

    if (!key) {
        fprintf (stderr, "DefaultHash: Invalid Arguments\n");
        return (-1); 
    }

    while ((c = *key++) != '\0') {
        index += c;
    }
    index &= (_n_buckets-1);
    return index;
}

/*
 * hash__Create: create the hash structure
 */

Hash *hash__Create(n_buckets)
size_t n_buckets;
{
 static size_t hashsize = HASHSIZE;
 Hash *htbl;
 size_t i = 1;

   if (n_buckets <= 0) {
	_InvalidArg ("hash_Create: Invalid arguments\n");
	return (NULL);
   }

   while (i < n_buckets)
	i <<= 1;

#ifdef DEBUG
   fprintf (stderr, "n_buckest: old = %u, new = %u\n", n_buckets, i);
#endif
   n_buckets = i;


   if ((htbl = (Hash *) malloc (hashsize)) == NULL) {
	fprintf (stderr, "hash_Create: not enough memory\n");
	return (NULL);
   }

   if (!(htbl->buckets =
		(_SimpleList **) malloc (n_buckets * sizeof (_SimpleList *)))) {
	fprintf (stderr, "hash_Create: not enough memory\n");
	free (htbl);
	return (NULL);
   }
   memset ((char *) htbl->buckets, '\0', n_buckets * sizeof (_SimpleList *));
   htbl->n_buckets = _n_buckets = n_buckets;
   htbl->hash = DefaultHash;

   return (htbl);
}


/*
 * hash__Replace: replace value with newvalue
 */

Egg *hash__Replace(self,key,newvalue)
struct hash *self;
char *key;
char *newvalue;
{
    struct egg *egg;

    egg = hash__EggLookup (self, key);
    if (!egg)
	return (NULL);

    egg->value = newvalue;
    return (egg);
}


/*
 * hash__Store: store value
 */

Egg *hash__Store(self,key,value)
struct hash *self;
char *key;
char *value;
{
    int bucket;
    struct egg *egg;

    if (!self || !key) {
	_InvalidArg ("hash__Store: Invalid Arguments\n");
	return (NULL);
    }

    bucket = (*self->hash)(key);

    egg = (struct egg *)malloc(sizeof(struct egg)); 

    if (!egg) {
	fprintf (stderr, "hash__Store: not enough memory\n");
	return (NULL);
    }

    egg->key = (char *)malloc(strlen(key)+1); 

    if (!(egg->key)) {
	fprintf (stderr, "hash__Store: not enough memory\n");
	free (egg);
        return (NULL);
    }

    strcpy(egg->key,key);
    egg->value = value;

    if (self->buckets[bucket] == NULL)
        self->buckets[bucket] = _SListCreate(); 

    _SListAppend(self->buckets[bucket],egg);
    return (egg);
}

/*
 * FindEgg: verify match
 */

static int FindEgg(key, egg)
char *key;
struct egg *egg;
{
    return (strcmp(key, egg->key));
}



/*
 * hash__Lookup: search for key
 */

char *hash__Lookup(self,key)
struct hash *self;
char *key;
{
    int bucket;
    struct egg *egg;

    if (!self || !key) {
        _InvalidArg ("hash__Lookup: Invalid Arguments\n");
        return (NULL);
    }

    bucket = (*self->hash)(key);

    if (self->buckets[bucket] == NULL) {
        return NULL;
    }
    else {
        egg = (struct egg *) _SListFind(self->buckets[bucket],key,FindEgg,
				_LST_BEGINNING);
        if (egg != NULL) {
            return egg->value;
        }
        else {
            return NULL;
        }
    }
}

struct egg *hash__EggLookup(self,key)
struct hash *self;
char *key;
{
    int bucket;
    struct egg *egg;

    if (!self || !key) {
        _InvalidArg ("hash__Lookup: Invalid Arguments\n");
        return (NULL);
    }

    bucket = (*self->hash)(key);

    if (self->buckets[bucket] == NULL) {
        return NULL;
    }
    else {
        egg = (struct egg *) _SListFind(self->buckets[bucket],key,FindEgg,
				_LST_BEGINNING);
	return (egg);
        }
}

/*
 * hash__Delete: delete key
 */

int hash__Delete(self,key)
struct hash *self;
char *key;
{
    int bucket;
    struct egg *egg;

    if (!self || !key) {
        _InvalidArg ("hash__Delete: Invalid Arguments\n");
        return (NULL);
    }

    bucket = (*self->hash)(key);

    if (self->buckets[bucket] == NULL)
        return (0);
    egg = (struct egg *) _SListFind(self->buckets[bucket],key,FindEgg,
				_LST_BEGINNING);
    if (egg != NULL) {
	free (egg->key);
	egg->key = NULL;
        _SListDelete(self->buckets[bucket],egg);
	return (1);
    }
    return (0);
}

/*
 * terminator: destroy element
 */

int terminator (iptr)
char *iptr;
{
   register struct egg *egg = (struct egg *) iptr;
	free (egg->key); egg->key = NULL;
}

/*
 * hash__Clear: clear hash table
 */

void hash__Clear(self)
struct hash *self;
{
    int i;

    if (!self) {
        _InvalidArg ("hash__Clear: Invalid Arguments\n");
	return;
    }

    for (i=0;i<self->n_buckets;i++) {
	if (self->buckets[i] != NULL)  {
	    _SListTraverse(self->buckets[i], terminator);
	    _SListDestroy(self->buckets[i]);
	    self->buckets[i] = NULL;
	}
    }
}

/*
 * hash__Destroy: destroy hash table
 */

void hash__Destroy(self)
struct hash *self;
{
   if (!self) {
        _InvalidArg ("hash__Destroy: Invalid Arguments\n");
	return;
   }
   hash__Clear(self);
   free (self->buckets);
   free (self);
}

/*
 * PrintAll: Print element
 */

PrintAll(egg)
struct egg *egg;
{
    printf("Egg (%s) contains (%s)\n",egg->key,egg->value);
}

/*
 * hash__Debug: Print hash table
 */

void hash__Debug(self)
struct hash *self;
{
 int i;
    if (!self) {
        _InvalidArg ("hash__Debug: Invalid Argument\n");
	return;
    }
    for (i=0;i<self->n_buckets;i++) {
        if (self->buckets[i] == NULL)
            printf("Bucket %d is NULL\n",i);
        else {
            printf("Bucket %d ------------------>\n",i);
            _SListTraverse(self->buckets[i],PrintAll);
        }
    }
}
