/***************************************************************************/
/*                                                                           */
/* ftplib.c - callable ftp access routines                                 */
/* Copyright (C) 1996, 1997 Thomas Pfau, pfau@cnj.digex.net                */
/*        73 Catherine Street, South Bound Brook, NJ, 08880                   */
/*                                                                           */
/* This program is free software; you can redistribute it and/or modify    */
/* it under the terms of the GNU General Public License as published by    */
/* the Free Software Foundation; either version 2 of the License, or       */
/* (at your option) any later version.                                     */
/*                                                                           */
/* This program is distributed in the hope that it will be useful,         */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of          */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the           */
/* GNU General Public License for more details.                            */
/*                                                                           */
/* You should have received a copy of the GNU General Public License       */
/* along with this program; if not, write to the Free Software             */
/* Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.               */
/*                                                                           */
/***************************************************************************/

#if defined(__unix__)
#include <sys/stat.h>
#include <unistd.h>
#include <time.h>
#endif
#if defined(__unix__) || defined(VMS)
#define GLOBALDEF
#define GLOBALREF extern
#elif defined(_WIN32)
#include <windows.h>
#define GLOBALDEF __declspec(dllexport)
#define GLOBALREF __declspec(dllimport)
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#if defined(__unix__)
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#elif defined(VMS)
#include <types.h>
#include <socket.h>
#include <in.h>
#include <netdb.h>
#elif defined(_WIN32)
#include <winsock.h>
#endif

#if !defined(_WIN32)
#include "ftplib.h"
#endif

#if defined(_WIN32)
#define SETSOCKOPT_OPTVAL_TYPE (const char *)
#else
#define SETSOCKOPT_OPTVAL_TYPE (void *)
#endif

#define ACCEPT_TIMEOUT 30

netbuf *DefaultNetbuf;

static char *version =
    "ftplib Release 2 3/15/97, copyright 1996, 1997 Thomas Pfau";

GLOBALDEF int ftplib_debug = 0;

GLOBALDEF char ftp_reason[256];

#if defined(__unix__) || defined(VMS)
#define net_read read
#define net_write write
#define net_close close
#elif defined(_WIN32)
#define net_read(x,y,z) recv(x,y,z,0)
#define net_write(x,y,z) send(x,y,z,0)
#define net_close closesocket
#endif

#if defined(VMS)
/*
 * VAX C does not supply a memccpy routine so I provide my own
 */
void *memccpy(void *dest, const void *src, int c, size_t n)
{
    int i=0;
    unsigned char *ip=src,*op=dest;
    while (i < n)
    {
        if ((*op++ = *ip++) == c)
            break;
        i++;
    }
    if (i == n)
        return NULL;
    return op;
}
#endif

/*
 * read a line of text
 *
 * return -1 on error or bytecount
 */
static int readline(char *buf,int max,netbuf *ctl)
{
    int x,retval = 0;
    char *end;
    int eof = 0;

    if (max == 0)
        return 0;
    do
    {
            if (ctl->cavail > 0)
            {
            x = (max >= ctl->cavail) ? ctl->cavail : max-1;
            end = memccpy(buf,ctl->cget,'\n',x);
            if (end != NULL)
                x = end - buf;
            retval += x;
            buf += x;
            *buf = '\0';
            max -= x;
            ctl->cget += x;
            ctl->cavail -= x;
            if (end != NULL)
                    break;
            }
            if (max == 1)
            {
            *buf = '\0';
            break;
            }
            if (ctl->cput == ctl->cget)
            {
            ctl->cput = ctl->cget = ctl->buf;
            ctl->cavail = 0;
            ctl->cleft = FTP_BUFSIZ;
            }
        if (eof)
        {
            if (retval == 0)
                retval = -1;
            break;
        }
            if ((x = net_read(ctl->handle,ctl->cput,ctl->cleft)) == -1)
            {
            perror("read");
            retval = -1;
            break;
            }
        if (x == 0)
            eof = 1;
            ctl->cleft -= x;
            ctl->cavail += x;
            ctl->cput += x;
    }
    while (1);
    return retval;
}

/*
 * read a response from the server
 *
 * return 0 if first char doesn't match
 * return 1 if first char matches
 */
static int readresp(char c, netbuf *nControl)
{
    char match[5];
    if (readline(nControl->response,256,nControl) == -1)
    {
        perror("Control socket read failed");
        return 0;
    }
    if (ftplib_debug > 1)
        fprintf(stderr,"%s",nControl->response);
    if (nControl->response[3] == '-')
    {
        strncpy(match,nControl->response,3);
        match[3] = ' ';
        match[4] = '\0';
        do
        {
            if (readline(nControl->response,256,nControl) == -1)
            {
                perror("Control socket read failed");
                return 0;
            }
            if (ftplib_debug > 1)
                fprintf(stderr,"%s",nControl->response);
        }
        while (strncmp(nControl->response,match,4));
    }
    if (nControl->response[0] == c)
        return 1;
    return 0;
}

/*
 * FtpInit for stupid operating systems that require it (Windows NT)
 */
GLOBALDEF void FtpInit(void)
{
#if defined(_WIN32)
    WORD wVersionRequested;
    WSADATA wsadata;
    int err;
    wVersionRequested = MAKEWORD(1,1);
    if ((err = WSAStartup(wVersionRequested,&wsadata)) != 0)
        fprintf(stderr,"Network failed to start: %d\n",err);
#endif
}

/*
 * FtpLastResponse - return a pointer to the last response received
 */
GLOBALDEF char *FtpLastResponse(netbuf *nControl)
{
    return nControl->response;
}

/*
 * FtpConnect - connect to remote server
 *
 * return 1 if connected, 0 if not
 */
GLOBALDEF int FtpConnect(char *host, netbuf **nControl)
{
    int sControl;
    struct sockaddr_in sin;
    struct hostent *phe;
    struct servent *pse;
    int on=1;
    netbuf *ctrl;

    memset(&sin,0,sizeof(sin));
    sin.sin_family = AF_INET;
    *nControl=NULL;
    ctrl = calloc(1,sizeof(netbuf));
    if (ctrl == NULL)
    {
        perror("calloc");
        close(sControl);
        return 0;
    }
#if defined(VMS)
    sin.sin_port = htons(21);
#else
    if ((pse = getservbyname("ftp","tcp")) == NULL)
    {
        strcpy(ctrl->response,"Ftplib[internal error]: getservbyname");
        return 0;
    }
    sin.sin_port = pse->s_port;
#endif
    if ((phe = gethostbyname(host)) == NULL)
    {
        strcpy(ctrl->response,"Ftplib[internal error]: gethostbyname");
        return 0;
    }
    memcpy((char *)&sin.sin_addr, phe->h_addr, phe->h_length);
    sControl = socket(PF_INET, SOCK_STREAM, IPPROTO_TCP);
    if (sControl == -1)
    {
        strcpy(ctrl->response,"Ftplib[internal error]: socket");
        return 0;
    }
    if (setsockopt(sControl,SOL_SOCKET,SO_REUSEADDR,
                   SETSOCKOPT_OPTVAL_TYPE &on, sizeof(on)) == -1)
    {
        strcpy(ctrl->response,"Ftplib[internal error]: setsockopt");
        close(sControl);
        return 0;
    }
    if (connect(sControl, (struct sockaddr *)&sin, sizeof(sin)) == -1)
    {
        strcpy(ctrl->response,"Ftplib[internal error]: connect");
        close(sControl);
        return 0;
    }
    ctrl->handle = sControl;
    if (readresp('2', ctrl) == 0)
    {
        close(sControl);
        free(ctrl);
        return 0;
    }
    *nControl = ctrl;
    return 1;
}

/*
 * FtpSendCmd - send a command and wait for expected response
 *
 * return 1 if proper response received, 0 otherwise
 */
static int FtpSendCmd(char *cmd, char expresp, netbuf *nControl)
{
    char buf[256];
    if (ftplib_debug > 2)
        fprintf(stderr,"%s\n",cmd);
    sprintf(buf,"%s\r\n",cmd);
    if (net_write(nControl->handle,buf,strlen(buf)) <= 0)
    {
        perror("write");
        return 0;
    }    
    return readresp(expresp, nControl);
}

/*
 * FtpLogin - log in to remote server
 *
 * return 1 if logged in, 0 otherwise
 */
GLOBALDEF int FtpLogin(char *user, char *pass, netbuf *nControl)
{
    char tempbuf[64];
    sprintf(tempbuf,"user %s",user);
    if (!FtpSendCmd(tempbuf,'3',nControl))
        return 0;
    sprintf(tempbuf,"pass %s",pass);
    return FtpSendCmd(tempbuf,'2',nControl);
}

/*
 * FtpPort - set up data connection
 *
 * return 1 if successful, 0 otherwise
 */
static int FtpPort(netbuf *nControl)
{
    int sData;
    union {
        struct sockaddr sa;
        struct sockaddr_in in;
    } sin;
    struct linger lng = { 0, 0 };
    int l;
    char buf[64];
    int on=1;
    char *cp;
    int v[6];

    l = sizeof(sin);
    memset(&sin, 0, l);
    sin.in.sin_family = AF_INET;
    if (!FtpSendCmd("pasv",'2',nControl))
        return -1;
    cp = strchr(nControl->response,'(');
    if (cp == NULL)
        return -1;
    cp++;
    sscanf(cp,"%d,%d,%d,%d,%d,%d",&v[2],&v[3],&v[4],&v[5],&v[0],&v[1]);
    (unsigned char)sin.sa.sa_data[2] = v[2];
    (unsigned char)sin.sa.sa_data[3] = v[3];
    (unsigned char)sin.sa.sa_data[4] = v[4];
    (unsigned char)sin.sa.sa_data[5] = v[5];
    (unsigned char)sin.sa.sa_data[0] = v[0];
    (unsigned char)sin.sa.sa_data[1] = v[1];
    sData = socket(PF_INET,SOCK_STREAM,IPPROTO_TCP);
    if (sData == -1)
    {
        perror("socket");
        return -1;
    }
    if (setsockopt(sData,SOL_SOCKET,SO_REUSEADDR,
                   SETSOCKOPT_OPTVAL_TYPE &on,sizeof(on)) == -1)
    {
        perror("setsockopt");
        net_close(sData);
        return -1;
    }
    if (setsockopt(sData,SOL_SOCKET,SO_LINGER,
                   SETSOCKOPT_OPTVAL_TYPE &lng,sizeof(lng)) == -1)
    {
        perror("setsockopt");
        net_close(sData);
        return -1;
    }
    if (connect(sData, &sin.sa, sizeof(sin.sa)) == -1)
    {
        perror("connect");
        net_close(sData);
        return -1;
    }
    return sData;
}

/*
 * FtpSite - send a SITE command
 *
 * return 1 if command successful, 0 otherwise
 */
GLOBALDEF int FtpSite(char *cmd, netbuf *nControl)
{
    char buf[256];
    sprintf(buf,"site %s",cmd);
    if (!FtpSendCmd(buf,'2',nControl))
        return 0;
    return 1;
}

/*
 * FtpMkdir - create a directory at server
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpMkdir(char *path, netbuf *nControl)
{
    char buf[256];
    sprintf(buf,"mkd %s",path);
    if (!FtpSendCmd(buf,'2', nControl))
        return 0;
    return 1;
}

/*
 * FtpPwd - get path at remote
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpPwd(char *path, netbuf *nControl)
{
    char buf[256];
    char *pstr=nControl->response;
    if (!FtpSendCmd("pwd",'2',nControl))
        return 0;
    while(*pstr!='\"' && *pstr!=0) pstr++;
    if(*pstr==0) return 0;
    pstr++;
    while(*pstr!=0 && *pstr!='\"')
         *path++=*pstr++;
    *path=0;
    return 1;
}

/*
 * FtpChdir - change path at remote
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpChdir(char *path, netbuf *nControl)
{
    char buf[256];
    sprintf(buf,"cwd %s",path);
    if (!FtpSendCmd(buf,'2',nControl))
        return 0;
    return 1;
}

/*
 * FtpCdup - change path to upper dir ('cd ..')
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpCdup(netbuf *nControl)
{
    if (!FtpSendCmd("cwd ..",'2',nControl))
        return 0;
    return 1;
}

GLOBALDEF int FtpRmdir(char *path, netbuf *nControl)
{
    char buf[256];
    sprintf(buf,"RMD %s",path);
    if (!FtpSendCmd(buf,'2',nControl))
        return 0;
    return 1;
}

/*
 * FtpNlst - issue an NLST command and write response to output
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpNlst(char *outputfile, char *path, netbuf *nControl)
{
    int sData,l;
    char buf[256];
    char *dbuf;
    FILE *output;

    if (!FtpSendCmd("type A",'2', nControl))
        return 0;
    if ((sData = FtpPort(nControl)) == -1)
        return 0;
    strcpy(buf,"nlst");
    if (path != NULL)
        sprintf(buf+strlen(buf)," %s",path);
    if (!FtpSendCmd(buf,'1', nControl))
        return 0;
    if (outputfile == NULL)
        output = stdout;
    else
    {
        output = fopen(outputfile,"w");
        if (output == NULL)
        {
            perror(outputfile);
            output = stdout;
        }
    }
    dbuf = malloc(FTP_BUFSIZ);
    while ((l = net_read(sData,dbuf,FTP_BUFSIZ)) > 0)
        fwrite(dbuf,1,l,output);
    if (outputfile != NULL)
        fclose(output);
    free(dbuf);
    shutdown(sData,2);
    net_close(sData);
    return readresp('2', nControl);
}

/*
 * FtpDir - issue a LIST command and write response to output
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpDir(char *outputfile, char *path, netbuf *nControl)
{
    int sData,l;
    char buf[256];
    char *dbuf;
    FILE *output;

    if (!FtpSendCmd("type A",'2', nControl))
        return 0;
    if ((sData = FtpPort(nControl)) == -1)
        return 0;
    strcpy(buf,"list");
    if (path != NULL)
        sprintf(buf+strlen(buf)," %s",path);
    if (!FtpSendCmd(buf,'1', nControl))
        return 0;
    if (outputfile == NULL)
        output = stdout;
    else
    {
        output = fopen(outputfile,"w");
        if (output == NULL)
        {
            perror(outputfile);
            output = stdout;
        }
    }
    dbuf = malloc(FTP_BUFSIZ);
    while ((l = net_read(sData,dbuf,FTP_BUFSIZ)) > 0)
        fwrite(dbuf,1,l,output);
    if (outputfile != NULL)
        fclose(output);
    free(dbuf);
    shutdown(sData,2);
    net_close(sData);
    return readresp('2', nControl);
}

int Ftp_convert_attr(char *sa)
{
 int mode=0,mask;
 if(*sa=='d')
   mode=S_IFDIR;
 if(*sa=='l')
   mode=S_IFLNK;
 sa++;
 mask=0400;
 while(mask)
 {
   if(*sa!='-') mode|=mask;
   mask>>=1;
   sa++;
 }
 return mode;
}

char *Ftp_getword(char *from,char* to)
{
 while(*from==' ' || *from=='\r' || *from=='\n') from++;
 while(*from!=' ' && *from!='\0' && *from!='\r' && *from!='\n') *to++=*from++;
 *to='\0';
 return from;
}

char *Ftp_nameword(char *from,char* to)
{
 while(*from==' ' || *from=='\r' || *from=='\n') from++;
 if(*from!='\'') 
        return NULL;
 from++;
 while(*from!='\0' && *from!='\r' && *from!='\n' && *from!='\'') *to++=*from++;
 *to='\0';
 return from+1;
}

int Ftp_getmonth(char *s)
{
 char *m[12]={"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
 int i;
 for(i=0;i<12;i++)
  if(strcmp(s,m[i])==0) return i;
 return 0;
}

void Ftp_gettime(char *s,struct tm *ttm)
{
 char *s2=s;
 while(*s2)
  if(*s2==':')    /*Wow! we found : - this is time in HH:MM format*/
  {
   *s2=0;
   sscanf(s,"%d",&ttm->tm_hour);
   ttm->tm_hour--;
   sscanf(s2+1,"%d",&ttm->tm_min);
   ttm->tm_min--;
   return;
  } else s2++;
 sscanf(s,"%d",&ttm->tm_year);     /*If we here then this is not a time, this is year i hope.*/
 ttm->tm_year-=1900;
}

GLOBALDEF int FtpLS(char *outputfile, char *path, netbuf *nControl)
{
    int sData,l;
    char buf[256];
    char buf2[100];
    char *dbuf,*buf1,*buf3;
    FILE *output,*fp;
    long fsiz;
    int mode;
    struct tm ttm;
    time_t tim;

    if (!FtpSendCmd("type A",'2', nControl))
        return 0;
    if ((sData = FtpPort(nControl)) == -1)
        return 0;
    strcpy(buf,"list");
    if (path != NULL)
        sprintf(buf+strlen(buf)," %s",path);
    if (!FtpSendCmd(buf,'1', nControl))
        return 0;
    if (outputfile == NULL)
        output = stdout;
    else
    {
        output = fopen(outputfile,"w");
        if (output == NULL)
        {
            perror(outputfile);
            output = stdout;
        }
    }
    dbuf = malloc(FTP_BUFSIZ);
    fp=fdopen(sData,"r");
    if(fp==NULL)
    {
     perror("fdopen");
     net_close(sData);
     return 0;
    } 
    while (fgets(dbuf,FTP_BUFSIZ,fp))
    {
      l=strlen(dbuf);
      dbuf[l-1]=0;l--;
      buf1=Ftp_getword(dbuf,buf2);
      if(strcmp(buf2,"total")==0) continue;        /*If "total" then skip this line*/
      mode=Ftp_convert_attr(buf2);                 /*Otherwise this is attributes*/

      buf1=Ftp_getword(buf1,buf2);                 /*Skip this*/
      buf1=Ftp_getword(buf1,buf2);                 /*owner ->Skip this*/
      buf1=Ftp_getword(buf1,buf2);                 /*group ->Skip this*/
      buf1=Ftp_getword(buf1,buf2);                 /*file size*/
      sscanf(buf2,"%d",&fsiz);

      tim=time(&tim);
      memcpy(&ttm,localtime(&tim),sizeof(struct tm));
      buf1=Ftp_getword(buf1,buf2);                 /*Month*/
      ttm.tm_mon=Ftp_getmonth(buf2);
      buf1=Ftp_getword(buf1,buf2);                 /*Day of month*/
      sscanf(buf2,"%d",&ttm.tm_mday);
      buf1=Ftp_getword(buf1,buf2);                 /*Time*/
      Ftp_gettime(buf2,&ttm);
      buf3=buf2;
      sscanf(buf1,"%[^\n]",buf2);
      l=strlen(buf2);
      if(buf2[l-1]=='\r')
              buf2[l-1]=0;
      tim=mktime(&ttm);
      if(strcmp(buf2,"."))        //Don't put '.' dir to file, skip it.
              fprintf(output,"%d %o %X %s\n",fsiz,mode,tim,buf2);
    }  

    if (outputfile != NULL)
        fclose(output);
    free(dbuf);
    shutdown(sData,2);
    fclose(fp);
    net_close(sData);
    return readresp('2', nControl);
}

/*
 * FtpGet - issue a GET command and write received data to output
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpGet(char *outputfile, char *path, char mode, netbuf *nControl)
{
    int sData;
    int l;
    char buf[256];
    char *dbuf;
    FILE *output;
    int size = 0;

    sprintf(buf,"type %c",mode);
    if (!FtpSendCmd(buf,'2', nControl))
        return 0;
    if ((sData = FtpPort(nControl)) == -1)
        return 0;
    sprintf(buf,"retr %s",path);
    if (!FtpSendCmd(buf,'1',nControl))
        return 0;
    if (dbuf = strchr(nControl->response,'('))
    {
        dbuf++;
        size = atoi(dbuf);
    }
    if (outputfile == NULL)
        output = stdout;
    else
    {
        output = fopen(outputfile,"w");
        if (output == NULL)
        {
            perror(outputfile);
            output = stdout;
        }
    }
    dbuf = malloc(FTP_BUFSIZ);
    if (mode == 'A')
    {
        netbuf *nData;
        int dl;
        nData = calloc(1,sizeof(netbuf));
        if (nData == NULL)
        {
            perror("calloc");
            net_close(sData);
            return 0;
        }
        nData->handle = sData;
        while ((dl = readline(dbuf,FTP_BUFSIZ,nData))!= -1)
        {
            if (strcmp(&dbuf[dl-2],"\r\n") == 0)
            {
                dl -= 2;
                dbuf[dl++] = '\n';
                dbuf[dl] = '\0';
            }
            fwrite(dbuf,1,dl,output);
        }
    }
    else
        while ((l = net_read(sData,dbuf,FTP_BUFSIZ)) > 0)
            fwrite(dbuf,1,l,output);
    if (outputfile != NULL)
            fclose(output);
    free(dbuf);
    shutdown(sData,2);
    net_close(sData);
    return readresp('2', nControl);
}

/*
 * FtpBinGet - issue a GET command and write received data to output
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpBinGet(char *outputfile, char *path, int chmode, netbuf *nControl)
{
    int sData;
    int l;
    char buf[256];
    char *dbuf;
    FILE *output;
    int size = 0,siz=0;

    if (!FtpSendCmd("type I",'2', nControl))
        return 0;
    if ((sData = FtpPort(nControl)) == -1)
        return 0;
    sprintf(buf,"retr %s",path);
    if (!FtpSendCmd(buf,'1',nControl))
        return 0;
    if (dbuf = strchr(nControl->response,'('))
    {
        dbuf++;
        size = atoi(dbuf);
    }
    if (outputfile == NULL)
        output = stdout;
    else
    {
        output = fopen(outputfile,"w");
        if (output == NULL)
        {
            perror(outputfile);
            output = stdout;
        }
    }
    dbuf = malloc(FTP_BUFSIZ);
        while ((l = net_read(sData,dbuf,FTP_BUFSIZ)) > 0)
        {
            fwrite(dbuf,1,l,output);
            siz+=l;
        };
    if(ftplib_debug>1)
      fprintf(stderr,"Size: %d   Actual size: %d\n",size,siz);
    if (outputfile != NULL)
            fclose(output);
    free(dbuf);
    shutdown(sData,2);
    net_close(sData);
    chmod(outputfile,chmode);
    return readresp('2', nControl);
}

/*
 * FtpPut - issue a PUT command and send data from input
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpPut(char *inputfile, char *path, char mode, netbuf *nControl)
{
    int sData;
    int l;
    char buf[256];
    char *dbuf;
    FILE *input;

    if (inputfile == NULL)
    {
        fprintf(stderr,"Must specify a file name to send\n");
        return 0;
    }
    input = fopen(inputfile,"r");
    if (input == NULL)
    {
        perror(inputfile);
        return 0;
    }
    sprintf(buf,"type %c",mode);
    if (!FtpSendCmd(buf,'2',nControl))
        return 0;
    if ((sData = FtpPort(nControl)) == -1)
        return 0;
    sprintf(buf,"stor %s",path);
    if (!FtpSendCmd(buf,'1',nControl))
        return 0;
    dbuf = malloc(FTP_BUFSIZ);
    if (mode == 'A')
    {
        netbuf *nData;
        int dl;
        nData = calloc(1,sizeof(netbuf));
        if (nData == NULL)
        {
            perror("calloc");
            net_close(sData);
            fclose(input);
            return 0;
        }
        nData->handle = fileno(input);
        while ((dl = readline(dbuf,FTP_BUFSIZ-1,nData))!= -1)
        {
            if (dbuf[dl-1] == '\n')
            {
                dbuf[dl-1] = '\r';
                dbuf[dl++] = '\n';
                dbuf[dl] = '\0';
            }
            if (net_write(sData,dbuf,dl) == -1)
            {
                perror("write");
                break;
            }
        }
        free(nData);
    }
    else
        while ((l = fread(dbuf,1,FTP_BUFSIZ,input)) != 0)
            if (net_write(sData,dbuf,l) == -1)
            {
                perror("write");
                break;
            }
    fclose(input);
    free(dbuf);
    if (shutdown(sData,2) == -1)
        perror("shutdown");
    if (net_close(sData) == -1)
        perror("close");
    return readresp('2', nControl);
}

/*
 * FtpRename - rename a file at remote
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpRename(char *src, char *dst, netbuf *nControl)
{
    char cmd[256];
    sprintf(cmd,"RNFR %s",src);
    if (!FtpSendCmd(cmd,'3',nControl))
        return 0;
    sprintf(cmd,"RNTO %s",dst);
    if (!FtpSendCmd(cmd,'2',nControl))
        return 0;
    return 1;
}

/*
 * FtpDelete - delete a file at remote
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF int FtpDelete(char *fnm, netbuf *nControl)
{
    char cmd[256];
    sprintf(cmd,"DELE %s",fnm);
    if (!FtpSendCmd(cmd,'2', nControl))
        return 0;
    return 1;
}

/*
 * FtpQuit - disconnect from remote
 *
 * return 1 if successful, 0 otherwise
 */
GLOBALDEF void FtpQuit(netbuf *nControl)
{
    FtpSendCmd("quit",'2',nControl);
    net_close(nControl->handle);
    free(nControl);
}
